//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Renderer2D.h"

#include <FeatStd/Util/StaticObject.h>
#include <Candera/Macros.h>
#include <Candera/Engine2D/Core/Scene2D.h>
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Renderer2DListener.h>
#include <Candera/Engine2D/Core/Camera2DRenderStrategy.h>
#include <Candera/Engine2D/Core/Camera2DListener.h>
#include <Candera/System/Monitor/MonitorPublicIF.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>
#include <Candera/Engine2D/Core/BitmapImage2D.h>
#include <Candera/Engine2D/Core/VertexBuffer2D.h>
#include <Candera/Engine2D/Core/RenderDirectionHelper.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine2D);

    bool Renderer2D::s_isRenderingCamera = false;
    Renderer2D::Statistics Renderer2D::s_statistics;
    Matrix3x2 Renderer2D::s_sourceSurfaceTransformationMatrix;
    Vector2 Renderer2D::s_cameraTranslationOffset = Vector2(0.0F, 0.0F);
#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
    Vector2 Renderer2D::s_preTranslate = Vector2(0.0F, 0.0F);
#endif

    void Renderer2D::OnCameraRenderTargetChanged(const Camera2D* camera, RenderTarget2D* previousRenderTarget)
    {
        if ((camera->GetCameraRenderStrategy() != 0) && (!camera->GetCameraRenderStrategy()->IsRenderPassCompleted())) {
            FEATSTD_LOG_WARN("RenderTarget3D of Camera changed while the render pass is not completed. Incomplete frame might be rendered.");
            if ((previousRenderTarget != 0) && (previousRenderTarget->m_activeCameraCount != 0)) {
                --previousRenderTarget->m_activeCameraCount;
                if (previousRenderTarget->m_activeCameraCount == 0) {
                    previousRenderTarget->m_currentRenderPass = RenderTarget2D::BeforeFirstRenderPass;
                }
            }
            camera->GetCameraRenderStrategy()->SetRenderPassCompleted(true);
        }
    }

    void Renderer2D::OnCameraRenderStrategyChanged(Camera2D* camera, const Camera2DRenderStrategy* previousCameraRenderStrategy)
    {
        RenderTarget2D* renderTarget = camera->GetRenderTarget();
        bool isCameraActive = (renderTarget != 0) &&
            (renderTarget->m_activeCameraCount != 0) &&
            (!previousCameraRenderStrategy->IsRenderPassCompleted());

        if (isCameraActive) {
            FEATSTD_LOG_WARN("RenderStrategy of Camera changed while the render pass is not completed. Incomplete frame might be rendered.");
            --renderTarget->m_activeCameraCount;
            if (renderTarget->m_activeCameraCount == 0) {
                renderTarget->m_currentRenderPass = RenderTarget2D::BeforeFirstRenderPass;
            }
        }

        if (camera->GetCameraRenderStrategy() != 0) {
            camera->GetCameraRenderStrategy()->SetRenderPassCompleted(true);
        }
    }

    class Renderer2D::RendererCameraListener: public Camera2DListener {
    public:
        virtual void OnPreRender(Camera2D* /*camera*/) override {}
        virtual void OnPostRender(Camera2D* /*camera*/) override {}

        virtual void OnRenderTargetChanged(Camera2D* camera, RenderTarget2D* previousRenderTarget) override
        {
            Renderer2D::OnCameraRenderTargetChanged(camera, previousRenderTarget);
        }

        virtual void OnCameraRenderStrategyChanged(Camera2D* camera, Camera2DRenderStrategy* previousCameraRenderStrategy) override
        {
            Renderer2D::OnCameraRenderStrategyChanged(camera, previousCameraRenderStrategy);
        }
    };

    void Renderer2D::RenderCamera(Camera2D* camera, bool resetStatistics, const Rectangle& dirtyArea)
    {
        if (resetStatistics) {
            s_statistics.Reset();
        }

        s_isRenderingCamera = true;
        RenderCameraInternal(camera, dirtyArea);
        s_isRenderingCamera = false;
    }

    Camera2D* Renderer2D::s_activeCamera = 0;

    /******************************************************************************
     *  RenderCamera
     ******************************************************************************/
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
    void Renderer2D::RenderCameraInternal(Camera2D* camera, const Rectangle& dirtyArea)
    {
        MONITOR_HANDLE_RUNNING();

        if ((camera != 0) && camera->IsEffectiveRenderingEnabled()) {

            CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderCamera2D, camera->GetName())); // don't record, if not rendered

            Matrix3x2 cameraTransformation(camera->GetViewMatrix());
            Camera2DRenderStrategy* const cameraRenderStrategy = camera->GetCameraRenderStrategy();
            const bool isCameraRenderStrategyAvailable = cameraRenderStrategy != 0;

            Scene2D* scene = camera->GetScene();
            if (scene != 0) {
                RenderTarget2D* renderTarget = camera->GetRenderTarget();
                if ((renderTarget != 0) && (renderTarget->Activate())) {
                    s_activeCamera = camera;
                    s_cameraTranslationOffset.SetZero();
                    PreRenderNodes(scene);
                    RenderOrder2D& renderOrder = scene->GetRenderOrder();
                    renderOrder.Validate();

                    {
                        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::ActivateCamera2D, camera->GetName()));
                        NotifyListenersOnCameraPreActivate(camera);
                        camera->NotifyListenersOnPreActivate();
                        renderTarget->Sync();
                        renderTarget->SafeBeginDraw(renderTarget);
                        camera->NotifyListenersOnPostActivate();
                        NotifyListenersOnCameraPostActivate(camera);
                    }

                    camera->NotifyListenersOnPreRender();

                    const Rectangle& cameraViewportRect = camera->GetViewport();
                    const bool clear = camera->IsClearColorEnabled() && (!(isCameraRenderStrategyAvailable && (!cameraRenderStrategy->IsRenderPassCompleted())));
                    Rectangle drawRect = cameraViewportRect;
                    Rectangle clearRect = cameraViewportRect;

                    const bool hasCameraRotation = (0.0F != camera->GetRotation());
                    const bool isDirtyAreaFullscreen = ((0.0F == dirtyArea.GetLeft()) && (0.0F == dirtyArea.GetTop()) &&
                        (-1.0F == dirtyArea.GetWidth()) && (-1.0F == dirtyArea.GetHeight()));
                    const bool useDirtyArea = (!hasCameraRotation) && (!isDirtyAreaFullscreen);
                    if (hasCameraRotation && (!isDirtyAreaFullscreen)) {
                        FEATSTD_LOG_WARN("A dirty area rectangle was supplied to the Renderer2D, but cannot be used because the Camera2D is rotated.");
                    }

                    ContextHandle2D context = renderTarget->Get2DContextHandle();

                    if (camera->IsScissoringEnabled() || useDirtyArea) {
                        Rectangle scissorRect = camera->IsScissoringEnabled() ? camera->GetScissorRectangle() : dirtyArea;
                        if (camera->IsScissoringEnabled() && useDirtyArea) {
                            scissorRect.Intersect(dirtyArea);
                        }

                        if (RenderDevice2D::SetScissorEnabled(context, true)) {
                            static_cast<void>(RenderDevice2D::SetScissorRectangle(context, scissorRect));
                        }
                        else {
                            // Scissor is not supported, use the viewport instead
                            if ((drawRect.GetWidth() < 0.0F) && (drawRect.GetHeight() < 0.0F)) {
                                drawRect = scissorRect;
                            }
                            else {
                                drawRect.Intersect(scissorRect);
                            }
                        }

                        if ((clearRect.GetWidth() < 0.0F) && (clearRect.GetHeight() < 0.0F)) {
                            clearRect = scissorRect;
                        }
                        else {
                            clearRect.Intersect(scissorRect);
                        }

                        if (useDirtyArea) {
                            Int renderTargetArea = renderTarget->GetWidth() * renderTarget->GetHeight();
                            Float drawArea = drawRect.GetWidth() * drawRect.GetHeight();
                            Float dirtyAreaFactor = drawArea / static_cast<Float>(renderTargetArea);
                            s_statistics.AverageDirtyAreaFactor *= static_cast<Float>(s_statistics.m_dirtyAreaCounter);
                            s_statistics.AverageDirtyAreaFactor += dirtyAreaFactor;
                            if (0 != ++s_statistics.m_dirtyAreaCounter) {
                                s_statistics.AverageDirtyAreaFactor /= static_cast<Float>(s_statistics.m_dirtyAreaCounter);
                            }
                            else {
                                s_statistics.AverageDirtyAreaFactor = 0.0F; // Overflow happened, reset the value.
                            }
                        }

                        s_cameraTranslationOffset.SetX(cameraViewportRect.GetLeft() - drawRect.GetLeft());
                        s_cameraTranslationOffset.SetY(cameraViewportRect.GetTop() - drawRect.GetTop());
                        cameraTransformation.Translate(s_cameraTranslationOffset.GetX(), s_cameraTranslationOffset.GetY());
                    }
                    else {
                        static_cast<void>(RenderDevice2D::SetScissorEnabled(context, false));
                    }

                    if (clear) {
                        MONITOR_INTERNAL_CLEAR_2D_BEGIN();
                        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::ClearCamera2D, camera->GetName()));
                        NotifyListenersOnCameraPreClear(camera);
                        camera->NotifyListenersOnPreClear();

                        static_cast<void>(RenderDevice2D::SetActiveArea(context,
                                          RenderDevice2D::DestinationSurface,
                                          clearRect.GetLeft(),
                                          clearRect.GetTop(),
                                          clearRect.GetWidth(),
                                          clearRect.GetHeight()));

                        const Color& color = camera->GetClearColor();
                        static_cast<void>(Clear(context,
                                  color.GetRed(),
                                  color.GetGreen(),
                                  color.GetBlue(),
                                  color.GetAlpha()));

                        camera->NotifyListenersOnPostClear();
                        NotifyListenersOnCameraPostClear(camera);
                        MONITOR_INTERNAL_CLEAR_2D_END();
                    }

                    if ((drawRect != clearRect) || (!clear)) {
                        static_cast<void>(RenderDevice2D::SetActiveArea(context,
                                          RenderDevice2D::DestinationSurface,
                                          drawRect.GetLeft(),
                                          drawRect.GetTop(),
                                          drawRect.GetWidth(),
                                          drawRect.GetHeight()));
                    }

                    renderOrder.IterationBegin();

                    // If camera render strategy is available then move render order iterator to the node where to start rendering from.
                    if (isCameraRenderStrategyAvailable) {
                        if (cameraRenderStrategy->IsRenderPassCompleted()) {
                            // Start rendering with first node.
                            cameraRenderStrategy->SetPreviousNodeRendered(0);
                            cameraRenderStrategy->OnRenderPassBegins();
                        }
                        else {
                            // Iterate to the successor of previous node rendered before the render pass has been paused by CameraRenderStrategy.
                            const Node2D* const previousNodeRendered = cameraRenderStrategy->GetPreviousNodeRendered();
                            Node2D* currentNode = 0;
                            while ((currentNode != previousNodeRendered) && (!renderOrder.IterationIsEnd())) {
                                currentNode = &renderOrder.IterationGetNode();
                                renderOrder.IterationMoveToNextNode();
                            }
                            cameraRenderStrategy->OnRenderPassResumes();
                        }
                    }

#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
                    bool resetActiveArea = false;
#endif
                    bool isRenderPassCompleted = true;
                    // Loop RenderOrder to render all Nodes. Only a RenderCameraStrategy object can interrupt the render loop.
                    for (/* Node from where rendering starts */; !renderOrder.IterationIsEnd(); renderOrder.IterationMoveToNextNode()) {
                        RenderNode& renderNode = renderOrder.IterationGetNode();

                        if (renderNode.IsEffectiveRenderingEnabled() && renderNode.IsInScopeOf(camera->GetScopeMask())) {
                            Matrix3x2 modelViewMatrix = renderNode.GetWorldTransform() * cameraTransformation;

                            if (renderNode.IsSnapToDevicePixelEnabled()) {
                                Int32 x = Int32(0.5F + modelViewMatrix.Get(2, 0));
                                modelViewMatrix.Set(2, 0, Float(x));
                                Int32 y = Int32(0.5F + modelViewMatrix.Get(2, 1));
                                modelViewMatrix.Set(2, 1, Float(y));
                            }

#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
                            s_preTranslate.SetZero();
                            if (renderNode.HasClippingRect()) {
                                Rectangle screenSpaceClippingRect(drawRect);
                                CalculateScreenSpaceClippingRect(renderNode, cameraTransformation, screenSpaceClippingRect);

                                static_cast<void>(RenderDevice2D::SetActiveArea(context,
                                                  RenderDevice2D::DestinationSurface,
                                                  screenSpaceClippingRect.GetLeft(),
                                                  screenSpaceClippingRect.GetTop(),
                                                  screenSpaceClippingRect.GetWidth(),
                                                  screenSpaceClippingRect.GetHeight()));

                                 // compute translation for render node based on original draw area (viewport and possible scissoring rect.) and the clipping area
                                const Float preTranslateX = drawRect.GetLeft() - screenSpaceClippingRect.GetLeft();
                                const Float preTranslateY = drawRect.GetTop() - screenSpaceClippingRect.GetTop();
                                modelViewMatrix.PreTranslate(preTranslateX, preTranslateY);

                                // set pre translation for further use in mask effects
                                s_preTranslate.SetX(preTranslateX);
                                s_preTranslate.SetY(preTranslateY);
                                resetActiveArea = true;
                            }
                            else if (resetActiveArea) {
                                static_cast<void>(RenderDevice2D::SetActiveArea(context,
                                    RenderDevice2D::DestinationSurface,
                                    drawRect.GetLeft(),
                                    drawRect.GetTop(),
                                    drawRect.GetWidth(),
                                    drawRect.GetHeight()));

                                resetActiveArea = false;
                            }
                            else {
                                /* empty to satisfy LINT */
                            }
#endif
                            // If Camera2DRenderStrategy is available, check if render pass shall proceed with current node.
                            if (isCameraRenderStrategyAvailable) {
                                const Camera2DRenderStrategy::RenderPassAction action = cameraRenderStrategy->GetRenderPassAction(&renderNode);
                                if (action == Camera2DRenderStrategy::ProceedRenderPass) {
                                    cameraRenderStrategy->SetPreviousNodeRendered(&renderNode);
                                }
                                else if (action == Camera2DRenderStrategy::PauseRenderPass) {
                                    isRenderPassCompleted = false;
                                    break;
                                }
                                else if (action == Camera2DRenderStrategy::SkipNode) {
                                    continue;
                                }
                                else { /*if (action == CameraRenderStrategy::StopRenderPass)*/
                                    break;
                                }
                            }
                            Internal::RenderDirectionHelper::SetCurrentRightToLeft(Internal::RenderDirectionHelper::IsRightToLeft(renderNode));
                            NotifyRendererListenersOnNodePreRender(&renderNode, &modelViewMatrix);
                            renderNode.Render(renderTarget, modelViewMatrix);
                            NotifyRendererListenersOnNodePostRender(&renderNode, &modelViewMatrix);
                        }
                    }

                    renderTarget->SafeEndDraw();

                    if (isCameraRenderStrategyAvailable) {
                        cameraRenderStrategy->SetRenderPassCompleted(isRenderPassCompleted);
                    }

                    bool isSwapRequired = (isRenderPassCompleted &&
                        (!renderTarget->IsAutoSwapEnabled()) &&
                        (camera->IsSwapEnabled()));

                    if (isSwapRequired) {
                        renderTarget->SwapBuffers();
                    }

                    camera->NotifyListenersOnPostRender();

                    NotifyListenersOnCameraPostRender(camera);
                    s_activeCamera = 0;
                    s_cameraTranslationOffset.SetZero();
                }
            }
        }
    }
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

    /******************************************************************************
     *  RenderAllCameras
     ******************************************************************************/
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
    void Renderer2D::RenderAllCameras(const RenderTarget2D* renderTarget /* = 0 */)
    {
#ifdef FEATSTD_MONITOR_ENABLED
        MONITOR_HANDLE_PAUSED_ID(2);
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::Frame_2D));
#endif

        s_statistics.Reset();

#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker cameraListLock(GetCameraListCriticalSection());
#endif
        CameraList& cameras = GetCameraList();
        const Int cameraListSize = static_cast<Int>(GetCameraCount());

        //Before the first render pass, count number of cameras that will render on
        // each RenderTarget.
        for (Int32 i = 0; i < cameraListSize; i++) {
            RenderTarget2D* localRenderTarget = cameras[i]->GetRenderTarget();
            if ((renderTarget != 0) && (renderTarget != localRenderTarget)) {
                localRenderTarget = 0;
            }
            if ((localRenderTarget != 0) &&
                (localRenderTarget->m_currentRenderPass == RenderTarget2D::BeforeFirstRenderPass)) {
                ++localRenderTarget->m_activeCameraCount;
            }
        }

        //Render each camera in camera list.
        for (Int32 i = 0; i < cameraListSize; i++) {
            Camera2D* camera = cameras[i];
            RenderTarget2D* localRenderTarget = camera->GetRenderTarget();
            if ((renderTarget != 0) && (renderTarget != localRenderTarget)) {
                localRenderTarget = 0;
            }
            Camera2DRenderStrategy* cameraRenderStrategy = camera->GetCameraRenderStrategy();

            //skip camera render if:
            bool skipCamera =
                // - it has no RenderTarget to render on OR
                (localRenderTarget == 0) ||
                //  - this is not the first render pass on the RenderTarget AND
                ((localRenderTarget->m_currentRenderPass >= RenderTarget2D::AfterFirstRenderPass) &&
                //   - it does not support multi pass rendering OR
                 ((cameraRenderStrategy == 0) ||
                 //   - it supports multi pass rendering, but the rendering is complete.
                  cameraRenderStrategy->IsRenderPassCompleted()));
            if (!skipCamera) {
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, localRenderTarget, CANDERA_LINT_REASON_OPERANDNOTNULL);
                //If nothing else has been rendered on the RenderTarget:
                if (localRenderTarget->m_currentRenderPass == RenderTarget2D::BeforeFirstRenderPass) {
                    //Clear RenderTarget if automatic clear is enabled.
                    if (localRenderTarget->IsAutoClearEnabled() && (localRenderTarget->GetSharedClearMode() != 0)) {
                        localRenderTarget->Sync();
                        localRenderTarget->SafeBeginDraw(localRenderTarget);

                        static_cast<void>(RenderDevice2D::SetActiveArea(localRenderTarget->Get2DContextHandle(),
                            RenderDevice2D::DestinationSurface,
                            Float(0),
                            Float(0),
                            Float(-1),
                            Float(-1)));
                        const Color& color = localRenderTarget->GetSharedClearMode()->GetClearColor();
                        static_cast<void>(Clear(localRenderTarget->Get2DContextHandle(),
                            color.GetRed(),
                            color.GetGreen(),
                            color.GetBlue(),
                            color.GetAlpha()));

                        localRenderTarget->SafeEndDraw();
                    }
                    //Mark the first pass on the RenderTarget as started, no further clearing needed.
                    localRenderTarget->m_currentRenderPass = RenderTarget2D::DuringFirstRenderPass;
                }
                //Do the actual rendering.
                RenderCamera(camera, /* resetStatistics */ false);
                //If render pass is completed:
                if ((cameraRenderStrategy == 0) || cameraRenderStrategy->IsRenderPassCompleted()) {
                    //decrease number of cameras that need to further render on the RenderTarget.
                    --localRenderTarget->m_activeCameraCount;
                    //If no more cameras need to render,
                    if (localRenderTarget->m_activeCameraCount == 0) {
                        //if RenderTarget should be swapped automatically,
                        if (localRenderTarget->IsAutoSwapEnabled()) {
                            //swap the RenderTarget.
                            localRenderTarget->SwapBuffers();
                        }
                        //Mark the RenderTarget as fully rendered.
                        localRenderTarget->m_currentRenderPass = RenderTarget2D::AfterLastRenderPass;
                    }
                }
            }

            //RenderTarget should swap if no more cameras need to render.
            bool shouldSwap = (localRenderTarget != 0) &&
                (localRenderTarget->m_activeCameraCount == 0) &&
                (localRenderTarget->m_currentRenderPass != RenderTarget2D::AfterLastRenderPass);

            if (shouldSwap) {
                //if RenderTarget should be swapped automatically:
                if (localRenderTarget->IsAutoSwapEnabled()) {
                    //swap the RenderTarget.
                    localRenderTarget->SwapBuffers();
                }
                //Mark the RenderTarget as fully rendered.
                localRenderTarget->m_currentRenderPass = RenderTarget2D::AfterLastRenderPass;
            }
        }

        //After the full render pass, mark the RenderTargets that are not fully rendered.
        for (Int32 i = 0; i < cameraListSize; i++) {
            RenderTarget2D* localRenderTarget = cameras[i]->GetRenderTarget();
            if ((renderTarget != 0) && (renderTarget != localRenderTarget)) {
                localRenderTarget = 0;
            }
            if (localRenderTarget != 0) {
                if (localRenderTarget->m_currentRenderPass == RenderTarget2D::DuringFirstRenderPass) {
                    localRenderTarget->m_currentRenderPass = RenderTarget2D::AfterFirstRenderPass;
                }
                if (localRenderTarget->m_currentRenderPass == RenderTarget2D::AfterLastRenderPass) {
                    localRenderTarget->m_currentRenderPass = RenderTarget2D::BeforeFirstRenderPass;
                }
            }
        }
    }
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

    /******************************************************************************
     *  GetCameraCount
     ******************************************************************************/
    UInt32 Renderer2D::GetCameraCount()
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker cameraListLock(GetCameraListCriticalSection());
#endif
        return static_cast<UInt32>(GetCameraList().Size());
    }

    /******************************************************************************
     *  GetCamera
     ******************************************************************************/
    Camera2D* Renderer2D::GetCamera(UInt32 index)
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker cameraListLock(GetCameraListCriticalSection());
#endif
        return GetCameraList()[static_cast<Int>(index)];
    }


    typedef Internal::ListenerContainer<Renderer2DListener> ListenerContainer;

    /******************************************************************************
     *  GetListenerContainer
     ******************************************************************************/
    ListenerContainer& GetListenerContainer()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(ListenerContainer, s_listenerContainer);
        return s_listenerContainer;
    }

    ListenerContainer& s_forceInitListenerContainer = GetListenerContainer();

    /******************************************************************************
     *  AddRendererListener
     ******************************************************************************/
    bool Renderer2D::AddRendererListener(Renderer2DListener* listener)
    {
        if (listener != 0) {
            return GetListenerContainer().Add(listener);
        }
        return false;
    }

    /******************************************************************************
     *  RemoveRendererListener
     ******************************************************************************/
    bool Renderer2D::RemoveRendererListener(Renderer2DListener* listener)
    {
        ListenerContainer& listenerContainer = GetListenerContainer();
        if (listenerContainer.Remove(listener)) {
            if (0 == listenerContainer.Size()) {
                listenerContainer.Free();
            }

            return true;
        }

        return false;
    }


    /******************************************************************************
     *  OnNodePreRenderEvent
     ******************************************************************************/
    class Renderer2D::OnNodePreRenderEvent : public ListenerContainer::Event
    {
    public:
        OnNodePreRenderEvent(const RenderNode* node, const Matrix3x2* modelViewMatrix) :
            m_node(node),
            m_modelViewMatrix(modelViewMatrix)
        {}

        virtual void Notify(Renderer2DListener* listener) override
        {
            listener->OnNodePreRender(m_node, m_modelViewMatrix);
        }
    private:
        const RenderNode* m_node;
        const Matrix3x2* m_modelViewMatrix;
    };

    /******************************************************************************
     *  OnNodePostRenderEvent
     ******************************************************************************/
    class Renderer2D::OnNodePostRenderEvent : public ListenerContainer::Event
    {
    public:
        OnNodePostRenderEvent(const RenderNode* node, const Matrix3x2* modelViewMatrix) :
            m_node(node),
            m_modelViewMatrix(modelViewMatrix)
        {}

        virtual void Notify(Renderer2DListener* listener) override
        {
            listener->OnNodePostRender(m_node, m_modelViewMatrix);
        }
    private:
        const RenderNode* m_node;
        const Matrix3x2* m_modelViewMatrix;
    };

    /******************************************************************************
     *  OnCameraPostClearEvent
     ******************************************************************************/
    class Renderer2D::OnCameraPostRenderEvent : public ListenerContainer::Event
    {
    public:
        OnCameraPostRenderEvent(Camera2D *camera) :
            m_camera(camera) {}
        virtual void Notify(Renderer2DListener* listener) {
            listener->OnCameraPostRender(m_camera);
        }
    private:
        Camera2D* m_camera;
    };

    /******************************************************************************
    *  OnCameraPreActivateEvent
    ******************************************************************************/
    class Renderer2D::OnCameraPreActivateEvent : public ListenerContainer::Event
    {
    public:
        OnCameraPreActivateEvent(const Camera2D *camera) :
            m_camera(camera) {}
        virtual void Notify(Renderer2DListener* listener) {
            listener->OnCameraPreActivate(m_camera);
        }
    private:
        const Camera2D* m_camera;
    };

    /******************************************************************************
    *  OnCameraPostActivateEvent
    ******************************************************************************/
    class Renderer2D::OnCameraPostActivateEvent : public ListenerContainer::Event
    {
    public:
        OnCameraPostActivateEvent(const Camera2D *camera) :
            m_camera(camera) {}
        virtual void Notify(Renderer2DListener* listener) {
            listener->OnCameraPostActivate(m_camera);
        }
    private:
        const Camera2D* m_camera;
    };

    /******************************************************************************
    *  OnCameraPreClearEvent
    ******************************************************************************/
    class Renderer2D::OnCameraPreClearEvent : public ListenerContainer::Event
    {
    public:
        OnCameraPreClearEvent(const Camera2D *camera) :
            m_camera(camera) {}
        virtual void Notify(Renderer2DListener* listener) {
            listener->OnCameraPreClear(m_camera);
        }
    private:
        const Camera2D* m_camera;
    };

    /******************************************************************************
    *  OnCameraPostClearEvent
    ******************************************************************************/
    class Renderer2D::OnCameraPostClearEvent : public ListenerContainer::Event
    {
    public:
        OnCameraPostClearEvent(const Camera2D *camera) :
            m_camera(camera) {}
        virtual void Notify(Renderer2DListener* listener) {
            listener->OnCameraPostClear(m_camera);
        }
    private:
        const Camera2D* m_camera;
    };


    /******************************************************************************
     *  NotifyRendererListenersOnNodePreRender
     ******************************************************************************/
    void Renderer2D::NotifyRendererListenersOnNodePreRender(const RenderNode *node, const Matrix3x2 *modelViewMatrix)
    {
        OnNodePreRenderEvent event(node, modelViewMatrix);
        GetListenerContainer().Iterate(event);
    }

    /******************************************************************************
     *  NotifyRendererListenersOnNodePostRender
     ******************************************************************************/
    void Renderer2D::NotifyRendererListenersOnNodePostRender(const RenderNode *node, const Matrix3x2 *modelViewMatrix)
    {
        OnNodePostRenderEvent event(node, modelViewMatrix);
        GetListenerContainer().Iterate(event);
    }

    /******************************************************************************
     *  NotifyRendererListenersOnCameraPostRender
     ******************************************************************************/
    void Renderer2D::NotifyListenersOnCameraPostRender(Camera2D* camera)
    {
        OnCameraPostRenderEvent event(camera);
        GetListenerContainer().Iterate(event);
    }

    /******************************************************************************
    *  NotifyRendererListenersOnCameraPreActivate
    ******************************************************************************/

    void Renderer2D::NotifyListenersOnCameraPreActivate(const Camera2D *camera)
    {
        OnCameraPreActivateEvent event(camera);
        GetListenerContainer().Iterate(event);
    }

    /******************************************************************************
    *  NotifyRendererListenersOnCameraPostActivate
    ******************************************************************************/
    void Renderer2D::NotifyListenersOnCameraPostActivate(const Camera2D *camera)
    {
        OnCameraPostActivateEvent event(camera);
        GetListenerContainer().Iterate(event);
    }

    /******************************************************************************
    *  NotifyRendererListenersOnCameraPreClear
    ******************************************************************************/
    void Renderer2D::NotifyListenersOnCameraPreClear(const Camera2D *camera)
    {
        OnCameraPreClearEvent event(camera);
        GetListenerContainer().Iterate(event);
    }

    /******************************************************************************
    *  NotifyRendererListenersOnCameraPostClear
    ******************************************************************************/
    void Renderer2D::NotifyListenersOnCameraPostClear(const Camera2D *camera)
    {
        OnCameraPostClearEvent event(camera);
        GetListenerContainer().Iterate(event);
    }


    /******************************************************************************
     *  GetCameraList
     ******************************************************************************/
    Renderer2D::CameraList& Renderer2D::GetCameraList()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(CameraList, s_cameraList);
        return s_cameraList;
    }

    Renderer2D::CameraList& Renderer2D::s_forceInitCameraList = GetCameraList();

    FeatStd::Internal::CriticalSection* Renderer2D::GetCameraListCriticalSection()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, s_containerCriticalSection);
        return &s_containerCriticalSection;
    }

    void Renderer2D::PreRenderNodes(Scene2D* scene)
    {
        RenderOrder2D& renderOrder = scene->GetRenderOrder();
        renderOrder.Validate();

        for (renderOrder.IterationBegin(); !renderOrder.IterationIsEnd(); renderOrder.IterationMoveToNextNode()) {
            RenderNode& renderNode = renderOrder.IterationGetNode();

            if (renderNode.IsEffectiveRenderingEnabled()) {
                renderNode.PreRender();
            }
        }

    }

    /******************************************************************************
     *  GetCamera2dListenerInstance
     ******************************************************************************/
    Camera2DListener* Renderer2D::GetCamera2dListenerInstance()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(RendererCameraListener, instance);
        return &instance;
    }

    Camera2DListener* Renderer2D::s_forceInitCamera2dListenerInstance = GetCamera2dListenerInstance();

    /******************************************************************************
     *  Clear
     ******************************************************************************/
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
    bool Renderer2D::Clear(ContextHandle2D context, Float r, Float g, Float b, Float a)
    {
        bool success = RenderDevice2D::Clear(context, r, g, b, a);
        if (success) {
            ++s_statistics.Clears;
        }

        return success;
    }
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

    /******************************************************************************
     *  Blit
     ******************************************************************************/
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
    bool Renderer2D::Blit(ContextHandle2D context, bool forceBlit)
    {
        if (forceBlit || (!IsSourceSurfaceAreaCulled(context))) {

            if (RenderDevice2D::Blit(context)) {
                ++s_statistics.Blits;
                return true;
            }

            return false;
        }
        ++s_statistics.BlitsCulled;
        return true;
    }
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

    /******************************************************************************
     *  SetTransformationMatrix
     ******************************************************************************/
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
    bool Renderer2D::SetTransformationMatrix(ContextHandle2D context, RenderDevice2D::SurfaceType sidx, const Matrix3x2& mat)
    {
        if (RenderDevice2D::SourceSurface == sidx) {
            s_sourceSurfaceTransformationMatrix = mat;
        }

        return RenderDevice2D::SetTransformationMatrix(context, sidx, mat);
    }
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()

    /******************************************************************************
     *  IsSourceSurfaceAreaCulled
     ******************************************************************************/
    bool Renderer2D::IsSourceSurfaceAreaCulled(ContextHandle2D context)
    {
        if (0 == s_activeCamera) {
            return false;
        }

        Float srcLeft;
        Float srcTop;
        Float srcWidth;
        Float srcHeight;
        if (RenderDevice2D::GetActiveArea(context, RenderDevice2D::SourceSurface, &srcLeft, &srcTop, &srcWidth, &srcHeight)) {
            Float dstLeft;
            Float dstTop;
            Float dstWidth;
            Float dstHeight;
            if (RenderDevice2D::GetActiveArea(context, RenderDevice2D::DestinationSurface, &dstLeft, &dstTop, &dstWidth, &dstHeight)) {
                // Intersect DestinationSurface with scissor to calculate screen space rectangle which will be rendered into.
                Rectangle dstRect(dstLeft, dstTop, dstWidth, dstHeight);    // DestinationSurface in screen space.
                if (RenderDevice2D::IsScissorEnabled(context)) {
                    Rectangle scissor;
                    if (RenderDevice2D::GetScissorRectangle(context, scissor)) {
                        dstRect.Intersect(scissor);                         // Scissor in screen space.
                    }
                }

                // Transform SourceSurface to screen space
                Rectangle srcRect(srcLeft, srcTop, srcWidth, srcHeight);    // SourceSurface in oject space.
                srcRect.Transform(s_sourceSurfaceTransformationMatrix);     // Transform SourceSurface to view+offset space

                const Rectangle& viewport = s_activeCamera->GetViewport();  // Calculate view offset
                Float offsetX = s_cameraTranslationOffset.GetX() - viewport.GetLeft();
                Float offsetY = s_cameraTranslationOffset.GetY() - viewport.GetTop();
#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
                offsetX += s_preTranslate.GetX();                           // Add offset from clipping rectangle.
                offsetY += s_preTranslate.GetY();
#endif
                srcRect.SetLeft(srcRect.GetLeft() - offsetX);               // Transform to screen space by subtracting
                srcRect.SetTop(srcRect.GetTop() - offsetY);                 // the offsets.

                // Intersect SourceSurface with DestinationSurface in screen space to determine if it can be culled.
                srcRect.Intersect(dstRect);
                if (0.0F >= srcRect.GetHeight() * srcRect.GetWidth()) {
                    return true;
                }
            }
        }

        return false;
    }

#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
    void Renderer2D::CalculateScreenSpaceClippingRect(const RenderNode& renderNode, const Matrix3x2& viewMatrix, Rectangle& screenSpaceClippingRect)
    {
        Rectangle clippingRect(renderNode.GetClippingRect());

        // apply camera transformation
        clippingRect.Transform(renderNode.GetParent()->GetWorldTransform() * viewMatrix);

        // Round clippingRect to pixels for consistency with DestinationSurface handling in RenderDevice2D.
        Float flooredLeft = Math::Floor(clippingRect.GetLeft());
        clippingRect.SetWidth(Math::Ceil(clippingRect.GetWidth() + flooredLeft - clippingRect.GetLeft()));
        clippingRect.SetLeft(flooredLeft);
        Float flooredTop = Math::Floor(clippingRect.GetTop());
        clippingRect.SetHeight(Math::Ceil(clippingRect.GetHeight() + flooredTop - clippingRect.GetTop()));
        clippingRect.SetTop(flooredTop);

        // compute intersection
        screenSpaceClippingRect.IntersectChild(clippingRect);
    }
#endif



    bool Renderer2D::UploadVertexBuffer2D(VertexBuffer2D& vb, DeviceObject2D::LoadingHint loadingHint)
    {
        if ((DeviceObject2D::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
            FEATSTD_LOG_WARN("Uploading VertexBuffer2D (name: '%s') while rendering.", vb.GetName());
        }
        if (RenderDevice2D::UploadVertexBuffer2D(vb)) {
            return true;
        }
        return false;
    }

    bool Renderer2D::UnloadVertexBuffer2D(VertexBuffer2D& vb, DeviceObject2D::LoadingHint loadingHint)
    {
        if ((DeviceObject2D::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
            FEATSTD_LOG_WARN("Unloading VertexBuffer2D (name: '%s') while rendering.", vb.GetName());
        }
        if (RenderDevice2D::UnloadVertexBuffer2D(vb))
        {
            return true;
        }
        return false;
    }



    FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
    bool Renderer2D::UploadBitmapImage2D(BitmapImage2D& image, DeviceObject2D::LoadingHint loadingHint)
    {
        if ((DeviceObject2D::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
            FEATSTD_LOG_WARN("Uploading BitmapImage2D (name: '%s') while rendering.", image.GetName());
        }
        return RenderDevice2D::UploadBitmapImage2D(image);
    }
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

    FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
    bool Renderer2D::UnloadBitmapImage2D(BitmapImage2D& image, DeviceObject2D::LoadingHint loadingHint)
    {
        if ((DeviceObject2D::WarnWhileRendering == loadingHint) && (IsRenderingCamera())) {
            FEATSTD_LOG_WARN("Unloading BitmapImage2D (name: '%s') while rendering.", image.GetName());
        }
        return RenderDevice2D::UnloadBitmapImage2D(image);
    }
    FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

}   // namespace Candera
