//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RenderOrder2D_H)
    #define CANDERA_RenderOrder2D_H

#include <Candera/Environment.h>
#include <Candera/System/Container/Vector.h>


namespace Candera {
    class RenderNode;

    /**
     *  @brief  Class RenderOrder2D defines a set of nodes to be rendered.
     *
     *  Nodes are inserted into render order according to following rule:
     *  -# Nodes with higher depth value are inserted before nodes with lower depth value in render order.
     *  -# Nodes within same depth value are inserted with respect to their position in scene tree. Pre-order insertion
     *  assures that nodes located higher in vertical scene tree are rendered before nodes with lower position in scene
     *  tree.
     */
    class RenderOrder2D {
    public:
        /**
         *  Creates a RenderOrder2D with default values.
         *  @param   maxNodeCount    Maximum number of nodes to be stored.
         */
        RenderOrder2D(Int32 maxNodeCount);

        /**
         * Destructor
         */
        ~RenderOrder2D();

        /**
         * Set iterator of this RenderOrder2D to first node.
         */
        void IterationBegin() { m_nodeIndex = 0; }

        /**
         * Returns if there are more nodes to retrieve. This is true, as long as the last node has not been retrieved.
         * @return True if there are more nodes to retrieve, false otherwise.
         */
        bool IterationIsEnd() const { return static_cast<SizeType>(m_nodeIndex) >= m_nodeList.Size(); }

        /**
         * Move iterator to next node in this render order.
         */
        void IterationMoveToNextNode();

        /**
         * Retrieve the node the iterator is pointing to.
         * @return The RenderNode the iterator is pointing to.
         */
        RenderNode& IterationGetNode() { return *m_nodeList[m_nodeIndex]; }

        /**
         * Assign the node to the render order.
         * @param node The node to add
         * @return If node was added successfully. May fail if underlying list cannot allocate more memory.
         */
        bool AddNode(RenderNode* node);

        /**
         * Remove the node from the render order.
         * @param node The node to remove.
         * @return     True if node was removed, False if node wasn't found.
         */
        bool RemoveNode(const RenderNode* node);

        /**
         *  Invalidates the list of render nodes. Sorting has to be triggered by calling Validate() before next render pass.
         */
        void Invalidate() { m_isValid = false; }

        /**
         *  Validates the render order. If it has been marked as invalid, the list is sorted accordingly.
         *  This method is usually called before rendering.
         */
        void Validate();

        /**
         * Removes all Node2D associations.
         */
        void Clear() { m_nodeList.Clear(); }

    protected:
        RenderOrder2D(const RenderOrder2D& rhs);
        RenderOrder2D& operator = (const RenderOrder2D& rhs);

    private:
        bool m_isValid;
        Int32 m_nodeIndex;
        Internal::Vector<RenderNode*> m_nodeList;    // list of nodes sorted by depth value to allow rendering from back to front        

        UInt16 GetPositionInSceneTree(const RenderNode* node) const;
    };
}   // namespace Candera
#endif  // CANDERA_RenderOrder2D_H
