//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "RenderOrder2D.h"
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/Engine2D/Core/TreeTraverser2D.h>
#include <Candera/Engine2D/Core/Scene2D.h>

namespace Candera {

     class RendererOrder2DSortHelper : public TreeTraverser2D {
    public:
        RendererOrder2DSortHelper() : m_position(0) {}

        bool operator()(const RenderNode* a, const RenderNode* b) const
        {
            const Int16 renderOrderRankA = a->m_effectiveRenderOrderRank;
            const Int16 renderOrderRankB = b->m_effectiveRenderOrderRank;

            // nodes with smaller render order rank are rendered before others
            if (renderOrderRankA < renderOrderRankB) {
                return true;
            }
            else if (renderOrderRankA > renderOrderRankB) {
                return false;
            }
            else {
                // if the render order rank is the same, the position within the scene tree decides
                return a->m_positionInTree < b->m_positionInTree;
            }
        }

    protected:
        virtual TraverserAction ProcessNode(Node2D& node)
        {
            RenderNode* renderNode = Dynamic_Cast<RenderNode*>(&node);
            if (0 != renderNode) {
                renderNode->m_positionInTree = ++m_position;
                FEATSTD_DEBUG_ASSERT(m_position != 0); // check for overflow if more than (2^16) - 1 (!!!) render nodes are rendered
                renderNode->m_effectiveRenderOrderRank = static_cast<UInt16>(renderNode->GetEffectiveRenderOrderRank());
            }
            return ProceedTraversing;
        }

    private:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(RendererOrder2DSortHelper);

        UInt16 m_position;

    };
    class SceneTreeOrderTraverser : public ConstTreeTraverser2D {
        public:
            SceneTreeOrderTraverser(const RenderNode* node) : m_position(0), m_node(node) {}

            UInt16 GetPositionInSceneTree() const { return m_position; }

        protected:
            virtual TraverserAction ProcessNode(const Node2D& node) override
            {
                if (&node == m_node) {
                    return StopTraversing;
                }
                else {
                    if (node.IsTypeOf(RenderNode::GetTypeId())) {
                        m_position++;
                    }
                    else {
                        // nothing to do here
                    }
                }                

                return ProceedTraversing;
            }

        private:
            UInt16 m_position;
            const RenderNode* m_node;
    };

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    RenderOrder2D::RenderOrder2D(Int32 maxNodeCount) :
        m_isValid(false),
        m_nodeIndex(0),
        m_nodeList()
    {
        static_cast<void>(m_nodeList.Reserve(maxNodeCount));
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    RenderOrder2D::~RenderOrder2D()
    {
    }

    /******************************************************************************
     *  IterationMoveToNextNode
     ******************************************************************************/
    void RenderOrder2D::IterationMoveToNextNode()
    {
        if (!IterationIsEnd()) {
            m_nodeIndex++;
        }
    }

    /******************************************************************************
     *  AddNode
     ******************************************************************************/
    bool RenderOrder2D::AddNode(RenderNode* node)
    {
        if (node != 0) {
            Invalidate();
            return m_nodeList.Add(node);
        }

        return false;
    }

    /******************************************************************************
     *  RemoveNode
     ******************************************************************************/
    bool RenderOrder2D::RemoveNode(const RenderNode* node)
    {
        SizeType i = 0;
        const SizeType size = m_nodeList.Size();
        while ((i < size) && (m_nodeList[i] != node)) {
            i++;
        }

        if (i < size) {
            return m_nodeList.Remove(i);
        }
        else {
            return false;
        }
    }

    /******************************************************************************
     *  GetPositionInSceneTree
     ******************************************************************************/
    UInt16 RenderOrder2D::GetPositionInSceneTree(const RenderNode* node) const
    {
        UInt16 order = 0;

        Scene2D* scene = node->GetScene();
        if (scene != 0) {
            SceneTreeOrderTraverser traverser(node);
            traverser.Traverse(*scene);
            order = traverser.GetPositionInSceneTree();
        }

        return order;
    }

    /******************************************************************************
     *  Validate
     ******************************************************************************/
    void RenderOrder2D::Validate()
    {
        if (!m_isValid) {
            if (m_nodeList.Size() > 0) {
                RendererOrder2DSortHelper sortHelper;
                Scene2D* scene = m_nodeList[0]->GetScene();
                if (scene != 0) {
                    sortHelper.Traverse(*scene);
                }
                m_nodeList.Sort(sortHelper);
            }
            m_isValid = true;
        }

    }
}   // namespace Candera
