//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RenderNode_H)
    #define CANDERA_RenderNode_H

#include <Candera/Engine2D/Core/2DStringBufferAppenders.h>
#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/Engine2D/Effects/Effect2D.h>
#include <Candera/System/Container/Vector.h>

namespace Candera {
    class RendererOrder2DSortHelper;
/** @addtogroup Core2D
 *  @{
 */

    /**
     *  @brief RenderNode is a 2D node that links to an effect chain. Therefore this is the only type of 2D node
     *  which has content to be rendered.
     *
     *  All nodes of type RenderNode are added to a list which is sorted regarding each node's render order rank . See SetRenderOrderRank method.
     */
    class RenderNode : public Node2D {

        FEATSTD_TYPEDEF_BASE(Node2D);

    public:
        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance and possible children, if any.
         *  @return Pointer to the created object
         */
        static RenderNode* Create();

        /**
         *  Destructor
         */
        virtual ~RenderNode() override;

        virtual RenderNode* Clone() const override;

        /**
         *  @param  snapToDevicePixel   Enable or disable snapping of translation to device pixels.
         */
        void SetSnapToDevicePixelEnabled(bool snapToDevicePixel) { m_snapToDevicePixel = snapToDevicePixel; }

        /**
         *  @return     true, if snapping of translation to device pixels is enabled.
         */
        bool IsSnapToDevicePixelEnabled() const { return m_snapToDevicePixel; }

        /**
         *  Add the effect at the end of the list (predecessor is 0) or at position after predecessor.
         *  @param effect The Effect2D to add to the list
         *  @param predecessor The effect after which the new one shall be added (end of list if 0)
         *  @return true if added successfully
         */
        bool AddEffect(Effect2D* effect, Effect2D* predecessor = 0);

        /**
         *  Add the effect at front of the list.
         *  @param effect The Effect2D to add to the list
         *  @return true if added successfully
         */
        bool AddEffectAsFirst(Effect2D* effect) { return InsertEffectToList(0, effect); }

        /**
         *  Removes the effect from the list.
         *  @param effect The Effect2D to remove from the list
         *  @return true if removed successfully
         */
        bool RemoveEffect(Effect2D* effect) { return RemoveEffectFromList(effect); }

        /**
         * Gets the effect identified by idx from the list.
         * @param idx List index to get effect.
         * @return    Effect from list identified by list index idx.
         */
        Effect2D* GetEffect(UInt8 idx) const { return (idx < m_effectList.Size()) ? m_effectList[idx].GetPointerToSharedInstance() : 0; }

        // Override method from Node2D.
        virtual void GetBasePoint(Vector2& basePoint) const override;

        FEATSTD_LINT_NEXT_EXPRESSION(1735, "Virtual function uses same default parameter as abstract base class.")
        virtual void GetComputedBoundingRectangle(Rectangle& boundingRectangle, bool ignoreInvisible = false) const override;

#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
        void SetClippingRect(const Rectangle& clippingRect = Rectangle(0.0F, 0.0F, -1.0F, -1.0F));
        const Rectangle& GetClippingRect() const { return m_clippingRect; }
        bool HasClippingRect() const { return ((m_clippingRect.GetWidth() >= 0.0F) && (m_clippingRect.GetHeight() >= 0.0F)); }
#endif

#if defined(CANDERA_LAYOUT_ENABLED)
        virtual void GetComputedLayoutRectangle(Rectangle& rectangle) const override;
#endif

        FEATSTD_RTTI_DECLARATION();

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        RenderNode();
        explicit RenderNode(const RenderNode& renderNode);
        RenderNode& operator = (const RenderNode& renderNode);

        virtual void Render(RenderTarget2D* renderTarget, const Matrix3x2& localTransform);

        virtual bool UploadSelf() override;
        virtual bool UnloadSelf() override;

        /**
         *  Disposes this node only - not descendant children nodes.
         */
        virtual void DisposeSelf() override;

        /**
         *  Notification that an ancestor to this node has been added.
         *  An ancestor is each node up the tree hierarchy that has a parent relationship to this node.
         *  An ancestor can therefore be reached via GetParent(), recursively.
         *  Any derived class might override this function in order to react to this event.
         *  At the time the notification OnAncestorAdded is invoked, the parent of this Node is already set and the
         *  node's WorldTransformCache is invalidated.
         *  @param  scene       The scene node which became root of this node's scene graph.
         */
        virtual void OnAncestorAdded(Scene2D* scene) override;

        /**
         *  Notification that an ancestor to this node has been removed.
         *  An ancestor is each node up the tree hierarchy that has a parent relationship to this node.
         *  An ancestor can therefore be reached via GetParent(), recursively.
         *  Any derived class might override this function in order to react to this event.
         *  At the time the notification OnAncestorRemoved is invoked, the ancestor of this node is already
         *  disconnected to the scene.
         *  @param  scene       The scene node which previously had been the root of the scene graph this node was
         *                      connected with.
         */
        virtual void OnAncestorRemoved(Scene2D* scene) override;

    private:
        friend class Renderer2D;
        friend class RendererOrder2DSortHelper;

        typedef Internal::Vector<Effect2D::SharedPointer> EffectList;

        bool m_snapToDevicePixel;
        Int16 m_effectiveRenderOrderRank;
        UInt16 m_positionInTree;
        EffectList m_effectList;
#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
        Rectangle m_clippingRect;
#endif

        virtual void PreRender();

        void AddToRenderOrderList(Scene2D* scene);
        void RemoveFromRenderOrderList(Scene2D* scene) const;

        bool InsertEffectToList(SizeType pos, Effect2D* effect);
        bool RemoveEffectFromList(Effect2D* effect);


        CdaDynamicProperties(Candera::RenderNode, Candera::Node2D);
        CdaDynamicPropertiesEnd();
    };

 /** @} */ // end of Core2D

    inline void RenderNode::PreRender() {}
}   // namespace Candera

#endif  // CANDERA_RenderNode_H
