//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Node2D.h"
#include <Candera/Engine2D/Core/Scene2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/OS/StringPlatform.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageDescriptor.h>
#include <Candera/Engine2D/Mathematics/Math2D.h>
#include <Candera/Engine2D/Cloning/DeepNode2DCloneStrategy.h>
#include <Candera/Engine2D/Cloning/TreeCloner2D.h>
#include <FeatStd/Util/StaticObject.h>

#if defined(CANDERA_LAYOUT_ENABLED)
#include <Candera/EngineBase/Layout/Layout.h>
#endif

namespace Candera {
    using namespace Diagnostics;
    using namespace Candera::Internal;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine2D);

    class Node2DDynamicProperties
    {
    private:
        friend class Node2D;

        static bool SetAlphaValue(Node2D& node, Float alphaValue)
        {
            return node.SetValue(CdaDynamicPropertyInstance(AlphaValue), alphaValue);
        }

        static Float GetAlphaValue(const Node2D& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(AlphaValue));
        }

        static void SetBoundingRectangle(Node2D& node, const Rectangle& boundingRectangle)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(BoundingRectangle), boundingRectangle));
        }

        static const Rectangle& GetBoundingRectangle(const Node2D& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(BoundingRectangle));
        }

        static bool IsBoundingRectangleSet(const Node2D& node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(BoundingRectangle));
        }

        static bool ClearBoundingRectangle(Node2D& node)
        {
            return node.ClearValue(CdaDynamicPropertyInstance(BoundingRectangle));
        }

        static bool SetRenderOrderRank(Node2D& node, Int renderOrderRank)
        {
            return node.SetValue(CdaDynamicPropertyInstance(RenderOrderRank), static_cast<Int16>(renderOrderRank));
        }

        static Int GetRenderOrderRank(const Node2D& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(RenderOrderRank));
        }

        static void SetRenderBenchmark(Node2D& node, Float benchmark)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(RenderBenchmark), benchmark));
        }

        static Float GetRenderBenchmark(const Node2D& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(RenderBenchmark));
        }

#if defined(CANDERA_LAYOUT_ENABLED)
        static void SetLayouter(Node2D& node, Layouter* layouter)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(Layouter), layouter));
        }

        static Layouter* GetLayouter(const Node2D& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(Layouter));
        }

        static void SetPreferredSize(Node2D& node, const Vector2& preferredSize)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(PreferredSize), preferredSize));
        }

        static const Vector2& GetPreferredSize(const Node2D& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(PreferredSize));
        }

        static bool ClearLayouter(Node2D& node)
        {
            return node.ClearValue(CdaDynamicPropertyInstance(Layouter));
        }

        static void SetLayoutingRectangle(Node2D& node, const Rectangle& rectangle)
        {
            static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(LayoutingRectangle), rectangle));
        }

        static const Rectangle& GetLayoutingRectangle(const Node2D& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(LayoutingRectangle));
        }

        static bool IsLayoutingRectangleSet(const Node2D& node)
        {
            return node.IsValueSet(CdaDynamicPropertyInstance(LayoutingRectangle));
        }

        static bool ClearLayoutingRectangle(Node2D& node)
        {
            return node.ClearValue(CdaDynamicPropertyInstance(LayoutingRectangle));
        }

        static Layouter* GetDefaultLayouter() {
            return DefaultLayouter::GetInstance();
        }

        static Vector2& GetDefaultPreferredSize() {
            FEATSTD_UNSYNCED_STATIC_OBJECT(Vector2, s_preferredSizeDefault);
            return s_preferredSizeDefault;
        }
#endif
        static const Rectangle DefaultBoundingRectangleDefaultValue()
        {
            return Rectangle(0.0F, 0.0F, -1.0F, -1.0F);
        }
        static const Rectangle& GetDefaultBoundingRectangleDefault()
        {
            FEATSTD_UNSYNCED_STATIC_OBJECT(const Rectangle, s_boundingRectangleDefault, DefaultBoundingRectangleDefaultValue());
            return s_boundingRectangleDefault;
        }


        CdaDynamicPropertiesDefinition(Candera::Node2D, Candera::Node2D::Base);

            CdaDynamicProperty(AlphaValue, Float);
                CdaDynamicPropertyDefaultValue(1.0F);
                CdaDynamicPropertyDescription("The alpha value of this node.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(BoundingRectangle, Rectangle);
                CdaDynamicPropertyDefaultValue(GetDefaultBoundingRectangleDefault());
                CdaDynamicPropertyDescription("The manually set bounding rectangle for this node.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(RenderOrderRank, Int16);
                CdaDynamicPropertyDefaultValue(0);
                CdaDynamicPropertyDescription("The render order rank defines the relative value within render order. Nodes with " \
                                              "lower render order rank are placed in background and therefore are rendered first.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();

#if defined(CANDERA_LAYOUT_ENABLED)

#ifdef CANDERA_DEPRECATED_3_2_0
            CdaDynamicProperty(LayoutingRectangle, Rectangle);
                CdaDynamicPropertyDefaultValue(GetDefaultBoundingRectangleDefault());
                CdaDynamicPropertyDescription("The rectangle set when layout"
                     " this node. If not set, this is automatically computed.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();
#else
#error Remove this!
#endif

            CdaDynamicProperty(Layouter, Layouter*);
                CdaDynamicPropertyDefaultValue(GetDefaultLayouter());
                CdaDynamicPropertyDescription("Defines a layout strategy for the child nodes, e.g. a grid or stack layout.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

            CdaDynamicProperty(PreferredSize, Vector2);
                CdaDynamicPropertyDefaultValue(GetDefaultPreferredSize());
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();

#endif
            CdaDynamicProperty(RenderBenchmark, Float);
                CdaDynamicPropertyDefaultValue(0.0F);
                CdaDynamicPropertyDescription("Benchmark value related to render time, used BenchmarkCamera2DRenderStrategy. " \
                                              "Default value is 0.0F.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();
        CdaDynamicPropertiesEnd();
    };

    const Candera::DynamicProperties::PropertyHierarchyNode&  Node2D::GetPropertyHierarchy() const
    {
        return Node2DDynamicProperties::GetPropertyHierarchy();
    }

    Float Node2D::GetAlphaValue() const
    {
        return Node2DDynamicProperties::GetAlphaValue(*this);
    }

    void Node2D::SetBoundingRectangle(const Rectangle& boundingRectangle)
    {
        Node2DDynamicProperties::SetBoundingRectangle(*this, boundingRectangle);
    }

    const Rectangle& Node2D::GetBoundingRectangle() const
    {
        return Node2DDynamicProperties::GetBoundingRectangle(*this);
    }

    bool Node2D::IsBoundingRectangleSet() const
    {
        return Node2DDynamicProperties::IsBoundingRectangleSet(*this);
    }

    bool Node2D::ClearBoundingRectangle()
    {
        return Node2DDynamicProperties::ClearBoundingRectangle(*this);
    }

    Int Node2D::GetRenderOrderRank() const
    {
        return Node2DDynamicProperties::GetRenderOrderRank(*this);
    }

    Float Node2D::GetRenderBenchmark() const
    {
        return Node2DDynamicProperties::GetRenderBenchmark(*this);
    }

#if defined(CANDERA_LAYOUT_ENABLED)
    Layouter* Node2D::GetLayouter() const
    {
        return Node2DDynamicProperties::GetLayouter(*this);
    }

    void Node2D::SetPreferredSize(const Vector2& preferredSize)
    {
        Node2DDynamicProperties::SetPreferredSize(*this, preferredSize);
    }

    const Vector2& Node2D::GetPreferredSize() const
    {
        return Node2DDynamicProperties::GetPreferredSize(*this);
    }

    bool Node2D::ClearLayouter()
    {
        return Node2DDynamicProperties::ClearLayouter(*this);
    }
#endif

    /**
     *  OnAncestorAddedTraverser2D informs descendants that an ancestor has been added.
     */
    class OnAncestorAddedTraverser2D : public TreeTraverser2D
    {
        public:
            OnAncestorAddedTraverser2D(Scene2D* scene) :
                m_scene(scene)
            {
            }

        protected:
            virtual TraverserAction ProcessNode(Node2D& node) override
            {
                node.OnAncestorAdded(m_scene);
                return ProceedTraversing;
            }

        private:
            Scene2D* m_scene;
    };

    /**
     *  OnAncestorRemovedTraverser2D informs descendants that an ancestor has been removed.
     */
    class OnAncestorRemovedTraverser2D : public TreeTraverser2D
    {
        public:
            OnAncestorRemovedTraverser2D(Scene2D* scene) :
                m_scene(scene)
            {
            }

        protected:
            virtual TraverserAction ProcessNode(Node2D& node) override
            {
                node.OnAncestorRemoved(m_scene);
                return ProceedTraversing;
            }

        private:
            Scene2D* m_scene;
    };

    /**
     *  InvalidateWorldTransformCacheTraverser2D informs descendants that the world transform has changed.
     */
    class InvalidateWorldTransformCacheTraverser2D : public TreeTraverser2D
    {
        protected:
            virtual TraverserAction ProcessNode(Node2D& node) override
            {
                if (node.IsWorldTransformCacheValid()) {
                    node.InvalidateWorldTransformCache();
                    node.OnAncestorTransformChanged();
                    return ProceedTraversing;
                }

                return StopTraversingForDescendants;
            }
    };

    /**
     *  InvalidateAlphaValueCacheTraverser2D informs descendants that the alpha value has changed.
     */
    class InvalidateEffectiveAlphaValueCacheTraverser2D : public TreeTraverser2D
    {
        protected:
            virtual TraverserAction ProcessNode(Node2D& node) override
            {
                if (node.IsAlphaValueCacheValid()) {
                    node.InvalidateEffectiveAlphaValueCache();
                    return ProceedTraversing;
                }

                return StopTraversingForDescendants;
            }
    };

    /**
     *  UploadTraverser invokes an upload at the node given and all its descendants.
     */
    class UploadTraverser2D : public TreeTraverser2D
    {
        public:
            UploadTraverser2D(const ScopeMask& scopeMask) :
                m_isEffectiveResultSuccessful(true),
                m_scopeMask(scopeMask)
            {
            }

            bool IsEffectiveResultSuccessful() const { return m_isEffectiveResultSuccessful; }

        protected:
            virtual TraverserAction ProcessNode(Node2D& node) override
            {
                if (node.IsInScopeOf(m_scopeMask)) {
                    if (!node.UploadSelf()) {
                        FEATSTD_LOG_ERROR("Upload failed for node %s", node.GetName());
                        m_isEffectiveResultSuccessful = false;
                    }
                }

                return ProceedTraversing;
            }

        private:
            bool m_isEffectiveResultSuccessful;
            const ScopeMask m_scopeMask;

            //forbid assignment, make it private
            UploadTraverser2D& operator=(const UploadTraverser2D&);
    };

    /**
     *  UploadTraverser invokes an upload at the node given and all its descendants.
     */
    class UnloadTraverser2D : public TreeTraverser2D
    {
        public:
            UnloadTraverser2D(const ScopeMask& scopeMask) :
                m_isEffectiveResultSuccessful(true),
                m_scopeMask(scopeMask)
            {
            }

            bool IsEffectiveResultSuccessful() const { return m_isEffectiveResultSuccessful; }

        protected:
            virtual TraverserAction ProcessNode(Node2D& node) override
            {
                if (node.IsInScopeOf(m_scopeMask)) {
                    if (!node.UnloadSelf()) {
                        m_isEffectiveResultSuccessful = false;
                    }
                }

                return ProceedTraversing;
            }

        private:
            bool m_isEffectiveResultSuccessful;
            const ScopeMask m_scopeMask;

            //forbid assignment, make it private
            UnloadTraverser2D& operator=(const UnloadTraverser2D&);
    };

    /**
     *  SetScopeMaskTraverser2D invokes SetScopeMask at the node given and all its descendants.
     */
    class SetScopeMaskTraverser2D : public TreeTraverser2D
    {
        public:
            SetScopeMaskTraverser2D(const ScopeMask& scopeMask): m_scopeMask(scopeMask) {}

        protected:
            virtual TraverserAction ProcessNode(Node2D& node) override
            {
                node.SetScopeMask(m_scopeMask, Node2D::Flat);
                return ProceedTraversing;
            }

        private:
            const ScopeMask m_scopeMask;

            //forbid assignment, make it private
            SetScopeMaskTraverser2D& operator=(const SetScopeMaskTraverser2D&);
    };

    /**
     *  SetScopeEnabledTraverser2D invokes SetScopeEnabled at the node given and all its descendants.
     */
    class SetScopeEnabledTraverser2D : public TreeTraverser2D
    {
        public:
            SetScopeEnabledTraverser2D(UInt scopeIndex, bool enable) :
                m_scopeIndex(scopeIndex),
                m_enable(enable),
                m_isEffectiveResultSuccessful(true)
            {
            }

            bool IsEffectiveResultSuccessful() const { return m_isEffectiveResultSuccessful; }

        protected:
            virtual TraverserAction ProcessNode(Node2D& node) override
            {
                if (! node.SetScopeEnabled(m_scopeIndex, m_enable, Node2D::Flat)) {
                    m_isEffectiveResultSuccessful = false;
                }
                return ProceedTraversing;
            }

        private:
            const UInt m_scopeIndex;  // Scope index to update.
            const bool m_enable;       // Enable or disable scope at scope index given.
            bool m_isEffectiveResultSuccessful;

            //forbid assignment, make it private
            SetScopeEnabledTraverser2D& operator=(const SetScopeEnabledTraverser2D&);
    };

    /**
    *  FindByNameTraverser traverses to find a Node by the name given.
    */
    class FindByIdTraverser : public ConstTreeTraverser2D
    {
    public:
        void SetIdToFind(const Char* findId)
        {
            m_findId = findId;
            m_matchingNode = 0;
        }

        void SetRootNode(const Node2D* rootNode) { m_rootNode = rootNode; }

        const Node2D* GetMatchingNode() const { return m_matchingNode; }

    protected:
        virtual TraverserAction ProcessNode(const Node2D& node) override
        {
            if (&node == m_rootNode) {
                return ProceedTraversing;
            }
            const Char* const nodeName = node.GetName();
            const bool found = ((nodeName != 0) && (Candera::StringPlatform::CompareStrings(nodeName, m_findId) == 0));
            if (found) {
                m_matchingNode = &node;
            }
            return (found ? StopTraversing : ProceedTraversing);
        }

    private:
        const Char* m_findId;
        const Node2D* m_matchingNode;
        const Node2D* m_rootNode;
    };

#if defined(CANDERA_LAYOUT_ENABLED)
    /**
     *  LayoutTraverser traverses through all children until if finds a node having a Layouter. Therefore Layouter::Layout() is called,
     *  all its children are skipped and the traverser continues with the next sibling.
     */
    class LayoutTraverser : public TreeTraverser2D {
    protected:
        /**
         *  If the node has a layouter, Layout() is called but its children are ignored - they are handled by the layouter itself.
         */
        virtual TraverserAction ProcessNode(Node2D& node) override
        {
            Layouter* layouter = node.GetLayouter();
            if ((layouter != 0) && (layouter != DefaultLayouter::GetInstance())) {
                AbstractNodePointer abstractNode(&node);
                layouter->Layout(abstractNode);
                return StopTraversingForDescendants;
            }
            return ProceedTraversing;
        }
    };

#endif

    class AxisAlignedBoundingRectangleTraverser : public ConstTreeTraverser2D {
        public:
            const Rectangle& GetBoundingRectangle() const { return m_boundingRect; }

        protected:
            virtual TraverserAction ProcessNode(const Node2D& node) override{
                Rectangle boundingRect;
                node.GetWorldAxisAlignedBoundingRectangle(boundingRect);
                m_boundingRect.Union(boundingRect);

                return ProceedTraversing;
            }

        private:
            Rectangle m_boundingRect;
    };

    FEATSTD_RTTI_DEFINITION(Node2D, Transformable2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    Node2D::Node2D() :
        Base(),
        m_isWorldTransformCacheValid(false),
        m_isEffectiveAlphaValueCacheValid(false),
        m_isRenderingEnabled(true),
        m_childCount(0),
        m_parent(0),
        m_firstChild(0),
        m_effectiveAlphaValueCache(0.0F),
        m_scopeMask()
    {
    }

    Node2D::Node2D(const Node2D& node) :
        Base(node),
        m_isWorldTransformCacheValid(false),
        m_isEffectiveAlphaValueCacheValid(false),
        m_isRenderingEnabled(node.m_isRenderingEnabled),
        m_childCount(0),
        m_parent(0),
        m_firstChild(0),
        m_effectiveAlphaValueCache(0.0F),
        m_scopeMask(node.m_scopeMask)
    {
#if defined(CANDERA_LAYOUT_ENABLED)
        static_cast<void>(ClearLayouter());
#endif
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    Node2D::~Node2D()
    {
#if defined(CANDERA_LAYOUT_ENABLED)
        Layouter* layouter = GetLayouter();
        if (layouter != 0) {
            layouter->Dispose();
        }
#endif
        m_parent = 0;
        m_firstChild = 0;
        m_childCount = 0;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Node2D* Node2D::CloneTraversing(TraverseOperation traverseOperation /* = TraverseFlat */, CloneOperation cloneOperation /* = CloneShallow */) const
    {
        if (IsTypeOf(Scene2D::GetTypeId())) {
            return 0;   // cloning Scene2D is not allowed
        }

        if (traverseOperation == TraverseFlat) {
            Node2D* clone = Clone();
            if ((clone != 0) && (cloneOperation == CloneDeep)) {
                DeepNode2DCloneStrategy cloneStrategy;
                DeepNode2DCloneStrategy::Pair pair(this, clone);
                cloneStrategy.Execute(pair, pair);
            }
            return clone;
        }
        else {
            TreeCloner2D cloner;
            DeepNode2DCloneStrategy cloneStrategy;

            if (cloneOperation == CloneDeep) {
                cloner.SetNodeCloneStrategy(&cloneStrategy);
            }
            return cloner.CreateClone(*this);
        }
    }

    /******************************************************************************
     *  Dispose
     ******************************************************************************/
    void Node2D::Dispose()
    {
        // Dispose this node and descendants.
        Node2D* const root = this;
        Node2D* currentNode = this;
        bool done = false;
        while (!done) {
            Node2D* const firstChild = currentNode->GetFirstChild();
            const bool currentNodeIsLeaf = (firstChild == 0);
            if (currentNodeIsLeaf) {
                Node2D* const parentNode = currentNode->GetParent();
                if (parentNode != 0) {
                    static_cast<void>(parentNode->RemoveChild(currentNode));
                }
                currentNode->DisposeSelf();
                if (currentNode == root) {
                    done = true;
                }
                else {
                    // up to parent
                    currentNode = parentNode;
                }
            }
            else {
                // down to first child
                currentNode = firstChild;
            }
        }
    }

    /******************************************************************************
     *  GetAssociatedScene
     ******************************************************************************/
    Scene2D* Node2D::GetScene() const
    {
        // Search the Scene2D parent to this Node2D. This node must not be the associated scene.
        const Node2D* node = this;
        while (node->GetParent() != 0) {
            node = node->GetParent();
        }
        // Check if root node found is a Scene.
        return Dynamic_Cast<Scene2D*>(const_cast<Node2D*>(node));
    }

    /******************************************************************************
    *  GetChild
    ******************************************************************************/
    Node2D* Node2D::GetChild(const Char* name)
    {
        for (Node2D* currentNode = m_firstChild; currentNode != 0; currentNode = currentNode->GetNextSibling()) {
            if (StringPlatform::CompareStrings(currentNode->GetName(), name) == 0) {
                return currentNode;
            }
        }
        return 0;
    }

    const Node2D* Node2D::GetChild(const Char* name) const
    {
        for (Node2D* currentNode = m_firstChild; currentNode != 0; currentNode = currentNode->GetNextSibling()) {
            if (StringPlatform::CompareStrings(currentNode->GetName(), name) == 0) {
                return currentNode;
            }
        }
        return 0;
    }

    Node2D* Node2D::GetChild(Id id)
    {
        for (Node2D* currentNode = m_firstChild; currentNode != 0; currentNode = currentNode->GetNextSibling()) {
            if (currentNode->GetId() == id) {
                return currentNode;
            }
        }
        return 0;
    }

    const Node2D* Node2D::GetChild(Id id) const
    {
        for (Node2D* currentNode = m_firstChild; currentNode != 0; currentNode = currentNode->GetNextSibling()) {
            if (currentNode->GetId() == id) {
                return currentNode;
            }
        }
        return 0;
    }

    /******************************************************************************
    *  InsertBeforeChild
    ******************************************************************************/
    bool Node2D::InsertBeforeChild(Node2D* newChild, Node2D* beforeChild)
    {
        // Return error if child is null, child is this group, child has already a parent set, or child is a scene.
        // ToDo: Return error if child is ancestor to this group
        if ((newChild == 0) || (newChild == this) || (newChild->GetParent() != 0) || newChild->IsTypeOf(Scene2D::GetTypeId()) ) {
            if (newChild == 0) {
                FEATSTD_LOG_ERROR("Insert before not possible, child 0.");
            }
            else {
                if (newChild == this) {
                    FEATSTD_LOG_ERROR("Insert before not possible, child == this.");
                }
                if (newChild->GetParent() != 0) {
                    FEATSTD_LOG_ERROR("Insert before not possible, child already has a parent");
                }
                if (newChild->IsTypeOf(Scene2D::GetTypeId())) {
                    FEATSTD_LOG_ERROR("Insert before not possible, child is a Scene.");
                }
            }
            return false;
        }

        newChild->SetParent(this);

        InvalidateWorldTransformCacheTraverser2D iwtcTraverser;
        iwtcTraverser.Traverse(*newChild);

        InvalidateEffectiveAlphaValueCacheTraverser2D ieavcTraverser;
        ieavcTraverser.Traverse(*newChild);

        Candera::Internal::LinkedList<Node2D, &Node2D::m_childListNode> childList(m_childCount, m_firstChild);
        childList.Insert(newChild, beforeChild);
        m_firstChild = &*childList.Begin();
        m_childCount = static_cast<UInt16>(childList.GetSize());

        Scene2D *scene = GetScene();
        if (scene != 0) {
#if defined(CANDERA_LAYOUT_ENABLED)
            AbstractNodePointer abstractNode(this);
            abstractNode.InvalidateLayout();
#endif
            OnAncestorAddedTraverser2D oaaTraverser(scene);
            oaaTraverser.Traverse(*newChild);
        }

        NotifyListenersOnChildAdded(newChild);
        return true;
    }

    /******************************************************************************
     *  RemoveChild
     ******************************************************************************/
    bool Node2D::RemoveChild(Node2D* child)
    {
        if (child == 0) {
            FEATSTD_LOG_ERROR("Remove child failed, child == 0.");
            return false;
        }

        Candera::Internal::LinkedList<Node2D, &Node2D::m_childListNode> childList(m_childCount, m_firstChild);
        
        const bool removeSucceeded = childList.Remove(child);

        if (removeSucceeded) {
            child->SetParent(0);
        }

        m_firstChild = &*childList.Begin();
        m_childCount = static_cast<UInt16>(childList.GetSize());
        child->NotifyListenersOnNodeRemoved(this);  //Trigger "OnNodeRemoved" event on all attached Node2DListeners
                                                    //Now called even when remove was unsuccessfull, according to review request #495
        if (removeSucceeded) {
#if defined(CANDERA_LAYOUT_ENABLED)
            Scene2D *scene = GetScene();
            if (scene != 0) {
                AbstractNodePointer abstractNode(this);
                abstractNode.InvalidateLayout();
            }
#endif
            OnAncestorRemovedTraverser2D oarTraverser(GetScene());
            oarTraverser.Traverse(*child);
            for (Node2D* parent = this; parent != 0; parent = parent->GetParent()) {
                parent->OnDescendantRemoved(child);
            };
        }

        if (removeSucceeded) {
            NotifyListenersOnChildRemoved(child);
        }
        return removeSucceeded;
    }

    /******************************************************************************
     *  IsEffectiveRenderingEnabled
     ******************************************************************************/
    bool Node2D::IsEffectiveRenderingEnabled() const
    {
        const Node2D* currentNode = this;
        while (currentNode != 0) {
            if (!currentNode->IsRenderingEnabled()) {
                return false;
            }
            currentNode = currentNode->GetParent();
        }
        return true;
    }

    /******************************************************************************
     *  GetWorldTransform
     ******************************************************************************/
    const Matrix3x2& Node2D::GetWorldTransform() const
    {
        //compute world transform and update world transformation cache only, if not up-to-date
        if ((!IsWorldTransformCacheValid()) || (!IsCompositeTransformCacheValid())) {
            // Get local composite transformation of this Node first
            m_worldTransformCache = GetCompositeTransform();
            const Node2D* parent = GetParent();
            if ((parent != 0) && (!parent->IsTypeOf(Scene2D::GetTypeId()))) {
                // Multiply local composite transform with ancestors world transform
                m_worldTransformCache *= parent->GetWorldTransform();
            }
            m_isWorldTransformCacheValid = true;
        }
        return m_worldTransformCache;
    }

    /******************************************************************************
     *  GetWorldPosition
     ******************************************************************************/
    Candera::Vector2 Node2D::GetWorldPosition() const
    {
        const Matrix3x2& worldTransform = GetWorldTransform();
        return Vector2(worldTransform(2,0), worldTransform(2,1));
    }

    /******************************************************************************
     *  IsInsideBoundingRectangle
     ******************************************************************************/
    bool Node2D::IsInsideBoundingRectangle(const Vector2& point, bool ignoreInvisible) const
    {
        Matrix3x2 matrix(GetWorldTransform());
        matrix.Inverse();

        Rectangle boundingRect;
        GetEffectiveBoundingRectangle(boundingRect, ignoreInvisible);

        return boundingRect.Contains(matrix.Multiply(point));
    }

    /******************************************************************************
     *  IsPickIntersectingBoundingRectangle
     ******************************************************************************/
    bool Node2D::IsPickIntersectingBoundingRectangle(const Camera2D& camera, const Vector2& point, PickPointCoordinateSpace space, bool ignoreInvisible) const
    {
        Vector2 pointInRenderTargetSpace = point;
        if ((space == Absolute) || ((space == Automatic) && (!DevicePackageDescriptor::GetPlatformDescription().isWindowManagerUsed))) {
            const RenderTarget2D* renderTarget = camera.GetRenderTarget();
            if (renderTarget != 0) {
                pointInRenderTargetSpace = Math2D::TransformScreenToRenderTarget(
                    *renderTarget, pointInRenderTargetSpace);
            }
        }
        const Vector2 pointInWorldSpace =
            Math2D::TransformViewportToScene(camera,
                Math2D::TransformRenderTargetToViewport(camera,
                    pointInRenderTargetSpace));

        return IsInsideBoundingRectangle(pointInWorldSpace, ignoreInvisible);
    }

    /******************************************************************************
     *  GetAxisAlignedBoundingRectangle
     ******************************************************************************/
    void Node2D::GetWorldAxisAlignedBoundingRectangle(Rectangle& boundingRectangle, Traversing traverse /*= Flat*/) const
    {
        if (traverse == Flat) {
            GetEffectiveBoundingRectangle(boundingRectangle);
            boundingRectangle.Transform(GetWorldTransform());
        }
        else {  // Deep
            AxisAlignedBoundingRectangleTraverser traverser;
            traverser.Traverse(*this);
            boundingRectangle = traverser.GetBoundingRectangle();
        }
    }

    /******************************************************************************
     *  GetWorldOrientatedBoundingRectangle
     ******************************************************************************/
    void Node2D::GetWorldOrientatedBoundingRectangle(Vector2 points[4]) const
    {
        Rectangle boundingRect;
        GetEffectiveBoundingRectangle(boundingRect);

        const Float left = boundingRect.GetLeft();
        const Float top = boundingRect.GetTop();
        const Float width = boundingRect.GetWidth();
        const Float height = boundingRect.GetHeight();

        const Matrix3x2 worldTransform(GetWorldTransform());

        points[0] = worldTransform.Multiply(Vector2(left, top));
        points[1] = worldTransform.Multiply(Vector2(left + width , top));
        points[2] = worldTransform.Multiply(Vector2(left + width , top + height));
        points[3] = worldTransform.Multiply(Vector2(left, top + height));
    }

    /******************************************************************************
     *  GetBasePoint
     ******************************************************************************/
    void Node2D::GetBasePoint(Vector2& basePoint) const
    {
        Rectangle boundingRect;
        GetEffectiveBoundingRectangle(boundingRect);

        basePoint = boundingRect.GetPosition() + (boundingRect.GetSize() * 0.5F);
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool Node2D::Upload(const ScopeMask& scopeMask /*= ScopeMask()*/, Traversing traverse /*= Flat*/)
    {
        if (traverse == Flat) {
            return (IsInScopeOf(scopeMask) && UploadSelf());
        }
        else {
            UploadTraverser2D uploadTraverser(scopeMask);
            uploadTraverser.Traverse(*this);
            return uploadTraverser.IsEffectiveResultSuccessful();
        }
    }

    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool Node2D::Unload(const ScopeMask& scopeMask /*= ScopeMask()*/, Traversing traverse /*= Flat*/)
    {
        if (traverse == Flat) {
            return (IsInScopeOf(scopeMask) && UnloadSelf());
        }
        else {
            UnloadTraverser2D unloadTraverser(scopeMask);
            unloadTraverser.Traverse(*this);
            return unloadTraverser.IsEffectiveResultSuccessful();
        }
    }

    /******************************************************************************
     *  SetScopeMask
     ******************************************************************************/
    void Node2D::SetScopeMask(const ScopeMask& scopeMask, Traversing traverse /*= Flat*/)
    {
        if (traverse == Flat) {
            m_scopeMask = scopeMask;
        }
        else { //traverse == Deep
            SetScopeMaskTraverser2D setScopeMaskTraverser(scopeMask);
            setScopeMaskTraverser.Traverse(*this);
        }
    }

    /******************************************************************************
     *  SetScopeEnabled
     ******************************************************************************/
    bool Node2D::SetScopeEnabled(UInt scopeIndex, bool enable, Traversing traverse /*= Flat*/)
    {
        bool result;
        if (traverse == Flat) {
            result = m_scopeMask.SetScopeEnabled(scopeIndex, enable);
        }
        else { //traverse == Deep
            SetScopeEnabledTraverser2D setScopeEnabledTraverser(scopeIndex, enable);
            setScopeEnabledTraverser.Traverse(*this);
            result =  setScopeEnabledTraverser.IsEffectiveResultSuccessful();
        }
        return result;
    }

    /******************************************************************************
     *  Layout
     ******************************************************************************/
#if defined(CANDERA_LAYOUT_ENABLED)
    void Node2D::SetLayouter(Layouter* layouter)
    {
        Node2DDynamicProperties::SetLayouter(*this, layouter);
        AbstractNodePointer abstractNode(this);
        Candera::Layouter::InvalidateLayout(abstractNode);
    }

    void Node2D::Layout()
    {
        Internal::LayoutEvents::LayoutEvent layoutBeginEvent(Internal::LayoutEvents::LayoutEvent::LayoutValidationBegin, AbstractNodePointer(this)); // put to variable due to MULTI compile warning
        Layouter::GetEventSource().DispatchEvent(layoutBeginEvent);
        LayoutTraverser layoutTraverser;
        layoutTraverser.Traverse(*this);
        Internal::LayoutEvents::LayoutEvent layoutEndEvent(Internal::LayoutEvents::LayoutEvent::LayoutValidationEnd, AbstractNodePointer(this)); // put to variable due to MULTI compile warning
        Layouter::GetEventSource().DispatchEvent(layoutEndEvent);
    }
#endif
    /******************************************************************************
     *  OnCompositeTransformChanged
     ******************************************************************************/
    void Node2D::OnCompositeTransformChanged()
    {
        InvalidateWorldTransformCacheTraverser2D iwtcTraverser;
        iwtcTraverser.Traverse(*this);

        NotifyListenersOnTransformChange(this);
    }

    /******************************************************************************
     *  OnAncestorTransformChanged
     ******************************************************************************/
    void Node2D::OnAncestorTransformChanged()
    {
        // no default action
    }

    /******************************************************************************
     *  SetParent
     ******************************************************************************/
    void Node2D::SetParent(Node2D* parent)
    {
        m_parent = parent;
        NotifyListenersOnNodeAdded(parent);
    }

    /******************************************************************************
     *  SetAlphaValue
     ******************************************************************************/
    void Node2D::SetAlphaValue(Float alphaValue)
    {
        if (Node2DDynamicProperties::SetAlphaValue(*this, alphaValue)) {

            InvalidateEffectiveAlphaValueCacheTraverser2D traverser;
            traverser.Traverse(*this);
        }
    }

    /******************************************************************************
     *  GetEffectiveAlphaValue
     ******************************************************************************/
    Candera::Float Node2D::GetEffectiveAlphaValue() const
    {
        if (IsAlphaValueCacheValid()) {
            return m_effectiveAlphaValueCache;
        }

        m_effectiveAlphaValueCache = GetAlphaValue();
        const Node2D* parent = GetParent();
        if (parent != 0) {
            m_effectiveAlphaValueCache *= parent->GetEffectiveAlphaValue();
        }

        m_isEffectiveAlphaValueCacheValid = true;
        return m_effectiveAlphaValueCache;
    }


    /******************************************************************************
     *  GetEffectiveBoundingRectangle
     ******************************************************************************/
    void Node2D::GetEffectiveBoundingRectangle(Rectangle& boundingRectangle, bool ignoreInvisible) const
    {
        if (Node2DDynamicProperties::IsBoundingRectangleSet(*this)) {
            boundingRectangle = GetBoundingRectangle();
        }
        else {
            GetComputedBoundingRectangle(boundingRectangle, ignoreInvisible);
        }
    }

    /******************************************************************************
     *  GetLayoutingRectangle
     ******************************************************************************/
#if defined(CANDERA_LAYOUT_ENABLED)

    const Rectangle& Node2D::GetLayoutingRectangle() const
    {
        return GetLayoutingRectangleDeprecated();
    }

    void Node2D::GetEffectiveLayoutingRectangle(Rectangle& rectangle) const
    {
        GetEffectiveLayoutingRectangleDeprecated(rectangle);
    }
    void Node2D::GetComputedLayoutingRectangle(Rectangle& rectangle) const
    {
        GetComputedLayoutRectangle(rectangle);
    }
    const Rectangle& Node2D::GetLayoutingRectangleDeprecated() const
    {
        return Node2DDynamicProperties::GetLayoutingRectangle(*this);
    }
    bool Node2D::IsLayoutingRectangleSetDeprecated() const
    {
        return Node2DDynamicProperties::IsLayoutingRectangleSet(*this);
    }
    void Node2D::GetEffectiveLayoutingRectangleDeprecated(Rectangle& rectangle, bool ignoreInvisible) const
    {
        if (IsLayoutingRectangleSetDeprecated()) {
            rectangle = GetLayoutingRectangleDeprecated();
        }
        else {
            GetComputedLayoutRectangle(rectangle);
        }
        FEATSTD_UNUSED(ignoreInvisible);
    }
#endif
    /******************************************************************************
     *  SetRenderOrderRank
     ******************************************************************************/
    void Node2D::SetRenderOrderRank(Int renderOrderRank)
    {
        if (renderOrderRank != GetRenderOrderRank()) {
            if (Node2DDynamicProperties::SetRenderOrderRank(*this, static_cast<Int16>(renderOrderRank))) {
                Scene2D* scene = GetScene();
                if (scene != 0) {
                    scene->GetRenderOrder().Invalidate();
                }
            }
        }
    }

    /******************************************************************************
     *  GetEffectiveRenderOrderRank
     ******************************************************************************/
    Int Node2D::GetEffectiveRenderOrderRank() const
    {
        Int effectiveRenderOrderRank = GetRenderOrderRank();
        const Node2D* parent = GetParent();
        while (parent != 0) {
            effectiveRenderOrderRank += parent->GetRenderOrderRank();
            parent = parent->GetParent();
        }

        return effectiveRenderOrderRank;
    }

    /******************************************************************************
     *  SetRenderBenchmark
     ******************************************************************************/
    void Node2D::SetRenderBenchmark(Float benchmark) {
        if (!Math::FloatAlmostEqual(benchmark, GetRenderBenchmark())) {
            Node2DDynamicProperties::SetRenderBenchmark(*this, benchmark);
        }
    }
//-
    /******************************************************************************
    *  AddNodeListener
    ******************************************************************************/

    bool Node2D::AddNodeListener(Node2DListener* listener){
        bool isSuccessful = false;
        if (listener != 0) {
            isSuccessful = m_Node2DListeners.Add(listener);
        }
        return isSuccessful;
    }

    /******************************************************************************
    *  RemoveNodeListener
    ******************************************************************************/
   
    bool Node2D::RemoveNodeListener(Node2DListener* listener){
        bool isSuccessful = false;
        if (listener != 0) {
            isSuccessful = m_Node2DListeners.Remove(listener);
        }
        return isSuccessful;
    }

    /******************************************************************************
    * OnNodeRemoved Event
    ******************************************************************************/
    class Node2D::OnNodeRemovedEvent : public Node2D::Node2DListenerContainer::Event {
        Node2D* m_node;
        Node2D* m_parent;
    public:
        OnNodeRemovedEvent(Node2D* parent, Node2D* node): m_node(node), m_parent(parent) {}
        virtual void Notify(Node2DListener* listener) override {
            listener->OnNodeRemoved(m_parent, m_node);
        }
    };

    void Node2D::NotifyListenersOnNodeRemoved(Node2D* parent){
        OnNodeRemovedEvent event(parent, this);
        m_Node2DListeners.Iterate(event);
    }

    /******************************************************************************
    * OnNodeAddedEvent Event
    ******************************************************************************/
    class Node2D::OnNodeAddedEvent : public Node2D::Node2DListenerContainer::Event {
        Node2D* m_parent;
        Node2D* m_node;
    public:
        OnNodeAddedEvent(Node2D* parent, Node2D* node): m_parent(parent), m_node(node) {}
        virtual void Notify(Node2DListener* listener) override {
            listener->OnNodeAdded(m_parent, m_node);
        }
    };

    void Node2D::NotifyListenersOnNodeAdded(Node2D* parent)
    {
        OnNodeAddedEvent event(parent, this);
        m_Node2DListeners.Iterate(event);
    }

    /******************************************************************************
    * OnChildAddedEvent Event
    ******************************************************************************/
    class Node2D::OnChildAddedEvent : public Node2D::Node2DListenerContainer::Event {
        Node2D* m_child;
        Node2D* m_node;
    public:
        OnChildAddedEvent(Node2D* child, Node2D* node): m_child(child), m_node(node) {}
        virtual void Notify(Node2DListener* listener) override {
            listener->OnChildAdded(m_child, m_node);
        }
    };

    void Node2D::NotifyListenersOnChildAdded(Node2D* child)
    {
        OnChildAddedEvent event(child, this);
        m_Node2DListeners.Iterate(event);
    }

    /******************************************************************************
    * OnChildRemovedEvent Event
    ******************************************************************************/
    class Node2D::OnChildRemovedEvent : public Node2D::Node2DListenerContainer::Event {
        Node2D* m_child;
        Node2D* m_node;
    public:
        OnChildRemovedEvent(Node2D* child, Node2D* node): m_child(child), m_node(node) {}
        virtual void Notify(Node2DListener* listener) override {
            listener->OnChildRemoved(m_child, m_node);
        }
    };

    void Node2D::NotifyListenersOnChildRemoved(Node2D* child)
    {
        OnChildRemovedEvent event(child, this);
        m_Node2DListeners.Iterate(event);
    }

    /******************************************************************************
    * OnTransformChange Event
    ******************************************************************************/
    class Node2D::OnTransformChangeEvent : public Node2D::Node2DListenerContainer::Event{
        Node2D* m_node;
    public:
        OnTransformChangeEvent(Node2D* node) : m_node(node) {}
        virtual void Notify(Node2DListener* listener) override{
            listener->OnTransformChange(m_node);
        }
    };

    void Node2D::NotifyListenersOnTransformChange(Node2D* node){
        OnTransformChangeEvent event(node);
        m_Node2DListeners.Iterate(event);
    }
    
 //-
    /******************************************************************************
     *  GetDescendant
     ******************************************************************************/
    Node2D* Node2D::GetDescendant(const Char* id)
    {
        FindByIdTraverser finder;
        finder.SetIdToFind(id);
        finder.SetRootNode(this);
        finder.Traverse(*this);
        return const_cast<Node2D*>(finder.GetMatchingNode());
    }

    const Node2D* Node2D::GetDescendant(const Char* id) const
    {
        FindByIdTraverser finder;
        finder.SetIdToFind(id);
        finder.SetRootNode(this);
        finder.Traverse(*this);
        return finder.GetMatchingNode();
    }

}   // namespace Candera
