//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Group2D.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Engine2D/Core/TreeTraverser2D.h>

#ifdef CANDERA_LAYOUT_ENABLED
#include <Candera/EngineBase/Layout/Layouter.h>
#endif
#include <Candera/Engine2D/Core/RenderNode.h>

namespace Candera {
    class NodeRectTraverser : public ConstTreeTraverser2D {
    public:
        typedef void (Node2D::*GetNodeRectFunction)(Rectangle&, bool) const;

        typedef bool (Node2D::*IsNodeRectSetFunction)() const;

        NodeRectTraverser(const Node2D& root, GetNodeRectFunction getNodeRectFunc, IsNodeRectSetFunction isNodeRectSetFunc, bool ignoreInvisible) :
            m_root(root),
            m_getNodeRectFunc(getNodeRectFunc),
            m_isNodeRectSetFunc(isNodeRectSetFunc),
            m_ignoreInvisible(ignoreInvisible)
        {
        }

        void GetNodeRect(Rectangle& rect) const
        {
            rect = m_nodeRect;
        }

    protected:
        virtual TraverserAction ProcessNode(const Node2D& node) override
        {
            if (m_ignoreInvisible && (!node.IsEffectiveRenderingEnabled())) {
                return StopTraversingForDescendants;
            }
            if ((!node.IsTypeOf(Group2D::GetTypeId())) || (node.*m_isNodeRectSetFunc)()) {
                Rectangle boundingRect;
                (node.*m_getNodeRectFunc)(boundingRect, m_ignoreInvisible);

                Matrix3x2 transform(node.GetCompositeTransform());
                const Node2D* parent = node.GetParent();
                while ((parent != 0) && (parent != &m_root)) {
                    transform *= parent->GetCompositeTransform();
                    parent = parent->GetParent();
                }
                boundingRect.Transform(transform);
#ifdef CANDERA_LAYOUT_ENABLED
                if (m_ignoreInvisible) {
#ifdef CANDERA_LAYOUT_CLIPPING_ENABLED

                    const RenderNode* renderNode = Dynamic_Cast<const RenderNode*>(&node);
                    if (0 != renderNode) {
                        boundingRect.Intersect(renderNode->GetClippingRect());
                    }
                    else {
#endif // CANDERA_LAYOUT_CLIPPING_ENABLED
                        boundingRect.Intersect(Layouter::GetClippingArea(node));
#ifdef CANDERA_LAYOUT_CLIPPING_ENABLED
                    }
#endif // CANDERA_LAYOUT_CLIPPING_ENABLED
                }
#endif
                m_nodeRect.Union(boundingRect);
            }

            return ProceedTraversing;
        }

    private:
        Rectangle m_nodeRect;
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1725, CANDERA_LINT_REASON_ASSOCIATION)
        const Node2D& m_root;
        GetNodeRectFunction m_getNodeRectFunc;
        IsNodeRectSetFunction m_isNodeRectSetFunc;
        bool m_ignoreInvisible;

        //Forbid assignment, make it private
        NodeRectTraverser& operator=(const NodeRectTraverser&);
    };

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    Group2D::Group2D() :
        Base()
    {
    }

    Group2D::Group2D(const Group2D& group) :
        Base(group)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    Group2D::~Group2D()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    Group2D* Group2D::Create()
    {
        return FEATSTD_NEW(Group2D);
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Group2D* Group2D::Clone() const
    {
        return FEATSTD_NEW(Group2D)(*this);
    }

    /******************************************************************************
     *  DisposeSelf
     ******************************************************************************/
    void Group2D::DisposeSelf()
    {
        FEATSTD_DELETE(this);
    }

    /******************************************************************************
     *  GetComputedBoundingRectangle
     ******************************************************************************/
    void Group2D::GetComputedBoundingRectangle(Rectangle& boundingRectangle, bool ignoreInvisible) const
    {
        NodeRectTraverser traverser(*this, &Node2D::GetEffectiveBoundingRectangle, &Node2D::IsBoundingRectangleSet, ignoreInvisible);
        traverser.Traverse(*this);
        traverser.GetNodeRect(boundingRectangle);
    }

    /******************************************************************************
     *  GetCalculatedBoundingRectangle
     ******************************************************************************/
#if defined(CANDERA_LAYOUT_ENABLED)
    void Group2D::GetComputedLayoutRectangle(Rectangle& rectangle) const
    {
        NodeRectTraverser traverser(*this, &Node2D::GetEffectiveLayoutingRectangleDeprecated, &Node2D::IsLayoutingRectangleSetDeprecated, false);
        traverser.Traverse(*this);
        traverser.GetNodeRect(rectangle);
    }
#endif

    FEATSTD_RTTI_DEFINITION(Group2D, Node2D)
}   // namespace Candera
