//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DEVICE_OBJECT_2D_LISTENER_H)
#define CANDERA_DEVICE_OBJECT_2D_LISTENER_H

#include <Candera/Environment.h>

namespace Candera {

    class DeviceObject2D;

    /** @addtogroup Core2D
     *  @{
     */

    /**
     * @brief Interface for DeviceObject2D listener implementations.
     *
     * DeviceObject2DListener receives notifications whenever the object that is
     *  listened to is uploaded, unloaded or destroyed.
     */
    class DeviceObject2DListener {
            friend class DeviceObject2D;
        public:
            virtual ~DeviceObject2DListener() {}

        protected:
            DeviceObject2DListener() {}

            /**
             * @brief Notification of DeviceObject2D start uploading.
             *
             * OnPreUpload notification is triggered by a DeviceObject2D before
             *  its data is uploaded in VRAM. If Upload is called on an already
             *  uploaded device object, no new notification will be triggered.
             * @param deviceObject The DeviceObject2D that triggered the
             *  notification.
             * @see DeviceObject2D::Upload
             */
            virtual void OnPreUpload(DeviceObject2D& deviceObject) = 0;

            /**
             * @brief Notification of DeviceObject2D finish uploading.
             *
             * OnPostUpload notification is triggered by a DeviceObject2D after
             *  its data is uploaded in VRAM. The notification is triggered even
             *  if upload fails. DeviceObject2D::IsUploaded() can be called to
             *  check the upload state.
             * @param deviceObject The DeviceObject2D that triggered the
             *  notification.
             * @see DeviceObject2D::Upload
             */
            virtual void OnPostUpload(DeviceObject2D& deviceObject) = 0;

            /**
             * @brief Notification of DeviceObject2D start unloading.
             *
             * OnPreUnload notification is triggered by a DeviceObject2D before
             *  its data is unloaded from VRAM. If Unload does not free VRAM,
             *  but only decrease the upload reference count, no notification
             *  will be triggered.
             * @param deviceObject The DeviceObject2D that triggered the
             *  notification.
             * @see DeviceObject2D::Unload
             */
            virtual void OnPreUnload(DeviceObject2D& deviceObject) = 0;

            /**
             * @brief Notification of DeviceObject2D finish unloading.
             *
             * OnPostUnload notification is triggered by a DeviceObject2D after
             *  its data is unloaded from VRAM. The notification is triggered
             *  even if unload fails. DeviceObject2D::IsUploaded() can be called
             *  to check the upload state.
             * @param deviceObject The DeviceObject2D that triggered the
             *  notification.
             * @see DeviceObject2D::Unload
             */
            virtual void OnPostUnload(DeviceObject2D& deviceObject) = 0;

            /**
             * @brief Notification of DeviceObject2D destroy.
             *
             * OnDestruction notification is triggered by a DeviceObject2D
             *  before it is destroyed.
             * @param deviceObject The DeviceObject2D that triggered the
             *  notification.
             */
            virtual void OnDestruction(DeviceObject2D& deviceObject) = 0;
        private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(DeviceObject2DListener);
    };
    /** @} */ // end of Core2D
}

#endif
