//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "DeviceObject2D.h"

#include <Candera/Engine2D/Core/DeviceObject2DListener.h>

namespace Candera {

    FEATSTD_RTTI_DEFINITION(DeviceObject2D, CanderaObject)

    DeviceObject2D::DeviceObject2D() :
        m_uploadReferenceCount(0),
        m_isDisposedAfterUpload(false)
    {
    }

    DeviceObject2D::~DeviceObject2D()
    {
        NotifyListenersOnDestruction();
    }

    bool DeviceObject2D::Upload(LoadingHint loadingHint)
    {
        if (m_uploadReferenceCount > 0) {
            ++m_uploadReferenceCount;
            return true; // object is already uploaded
        }

        NotifyListenersOnPreUpload();
        m_uploadReferenceCount = UploadInternal(loadingHint) ? 1 : 0;
        NotifyListenersOnPostUpload();

        if (IsUploaded() && (m_isDisposedAfterUpload == true)) {
            DisposeInternal();
        }

        return IsUploaded();
    }

    bool DeviceObject2D::Unload(LoadingHint loadingHint)
    {
        if (Force == loadingHint) {
            if (m_uploadReferenceCount == 0) {
                return true;
            }

            NotifyListenersOnPreUnload();
            bool result = UnloadInternal(NoHint);
            m_uploadReferenceCount = result ? 0 : m_uploadReferenceCount;
            NotifyListenersOnPostUnload();

            return result;
        }
        else {
            if (m_uploadReferenceCount > 1) {
                --m_uploadReferenceCount;
                return true; // no need to unload yet
            }
            else if (m_uploadReferenceCount < 1) {
                return false; // tried to unload a not uploaded object
            }
            else {
                NotifyListenersOnPreUnload();
                bool result = UnloadInternal(loadingHint);
                m_uploadReferenceCount = result ? 0 : 1;
                NotifyListenersOnPostUnload(); 
                return result;
            }
        } 
    }

    void DeviceObject2D::NotifyListenersOnPreUpload()
    {
        for (ListenerList::Iterator it = m_listenerList.Begin();
            it != m_listenerList.End();
            ++it) {
                (*it)->OnPreUpload(*this);
        }
    }

    void DeviceObject2D::NotifyListenersOnPostUpload()
    {
        for (ListenerList::Iterator it = m_listenerList.Begin();
            it != m_listenerList.End();
            ++it) {
                (*it)->OnPostUpload(*this);
        }
    }

    void DeviceObject2D::NotifyListenersOnPreUnload()
    {
        for (ListenerList::Iterator it = m_listenerList.Begin();
            it != m_listenerList.End();
            ++it) {
                (*it)->OnPreUnload(*this);
        }
    }

    void DeviceObject2D::NotifyListenersOnPostUnload()
    {
        for (ListenerList::Iterator it = m_listenerList.Begin();
            it != m_listenerList.End();
            ++it) {
                (*it)->OnPostUnload(*this);
        }
    }

    void DeviceObject2D::NotifyListenersOnDestruction()
    {
        for (ListenerList::Iterator it = m_listenerList.Begin();
            it != m_listenerList.End();
            ++it) {
                (*it)->OnDestruction(*this);
        }
    }

} // namespace Candera
