//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if  !defined(CANDERA_COMPOSITEGROUP2D_H)
#define CANDERA_COMPOSITEGROUP2D_H

#include <FeatStd/Container/Vector.h>
#include <Candera/System/Container/ForwardIterator.h>
#include <Candera/System/MetaInfo/PropertyMetaInfo.h>
#include <Candera/EngineBase/Animation/AnimationPlayerBase.h>
#include <Candera/EngineBase/Common/StringIdentifier.h>
#include <Candera/Engine2D/Core/Group2D.h>

namespace Candera {

    class WidgetBase;

    namespace Internal {
        class CompositeGroup2DFunctions;
    }

    /** @addtogroup Core2D
     *  @{
     */

    /**
     * @brief CompositeGroup2D is a Group2D that allows direct access to a 
     *  number of descendants, called 'anchor points'.
     *
     * CompositeGroup2Ds are useful at design time to easily access, attach, 
     *  detach or transform subtrees attached to the anchor points.
     */
    class CompositeGroup2D: public Group2D
    {
    public:
        /**
         * @brief Anchor point definition.
         *
         * An anchor point is a descendant of a CompositeGroup2D to which anchors
         *  (nodes) can be attached.
         */
        struct AnchorPoint {
            Id m_id;
            const Char* m_name;
            Node2D* m_node;
        };
    private:
        FEATSTD_TYPEDEF_BASE(Group2D);
        typedef FeatStd::Internal::Vector<WidgetBase*> WidgetList;
        typedef FeatStd::Internal::Vector<AnchorPoint> AnchorPointList;
        typedef FeatStd::Internal::Vector<Animation::AnimationPlayerBase::SharedPointer> AnimationList;
    public:
        FEATSTD_RTTI_DECLARATION();
        typedef ForwardIterator<AnimationList::ConstIterator> AnimationIterator;
        typedef ForwardIterator<AnchorPointList::ConstIterator> AnchorPointIterator;
        typedef ForwardIterator<WidgetList::ConstIterator> WidgetIterator;

        /**
         * Destructor.
         */
        virtual ~CompositeGroup2D() override {}
      
        // overrides Node::Clone
        virtual CompositeGroup2D* Clone() const override;

        /**
         * Retrieve a ForwardIterator that points to the first AnchorPoint.
         *
         * @return First AnchorPoint ForwardIterator.
         */
        AnchorPointIterator GetAnchorPointIterator() const;

        /**
         * Retrieve an anchor point node by name.
         *
         * @param name Name of the anchor point.
         * @return anchor point node if it exists.
         */
        Node2D* GetAnchorPointNodeByName(const Char* name) const;

        /**
         * Retrieve an anchor point node by id.
         *
         * @param id Id of the anchor point.
         * @return anchor point node if it exists.
         */
        Node2D* GetAnchorPointNode(Id id) const;

        /**
         * Retrieve a ForwardIterator that points to the first animation.
         *
         * @return First animation ForwardIterator.
         */
        AnimationIterator GetAnimationIterator() const;

        /**
         * Retrieve an animation by name.
         *
         * @param name Name of the animation.
         * @return animation with the given name.
         */
        Animation::AnimationPlayerBase::SharedPointer GetAnimationByName(const Char* name) const;

        /**
         * Retrieve an animation by id.
         *
         * @param id Id of the animation.
         * @return animation with the given id.
         */
        Animation::AnimationPlayerBase::SharedPointer GetAnimation(Id id) const;

        /**
         * Retrieve a ForwardIterator that points to the first widget.
         *
         * @return First widget ForwardIterator.
         */
        WidgetIterator GetWidgetIterator() const;

        /**
         * Retrieve an widget by name.
         *
         * @param name Name of the widget.
         * @return widget with the given name.
         */
        WidgetBase* GetWidgetByName(const Char* name) const;

        /**
         * Retrieve an widget by id.
         *
         * @param id Id of the widget.
         * @return widget with the given id.
         */
        WidgetBase* GetWidget(Id id) const;

        /**
         * Retrieve the string identifier.
         *
         * @return string identifier
         */
        const StringIdentifier* GetStringId() const { return &m_stringIdentifier; }

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        CompositeGroup2D() {};
        CompositeGroup2D(const CompositeGroup2D& other);
        CompositeGroup2D& operator = (const CompositeGroup2D& other);
        
        // overrides Node2D::DisposeSelf
        virtual void DisposeSelf() override;

        // overrides Node2D::OnDescendantRemoved
        virtual void OnDescendantRemoved(Node2D* descendant) override;

        /**
         * The anchor point Node2D must be a descendant of the CompositeGroup2D in
         *  order to be successfully defined.
         *
         * @param id Id of the anchor point.
         * @param name Name of the anchor point.
         * @param node Node2D to be marked as anchor point.
         * @return true if anchor point is defined.
         * @return false if anchor point could not be defined.
         */
        bool DefineAnchorPoint(Id id, const Char* name, Node2D* node);

        /**
         * Update AnchorPoint id.
         * 
         * @param oldId id of the anchor point of which the id should be changed.
         * @param newId new id for the anchor point.
         * @return true if anchor point with oldId is found and id is changed, false otherwise.
         */
        bool UpdateAnchorPointId(Id oldId, Id newId);

        /**
         * @param widget Widget to be associated with this CompositeGroup.
         * @return whether the widget was added successfully or not.
         */
        bool AddWidget(WidgetBase* widget);

        /**
         * @param animation AnimationPlayerBase to be associated with this CompositeGroup.
         * @return whether the animation was added successfully or not.
         */
        void AddAnimation(const Animation::AnimationPlayerBase::SharedPointer& animation);

    private:
        // The function for creating composite groups is internal.
        // See Internal::CompositeGroupFunctions.
        static CompositeGroup2D* Create();

        AnchorPointList m_anchorPointList;
        AnimationList m_animationList;
        WidgetList m_widgetList;
        StringIdentifier m_stringIdentifier;

        friend class Candera::Internal::CompositeGroup2DFunctions;
    };
    
/** @} */ // end of Core2D

    namespace Internal {
        class CompositeGroup2DFunctions {
        public:
            static CompositeGroup2D* Create() {
                return CompositeGroup2D::Create();
            }

            static bool DefineAnchorPoint(CompositeGroup2D& compositeGroup, Id id, const Char* name, Node2D* node) {
                return compositeGroup.DefineAnchorPoint(id, name, node);
            }
            
            static bool AddWidget(CompositeGroup2D& compositeGroup, WidgetBase* widget) {
                return compositeGroup.AddWidget(widget);
            }

            static void AddAnimation(CompositeGroup2D& compositeGroup, const Animation::AnimationPlayerBase::SharedPointer& animation) {
                compositeGroup.AddAnimation(animation);
            }

            static bool UpdateAnchorPointId(CompositeGroup2D& compositeGroup, Id oldId, Id newId) {
                return compositeGroup.UpdateAnchorPointId(oldId, newId);
            }

            static StringIdentifier& CompositeStringIdentifier(CompositeGroup2D& compositeGroup) {
                return compositeGroup.m_stringIdentifier;
            }
        };
    }
}

#endif //CANDERA_COMPOSITEGROUP2D_H

