//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CompositeGroup2D.h"

#include <CanderaWidget/WidgetBase/WidgetBase.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaEngine2D);
    FEATSTD_RTTI_DEFINITION(CompositeGroup2D, Group2D)

    static bool IsDescendant(const Node2D* parent, Node2D* child)
    {
        while ((child != 0) && (child != parent)) {
            child = child->GetParent();
        }

        return (child != 0);
    }

    void CompositeGroup2D::DisposeSelf()
    {
        FEATSTD_DELETE(this);
    }

    CompositeGroup2D* CompositeGroup2D::Clone() const
    {
        return FEATSTD_NEW(CompositeGroup2D)(*this);
    }

    CompositeGroup2D::CompositeGroup2D(const Candera::CompositeGroup2D &other) :
        Base(other)
    {
        m_anchorPointList.Clear();
        m_animationList.Clear();
        m_widgetList.Clear();
    }

    bool CompositeGroup2D::DefineAnchorPoint(Id id, const Char* name, Node2D* node)
    {
        bool isAnchorPointValid =
            ((name != 0) &&
             (node != 0) &&
             (IsDescendant(this, node)) &&
             (GetAnchorPointNode(id) == 0));

        if (!isAnchorPointValid) {
            return false;
        }

        AnchorPoint anchorEntry = {id, name, node};
        return m_anchorPointList.Add(anchorEntry);
    }

    void CompositeGroup2D::OnDescendantRemoved(Node2D* descendant)
    {
        for (Int index = static_cast<Int>(m_anchorPointList.Size()) - 1; index >= 0; --index) {
            if (IsDescendant(descendant, m_anchorPointList[index].m_node)) {
                static_cast<void>(m_anchorPointList.Remove(index));
            }
        }
    }

    CompositeGroup2D::AnchorPointIterator CompositeGroup2D::GetAnchorPointIterator() const
    {
        return AnchorPointIterator(m_anchorPointList.ConstBegin(), m_anchorPointList.ConstEnd());
    }

    Node2D* CompositeGroup2D::GetAnchorPointNodeByName(const Char* name) const
    {
        if (name == 0) {
            return 0;
        }

        for (AnchorPointIterator iterator = GetAnchorPointIterator(); iterator.IsValid(); ++iterator) {
            if (StringPlatform::CompareStrings(iterator->m_name, name) == 0) {
                return iterator->m_node;
            }
        }

        return 0;
    }

    Node2D* CompositeGroup2D::GetAnchorPointNode(Id id) const
    {
        for (AnchorPointIterator iterator = GetAnchorPointIterator(); iterator.IsValid(); ++iterator) {
            if (iterator->m_id == id) {
                return iterator->m_node;
            }
        }

        return 0;
    }

    CompositeGroup2D::AnimationIterator CompositeGroup2D::GetAnimationIterator() const
    {
        return AnimationIterator(m_animationList.ConstBegin(), m_animationList.ConstEnd());
    }

    Animation::AnimationPlayerBase::SharedPointer CompositeGroup2D::GetAnimationByName(const Char* name) const
    {
        Animation::AnimationPlayerBase::SharedPointer nullAnimation(0);

        if (name == 0) {
            return nullAnimation;
        }

        for (AnimationIterator iterator = GetAnimationIterator(); iterator.IsValid(); ++iterator) {
            if (StringPlatform::CompareStrings((*iterator)->GetName(), name) == 0) {
                return *iterator;
            }
        }

        return nullAnimation;
    }

    Animation::AnimationPlayerBase::SharedPointer CompositeGroup2D::GetAnimation(Id id) const
    {
        for (AnimationIterator iterator = GetAnimationIterator(); iterator.IsValid(); ++iterator) {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1058, const Candera::StringIdentifier &, False positive because GetId is from Candera::CanderaObject and not Candera::StringIdentifier.);
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(64, Candera::StringIdentifier = Candera::Animation::AnimationPlayerBase, False positive because GetId is from Candera::CanderaObject and not Candera::StringIdentifier.);
            if ((*iterator)->GetId() == id) {
                return *iterator;
            }
        }

        return Animation::AnimationPlayerBase::SharedPointer();
    }

    CompositeGroup2D::WidgetIterator CompositeGroup2D::GetWidgetIterator() const
    {
        return WidgetIterator(m_widgetList.ConstBegin(), m_widgetList.ConstEnd());
    }

    WidgetBase* CompositeGroup2D::GetWidgetByName(const Char* name) const
    {
        if (name == 0) {
            return 0;
        }

        for (WidgetIterator iterator = GetWidgetIterator(); iterator.IsValid(); ++iterator) {
            if (StringPlatform::CompareStrings((*iterator)->GetName(), name) == 0) {
                return *iterator;
            }
        }

        return 0;
    }

    WidgetBase* CompositeGroup2D::GetWidget(Id id) const
    {
        for (WidgetIterator iterator = GetWidgetIterator(); iterator.IsValid(); ++iterator) {
            if ((*iterator)->GetId() == id) {
                return *iterator;
            }
        }

        return 0;
    }

    CompositeGroup2D* CompositeGroup2D::Create()
    {
        return FEATSTD_NEW(CompositeGroup2D);
    }

    bool CompositeGroup2D::UpdateAnchorPointId(Id oldId, Id newId)
    {
        for (AnchorPointList::Iterator it = m_anchorPointList.Begin(); it != m_anchorPointList.End(); ++it) {
            if ((*it).m_id == oldId) {
                (*it).m_id = newId;
                return true;
            }
        }

        return false;
    }

    bool CompositeGroup2D::AddWidget(WidgetBase* widget)
    {
        widget->SetOwnerId(&m_stringIdentifier);
        return m_widgetList.Add(widget);
    }

    void CompositeGroup2D::AddAnimation(const Animation::AnimationPlayerBase::SharedPointer& animation)
    {
        animation->SetOwnerId(&m_stringIdentifier);
        static_cast<void>(m_animationList.Add(animation));
    }

}
