//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_BITMAP_IMAGE_2D_H)
#define CANDERA_BITMAP_IMAGE_2D_H

#include <Candera/Engine2D/Core/Image2D.h>
#include <FeatStd/Diagnostics/Measurable.h>

namespace Candera {
    /** @addtogroup Core2D
     *  @{
     */

    class Bitmap;
    class RenderDevice2D;
    /**
     * @brief  Supplies an Image2D interface to a Bitmap.
     */
    class BitmapImage2D : public Image2D, public FeatStd::Diagnostics::Measurable {

            FEATSTD_TYPEDEF_BASE(Image2D);
            friend class RenderDevice2D;
            
        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(BitmapImage2D);

            /**
             *  Define which memory pool to use.
             */
            enum MemoryPool {
                VideoMemory = 0,   ///< The buffer is in video memory.
                ClientMemory = 2   ///< The buffer is in client memory.
            };

            /**
             * Creates an instance of this class.
             * Use Dispose() to delete the instance and possible children, if any.
             * @return MemoryManagement::SharedPointer to the created object
             */
            static BitmapImage2D::SharedPointer Create();

            /**
             * Destructor
             */
            ~BitmapImage2D();

            /**
             * Assign a bitmap. If Upload was called, one must call Unload before
             * setting a new bitmap.
             * @param bitmap    Assigned bitmap.
             * @return          True if bitmap was uploaded, False if it's already uploaded.
             */
            bool SetBitmap(const Bitmap::SharedPointer& bitmap);

            /**
             * Retrieves the surface handle obtained upon uploading.
             * @return The surface handle to the uploaded data
             */
            virtual SurfaceHandle Get2DSurfaceHandle() const override {
                return m_surfaceHandle;
            }

            /**
             * Retrieves bitmap used by this BitmapImage object.
             * @return The Bitmap
             *
             */
            const Bitmap::SharedPointer& GetBitmap() const{
                return m_bitmap;
            }

            /**
             * Retrieves bitmap used by this BitmapImage object in case it is mutable
             * @return  The Bitmap if it is mutable (Bitmap::IsMutable() == true), otherwise NULL.
             */
            Bitmap::SharedPointer GetMutableBitmap();

            /**
             *  Defines the memory pool for the bitmap's data. See Bitmap::MemoryPool.
             *  @param  memoryPool      The memory pool for the bitmap.
             */
            void SetMemoryPool(MemoryPool memoryPool) {
                m_memoryPool = memoryPool;
            }

            /**
             *  Retrieves the memory pool for the bitmap's data. See Bitmap::MemoryPool.
             *  @return     The memory pool for the bitmap.
             */
            MemoryPool GetMemoryPool() const  {
                return static_cast<MemoryPool>(m_memoryPool);
            }

            /**VertexGeometryDisposerFn
             *  Get the attached surface Width.
             *  @return The width of the bitmap image
             */
            virtual UInt32 GetWidth() const override;

            /**
             *  Get the attached surface Height.
             *  @return The height of the bitmap image
             */
            virtual UInt32 GetHeight() const override;

            /**
             *  Get the size of of the bitmap.
             *  @return estimate of the video memory used by the bitmap.
             */
            virtual UInt GetSize() const;

            /**
             *  Defines if MipMap-chain shall be attached to Image2D.
             *  Call this function before Texture::Upload() to achieve desired effect, see return param for further details.
             *  @param   enableMipMapping defines if MipMap chain will be used or not when the image is uploaded.
             *  @return  true, if @param enableMipMapping is applied when the Texture will be uploaded.
             *  @return  false, if @param enableMipMapping is ignored, as Texture has already been uploaded.
             */
            bool SetMipMappingEnabled(bool enableMipMapping);

            // Override function from Image2D
            virtual bool IsMipMappingEnabled() const override { return m_isMipMappingEnabled; }

            /**
             *  Updates data within an already uploaded BitmapImage2D directly in VRAM.
             *  The source data is taken from the attached Bitmap in the BitmapImage2D.
             *  @return True if updating this BitmapTextureImage and setting every texture sub image succeeded. False otherwise.
             */
            bool Update();

            /**
             *  Updates data within an already uploaded BitmapImage2D directly in VRAM.
             *  @param xOffset     The x-Offset inside the Texture.
             *  @param yOffset     The y-Offset inside the Texture.
             *  @param width       The width to update. The width relates also to the width of the source data.
             *  @param height      The height to update. The height relates also to the height of the source data.
             *  @param data        The source data to update. The source data must have the same pack order and pack
             *                     alignment as the data supplied to RenderDevice2D::CreateBuffer, which was used to
             *                     create the surface.
             *  @return            True if updating every texture sub image succeeded. False if updating failed, texture isn't uploaded,
             *                     no data is passed.
             */
            bool Update(Int xOffset, Int yOffset, UInt width, UInt height, const UInt8* data) const;

            FEATSTD_RTTI_DECLARATION();
        protected:
            BitmapImage2D();

            virtual bool UploadInternal(LoadingHint loadingHint) override;
            virtual bool UnloadInternal(LoadingHint loadingHint) override;
            virtual void DisposeInternal() override;

        private:
            bool m_isMipMappingEnabled : 1;

            // UInt for enum. According to MISRA 9-6-2 Bit Fields must be explicitly signed or unsigned. MISRA 9-6-3: Bit-fields shall not have enum type.
            UInt m_memoryPool : 3;
            SurfaceHandle m_surfaceHandle;
            Bitmap::SharedPointer m_bitmap;

            void SetSurfaceHandle(SurfaceHandle surfaceHandle) { m_surfaceHandle = surfaceHandle; }
    };
    /** @} */ // end of Core2D
}

#endif //CANDERA_BITMAP_IMAGE_2D_H
