//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_Transformable2DRelativeTranslatePropertySetter_H)
#define CANDERA_Transformable2DRelativeTranslatePropertySetter_H

#include <Candera/Engine2D/AnimationPropertySetters/BaseTransformable2DRelativePropertySetter.h>
#include <Candera/System/Mathematics/Vector2.h>

namespace Candera { namespace Animation {

    /**
     *  @addtogroup AnimationPropertySetters2D
     *  @{
     */

    /**
     *  @brief Relative Translate AnimationPropertySetter.
     *
     *  The property setter applies a translation to the associated Transformable2D object, relative to the object translation
     *   cached at PropertySetter initialization time.
     */
    class Transformable2DRelativeTranslatePropertySetter: public BaseTransformable2DRelativePropertySetter {
        FEATSTD_TYPEDEF_BASE(BaseTransformable2DRelativePropertySetter);

        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(Transformable2DRelativeTranslatePropertySetter);

            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            FEATSTD_RTTI_DECLARATION();

            Transformable2DRelativeTranslatePropertySetter() : m_position(Vector2(0.0F, 0.0F)) {}
            virtual ~Transformable2DRelativeTranslatePropertySetter() override {}

            /**
             * Set the relative translation for the the Transformable2D object.
             *
             * @param value     Pointer to a float array that represents the offsets that needs to be added to
             *      the cached Transformable2D position value (value[0] for X axis, value[1] for Y axis).
             */
            virtual void Set(const Float* value) override;

            /**
             * Initialize the PropertySetter.
             *
             * The PropertySetter caches the position value of the current associated Transformable2D object, to
             *  be used by future Set calls. If an Transformable2D object is not associated at initialization time,
             *  the cached value is set to a default position of (0.0F, 0.0F).
             */
            virtual void Initialize() override;

        private:
            Vector2 m_position;
    };

    /**
     *  @brief Relative Translate AnimationPropertySetter on X axis.
     *
     *  The property setter applies a translation on the X axis to the associated Transformable2D object, relative
     *   to the object translation cached at PropertySetter initialization time.
     */
    class Transformable2DRelativeTranslateXPropertySetter : public BaseTransformable2DRelativePropertySetter {
            typedef BaseTransformable2DRelativePropertySetter Base;

        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(Transformable2DRelativeTranslateXPropertySetter);

            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            FEATSTD_RTTI_DECLARATION();

            Transformable2DRelativeTranslateXPropertySetter():m_positionX(0.0F) {}
            virtual ~Transformable2DRelativeTranslateXPropertySetter() override {}

            /**
             * Set the relative translation on the X axis for the the Transformable2D object.
             *
             * @param value     Pointer to a float value that represents the offset that needs to be added to
             *      the cached Transformable2D X axis position value.
             */
            virtual void Set(const Float* value) override;

            /**
             * Initialize the PropertySetter.
             *
             * The PropertySetter caches the position value on the X axis of the current associated Transformable2D object, to
             *  be used by future Set calls. If an Transformable2D object is not associated at initialization time,
             *  the cached value is set to a default position of 0.0F.
             */
            virtual void Initialize() override;

        private:
            Float m_positionX;
    };

    /**
     *  @brief Relative Translate AnimationPropertySetter on Y axis.
     *
     *  The property setter applies a translation on the Y axis to the associated Transformable2D object, relative
     *   to the object translation cached at PropertySetter initialization time.
     */
    class Transformable2DRelativeTranslateYPropertySetter : public BaseTransformable2DRelativePropertySetter {
            typedef BaseTransformable2DRelativePropertySetter Base;

        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(Transformable2DRelativeTranslateYPropertySetter);

            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            FEATSTD_RTTI_DECLARATION();

            Transformable2DRelativeTranslateYPropertySetter():m_positionY(0.0F) {}
            virtual ~Transformable2DRelativeTranslateYPropertySetter() override {}

            /**
             * Set the relative translation on the Y axis for the the Transformable2D object.
             *
             * @param value     Pointer to a float value that represents the offset that needs to be applied to
             *      the cached Transformable2D Y axis position value.
             */
            virtual void Set(const Float* value) override;

            /**
             * Initialize the PropertySetter.
             *
             * The PropertySetter caches the position value on the Y axis of the current associated Transformable2D object, to
             *  be used by future Set calls. If an Transformable2D object is not associated at initialization time,
             *  the cached value is set to a default position of 0.0F.
             */
            virtual void Initialize() override;

        private:
            Float m_positionY;

    };

    /**
     * @}
     */ // end of addtogroup AnimationPropertySetters2D
    } //namespace Animation

}   // namespace Candera

#endif  // CANDERA_Transformable2DRelativeTranslatePropertySetter_H
