/******************************************************************************
*
* FILE:          TunerControlDBusHandler.c
*   
* PROJECT:       Generic gateway
*
* DESCRIPTION:   Dbus server implemenatation for TunerControl
*
* AUTHOR:        BCP9KOR
*
* COPYRIGHT:     (c) 2016 Robert Bosch GmbH, Hildesheim
*
*****************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#include <gio/gio.h>
#include <stdlib.h>

#include "TunerControlDBusHandler.h"
#include "AutomDBusServer.h"
#include "Tuner_GenericClientInterface.h"
#include "ap-tuner-error-enums.h"
#include "ap-tuner-enums.h"

//Dlt
#include "dlt/dlt.h"
//Dlt Context
DLT_IMPORT_CONTEXT(AGW_Tuner);

GDBusMethodInvocation *g_invocation_save_preset = NULL;
GDBusMethodInvocation *g_invocation_tune = NULL;
GDBusMethodInvocation *g_invocation_cancel_search = NULL;
GDBusMethodInvocation *g_invocation_cancel_seek = NULL;
GDBusMethodInvocation *g_invocation_set_band = NULL;
GDBusMethodInvocation *g_invocation_set_config_settings = NULL;
GDBusMethodInvocation *g_invocation_set_frequency = NULL;
GDBusMethodInvocation *g_invocation_tune_to_preset = NULL;
GDBusMethodInvocation *g_invocation_tune_to_pi = NULL;


/******************************************************************************
 * Function:     handle_save_preset
 * Description:  Handler for save preset method
 * Parameters:   proxy object pointer, invocation pointer, index, band, freq,
                 programID, secondaryPI
 * Return:       bool
 *****************************************************************************/
gboolean handle_save_preset (TunerControl *object,
                             GDBusMethodInvocation *invocation,
                             guchar arg_index,
                             guchar arg_band,
                             guint arg_frequency,
                             guint arg_program_id,
                             guint arg_secondary_program_id)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+handle_save_preset"));

    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("index:"),DLT_INT(arg_index),
            DLT_STRING("band:"),DLT_INT(arg_band),DLT_STRING("freq:"),
            DLT_UINT(arg_band),
            DLT_STRING("pi:"),DLT_UINT(arg_program_id),DLT_STRING("secpi:"),
            DLT_UINT(arg_secondary_program_id));

    g_invocation_save_preset = g_object_ref(invocation);
    bSavePreset(arg_index, (ApTunerBand) arg_band, arg_frequency,
                arg_program_id, arg_secondary_program_id);

    return TRUE;
}                            

/******************************************************************************
 * Function:     handle_tune
 * Description:  Handler for tune method
 * Parameters:   proxy object pointer, invocation pointer, direction, step
 * Return:       bool
 *****************************************************************************/
gboolean handle_tune (TunerControl *object,
                      GDBusMethodInvocation *invocation,
                      guchar arg_direction,
                      guint arg_step_size)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+handle_tune"));
    g_invocation_tune = g_object_ref(invocation);

    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("direction:"),
            DLT_UINT8(arg_direction),
            DLT_STRING("step:"),DLT_UINT(arg_step_size));

    if(!bTuneWithFreqStep(arg_direction, arg_step_size))
    {
        vSendTuneError(AP_TUNER_ERROR_BAD_PARAMS);
    }

    return TRUE;
}                     

/******************************************************************************
 * Function:     handle_cancel_search_stations
 * Description:  Handler for cancel search method
 * Parameters:   proxy object pointer, invocation pointer
 * Return:       bool
 *****************************************************************************/
gboolean handle_cancel_search_stations (TunerControl *object,
                                        GDBusMethodInvocation *invocation)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,
            DLT_STRING("+handle_cancel_search_stations"));

    g_invocation_cancel_search = g_object_ref(invocation);
    bCancelSearch();
    return TRUE;
}                                       

/******************************************************************************
 * Function:     handle_cancel_seek
 * Description:  Handler for cancel seek method
 * Parameters:   proxy object pointer, invocation pointer
 * Return:       bool
 *****************************************************************************/
gboolean handle_cancel_seek (TunerControl *object,
                             GDBusMethodInvocation *invocation)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+handle_cancel_seek"));
    g_invocation_cancel_seek = g_object_ref(invocation);
    bCancelSeek();
    return TRUE;
}                             

/******************************************************************************
 * Function:     handle_search_available_stations
 * Description:  Handler for search available stations method
 * Parameters:   proxy object pointer, invocation pointer
 * Return:       bool
 *****************************************************************************/
gboolean handle_search_available_stations (TunerControl *object,
                                           GDBusMethodInvocation *invocation)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,
            DLT_STRING("+handle_search_available_stations"));

    vScan();
    tuner_control_complete_search_available_stations(object, invocation);
    return TRUE;
}                                           

/******************************************************************************
 * Function:     handle_seek
 * Description:  Handler for seek method
 * Parameters:   proxy object pointer, invocation pointer, direction
 * Return:       bool
 *****************************************************************************/
gboolean handle_seek (TunerControl *object,
                      GDBusMethodInvocation *invocation,
                      guchar arg_direction)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+handle_seek"));
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("direction:"),
            DLT_INT(arg_direction));

    if(!bSeekToNextStation(arg_direction))
    {
        g_dbus_method_invocation_return_error (invocation,
                                               AP_TUNER_ERROR,
                                               AP_TUNER_ERROR_BAD_PARAMS,
                                               "ERROR in seek");
    }
    else
    {
        tuner_control_complete_seek(object, invocation);
    }
    return TRUE;
}

/******************************************************************************
 * Function:     handle_set_band
 * Description:  Handler for set band method
 * Parameters:   proxy object pointer, invocation pointer, band
 * Return:       bool
 *****************************************************************************/
gboolean handle_set_band (TunerControl *object,
                          GDBusMethodInvocation *invocation,
                          guchar arg_band)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+handle_set_band"));
    g_invocation_set_band = g_object_ref(invocation);
    bSetBand(arg_band);
    return TRUE;
}

/******************************************************************************
 * Function:     handle_set_configuration_settings
 * Description:  Handler for set configuration settings method
 * Parameters:   proxy object pointer, invocation pointer, settings variant
 * Return:       bool
 *****************************************************************************/
gboolean handle_set_configuration_settings (TunerControl *object,
                                            GDBusMethodInvocation *invocation,
                                            GVariant *arg_settings)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,
            DLT_STRING("+handle_set_configuration_settings"));

    g_invocation_set_config_settings = g_object_ref(invocation);
    if(arg_settings == NULL)
    {
        return FALSE;
    }
    else
    {
        bSetConfigurationSettings(arg_settings);
    }
    return TRUE;
}

/******************************************************************************
 * Function:     handle_set_frequency
 * Description:  Handler for set frequency method
 * Parameters:   proxy object pointer, invocation pointer, band, frequency
 * Return:       bool
 *****************************************************************************/
gboolean handle_set_frequency (TunerControl *object,
                               GDBusMethodInvocation *invocation,
                               guchar arg_band,
                               guint arg_frequency)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+handle_set_frequency"));
    gboolean bRet;

    g_invocation_set_frequency = g_object_ref(invocation);
    bRet = bSetFrequency(arg_frequency,arg_band);

    if(bRet == FALSE)
    {
        vSendSetFreqError(AP_TUNER_ERROR_UNKOWN);
    }

    return TRUE;
}

/******************************************************************************
 * Function:     handle_tune_to_preset
 * Description:  Handler for tune to preset method
 * Parameters:   proxy object pointer, invocation pointer, preset element
 * Return:       bool
 *****************************************************************************/
gboolean handle_tune_to_preset (TunerControl *object,
                                GDBusMethodInvocation *invocation,
                                GVariant *arg_preset_element)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+handle_tune_to_preset"));
    g_invocation_tune_to_preset = g_object_ref(invocation);
    unsigned char eBand;
    unsigned char iIndex;
    g_variant_get (arg_preset_element, "(yy)", &iIndex, &eBand);

    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,
            DLT_STRING("handle_tune_to_preset index:"),
            DLT_INT(iIndex),DLT_STRING("band"),DLT_INT(eBand));

    bTuneToPreset(iIndex,(ApTunerBand)eBand);
    return TRUE;
}

/******************************************************************************
 * Function:     handle_tune_to_program_id
 * Description:  Handler for tune to program ID method
 * Parameters:   proxy object pointer, invocation pointer, programID
 * Return:       bool
 *****************************************************************************/
gboolean handle_tune_to_program_id (TunerControl *object,
                                    GDBusMethodInvocation *invocation,
                                    guint arg_program_id)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+handle_tune_to_program_id"));
    g_invocation_tune_to_pi = g_object_ref(invocation);
    return bTuneToPI(arg_program_id);
}                                    

/******************************************************************************
 * Function:     vSendSearchStatus
 * Description:  Emits search station signal
 * Parameters:   SearchStatus enum
 * Return:       void
 *****************************************************************************/
void vSendSearchStatus(ApTunerSearchStationStatus eSearchStatus)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendSearchStatus"));

    TunerControl* poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(poTunerControlProxyObj == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,DLT_STRING("Proxy object is null"));
        return;
    }
    else
    {
        tuner_control_emit_search_status(poTunerControlProxyObj,
                                         eSearchStatus);
    }
}

/******************************************************************************
 * Function:     vSendSeekStatus
 * Description:  Emits seek station signal
 * Parameters:   SeekStatus enum
 * Return:       void
 *****************************************************************************/
void vSendSeekStatus(ApTunerSeekStatus eSeekStatus)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendSeekStatus"));

    TunerControl* poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(poTunerControlProxyObj == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,DLT_STRING("Proxy object is null"));
        return;
    }
    else
    {
        tuner_control_emit_seek_status (poTunerControlProxyObj,
                                        eSeekStatus);
    }
}

/******************************************************************************
 * Function:     vSendSavePresetResult
 * Description:  Send method complete for save preset
 * Parameters:   void
 * Return:       void
 *****************************************************************************/
void vSendSavePresetResult()
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendSavePresetResult"));
    TunerControl* poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(poTunerControlProxyObj == NULL || g_invocation_save_preset == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Proxy object or invocation pointer is null"));
    }
    else
    {
        tuner_control_complete_save_preset (poTunerControlProxyObj,
                                            g_invocation_save_preset);
    }
	
	g_clear_object (&g_invocation_save_preset);

}

/******************************************************************************
 * Function:     vSendTuneResult
 * Description:  Send method complete for tune
 * Parameters:   void
 * Return:       void
 *****************************************************************************/
void vSendTuneResult()
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendTuneResult"));
    TunerControl* poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(poTunerControlProxyObj == NULL || g_invocation_tune == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Proxy object or invocation pointer is null"));
    }
    else
    {
        tuner_control_complete_tune (poTunerControlProxyObj,
                                     g_invocation_tune);
    }
	
	g_clear_object( &g_invocation_tune);
}

/******************************************************************************
 * Function:     vSendCancelSearchStnResult
 * Description:  Send method complete for cancel search
 * Parameters:   void
 * Return:       void
 *****************************************************************************/
void vSendCancelSearchStnResult()
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendCancelSearchStnResult"));
    TunerControl* poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(poTunerControlProxyObj == NULL || g_invocation_cancel_search == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Proxy object or invocation pointer is null"));
    }
    else
    {
        tuner_control_complete_cancel_search_stations (
                    poTunerControlProxyObj,
                    g_invocation_cancel_search);
    }
	
	g_clear_object( &g_invocation_cancel_search);

}

/******************************************************************************
 * Function:     vSendCancelSeekResult
 * Description:  Send method complete for cancel seek
 * Parameters:   void
 * Return:       void
 *****************************************************************************/
void vSendCancelSeekResult()
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendCancelSeekResult"));
    TunerControl* poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(poTunerControlProxyObj == NULL || g_invocation_cancel_seek == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Proxy object or invocation pointer is null"));
    }
    else
    {
        tuner_control_complete_cancel_seek (
                    poTunerControlProxyObj,
                    g_invocation_cancel_seek);
    }
	g_clear_object( &g_invocation_cancel_seek);
}

/******************************************************************************
 * Function:     vSendSetBandResult
 * Description:  Send method complete for set band
 * Parameters:   void
 * Return:       void
 *****************************************************************************/
void vSendSetBandResult()
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendSetBandResult"));
    TunerControl* poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(poTunerControlProxyObj == NULL || g_invocation_set_band == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Proxy object or invocation pointer is null"));
    }
    else
    {
        tuner_control_complete_set_band (
                    poTunerControlProxyObj,
                    g_invocation_set_band);
    }
	
	g_clear_object( &g_invocation_set_band);
}

/******************************************************************************
 * Function:     vSendSetConfigResult
 * Description:  Send method complete for set configuration
 * Parameters:   void
 * Return:       void
 *****************************************************************************/
void vSendSetConfigResult()
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendSetConfigResult"));
    TunerControl* poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(poTunerControlProxyObj == NULL ||
            g_invocation_set_config_settings == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Proxy object or invocation pointer is null"));
    }
    else
    {
        tuner_control_complete_set_configuration_settings (
                    poTunerControlProxyObj,
                    g_invocation_set_config_settings);
    }
	g_clear_object( &g_invocation_set_config_settings);
}

/******************************************************************************
 * Function:     vSendSetFreqResult
 * Description:  Send method complete for set frequency
 * Parameters:   void
 * Return:       void
 *****************************************************************************/
void vSendSetFreqResult()
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendSetFreqResult"));
    TunerControl* poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(poTunerControlProxyObj == NULL || g_invocation_set_frequency == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Proxy object or invocation pointer is null"));
    }
    else
    {
        tuner_control_complete_set_frequency (
                    poTunerControlProxyObj,
                    g_invocation_set_frequency);
    }
	
	g_clear_object( &g_invocation_set_frequency);
}

/******************************************************************************
 * Function:     vSendTuneToPresetResult
 * Description:  Send method complete for tune to preset
 * Parameters:   void
 * Return:       void
 *****************************************************************************/
void vSendTuneToPresetResult()
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendTuneToPresetResult"));
    TunerControl* poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(poTunerControlProxyObj == NULL || g_invocation_tune_to_preset == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Proxy object or invocation pointer is null"));
    }
    else
    {
        tuner_control_complete_tune_to_preset (
                    poTunerControlProxyObj,
                    g_invocation_tune_to_preset);
    }
	g_clear_object( &g_invocation_tune_to_preset);
}

/******************************************************************************
 * Function:     vSendTuneToPIResult
 * Description:  Send method complete for tune to PI
 * Parameters:   void
 * Return:       void
 *****************************************************************************/
void vSendTuneToPIResult()
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendTuneToPIResult"));
    TunerControl* poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(poTunerControlProxyObj == NULL || g_invocation_tune_to_pi == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Proxy object or invocation pointer is null"));
    }
    else
    {
        tuner_control_complete_tune_to_program_id (
                    poTunerControlProxyObj,
                    g_invocation_tune_to_pi);
    }
	g_clear_object( &g_invocation_tune_to_pi);
}

/******************************************************************************
 * Function:     vSendTuneToPIResult
 * Description:  Send method complete for tune to PI
 * Parameters:   Error enum, char* Error msg
 * Return:       void
 *****************************************************************************/
void vSendSavePresetError(APTunerError eError, char* sErrMsg)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendSavePresetError"));
    if(g_invocation_save_preset == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Invocation pointer is null"));
        return;
    }
    else
    {
        if(sErrMsg != NULL)
        {
            g_dbus_method_invocation_return_error (g_invocation_save_preset,
                                                   AP_TUNER_ERROR,
                                                   (int)eError,
                                                   sErrMsg);
        }
        else
        {
            g_dbus_method_invocation_return_error (g_invocation_save_preset,
                                                   AP_TUNER_ERROR,
                                                   (int)eError,
                                                   "ERROR in save preset");
        }
		g_clear_object( &g_invocation_save_preset);
    }
}

/******************************************************************************
 * Function:     vSendTuneToPIResult
 * Description:  Send method complete for tune to PI
 * Parameters:   Error enum
 * Return:       void
 *****************************************************************************/
void vSendTuneError(APTunerError eError)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendTuneError"));
    if(g_invocation_tune == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Invocation pointer is null"));
        return;
    }
    else
    {
        g_dbus_method_invocation_return_error (g_invocation_tune,
                                               AP_TUNER_ERROR,
                                               (int)eError,
                                               "ERROR in Tune");
	    g_clear_object( &g_invocation_tune);
    }
}

/******************************************************************************
 * Function:     vSendCancelSearchStnError
 * Description:  Send method error for tune to cancel search station
 * Parameters:   Error enum
 * Return:       void
 *****************************************************************************/
void vSendCancelSearchStnError(APTunerError eError)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendCancelSearchStnError"));
    if(g_invocation_cancel_search == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Invocation pointer is null"));
        return;
    }
    else
    {
        g_dbus_method_invocation_return_error (g_invocation_cancel_search,
                                               AP_TUNER_ERROR,
                                               (int)eError,
                                               "ERROR in Cancel Search");
	    g_clear_object( &g_invocation_cancel_search);
    }
}

/******************************************************************************
 * Function:     vSendCancelSeekError
 * Description:  Send method error for tune to cancel seek
 * Parameters:   Error enum
 * Return:       void
 *****************************************************************************/
void vSendCancelSeekError(APTunerError eError)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendCancelSeekError"));
    if(g_invocation_cancel_seek == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Invocation pointer is null"));
        return;
    }
    else
    {
        g_dbus_method_invocation_return_error (g_invocation_cancel_seek,
                                               AP_TUNER_ERROR,
                                               (int)eError,
                                               "ERROR in Cancel Seek");
	    g_clear_object( &g_invocation_cancel_seek);
    }
}

/******************************************************************************
 * Function:     vSendSetBandError
 * Description:  Send method error for set band
 * Parameters:   Error enum
 * Return:       void
 *****************************************************************************/
void vSendSetBandError(APTunerError eError)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendSetBandError"));
    if(g_invocation_set_band == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Invocation pointer is null"));
        return;
    }
    else
    {
        g_dbus_method_invocation_return_error (g_invocation_set_band,
                                               AP_TUNER_ERROR,
                                               (int)eError,
                                               "ERROR in set band");
	    g_clear_object( &g_invocation_set_band);
    }
}

/******************************************************************************
 * Function:     vSendSetConfigError
 * Description:  Send method error for set configuration
 * Parameters:   Error enum
 * Return:       void
 *****************************************************************************/
void vSendSetConfigError(APTunerError eError)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendSetConfigError"));
    if(g_invocation_set_config_settings == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Invocation pointer is null"));
        return;
    }
    else
    {
        g_dbus_method_invocation_return_error (g_invocation_set_config_settings,
                                               AP_TUNER_ERROR,
                                               (int)eError,
                                               "ERROR in set config");
	    g_clear_object( &g_invocation_set_config_settings);
    }
}

/******************************************************************************
 * Function:     vSendSetFreqError
 * Description:  Send method error for set frequency
 * Parameters:   Error enum
 * Return:       void
 *****************************************************************************/
void vSendSetFreqError(APTunerError eError)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendSetFreqError"));
    if(g_invocation_set_frequency == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Invocation pointer is null"));
        return;
    }
    else
    {
        g_dbus_method_invocation_return_error (g_invocation_set_frequency,
                                               AP_TUNER_ERROR,
                                               (int)eError,
                                               "ERROR in set frequency");
		g_clear_object( &g_invocation_set_frequency);
    }
}

/******************************************************************************
 * Function:     vSendTuneToPresetError
 * Description:  Send method error for tune to preset
 * Parameters:   Error enum
 * Return:       void
 *****************************************************************************/
void vSendTuneToPresetError(APTunerError eError)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendTuneToPresetError"));
    if(g_invocation_tune_to_preset == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Invocation pointer is null"));
        return;
    }
    else
    {
        g_dbus_method_invocation_return_error (g_invocation_tune_to_preset,
                                               AP_TUNER_ERROR,
                                               (int)eError,
                                               "ERROR in tune to preset");
		g_clear_object(&g_invocation_tune_to_preset);
    }
}

/******************************************************************************
 * Function:     vSendTuneToPIError
 * Description:  Send method error for tune to PI error
 * Parameters:   Error enum
 * Return:       void
 *****************************************************************************/
void vSendTuneToPIError(APTunerError eError)
{
    DLT_LOG(AGW_Tuner,DLT_LOG_INFO,DLT_STRING("+vSendTuneToPIError"));
    if(g_invocation_tune_to_pi == NULL)
    {
        DLT_LOG(AGW_Tuner,DLT_LOG_ERROR,
                DLT_STRING("Invocation pointer is null"));
        return;
    }
    else
    {
        g_dbus_method_invocation_return_error (g_invocation_tune_to_pi,
                                               AP_TUNER_ERROR,
                                               (int)eError,
                                               "ERROR in tune to PI");
		g_clear_object(&g_invocation_tune_to_pi);
    }
}
#endif //#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC