/******************************************************************************
*
* FILE:          TTSDBusHandler.c
*   
* PROJECT:       Generic gateway
*
* DESCRIPTION:   Dbus server implemenatation for TTS
*
* AUTHOR:        
*
* COPYRIGHT:     (c) 2015 Robert Bosch GmbH, Hildesheim
*
*****************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#include <gio/gio.h>
#include <stdlib.h>


#include "TTSDBusHandler.h"
#include "AutomDBusServer.h"
#include "TextToSpeech_ClientInterface.h"
#include "ap-tts-error-enums.h"
#include "ap-tts-enums.h"

//Dlt
#include "dlt/dlt.h"
//Dlt Context
DLT_IMPORT_CONTEXT(AGW_TTS);


static guint32 g_iSessionId = 0;
static GDBusMethodInvocation *g_invocation_get_tts_state = NULL;
static GDBusMethodInvocation *g_invocation_req_tts = NULL;
static GDBusMethodInvocation *g_invocation_cancel = NULL;

static TextToSpeech*     TTSProxyObj = NULL;

/*******************************************************************************
 * Function:    TTS
 * Description: Gets the TTS proxy object
 * Parameters:
 * Return:
 ******************************************************************************/

static TextToSpeech* poGetTTSProxyObj()
{
    if(TTSProxyObj == NULL)
    {
        TTSProxyObj = text_to_speech_skeleton_new ();
    }

    return TTSProxyObj;
}

/******************************************************************************
* Function:     handle_request_textto_speech
* Description:  Handler for request text to speech method call
* Parameters:   TextToSpeech object,invocation pointer,text_to_speak,priority
                Language,VoiceRate,Volume,VoicePitch,VoiceCategory
* Return:       bool
*****************************************************************************/

gboolean
handle_request_text_to_speech (TextToSpeech *object,
                               GDBusMethodInvocation *invocation,
                               const gchar *arg_text_to_speak,
                               guchar arg_priority,
                               const gchar *arg_language,
                               gint arg_voice_rate,
                               gint arg_volume,
                               gint arg_voice_pitch,
                               guchar arg_voice_category)
{

    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("TTS:handle_request_textto_speech"));
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("Text = "),
            DLT_STRING(arg_text_to_speak));
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("Rate = "),
            DLT_INT(arg_voice_rate));
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("volume = "),
            DLT_INT(arg_volume));


    g_invocation_req_tts = g_object_ref (invocation);

    gboolean bRet = FALSE;
    bRet = bStartTextToSpeech(arg_text_to_speak,arg_priority,arg_language,
                              arg_voice_rate,arg_volume,arg_voice_pitch,
                              arg_voice_category);

    if(!bRet)
    {
        vRequestTextToSpeechStateMethodError(AP_TTS_ERROR_UNKNOWN);
    }

    return TRUE;
}

/******************************************************************************
* Function:    vRequestTextToSpeechStateMethodResult
* Description: request text to speech method result call back 
* Parameters:  void
* Return:      void
******************************************************************************/
void vRequestTextToSpeechStateMethodResult()
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,
            DLT_STRING("TTS:vRequestTextToSpeechStateMethodResult"));
    g_iSessionId = g_random_int ();
    TextToSpeech* l_poTTSProxyObj =  poGetTTSProxyObj();
    if(g_invocation_req_tts != NULL && l_poTTSProxyObj != NULL)
    {
        text_to_speech_complete_request_text_to_speech (
                    l_poTTSProxyObj,
                    g_invocation_req_tts,
                    g_iSessionId);
        g_clear_object (&g_invocation_req_tts);
    }
    else
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                    "TTS:request invocation pointer or proxy object is null"));
    }
}

/******************************************************************************
* Function:    vRequestTextToSpeechStateMethodError
* Description: request text to speech method error call back
* Parameters:  state  
* Return:      void
******************************************************************************/
void vRequestTextToSpeechStateMethodError(ApTtsError eState)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,
            DLT_STRING("TTS:vRequestTextToSpeechStateMethodError"));
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,
            DLT_STRING("Error code sent :"),DLT_INT(eState));
    if(g_invocation_req_tts != NULL)
    {
        g_dbus_method_invocation_return_error (g_invocation_req_tts,
                                               AP_TTS_ERROR,
                                               (int)eState,
                                               "ERROR in request TTS");
        g_clear_object (&g_invocation_req_tts);
    }
    else
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                    "TTS:request invocation pointer is null"));
    }
}

/******************************************************************************
* Function:    handle_cancel
* Description: Handler for cancel method
* Parameters:  TextToSpeech object,invocation pointer,SessionId
* Return:      bool
*******************************************************************************/
gboolean handle_cancel (TextToSpeech *object,
                        GDBusMethodInvocation *invocation,
                        guint arg_session_id)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("TTS:handle_cancel"));
    if(arg_session_id == g_iSessionId && g_iSessionId != 0)
    {
        g_invocation_cancel = g_object_ref (invocation);
        if(!bCancel())
        {
            vCancelMethodError(AP_TTS_ERROR_UNKNOWN);
        }
    }
    else
    {
        g_dbus_method_invocation_return_error (invocation,
                                               AP_TTS_ERROR,
                                               AP_TTS_ERROR_INVALID_PARAMETERS,
                                               "SessionId is not valid");
    }
    return TRUE;
}  


/******************************************************************************
* Function:    vCancelMethodResult
* Description: Cancel method result call back
* Parameters:  void 
* Return:      void
******************************************************************************/
void vCancelMethodResult()
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("TTS:vCancelMethodResult"));

    TextToSpeech* l_poTTSProxyObj =  poGetTTSProxyObj();

    if(g_invocation_cancel != NULL && l_poTTSProxyObj != NULL)
    {
        text_to_speech_complete_cancel (l_poTTSProxyObj,
                                        g_invocation_cancel);
        g_clear_object (&g_invocation_cancel);
    }
    else
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                    "TTS:cancel invocation pointer or Proxy Object is null"));
    }
}

/******************************************************************************
* Function:    vCancelMethodError
* Description: Cancel method error call back
* Parameters:  ApTtsError (enum)
* Return:      void
******************************************************************************/
void vCancelMethodError(ApTtsError eState)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("TTS:vCancelMethodError"));
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("Error code sent :"),
            DLT_INT(eState));

    if(g_invocation_cancel != NULL)
    {
        g_dbus_method_invocation_return_error (g_invocation_cancel,
                                               AP_TTS_ERROR,
                                               eState,
                                               "ERROR in Cancel TTS");
        g_clear_object (&g_invocation_cancel);
    }
    else
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                    "TTS:cancel invocation pointer is null"));
    }
}


/******************************************************************************
* Function:    handle_get_text_to_speech_state
* Description: Handler for get_textto_speech_state method
* Parameters:  TextToSpeech object,invocation object,SessionId
* Return:      bool
******************************************************************************/
gboolean
handle_get_text_to_speech_state (TextToSpeech *object,
                                 GDBusMethodInvocation *invocation,
                                 guint arg_session_id)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,
            DLT_STRING("TTS:handle_get_textto_speech_state"));
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,
            DLT_STRING("SessionId = "),DLT_UINT(arg_session_id));

    if(arg_session_id == g_iSessionId && g_iSessionId != 0)
    {
        g_invocation_get_tts_state = g_object_ref (invocation);
        if(!bGetTextToSpeechStateMethodStart())
        {
            g_dbus_method_invocation_return_error (
                        g_invocation_get_tts_state,
                        AP_TTS_ERROR,
                        AP_TTS_ERROR_INVALID_PARAMETERS,
                        "Invalid parameter recieved by component");

            g_clear_object (&g_invocation_get_tts_state);

        }
    }
    else
    {
        g_dbus_method_invocation_return_error (invocation,
                                               AP_TTS_ERROR,
                                               AP_TTS_ERROR_INVALID_PARAMETERS,
                                               "SessionId is not valid");
    }
    return TRUE;
}

/******************************************************************************
* Function:    vGetTextToSpeechStateMethodResult
* Description: GetTextToSpeechState method result call back
* Parameters:  ApTtsState (enum)
* Return:      void
******************************************************************************/
void vGetTextToSpeechStateMethodResult(ApTtsState eState)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,
            DLT_STRING("TTS:vGetTextToSpeechStateMethodResult"));

    TextToSpeech* l_poTTSProxyObj =  poGetTTSProxyObj();
    if(g_invocation_get_tts_state != NULL && l_poTTSProxyObj != NULL)
    {
        text_to_speech_complete_get_text_to_speech_state (
                    l_poTTSProxyObj,
                    g_invocation_get_tts_state,
                    eState);
        g_clear_object (&g_invocation_get_tts_state);
    }
    else
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,
                DLT_STRING("TTS:invocation or proxy object is null"));
    }

}

/******************************************************************************
* Function:    bNotifyTextToSpeechState
* Description: emits notify_textto_speech
* Parameters:  enum ApTtsState 
* Return:      bool
*******************************************************************************/
bool
bNotifyTextToSpeechState (ApTtsState eState)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("TTS:send_notify_textto_speech"));

    bool bRet = TRUE;
    TextToSpeech* l_poTTSProxyObj =  poGetTTSProxyObj();
    if(g_iSessionId != 0 && l_poTTSProxyObj != NULL)
    {
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_UINT(g_iSessionId));

        text_to_speech_emit_notify_text_to_speech_state(
                    l_poTTSProxyObj,
                    g_iSessionId,
                    eState);
    }
    else
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,
                DLT_STRING("TTS:SessionID is invalid or proxy object is null"));
        bRet = FALSE;
    }

    if(eState == AP_TTS_STATE_FINISHED || eState == AP_TTS_STATE_CANCELLED)
    {
        g_iSessionId = 0;
    }
    return bRet;

}

/******************************************************************************
* Function:    vExportAGWTTSInterface
* Description: function to expose TTS interface
* Parameters:  void
* Return:      void
******************************************************************************/
void vExportAGWTTSInterface()
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("TTS:vExportAGWTTSInterface"));
    GDBusConnection* poConn = poGetGDBusConnection();
    GError *error = NULL;

    TextToSpeech* l_poTTSProxyObj =  poGetTTSProxyObj();

    if(l_poTTSProxyObj == NULL)
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                    "TTS : - Proxy Object is NULL. Export Failed"));
    }

    if(!g_dbus_interface_skeleton_export (
                G_DBUS_INTERFACE_SKELETON ((TextToSpeech*)poGetTTSProxyObj()),
                poConn,
                "/com/bosch/AutomotiveProxy/TextToSpeech",
                &error))
    {
        if(error != NULL)
        {
            DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                        "TTS : - ERROR-- Interface skeleton export failed - "),
                    DLT_STRING(error->message));
            g_clear_error (&error);
        }
        return;
    }
    else
    {
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                    "TTS : - Object created & exported"));
    }

    /* signal connection for TTS */
    g_signal_connect (l_poTTSProxyObj,
                      "handle-request-text-to-speech",
                      G_CALLBACK (handle_request_text_to_speech), NULL);

    g_signal_connect (l_poTTSProxyObj,
                      "handle-cancel",
                      G_CALLBACK (handle_cancel), NULL);

    g_signal_connect (l_poTTSProxyObj,
                      "handle-get-text-to-speech-state",
                      G_CALLBACK (handle_get_text_to_speech_state), NULL);
	
	g_signal_connect ((TextToSpeech*)poGetTTSProxyObj(),
                       "handle-set-preferences",
                       G_CALLBACK (handle_set_preferences), NULL);
					  
					  
}

/******************************************************************************
* Function:    vUnexportAGWTTSInterface
* Description: function to unexport TTS interface
* Parameters:  void
* Return:      void
******************************************************************************/
void vUnexportAGWTTSInterface()
{
    DLT_LOG(AGW_TTS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

    TextToSpeech* l_poTTSProxyObj =  poGetTTSProxyObj();
    GDBusConnection* poConn = poGetGDBusConnection();
	
    if(l_poTTSProxyObj != NULL)
    {
		
		if(g_dbus_interface_skeleton_has_connection
                               (G_DBUS_INTERFACE_SKELETON (l_poTTSProxyObj),
							    poConn))
		{
            g_dbus_interface_skeleton_unexport(
                    G_DBUS_INTERFACE_SKELETON (l_poTTSProxyObj));
		}
		else
		{
			DLT_LOG(AGW_TTS, DLT_LOG_INFO,DLT_STRING(
			        "Interface is not exported on connection"));
		}
		
    }

    DLT_LOG(AGW_TTS, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));

}

void
vUninitializeTTS()
{
	DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING( "\n"),
	                          DLT_STRING( __FUNCTION__));
    if(TTSProxyObj != NULL)
           g_clear_object (&TTSProxyObj);
}

/******************************************************************************
* Function:    handle_set_preferences
* Description: function to handle preferences
* Parameters:  TextToSpeech,invocation object,preferences
* Return:      void
******************************************************************************/
gboolean  handle_set_preferences(TextToSpeech *object, 
								GDBusMethodInvocation *invocation,
								GVariant *arg_preferences)
{
	DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("Entered set_preferences"));
    GVariantIter iter;
    GVariant *value = NULL;
    gchar *key = NULL;
	gboolean voice_rate_set = TRUE;
    gboolean volume_set = TRUE;
	gboolean typecheck_volume = FALSE;
	gboolean typecheck_voice_rate = FALSE;
	gint voice_rate;
	gint volume;
    g_variant_iter_init(&iter,arg_preferences);
	
	while (g_variant_iter_next(&iter,"{sv}",&key,&value))
    {
        //supported range is 50-400
        if(!g_strcmp0(key,"voice_rate"))
        {
			
			if(g_variant_is_of_type(value,G_VARIANT_TYPE_INT32))
			{
			g_variant_get(value,"i",&voice_rate);
		    typecheck_voice_rate = TRUE;
			}
			else
			{
			typecheck_voice_rate = FALSE;
			}	
		}	
		if(!g_strcmp0(key,"volume"))
        {				
			if(g_variant_is_of_type(value,G_VARIANT_TYPE_INT32))
			{
			g_variant_get(value,"i",&volume);		
			typecheck_volume = TRUE;	
			}
			else
			{
			typecheck_volume = FALSE;
			}
		}
		    
      g_variant_unref (value);
      g_free(key);
	  
	   
	}
	if(!(typecheck_volume && typecheck_voice_rate))
	   {
		    g_dbus_method_invocation_return_error(invocation,
                           AP_TTS_ERROR,
                           AP_TTS_ERROR_INVALID_PARAMETERS,
                		 "Type of message does not match expected type '(i)'");
			DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("Data Type Error "));
			return TRUE;								   
	   }
	
     if(!(voice_rate == (gint)AP_TTS_VOICE_RATE_SLOW || 
		  voice_rate ==  (gint)AP_TTS_VOICE_RATE_NORMAL|| 
		  voice_rate ==  (gint)AP_TTS_VOICE_RATE_FAST))
            
             {
                voice_rate_set=FALSE;
             }
        
        
            if(!(volume>=0  &&  volume<= 100))
            {
               volume_set =FALSE;
            }
			
	DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("preferred volume = "),
                                    DLT_INT(volume));	 
	DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("voice rate "),
                                    DLT_INT(voice_rate));
    

	  
									
	 if( volume_set==FALSE &&  voice_rate_set==FALSE)
            {
				g_dbus_method_invocation_return_error(invocation,
                                               AP_TTS_ERROR,
                                               AP_TTS_ERROR_INVALID_PARAMETERS,
                                               "invalid parameters");
				DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("Invalid parameters"));							   
				return TRUE;
			}
    else if(volume_set==FALSE)
	 {
		 g_dbus_method_invocation_return_error(invocation,
                                               AP_TTS_ERROR,
                                               AP_TTS_ERROR_INVALID_PARAMETERS,
                                               "volume out of range");
				DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("Volume out of range"));							   
				return TRUE;							   
		 
	 }
	else if(voice_rate_set==FALSE)
	 {
		 g_dbus_method_invocation_return_error(invocation,
                                               AP_TTS_ERROR,
                                               AP_TTS_ERROR_INVALID_PARAMETERS,
                                               "speech rate not supported");
				DLT_LOG(AGW_TTS,DLT_LOG_INFO,
										DLT_STRING("Speech rate out of range"));								   
				return TRUE;
	 }
	 else
	 {
		 vPreferences(voice_rate,volume); 
	 }	 
	DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("Left set_preferences"));
	text_to_speech_complete_set_preferences( (TextToSpeech*)poGetTTSProxyObj(),invocation);
	g_clear_object (&invocation);
	return TRUE;
 }
#endif//VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC








