/*****************************************************************************
* FILE:         FC_Gateway_TTSClientHandler.h
* PROJECT:      G3G project
* SW-COMPONENT: generic gateway
* DESCRIPTION:  Enables TTS session
*               CCA client handler for TTS
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2016 Robert Bosch GmbH, Hildesheim
*****************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#define SDS_FI_S_IMPORT_INTERFACE_SDS_TTSFI_TYPES
#define SDS_FI_S_IMPORT_INTERFACE_SDS_TTSFI_FUNCTIONIDS
#define SDS_FI_S_IMPORT_INTERFACE_SDS_TTSFI_ERRORCODES
#define SDS_FI_S_IMPORT_INTERFACE_SDS_TTSFI_SERVICEINFO


#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE

#define FI_S_IMPORT_INTERFACE_FI_VERSION
#include "fi_msgfw_if.h"

#define READ_MESSAGE_ALOUD_CONTROL_SEQUENCE           "\x1B\\tn=sms\\ "

/*****************************************************************************/
/* INCLUDES                                                                  */
/*****************************************************************************/
#include "FC_Gateway_AudioRouteHandler.h"
#include "TextToSpeech_ClientInterface.h"
#include "procgenericgateway_tclMainApp.h"

#include <sstream> // for ostringstream

//Dlt
#include "dlt/dlt.h"
//Dlt Context
DLT_DECLARE_CONTEXT(AGW_TTS); //TTS

BEGIN_MSG_MAP(FC_Gateway_TTSClientHandler, ahl_tclBaseWork)

/*SynthesizeSpeak*/
ON_MESSAGE_SVCDATA(SDS_TTSFI_C_U16_SYNTHESIZESPEAK,
                   AMT_C_U8_CCAMSG_OPCODE_ERROR,
                   vHandleSynthesizeSpeak)

ON_MESSAGE_SVCDATA(SDS_TTSFI_C_U16_SYNTHESIZESPEAK,
                   AMT_C_U8_CCAMSG_OPCODE_METHODRESULTFIRST,
                   vHandleSynthesizeSpeak)

ON_MESSAGE_SVCDATA(SDS_TTSFI_C_U16_SYNTHESIZESPEAK,
                   AMT_C_U8_CCAMSG_OPCODE_METHODRESULTMIDDLE,
                   vHandleSynthesizeSpeak)

ON_MESSAGE_SVCDATA(SDS_TTSFI_C_U16_SYNTHESIZESPEAK,
                   AMT_C_U8_CCAMSG_OPCODE_METHODRESULTLAST,
                   vHandleSynthesizeSpeak)

/*Abort Prompt*/
ON_MESSAGE_SVCDATA(SDS_TTSFI_C_U16_ABORTPROMPT,
                   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,
                   vHandleAbortPrompt)

ON_MESSAGE_SVCDATA(SDS_TTSFI_C_U16_ABORTPROMPT,
                   AMT_C_U8_CCAMSG_OPCODE_ERROR,
                   vHandleAbortPrompt)

/*Prepare voice output*/
ON_MESSAGE_SVCDATA(SDS_TTSFI_C_U16_PREPAREVOICEOUTPUT,
                   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,
                   vHandlePrepareVoiceOutput)

ON_MESSAGE_SVCDATA(SDS_TTSFI_C_U16_PREPAREVOICEOUTPUT,
                   AMT_C_U8_CCAMSG_OPCODE_ERROR,
                   vHandlePrepareVoiceOutput)

/*TTS STATUS*/
ON_MESSAGE_SVCDATA(SDS_TTSFI_C_U16_TTSSTATUS,
                   AMT_C_U8_CCAMSG_OPCODE_STATUS,
                   vHandleTTsStatus)

ON_MESSAGE_SVCDATA(SDS_TTSFI_C_U16_TTSSTATUS, 
                   AMT_C_U8_CCAMSG_OPCODE_ERROR,
                   vHandleTTsStatus)

END_MSG_MAP()

FC_Gateway_TTSClientHandler* 
FC_Gateway_TTSClientHandler::m_poFC_Gateway_TTSClientHandler = 0;

tclSpeechProperty* tclSpeechProperty::m_potclSpeechProperty = 0;

/* ---------------- */
/* public functions */
/* ---------------- */

/******************************************************************************
*
* FUNCTION:     getInstance(procgenericgateway_tclMainApp* poMainAppl)
*
* DESCRIPTION:  Gets pointer to Client handler object
*
* PARAMETER:    Main application pointer
*
* RETURNVALUE:  FC_Gateway_TTSClientHandler object instance.
*
******************************************************************************/
FC_Gateway_TTSClientHandler* 
FC_Gateway_TTSClientHandler::getInstance(procgenericgateway_tclMainApp* 
                                         poMainAppl)
{

    if(m_poFC_Gateway_TTSClientHandler == NULL)
    {
        m_poFC_Gateway_TTSClientHandler =
                new FC_Gateway_TTSClientHandler(poMainAppl);

        DLT_REGISTER_CONTEXT(AGW_TTS,"GWTTS","TextToSpeech"
                             "context for DLT Logging");
    }

    return m_poFC_Gateway_TTSClientHandler;
}

/******************************************************************************
*
* FUNCTION:     getInstance(procgenericgateway_tclMainApp* poMainAppl)
*
* DESCRIPTION:  Gets pointer to tclSpeechProperty class object
*
* PARAMETER:    none
*
* RETURNVALUE:  tclSpeechProperty object instance.
*
******************************************************************************/
tclSpeechProperty* tclSpeechProperty::getInstance()
{
    if(m_potclSpeechProperty == NULL)
    {
        m_potclSpeechProperty = new tclSpeechProperty();
    }

    return m_potclSpeechProperty;
}

/******************************************************************************
*
* FUNCTION:     getInstance(procgenericgateway_tclMainApp* poMainAppl)
*
* DESCRIPTION:  Constructor for tclSpeechProperty
*
******************************************************************************/
tclSpeechProperty::tclSpeechProperty()
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "tclSpeechProperty::tclSpeechProperty() entered."));
    m_sText.clear();
    m_sLanguage.clear();
    m_cPriority = '\0';
    m_iVoiceRate = 0;
    m_iVolume = 0;
    m_iVoicePitch = 0;
    m_cVoiceCategory = '\0';
    m_iPreferedVolume=999;
    m_iPreferedVoiceRate=999;
}

/******************************************************************************
*
* FUNCTION:     getInstance(procgenericgateway_tclMainApp* poMainAppl)
*
* DESCRIPTION:  Destructor for tclSpeechProperty
*
******************************************************************************/
tclSpeechProperty::~tclSpeechProperty()
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "tclSpeechProperty::tclSpeechProperty() entered."));
    delete m_potclSpeechProperty;
    m_potclSpeechProperty = NULL;
}

/******************************************************************************
*
* FUNCTION:     FC_Gateway_TTSClientHandler::FC_Gateway_TTSClientHandler
*
* DESCRIPTION:  Parameterized constructor for FC_Gateway_TTSClientHandler
*
******************************************************************************/
FC_Gateway_TTSClientHandler::FC_Gateway_TTSClientHandler(
        procgenericgateway_tclMainApp* poMainAppl)
    :ahl_tclBaseOneThreadClientHandler(
         poMainAppl,
         CCA_C_U16_SRV_SDS_TTS,
         SDS_TTSFI_C_U16_SERVICE_MAJORVERSION,
         SDS_TTSFI_C_U16_SERVICE_MINORVERSION)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "FC_Gateway_TTSClientHandler:: FC_Gateway_TTSClientHandler() entered."));
    m_poMainApp = (procgenericgateway_tclMainApp*) poMainAppl;
    m_eCurrentTTSStatus = AP_TTS_STATE_IDLE;
    m_oTTSStatus.TTSStatus.enType = sds_fi_tcl_e8_TTSStatus::FI_EN_IDLE;
    vAddAutoRegisterForProperty(SDS_TTSFI_C_U16_TTSSTATUS);
}

/******************************************************************************
*
* FUNCTION:     FC_Gateway_TTSClientHandler::~FC_Gateway_TTSClientHandler
*
* DESCRIPTION:  destructor for FC_Gateway_TTSClientHandler
*
******************************************************************************/
FC_Gateway_TTSClientHandler::~FC_Gateway_TTSClientHandler()
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "FC_Gateway_TTSClientHandler::~FC_Gateway_TTSClientHandler() entered."));
    vRemoveAutoRegisterForProperty(SDS_TTSFI_C_U16_TTSSTATUS);
    DLT_UNREGISTER_CONTEXT(AGW_TTS);
    if(m_poFC_Gateway_TTSClientHandler!= NULL)
    {
        delete m_poFC_Gateway_TTSClientHandler;
        m_poFC_Gateway_TTSClientHandler = NULL;
    }
}

/*******************************************************************************
*
* FUNCTION: tVoid FC_Gateway_TTSClientHandler::vOnServiceAvailable()
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              this client-handler has registered for has become available.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceAvailable().
*******************************************************************************/
tVoid FC_Gateway_TTSClientHandler::vOnServiceAvailable()
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("vOnServiceAvailable() entered."));

    vExportAGWTTSInterface();
}

/*******************************************************************************
*
* FUNCTION: tVoid FC_Gateway_TTSClientHandler::vOnServiceUnavailable()
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              this client-handler has registered for has become unavailable.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceUnavailable().
*******************************************************************************/
tVoid FC_Gateway_TTSClientHandler::vOnServiceUnavailable()
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("vOnServiceUnavailable() entered."));

    vUnexportAGWTTSInterface();

}

/*******************************************************************************
*
* FUNCTION:     FC_Gateway_TTSClientHandler::bPrepareVoiceOutput_MethodStart()
*
* DESCRIPTION:  Handles in Posting message to PrepareVoiceOutput Method Start
*
* PARAMETER:    None.
*
* RETURNVALUE:  tBool.
*
*******************************************************************************/

tBool FC_Gateway_TTSClientHandler::bPrepareVoiceOutput_MethodStart()
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "FC_Gateway_TTSClientHandler::bPrepareVoiceOutput_MethodStart"));

    sds_ttsfi_tclMsgPrepareVoiceOutputMethodStart oNewFiDataObject;
    sds_fi_tclString Device;
    tBool bResult = FALSE;

    oNewFiDataObject.Device =  Device;
    fi_tclVisitorMessage oVisitorMessage(oNewFiDataObject,
                                         SDS_TTSFI_C_U16_SERVICE_MAJORVERSION);


    vInitServiceData(oVisitorMessage, 0,
                     SDS_TTSFI_C_U16_PREPAREVOICEOUTPUT,
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

    ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

    if (bIfServiceAvailable())
    {
        enResult = m_poMainApp->enPostMessage(&oVisitorMessage, TRUE);
    }
    else
    {
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                    "bPrepareVoiceOutput_MethodStart(): Service unavailable"));
    }

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                    "bPrepareVoiceOutput_MethodStart() :"
                    "Unable to post PREPAREVOICEOUTPUT_METHODSTART msg"));
        if (!oVisitorMessage.bDelete())
        {
            DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                        "bPrepareVoiceOutput_MethodStart() : Message deletion failed."));
        }
    }
    else
    {
        bResult = TRUE;
    }
    oNewFiDataObject.vDestroy();

    return bResult;
}

/******************************************************************************
*
* FUNCTION:    tVoid  FC_Gateway_TTSClientHandler::bSynthesizeSpeak_MethodStart
*              (poInternalDataObj)
*
* DESCRIPTION: Handles in Posting message to SynthesizeSpeak Method Start
*
* PARAMETER:   std::string Text Content
*
* RETURNVALUE: tBool.
*
*******************************************************************************/

tBool 
FC_Gateway_TTSClientHandler::bSynthesizeSpeak_MethodStart(
        std::string sTextToSpeak)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,
            DLT_STRING("FC_Gateway_TTSClientHandler::"
                       "bSynthesizeSpeak_MethodStart sTextToSpeak ::"),
            DLT_STRING( sTextToSpeak.c_str()));

    tBool bResult = FALSE;

    sds_fi_tclString szText;

    szText.bSet(sTextToSpeak.c_str(), sds_fi_tclString::FI_EN_UTF8);

    DLT_LOG(AGW_TTS,DLT_LOG_INFO,
            DLT_STRING("FC_Gateway_TTSClientHandler::"
                       "bSynthesizeSpeak_MethodStart szText ::"),
            DLT_STRING( szText.szValue));

    sds_ttsfi_tclMsgSynthesizeSpeakMethodStart oNewFiDataObject;
    sds_fi_tcl_e8_TextStyleOption e8TextStyleOption;

    e8TextStyleOption.enType =  sds_fi_tcl_e8_TextStyleOption::FI_EN_PLAIN;

    oNewFiDataObject.TextAndTags.bSet(szText.szValue,
                                      sds_fi_tclString::FI_EN_UTF8);
    oNewFiDataObject.TextStyle = e8TextStyleOption; //Plain/SMS?
    oNewFiDataObject.AudioOutputLocation.u8Value = 255;
    oNewFiDataObject.ExpDate = 0x0000; //Same as in SDS
    oNewFiDataObject.StatusInfo.enType =
            sds_fi_tcl_e8_StatusInfoOption::FI_EN_NOINFO;
    oNewFiDataObject.LastPackage = TRUE; //This is the only job

    fi_tclVisitorMessage
            oVisitorMessage(oNewFiDataObject,
                            SDS_TTSFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oVisitorMessage, 0,
                     SDS_TTSFI_C_U16_SYNTHESIZESPEAK,
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

    ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

    if (bIfServiceAvailable())
    {
        enResult = m_poMainApp->enPostMessage(&oVisitorMessage, TRUE);
    }
    else
    {
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                    "bSynthesizeSpeak_MethodStart() Service unavailable"));
    }

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                    "bSynthesizeSpeak_MethodStart() : Unable to post"
                    " SYNTHESIZESPEAK_METHODSTART message."));
        if (!oVisitorMessage.bDelete())
        {
            DLT_LOG(AGW_TTS,DLT_LOG_INFO,
                    DLT_STRING("bSynthesizeSpeak_MethodStart()"
                               ": Message deletion failed."));
        }
    }
    else
    {
        bResult = TRUE;
    }
    oNewFiDataObject.vDestroy();

    return bResult;
}


/******************************************************************************
*
* FUNCTION:     tVoid  FC_Gateway_TTSClientHandler::bAbortPrompt_MethodStart
*
* DESCRIPTION:  Handles in Posting message to AbortPrompt Method Start 
*
* PARAMETER:    Pointer to the Message
*
* RETURNVALUE:  None.
*
******************************************************************************/
tBool FC_Gateway_TTSClientHandler::bAbortPrompt_MethodStart()
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("FC_Gateway_TTSClientHandler::"
                                            "bAbortPrompt_MethodStart"));

    sds_ttsfi_tclMsgAbortPromptMethodStart oNewFiDataObject;
    tBool bResult = FALSE;

    oNewFiDataObject.AbortPromptOption.enType =
            sds_fi_tcl_e8_PromptAbortOption::FI_EN_IMMEDIATE;

    fi_tclVisitorMessage
            oVisitorMessage(oNewFiDataObject,
                            SDS_TTSFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oVisitorMessage, 0,
                     SDS_TTSFI_C_U16_ABORTPROMPT,
                     AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

    ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

    if (bIfServiceAvailable())
    {
        enResult = m_poMainApp->enPostMessage(&oVisitorMessage, TRUE);
    }
    else
    {
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                    "bAbortPrompt_MethodStart() Service unavailable"));
    }

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                    "bAbortPrompt_MethodStart() :  Unable to"
                    " post ABORTPROMPT_METHODSTART message."));
        if (!oVisitorMessage.bDelete())
        {
            DLT_LOG(AGW_TTS,DLT_LOG_ERROR,
                    DLT_STRING("bAbortPrompt_MethodStart() :"
                               " Message deletion failed."));
        }
    }
    else
    {
        bResult = TRUE;
    }
    oNewFiDataObject.vDestroy();

    return bResult;
}

/******************************************************************************
*
* FUNCTION:     tVoid FC_Gateway_TTSClientHandler:: vHandlePrepareVoiceOutput
*               (amt_tclServiceData* poMessage)
*
* DESCRIPTION:  Handler function to process the arrival of 
*               PrepareVoiceOutput Method Result.
*
* PARAMETER:    Pointer to the Message.
*
* RETURNVALUE:  void
*
******************************************************************************/
tVoid 
FC_Gateway_TTSClientHandler::vHandlePrepareVoiceOutput(amt_tclServiceData* 
                                                       poMessage)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "FC_Gateway_TTSClientHandler::vHandlePrepareVoiceOutput"));

    tU8 u8OpCode = poMessage->u8GetOpCode();

    switch (u8OpCode)
    {
    case AMT_C_U8_CCAMSG_OPCODE_ERROR:
    {
        amt_tclServiceDataError oErrorMsg(poMessage);
        tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,
                DLT_STRING("vHandlePrepareVoiceOutput - "
                           "PrepareVoiceOutput Failed!! Error ::"),
                DLT_INT( u16ErrorCode));
        switch(u16ErrorCode)
        {
        case SDS_TTSFI_C_U16_ERROR_ENGINENOTINITIALIZED:
            //error codes from sds_tts_fi
            vRequestTextToSpeechStateMethodError(
                        AP_TTS_ERROR_TTS_ENGINE_NOT_STARTED);
            break;
        default:
            vRequestTextToSpeechStateMethodError(AP_TTS_ERROR_UNKNOWN);
        }
    }
        break;

    case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
    {

        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("vHandlePrepareVoiceOutput - "
                                                "PrepareVoiceOutput Success!!"));
        tclSpeechProperty* poSpeechProperty = tclSpeechProperty::getInstance();
        std::ostringstream sTags;



        if(poSpeechProperty->iGetVoiceRate() == AP_TTS_VOICE_RATE_SLOW && poSpeechProperty->bCheckVolumeRange())
        {
            DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("voice rate slow"));
            sTags << "\x1b\\vol=" << poSpeechProperty->iGetVolume()
                  <<"\\ "<< "\x1b\\rate=" <<50<<"\\ ";
        }
        else if(poSpeechProperty->iGetVoiceRate() == AP_TTS_VOICE_RATE_NORMAL && poSpeechProperty->bCheckVolumeRange())
        {
            DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("voice rate normal"));
            sTags << "\x1b\\vol=" << poSpeechProperty->iGetVolume()
                  <<"\\ "<< "\x1b\\rate=" <<100<<"\\ ";
        }
        else if(poSpeechProperty->iGetVoiceRate() == AP_TTS_VOICE_RATE_FAST && poSpeechProperty->bCheckVolumeRange())
        {
            DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("voice rate fast"));
            sTags << "\x1b\\vol=" << poSpeechProperty->iGetVolume()
                  <<"\\ "<< "\x1b\\rate=" <<150<<"\\ ";
        }
        else
        {

            if(poSpeechProperty->iGetPreferedVolume()==999
                    && poSpeechProperty->iGetPreferedVoiceRate()==999)
            {
                vRequestTextToSpeechStateMethodError( AP_TTS_PREFERENCES_NOT_SET);

                DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("preferences are not set error"));
                fc_gateway_tclAudioRouteHandler *poAudioClientHandlerObj =
                        procgenericgateway_tclMainApp::poGetInstance()->poGetAudioRouteHandler();
                poAudioClientHandlerObj->bAudioReleaseRequest(TTS_AUDIO_SOURCE);
                break;
            }
            else if(poSpeechProperty->iGetPreferedVolume()==999
                    && poSpeechProperty->iGetPreferedVoiceRate()!=999)
            {
                sTags << "\x1b\\vol=" <<poSpeechProperty->iGetVolume()
                      <<"\\ "<< "\x1b\\rate=" <<
                        poSpeechProperty->iGetPreferedVoiceRate()<<"\\ ";
                DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("preferences are taken for  voice rate"));
            }
            else if(poSpeechProperty->iGetPreferedVolume()!=999
                    && poSpeechProperty->iGetPreferedVoiceRate()==999)
            {
                sTags << "\x1b\\vol=" <<poSpeechProperty->iGetPreferedVolume()
                      <<"\\ "<< "\x1b\\rate=" <<0<<"\\ ";
                DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("preferences are taken for volume"));
            }
            else
            {
                sTags << "\x1b\\vol=" <<poSpeechProperty->iGetPreferedVolume()
                      <<"\\ "<< "\x1b\\rate=" <<
                        poSpeechProperty->iGetPreferedVoiceRate()<<"\\ ";
                DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("preferences are taken for  voice rate & volume "));
            }

            
        }
        std::string prefix(READ_MESSAGE_ALOUD_CONTROL_SEQUENCE);
        std::string l_sText;
        l_sText = prefix+sTags.str()+poSpeechProperty->sGetText();
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,
                DLT_STRING("Calling "
                           "bSynthesizeSpeak_MethodStart with text as "),
                DLT_STRING(l_sText.c_str()) );
        bSynthesizeSpeak_MethodStart(l_sText);
    }
        break;

    default:
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,
                DLT_STRING("vHandlePrepareVoiceOutput(): "
                           " Invalid OpCode =  received"),
                DLT_INT( poMessage->u8GetOpCode()));
    }
    }
}



/******************************************************************************
*
* FUNCTION:     FC_Gateway_TTSClientHandler:: vHandleSynthesizeSpeak
*               (amt_tclServiceData* poMessage)
*
* DESCRIPTION:  Handler function to process the arrival of SynthesizeSpeak 
*               Method Result 
*
* PARAMETER:    Pointer to the Message.
*
* RETURNVALUE:  void
*
******************************************************************************/
tVoid 
FC_Gateway_TTSClientHandler::vHandleSynthesizeSpeak(amt_tclServiceData* 
                                                    poMessage)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("FC_Gateway_TTSClientHandler::"
                                            "vHandleSynthesizeSpeak"));

    tU8 u8OpCode = poMessage->u8GetOpCode();

    switch (u8OpCode)
    {
    case AMT_C_U8_CCAMSG_OPCODE_ERROR :
    {
        if(m_eCurrentTTSStatus != AP_TTS_STATE_CANCELLED)
        {
            bNotifyTextToSpeechState(AP_TTS_STATE_FAILED);
            m_eCurrentTTSStatus = AP_TTS_STATE_FAILED;
        }
        amt_tclServiceDataError oErrorMsg(poMessage);
        tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,
                DLT_STRING("vHandleSynthesizeSpeak - "
                           "SynthesizeSpeak Failed!! Error Code ::"),
                DLT_INT( u16ErrorCode));

        switch(u16ErrorCode)
        {
        case SDS_TTSFI_C_U16_ERROR_TRANSMITTEDSTRINGTOOBIG:
            vRequestTextToSpeechStateMethodError(
                        AP_TTS_ERROR_TEXT_LENGTH_EXCEEDED);
            break;
        case SDS_TTSFI_C_U16_ERROR_NOSHAREDMEMORYAVAILABLE:
        case  SDS_TTSFI_C_U16_ERROR_NOTENOUGHMEMORYAVAILABLE:
            vRequestTextToSpeechStateMethodError(
                        AP_TTS_ERROR_OUT_OF_MEMORY);
            break;
        default:
            vRequestTextToSpeechStateMethodError(
                        AP_TTS_ERROR_SPEECH_PLAY_FAILED);
            fc_gateway_tclAudioRouteHandler *poAudioClientHandlerObj =
                    procgenericgateway_tclMainApp::poGetInstance()->poGetAudioRouteHandler();
            poAudioClientHandlerObj->bAudioReleaseRequest(TTS_AUDIO_SOURCE);
            break;
        }

    }
        break;

    case AMT_C_U8_CCAMSG_OPCODE_METHODRESULTFIRST :
    {
        sds_ttsfi_tclMsgSynthesizeSpeakMethodResultFirst oFiDataObject;
        fi_tclVisitorMessage oVisitorMsg(poMessage);

        if (oVisitorMsg.s32GetData(oFiDataObject,
                                   SDS_TTSFI_C_U16_SERVICE_MAJORVERSION)
                != OSAL_ERROR)
        {
            sds_fi_tcl_e8_JobStatus
                    oJobStatus = oFiDataObject.JobProgress.JobStatus;

            DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                        "vHandleSynthesizeSpeak JobStatus ::"),
                    DLT_INT( oJobStatus.enType));

            vRequestTextToSpeechStateMethodResult();
        }
        oFiDataObject.vDestroy();
    }
        break;

    case AMT_C_U8_CCAMSG_OPCODE_METHODRESULTMIDDLE :
    {
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                    "vHandleSynthesizeSpeak MIDDLE"));
        sds_ttsfi_tclMsgSynthesizeSpeakMethodResultMiddle oFiDataObject;
        fi_tclVisitorMessage oVisitorMsg(poMessage);

        if (oVisitorMsg.s32GetData(oFiDataObject,
                                   SDS_TTSFI_C_U16_SERVICE_MAJORVERSION)
                != OSAL_ERROR)
        {
            sds_fi_tcl_e8_JobStatus
                    oJobStatus = oFiDataObject.JobProgress.JobStatus;
            DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                        "vHandleSynthesizeSpeak JobStatus ::"),
                    DLT_INT(oJobStatus.enType));
            
        }
        oFiDataObject.vDestroy();
    }
        break;

    case AMT_C_U8_CCAMSG_OPCODE_METHODRESULTLAST :
    {
        sds_ttsfi_tclMsgSynthesizeSpeakMethodResultLast oFiDataObject;
        fi_tclVisitorMessage oVisitorMsg(poMessage);

        if (oVisitorMsg.s32GetData(oFiDataObject,
                                   SDS_TTSFI_C_U16_SERVICE_MAJORVERSION)
                != OSAL_ERROR)
        {
            sds_fi_tcl_e8_JobStatus
                    oJobStatus = oFiDataObject.JobProgress.JobStatus;
            DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                        "vHandleSynthesizeSpeak JobStatus ::"),
                    DLT_INT(oJobStatus.enType));
        }
        oFiDataObject.vDestroy();
        bNotifyTextToSpeechState(AP_TTS_STATE_FINISHED);
        fc_gateway_tclAudioRouteHandler *poAudioClientHandlerObj =
                procgenericgateway_tclMainApp::poGetInstance()->poGetAudioRouteHandler();
        poAudioClientHandlerObj->bAudioReleaseRequest(TTS_AUDIO_SOURCE);

    }
        break;

    default:
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                    "vHandleSynthesizeSpeak Invalid OpCode received"),
                DLT_INT( poMessage->u8GetOpCode()));
    }
    }
}
/******************************************************************************
*
* FUNCTION:     tVoid FC_Gateway_TTSClientHandler::vHandleAbortPrompt
*               (amt_tclServiceData* poMessage)
*
* DESCRIPTION:  Handler function to process the arrival of 
*               AbortPrompt Method Result.
*
* PARAMETER:    Pointer to the Message.
*
* RETURNVALUE:  void
*
******************************************************************************/

tVoid 
FC_Gateway_TTSClientHandler::vHandleAbortPrompt(amt_tclServiceData* poMessage)
{
    tU8 u8OpCode = poMessage->u8GetOpCode();

    switch (u8OpCode)
    {
    case AMT_C_U8_CCAMSG_OPCODE_ERROR:
    {
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                    "vHandleAbortPrompt:: AbortPrompt Failed!!"));

        //Should be same as SynthesizeSpeak_MethodResult_Last
        amt_tclServiceDataError oErrorMsg(poMessage);
        tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
        DLT_LOG(AGW_TTS,DLT_LOG_ERROR,DLT_STRING(
                    "vHandleAbortPrompt - AbortPrompt Failed!! Error ::"),
                DLT_INT(u16ErrorCode));

        switch(u16ErrorCode)
        {
        case SDS_TTSFI_C_U16_ERROR_NOJOBRUNNING:
            vCancelMethodError(AP_TTS_ERROR_TTS_ENGINE_NOT_STARTED);
            break;
        case SDS_TTSFI_C_U16_ERROR_NOTENOUGHMEMORYAVAILABLE:
            vCancelMethodError(AP_TTS_ERROR_OUT_OF_MEMORY);
            break;
        default:
            vCancelMethodError(AP_TTS_ERROR_UNKNOWN);
            break;
        }

    }
        break;

    case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
    {
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                    "vHandleAbortPrompt - AbortPrompt Success!!"));
        m_eCurrentTTSStatus = AP_TTS_STATE_CANCELLED;
        bNotifyTextToSpeechState(AP_TTS_STATE_CANCELLED);
        vCancelMethodResult();
        fc_gateway_tclAudioRouteHandler *poAudioClientHandlerObj =
                procgenericgateway_tclMainApp::poGetInstance()->poGetAudioRouteHandler();
        poAudioClientHandlerObj->bAudioReleaseRequest(TTS_AUDIO_SOURCE);

    }
        break;

    default:
    {
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                    "vHandleAbortPrompt() Invalid  OpCode =  received"),
                DLT_INT(poMessage->u8GetOpCode()));
    }
    }
}

/******************************************************************************
*
* FUNCTION:     tVoid FC_Gateway_TTSClientHandler::vHandleTTsStatus
*               (amt_tclServiceData* poMessage)
*
* DESCRIPTION:  Handler function to TTsStatus change notification.
*
* PARAMETER:    Pointer to the Message.
*
* RETURNVALUE:  void
*
******************************************************************************/
tVoid 
FC_Gateway_TTSClientHandler::vHandleTTsStatus(amt_tclServiceData* poMessage)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("FC_Gateway_TTSClientHandler : \n"),
            DLT_STRING( __FUNCTION__));
    fi_tclVisitorMessage oVisitorMsg(poMessage);
    sds_ttsfi_tclMsgTTSStatusStatus oTtsStatus;
    if (oVisitorMsg.s32GetData(oTtsStatus,
                               SDS_TTSFI_C_U16_SERVICE_MAJORVERSION)
            != OSAL_ERROR)
    {

        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                    "FC_Gateway_TTSClientHandler TTS Status : \n"),
                DLT_INT(oTtsStatus.TTSStatus.enType));

        if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_STATUS )
        {
            m_oTTSStatus = oTtsStatus;
            DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("AMT_C_U8_CCAMSG_OPCODE_STATUS"));
            switch(oTtsStatus.TTSStatus.enType)
            {
            case sds_fi_tcl_e8_TTSStatus::FI_EN_IDLE:
                bNotifyTextToSpeechState(AP_TTS_STATE_IDLE);
                m_eCurrentTTSStatus = AP_TTS_STATE_IDLE;
                break;
            case sds_fi_tcl_e8_TTSStatus::FI_EN_SYNTHESIZING:
                bNotifyTextToSpeechState(AP_TTS_STATE_PLAYING);
                m_eCurrentTTSStatus = AP_TTS_STATE_PLAYING;
                break;
            case sds_fi_tcl_e8_TTSStatus::FI_EN_ABORTING:
                bNotifyTextToSpeechState(AP_TTS_STATE_CANCELLED);
                break;
            case sds_fi_tcl_e8_TTSStatus::FI_EN_PAUSED:
                bNotifyTextToSpeechState(AP_TTS_STATE_PAUSED);
                m_eCurrentTTSStatus = AP_TTS_STATE_PAUSED;
                break;
            case sds_fi_tcl_e8_TTSStatus::FI_EN_DONE:
                //bNotifyTextToSpeechState(AP_TTS_STATE_FINISHED);
                break;
            case sds_fi_tcl_e8_TTSStatus::FI_EN_CONFIGFAILED:
                bNotifyTextToSpeechState(AP_TTS_STATE_FAILED);
                m_eCurrentTTSStatus = AP_TTS_STATE_FAILED;
                break;
            default:
                DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                            "FC_Gateway_TTSClientHandler TTS Status :  is not forwarded\n"),
                        DLT_INT(oTtsStatus.TTSStatus.enType));
                break;
            }
        }
        else
        {
            DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                        "FC_Gateway_TTSClientHandler: Invalid opcode received"));
        }
    }
    oTtsStatus.vDestroy();

}

/******************************************************************************
*
* FUNCTION:     tBool FC_Gateway_TTSClientHandler::bGetTTsStatus()
*
* DESCRIPTION:  Function used to get TTS Status.
*
* PARAMETER:    void
*
* RETURNVALUE:  tBool
*
******************************************************************************/
tBool FC_Gateway_TTSClientHandler::bGetTTsStatus()
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("FC_Gateway_TTSClientHandler : \n"),
            DLT_STRING( __FUNCTION__));

    switch(m_oTTSStatus.TTSStatus.enType)
    {
    case sds_fi_tcl_e8_TTSStatus::FI_EN_IDLE:
        vGetTextToSpeechStateMethodResult(AP_TTS_STATE_IDLE);
        break;
    case sds_fi_tcl_e8_TTSStatus::FI_EN_SYNTHESIZING:
        vGetTextToSpeechStateMethodResult(AP_TTS_STATE_PLAYING);
        break;
    case sds_fi_tcl_e8_TTSStatus::FI_EN_ABORTING:
        vGetTextToSpeechStateMethodResult(AP_TTS_STATE_CANCELLED);
        break;
    case sds_fi_tcl_e8_TTSStatus::FI_EN_PAUSED:
        vGetTextToSpeechStateMethodResult(AP_TTS_STATE_PAUSED);
        break;
    case sds_fi_tcl_e8_TTSStatus::FI_EN_DONE:
        vGetTextToSpeechStateMethodResult(AP_TTS_STATE_FINISHED);
        break;
    case sds_fi_tcl_e8_TTSStatus::FI_EN_CONFIGFAILED:
        vGetTextToSpeechStateMethodResult(AP_TTS_STATE_FAILED);
        break;
    default:
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                    "FC_Gateway_TTSClientHandler TTS Status :  is not forwarded"),
                DLT_INT(m_oTTSStatus.TTSStatus.enType));
        return FALSE;
        break;
    }
    return TRUE;
}

/******************************************************************************
*
* FUNCTION:     tclSpeechProperty::vSetProperty
*
* DESCRIPTION:  Setter function speech property
*
* PARAMETER:    speech properties
*
* RETURNVALUE:  void
*
******************************************************************************/
void tclSpeechProperty::vSetProperty(const char* arg_sText,
                                     unsigned char arg_cPriority,
                                     const char* arg_sLanguage,
                                     int arg_iVoiceRate,
                                     int arg_iVolume,
                                     int arg_iVoicePitch,
                                     unsigned char arg_cVoiceCategory)
{
    m_sText.assign(arg_sText);
    m_cPriority = arg_cPriority;
    m_sLanguage.assign(arg_sLanguage);
    m_iVoiceRate = arg_iVoiceRate;
    m_iVolume = arg_iVolume;
    m_iVoicePitch = arg_iVoicePitch;
    m_cVoiceCategory = arg_cVoiceCategory;
}
/******************************************************************************
*
* FUNCTION:     tclSpeechProperty::sGetText()
*
* DESCRIPTION:  Getter functions for a speech property
*
* PARAMETER:    void
*
* RETURNVALUE:  Required property
*
******************************************************************************/
std::string tclSpeechProperty::sGetText() const
{
    return m_sText;
}

unsigned char tclSpeechProperty::cGetPriority() const
{
    return m_cPriority;
}

std::string tclSpeechProperty::sGetLanguage() const
{
    return m_sLanguage;
}

unsigned char tclSpeechProperty::cGetVoiceCategory() const
{
    return m_cVoiceCategory;
}

int tclSpeechProperty::iGetVoiceRate() const
{
    return m_iVoiceRate;
}

int tclSpeechProperty::iVoicePitch() const
{
    return m_iVoicePitch;
}

int tclSpeechProperty::iGetVolume() const
{
    return m_iVolume;
}
int tclSpeechProperty::iGetPreferedVolume() const
{
    return m_iPreferedVolume;
}
int tclSpeechProperty::iGetPreferedVoiceRate() const
{
    return m_iPreferedVoiceRate;
}
bool  tclSpeechProperty::bCheckVolumeRange() 
{
    if(m_iVolume>=0 && m_iVolume<=100)
    {
        return true;
    }
    else
    {
        return false;
    }
}


#ifdef __cplusplus
extern "C"
{
#endif
/******************************************************************************
*
* FUNCTION:    bStartTextToSpeech
*
* DESCRIPTION: Wrapper function used to call StartTextToSpeech
*
* PARAMETER:   speech properties
*
* RETURNVALUE: bool
*
******************************************************************************/
bool bStartTextToSpeech(const char *arg_text_to_speak,
                        unsigned char arg_priority,
                        const char *arg_language,
                        int arg_voice_rate,
                        int arg_volume,
                        int arg_voice_pitch,
                        unsigned char arg_voice_category)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "bStartTextToSpeech entered with text :"),
            DLT_STRING(arg_text_to_speak));

    tclSpeechProperty *poSpeechProperty = tclSpeechProperty::getInstance();
    poSpeechProperty->vSetProperty(arg_text_to_speak,
                                   arg_priority,
                                   arg_language,
                                   arg_voice_rate,
                                   arg_volume,
                                   arg_voice_pitch,
                                   arg_voice_category);

    FC_Gateway_TTSClientHandler* poTTSClientHandlerObj =
            procgenericgateway_tclMainApp::poGetInstance()->poGetTTSClientHandler();

    sds_fi_tcl_e8_TTSStatus
            l_e8TTSStatus = poTTSClientHandlerObj->eGetCurrTTSStatus();

    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING("bStartTextToSpeech TTS Status:"),
            DLT_INT(l_e8TTSStatus.enType));
    if( (l_e8TTSStatus.enType == sds_fi_tcl_e8_TTSStatus::FI_EN_SYNTHESIZING)
            || (l_e8TTSStatus.enType == sds_fi_tcl_e8_TTSStatus::FI_EN_PAUSED))
    {
        DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                    "TTS Status is playing or paused"));
        return poTTSClientHandlerObj->bPrepareVoiceOutput_MethodStart();
    }
    else
    {
        fc_gateway_tclAudioRouteHandler* poAudioClientHandlerObj =
                procgenericgateway_tclMainApp::poGetInstance()->poGetAudioRouteHandler();
        return poAudioClientHandlerObj->bAudioRouteRequest(TTS_AUDIO_SOURCE);
    }
}



/******************************************************************************
*
* FUNCTION:    vPreferences
*
* DESCRIPTION: Function to set predefined preferences
*
* PARAMETER:   int,int
*
* RETURNVALUE: void
*
******************************************************************************/
void vPreferences(int arg_voice_rate,int arg_volume)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "call from dbus Setting Preferences"));
    tclSpeechProperty *poSpeechProperty = tclSpeechProperty::getInstance();
    poSpeechProperty->vSetPreferences(arg_voice_rate,arg_volume);

}

/******************************************************************************
*
* FUNCTION:    vSetPreferences
*
* DESCRIPTION: Function to set preferences
*
* PARAMETER:   int,int
*
* RETURNVALUE: void
*
******************************************************************************/
void tclSpeechProperty::vSetPreferences(int voice_rate,int volume)
{
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "Entered tclSpeechProperty::vSetPreferences"));
    m_iPreferedVolume=volume;
    m_iPreferedVoiceRate=voice_rate;
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "volume"),DLT_INT(m_iPreferedVolume));
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "voice_rate"),DLT_INT(m_iPreferedVoiceRate));
    DLT_LOG(AGW_TTS,DLT_LOG_INFO,DLT_STRING(
                "Left tclSpeechProperty::vSetPreferences"));
}
/******************************************************************************
*
* FUNCTION:    bCancel
*
* DESCRIPTION: Wrapper function used to call Cancel
*
* PARAMETER:   void
*
* RETURNVALUE: bool
*
******************************************************************************/
bool bCancel()
{
    FC_Gateway_TTSClientHandler *poTTSClientHandlerObj =
            procgenericgateway_tclMainApp::poGetInstance()->poGetTTSClientHandler();
    return poTTSClientHandlerObj->bAbortPrompt_MethodStart();
}

/******************************************************************************
*
* FUNCTION:    bGetTextToSpeechStateMethodStart
*
* DESCRIPTION: Wrapper function used to call GetTextToSpeechStateMethod
*
* PARAMETER:   void
*
* RETURNVALUE: bool
*
******************************************************************************/
bool bGetTextToSpeechStateMethodStart()
{
    FC_Gateway_TTSClientHandler *poTTSClientHandlerObj =
            procgenericgateway_tclMainApp::poGetInstance()->poGetTTSClientHandler();
    return poTTSClientHandlerObj->bGetTTsStatus();
}

#ifdef __cplusplus
}
#endif //__cplusplus

#ifdef __cplusplus
/******************************************************************************
* Function:    poGetTTSInstance
* Description: function to get TTS instance
* Parameters:  void
* Return:      void
******************************************************************************/
FC_Gateway_TTSClientHandler*  poGetTTSInstance(
        procgenericgateway_tclMainApp* poMainAppl)
{
    FC_Gateway_TTSClientHandler* oTTSObj  =
            FC_Gateway_TTSClientHandler::getInstance(poMainAppl);

    return oTTSObj;
}

/******************************************************************************
* Function:    vOnTTSServiceAvailable
* Description: function to invoke TTS service available. (Test purpose)
* Parameters:  void
* Return:      void
******************************************************************************/

void  vOnTTSServiceAvailable()
{
    FC_Gateway_TTSClientHandler* oTTSObj  =
            procgenericgateway_tclMainApp::poGetInstance()->poGetTTSClientHandler();
    oTTSObj->vOnServiceAvailable();
}

/******************************************************************************
* Function:    vOnTTSServiceUnAvailable
* Description: function to invoke TTS service unavailable. (Test purpose)
* Parameters:  void
* Return:      void
******************************************************************************/

void  vOnTTSServiceUnAvailable()
{
    FC_Gateway_TTSClientHandler* oTTSObj  =
            procgenericgateway_tclMainApp::poGetInstance()->poGetTTSClientHandler();
    oTTSObj->vOnServiceUnavailable();
}

#endif
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
