/******************************************************************************
*
* FILE:          SDC_DbusHandler.h
*
* PROJECT:       FC_GENERIC_GATEWAY
*
* DESCRIPTION:   DBUS service handler for SDC.
*
* AUTHOR:        IPD5KOR (RBEI/ECO2)
*
* COPYRIGHT:     (c) 2017 Robert Bosch GmbH, Hildesheim
*
******************************************************************************/

#include <gio/gio.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <openssl/ssl.h>
#include "SDC_DbusHandler.h"
#include "agwcl.h"
#include "dlt/dlt.h"
#include "ECM_ClientInterface.h"
#include "ap-sdc-enums.h"
#include "ap-ecm-enums.h"
#include "AutomDBusServer.h"

DLT_DECLARE_CONTEXT(AGW_SDC);

static SecureDataCommunication*   poSDCProxyObj = NULL;

typedef struct 
{
	ApCheckCertResult sdcCode;
	ApECMError ecmCode;
}CheckCertResultCode; 

static CheckCertResultCode sdc_ecm_result_map[] = { 
                {AP_CC_SUCCESS ,           AP_ECM_CERT_SUCCESS},
   	            {AP_CC_CERT_NOT_PRESENT,   AP_ECM_CERT_NOT_PRESENT },
	            {AP_CC_CERT_FORMAT_ERROR,  AP_ECM_CERT_FORMAT_ERROR},
                {AP_CC_INVALID_CERT,       AP_ECM_CERT_INVALID},
                {AP_CC_CRL_NOT_VALID,      AP_ECM_CRL_NOT_VALID_YET},
                {AP_CC_SECURE_TIME_ERROR,  AP_ECM_SECURE_TIME_UNAVAILABLE}
                };
		  
/******************************************************************************
* Function:    handle_get_device_certificate
* Description: The method provides the device certificate data as byte array
* Parameters:  proxy object, invocation pointer, algorithm, user_data
* Return:      gboolean
*****************************************************************************/
gboolean
handle_get_device_certificate (SecureDataCommunication *object,
                               GDBusMethodInvocation *invocation,
                               guint64 algorithm,
                               gpointer user_data)
{
        DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__),
		           DLT_STRING(": algorithm = "), DLT_UINT64(algorithm));
				   
        if(poGetSDCProxyObj() == NULL)
        {
            DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                         DLT_STRING("():NULL Proxy") );
            g_dbus_method_invocation_return_error( invocation,
                                                   AP_SDC_ERROR,
                                                  (int) AP_SDC_ERROR_UNKNOWN,
                                                  "null proxy object" );
            return TRUE;
        }
   //Removing algorithm validation as per Matthias suggestion
   // All operations on algorithm will be in AGWCL /SDC   
    /*    if(!is_valid_algorithm(algorithm, "getCert"))
        {
            DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Invalid algorithm"));
            g_dbus_method_invocation_return_error(
    			                              invocation,
                                              AP_SDC_ERROR,
                                              (int) AP_SDC_ERROR_INVALID_INPUT,
                                              "Invalid Algorithm" );
            return TRUE;
        } 
    */        
        gint result = 0;
        unsigned char* certData = NULL;
        size_t certLength = 0;
        int index = 0;
        GVariant* dataOut = NULL;
		GVariantBuilder builder;
		g_variant_builder_init(&builder,G_VARIANT_TYPE ("ay"));
        result = get_client_cert(&certData, &certLength, algorithm);

        DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                  		DLT_STRING("():Result is "),DLT_INT(result));
 
        if((certData == NULL) || (certLength == 0) )
        {
           DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
		                                DLT_STRING("NULL cert recieved"));
           g_variant_builder_add(&builder, "y", '\0');
           dataOut = g_variant_builder_end(&builder);
           dataOut =g_variant_ref_sink(dataOut);


           secure_data_communication_complete_get_device_cert(poGetSDCProxyObj(),
                                                              invocation,
                                                              dataOut,
                                                             (guint16) result );
			g_variant_unref(dataOut);												 
           return TRUE;
        }
         
        DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
              		DLT_STRING("():Cert length= "),  DLT_UINT(certLength)); 

        for(index = 0; index < (int) certLength  ; index++)
        {
          g_variant_builder_add(&builder, "y", certData[index]);
        }

        dataOut = g_variant_builder_end(&builder);
		free(certData);
        dataOut =g_variant_ref_sink(dataOut);

        secure_data_communication_complete_get_device_cert( poGetSDCProxyObj(),
                                                            invocation,
                                                            dataOut,
                                                          (guint16)  result );
       g_variant_unref(dataOut);
     
       DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));	
       return TRUE;
}         
            
/******************************************************************************
* Function:    handle_sign_device_key
* Description: The method generates signature on the gioven data
* Parameters:  proxy object, invocation pointer, data to be signed in,
*              algorithm, user_data
* Return:      gboolean
*****************************************************************************/
gboolean
handle_sign_with_device_key ( SecureDataCommunication *object,
                              GDBusMethodInvocation   *invocation,
                              GVariant    *data,
                              guint64     algorithm,
                              gpointer    user_data)
{
       DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__),
		           DLT_STRING(": algorithm = "), DLT_UINT64(algorithm)); 

        if(poGetSDCProxyObj() == NULL)
        {
            DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                         DLT_STRING("():NULL Proxy") );
            g_dbus_method_invocation_return_error( invocation,
                                                   AP_SDC_ERROR,
                                                  (int) AP_SDC_ERROR_UNKNOWN,
                                                  "null proxy object" );
        }

        if ((!data))
        {
            g_dbus_method_invocation_return_error( 
			                                 invocation,
                                             AP_SDC_ERROR,
                                             (int) AP_SDC_ERROR_INVALID_INPUT,
                                             "Invalid Input: Null data");
           
           DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
		      DLT_STRING("(): Invalid Input - Null data"));
                return TRUE;
        }
   //Removing algorithm validation as per Matthias suggestion
   // All operations on algorithm will be in AGWCL /SDC  
   
    /*  if(!is_valid_algorithm(algorithm, "sign"))
        {
            DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                                    DLT_STRING("(): Invalid algorithm"));
            g_dbus_method_invocation_return_error( 
			                               invocation,
                                           AP_SDC_ERROR,
                                           (int) AP_SDC_ERROR_INVALID_INPUT,
                                           "Invalid Algorithm" );
            return TRUE;
        }
    */
        size_t data_len = g_variant_n_children(data);
        unsigned char* dataStr = NULL;
        gsize index = 0;

        dataStr = malloc(data_len);

        if(dataStr == NULL)
        {
           DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                       DLT_STRING("malloc failed for dataStr"));
           g_dbus_method_invocation_return_error( invocation,
                                                   AP_SDC_ERROR,
                                                  (int) AP_SDC_ERROR_BAD_DATA,
                                                  "Memory allocation Error" );
           return TRUE;
        }
        memset(dataStr, '\0' , data_len);
        for(index=0; index < data_len; index++)
        {
             g_variant_get_child(data, index,"y", &dataStr[index]);
        }

        DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
		      DLT_STRING("(): data = "), DLT_RAW(dataStr, (uint16_t)data_len));

        gint result = 0;
        GVariantBuilder builder ;
		g_variant_builder_init(&builder,G_VARIANT_TYPE ("ay"));
        unsigned char* signature_str = NULL;
        GVariant*      signature_var = NULL;
        size_t signature_len = 0;

        result = sign_with_device_key( dataStr,
               		                   data_len,
          						       &signature_str,
 									   &signature_len, 
									   algorithm 
									 );

        DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
		                        DLT_STRING("():Result is "),DLT_INT(result));

        if((signature_str == NULL) || (signature_len == 0))
        {  
            DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
	 	      DLT_STRING("(): NULL signature received"));

            g_variant_builder_add(&builder, "y", '\0');
            signature_var = g_variant_builder_end(&builder);
            

            secure_data_communication_complete_sign_with_device_key( 
			                                               poGetSDCProxyObj(),
                                                           invocation,
                                                           signature_var,
                                                           (guint16)  result );
            free(dataStr);
            return TRUE;             
        }

        DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING(
		            "():Signature length= "), DLT_UINT(signature_len)); 

        for(index = 0; index < signature_len ; index++)
        {
          g_variant_builder_add(&builder, "y", signature_str[index]);
        }

        signature_var = g_variant_builder_end(&builder);
        

        secure_data_communication_complete_sign_with_device_key(  
		                                                  poGetSDCProxyObj(),
                                                                  invocation,
                                                               signature_var,
                                                           (guint16)  result );
		free(dataStr);
		free(signature_str);
        DLT_LOG(AGW_SDC,DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));	
        return TRUE;
}

/******************************************************************************
* Function:    handle_decrypt_with_device_key
* Description: The method decrypts the cipher text.
* Parameters:  proxy object, invocation pointer,
*              cipherText, algorithm, user data
* Return:      gboolean
*****************************************************************************/
gboolean
handle_decrypt_with_device_key ( SecureDataCommunication *object,
                                 GDBusMethodInvocation   *invocation,
                                 GVariant* cipherText,
                                 guint64   algorithm,
                                 gpointer  user_data )
{
        DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__),
                   DLT_STRING(": algorithm = "), DLT_UINT64(algorithm));
       
        if ((!cipherText))
        {
             g_dbus_method_invocation_return_error( 
			                                 invocation,
                                             AP_SDC_ERROR,
                                             (int) AP_SDC_ERROR_INVALID_INPUT,
                                             "Invalid Input - Null cipherText");

             DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
		        DLT_STRING("(): Invalid Input - Null cipherText"));
             return FALSE;
        }
      //Removing algorithm validation as per Matthias suggestion
      // All operations on algorithm will be in AGWCL /SDC  
      /* if(!is_valid_algorithm(algorithm, "decrypt"))
         {
            DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): Invalid algorithm"));
            g_dbus_method_invocation_return_error( 
			                              invocation,
                                          AP_SDC_ERROR,
                                          (int) AP_SDC_ERROR_INVALID_INPUT,
                                          "Invalid Algorithm" );
            return TRUE;
         }
      */  
        size_t cipher_len = g_variant_n_children(cipherText);
        unsigned char* cipherStr = NULL;
        gsize index = 0;

        cipherStr = malloc(cipher_len);

        if(cipherStr == NULL)
        {
           DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                       DLT_STRING("malloc failed for cipherStr"));
           g_dbus_method_invocation_return_error(  invocation,
                                                   AP_SDC_ERROR,
                                                  (int) AP_SDC_ERROR_BAD_DATA,		   "Error in SDC" ); 									  
           return TRUE;
        }
  
        for(index = 0; index < cipher_len; index++)
        {
          g_variant_get_child(cipherText, index, "y", &cipherStr[index]);
        }

        DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING(
		    "(): cipherText = "),  DLT_RAW(cipherStr, (uint16_t)cipher_len));

        gint            result = 0;
        GVariantBuilder builder ;
		g_variant_builder_init(&builder,G_VARIANT_TYPE ("ay"));
        unsigned char*  plainData_str = NULL;
        GVariant*       plainData_var = NULL;
        size_t          plainData_len = 0;

        result = (guint16)decrypt_with_device_key (  cipherStr,
                                                     cipher_len,
                                                     &plainData_str,
                                                     &plainData_len,
                                                     algorithm );

        DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
		      DLT_STRING("():Result is: "), DLT_INT(result));
		free(cipherStr);
        if((plainData_str == NULL) || (plainData_len == 0))
        {
             DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
			     DLT_STRING("(): NULL plain data recieved"));
             g_variant_builder_add(&builder, "y", '\0');
             plainData_var = g_variant_builder_end(&builder);
             
				 
	     secure_data_communication_complete_decrypt_with_device_key( object,
                                                                 invocation,
                                                              plainData_var,
                                                             (guint16) result); 
														   
  	     return TRUE;
        }
        DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
                 DLT_STRING("(): plainData_len is "), DLT_UINT(plainData_len));	   

        for(index = 0; index < (int)plainData_len ; index++)
        {
           g_variant_builder_add(&builder, "y", plainData_str[index]);
        }
        plainData_var = g_variant_builder_end(&builder);
                 
              				
		 secure_data_communication_complete_decrypt_with_device_key( object,
                                                                    invocation,
                                                                 plainData_var,
                                                             (guint16) result );
		 free(plainData_str);
         DLT_LOG(AGW_SDC,DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
         return TRUE;
}

/******************************************************************************
* Function:    handle_check_certificate
* Description: The method checks whether given certificate is valid
* Parameters:  proxy object, invocation pointer, certificate, user_data
* Return:      gboolean
*****************************************************************************/
gboolean
handle_check_certificate ( SecureDataCommunication *object,
                           GDBusMethodInvocation   *invocation,
                           GVariant  *certificate,
                           gpointer  user_data )
{
    DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

#if ENABLE_NON_CMD_IN_GATEWAY
    user_data = g_object_ref(invocation);

    if ((certificate == NULL) )
    {
        g_dbus_method_invocation_return_error( invocation,
                                               AP_SDC_ERROR,
                                               (int) AP_SDC_ERROR_INVALID_INPUT,
                                               "Invalid Input to SDC");

         DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                      DLT_STRING("(): Invalid Input"));
         return FALSE;
    }

    guint certLength = (guint) g_variant_n_children(certificate); 
    GDBusProxy* pECMProxyObject = NULL;

    if( (pECMProxyObject = getECMClientProxy()) == NULL)
    {
        g_dbus_method_invocation_return_error( invocation,
                                               AP_SDC_ERROR,
                                               (int) AP_SDC_ERROR_UNKNOWN,
                                               "ECM unavailable");

        DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
                         DLT_STRING("(): ECM Proxy is NULL"));
        return FALSE;
    }     

    /* Dbus call to ECM */ 
    g_dbus_proxy_call ( pECMProxyObject,
                        "checkCert",
                        g_variant_new ("(@ayu)",
                             certificate, certLength),
                        G_DBUS_CALL_FLAGS_NONE,
                        -1,
                        NULL,
                        check_cert_result,
                        user_data );

    DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
    return TRUE;
#else
    g_dbus_method_invocation_return_error(
                                       invocation,
                                       AP_SDC_ERROR,
                                      (int) AP_SDC_ERROR_UNKNOWN,
                                      "ECM unavailable in non-psa environment"
									     );
	return FALSE;
#endif
}
                        
/******************************************************************************
* Function:    check_cert_result
* Description: The method sends the result of checkCert
* Parameters:  proxy object, result object, user_data
* Return:      gboolean
*****************************************************************************/
void check_cert_result ( GObject* pObj,
                       GAsyncResult *res,
                       gpointer user_data )
{
    DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    GError* error =   NULL;
    gchar* errorMsg = NULL;
    GVariant *_ret;
    guint32 certValidity = 0;
    ApCheckCertResult  result = AP_CC_INVALID_CERT;
    guint32 mapLength  = sizeof(sdc_ecm_result_map) /
	                       sizeof(sdc_ecm_result_map[0]);

    _ret = g_dbus_proxy_call_finish (G_DBUS_PROXY(pObj), res, &error);
    if (_ret == NULL)
    {
                if (error != NULL)
                {
                    DLT_LOG(AGW_SDC,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
					      DLT_STRING("():Failed getting result, error = "), 
					      DLT_STRING(error->message));
                    g_clear_error(&error);
                }
                return;
    }
    else
    {
       g_variant_get (_ret, "(us)", &certValidity, &errorMsg );
       g_variant_unref (_ret);
    }
    DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
	     DLT_STRING("():certValidity = "), DLT_UINT(certValidity));

	int i =0;	  
    for(i= 0; i < mapLength; i++)
	{
		if((guint32)sdc_ecm_result_map[i].ecmCode == certValidity)
		{
			result = sdc_ecm_result_map[i].sdcCode;
			break;
		}	
	}

    if((poGetSDCProxyObj() != NULL) && (user_data != NULL))
    {
        secure_data_communication_complete_check_cert( poGetSDCProxyObj(),
                                              (GDBusMethodInvocation*)user_data,
                                              (guint16) result );
    }
    else{
        DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
             DLT_STRING("chcek cert result not sent, "
                        "SDC proxy pointer/invocation ptr is NULL"));
    }
    
    g_object_unref(user_data);
	if(errorMsg!=NULL)
	free(errorMsg);	
    DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

//Removing algorithm validation as per Matthias suggeston
// All operations on algorithm will be in AGWCL /SDC  
/*
gboolean is_valid_algorithm(guint64   algorithm,
                            gchar*    reqType)
{
   DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
      DLT_STRING("(): Algorithm = "), DLT_UINT64(algorithm), 
	  DLT_STRING(" request type= "), DLT_STRING(reqType) );

   guint bit =1, bitValue = 0;
   guint64 quotient = algorithm;
   while (quotient != 0)
   {
      bitValue = (guint)(quotient % 16);
      if(bitValue > 6)
        return FALSE;
      if(bit > 4)
        return FALSE;
      if((bit==1) && (bitValue > 1))
           return FALSE;
      if((bit==3) && (bitValue > 3))
           return FALSE;
      if((bit==4) && (bitValue > 4))
           return FALSE;

     if((strcmp(reqType, "sign")== 0) ||
           (strcmp(reqType, "decrypt")== 0))
     {
        if((bit == 4) && (bitValue == 3))
          return FALSE;
     }
     else 
     {
        if((bit==4) && (bitValue != 3))
          return FALSE;
     }
     quotient = quotient / 16;
     bit++;
  }
  DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("-"), DLT_STRING(__FUNCTION__));
  return TRUE;
}    
*/
             
/******************************************************************************
* Function:    bInitializeSDC
* Description: Intializes the handler by exporting skeleton object
* Parameters:  void
* Return:      gboolean - TRUE when skeleton export success
                        - FALSE when skeleton export fails
*****************************************************************************/
gboolean bInitializeSDC()
{
   DLT_REGISTER_CONTEXT(AGW_SDC,"GWSDC","Gateway SDC"
                         "context for DLT Logging");
						 
   DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

   GError *error = NULL;
   GDBusConnection* connection = NULL;

   if((connection = poGetGDBusConnection()) == NULL)
   {
        DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): poGetGDBusConnection returned NULL"));
        return FALSE;
   }

   if(poGetSDCProxyObj() == NULL)
   {
	   DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
       	DLT_STRING("(): SecureDataCommunication proxy object creation failed"));
	                                                                    
	   return FALSE;
   }

   if (!g_dbus_interface_skeleton_export (
                          G_DBUS_INTERFACE_SKELETON (poGetSDCProxyObj()),
                          connection,
                          "/com/bosch/AutomotiveProxy/SecureDataCommunication",
                          &error) )
   {
       if(error != NULL)
	   {
           DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            	DLT_STRING("(): "), DLT_STRING(error->message));    
		   g_clear_error(&error);
	   }
       return FALSE;     
   }
   if(!SSL_library_init())
   {
	   DLT_LOG(AGW_SDC, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
	      DLT_STRING("(): SSL_library_init Failed"));
   }
   g_signal_connect (poGetSDCProxyObj(), "handle-sign-with-device-key",
                     G_CALLBACK (handle_sign_with_device_key), NULL);

   g_signal_connect (poGetSDCProxyObj(), "handle-decrypt-with-device-key",
                     G_CALLBACK (handle_decrypt_with_device_key), NULL);

   g_signal_connect (poGetSDCProxyObj(),"handle-get-device-cert",
                     G_CALLBACK (handle_get_device_certificate), NULL);					 

   g_signal_connect (poGetSDCProxyObj(),"handle-check-cert",
                     G_CALLBACK (handle_check_certificate), NULL);

   DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
   return TRUE;

}

/******************************************************************************
* Function:    poGetSDCProxyObj
* Description: provides the communication protocol proxy object
* Parameters:  void 
* Return:      SecureDataCommunication* 
*****************************************************************************/
SecureDataCommunication* poGetSDCProxyObj()
{
   DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
   if(poSDCProxyObj == NULL)
       poSDCProxyObj = secure_data_communication_skeleton_new();
    DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
    return poSDCProxyObj;
}

/******************************************************************************
* Function:    uninitializeSDC
* Description: unintializes the SDC component by releasing proxy
* Parameters:  void
* Return:      void
*****************************************************************************/
void uninitializeSDC()
{
   DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 

   if(poSDCProxyObj)
   {
       g_dbus_interface_skeleton_unexport(
                        G_DBUS_INTERFACE_SKELETON(poSDCProxyObj));
       g_clear_object(&poSDCProxyObj);
   }

   DLT_LOG(AGW_SDC, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
   DLT_UNREGISTER_CONTEXT(AGW_SDC);
}


