/******************************************************************************
*
* FILE:          PhoneDBusHandler.c
*
* PROJECT:       FC_GENERIC_GATEWAY
*
* DESCRIPTION:   DBus server implementation for phone
*
* AUTHOR:        KRJ5KOR(RBEI/ECO2)
*
* COPYRIGHT:     (c) 2015 Robert Bosch GmbH, Hildesheim
*
******************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#include <gio/gio.h>
#include <stdlib.h>

#include "Utility.h"
#include "Phone_ClientInterface.h"
#include "PhoneDBusHandler.h"
#include "PhoneDataTypes.h"
#include "AutomDBusServer.h"

#include "dlt/dlt.h"
//Dlt Context
DLT_IMPORT_CONTEXT(AGW_Phone);

static GDBusMethodInvocation *g_invocation_prepare_call_history = NULL;
static GDBusMethodInvocation *g_invocation_get_call_history = NULL;
static GDBusMethodInvocation *g_invocation_prepare_contacts_list = NULL;
static GDBusMethodInvocation *g_invocation_get_contacts_list = NULL;
static GDBusMethodInvocation *g_invocation_hangup_call = NULL;
static GDBusMethodInvocation *g_invocation_prepare_get_messages_list = NULL;
static GDBusMethodInvocation *g_invocation_get_messages_list = NULL;
static GDBusMethodInvocation *g_invocation_read_message = NULL;


/*******************************************************************************
 * Function:    handle_prepare_call_history_list
 * Description: Prepare the call history list.
 * Parameters:  *invocation, user_data
 * Return:      boolean
 ******************************************************************************/
gboolean
handle_prepare_call_history_list (CallManager *object,
                                  GDBusMethodInvocation *invocation,
                                  GVariant *arg_options,
                                  gpointer user_data)
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    g_invocation_prepare_call_history = g_object_ref(invocation);
    if(arg_options == NULL)
    {
        vPrepareCallHistoryError(AP_TELEPHONY_ERROR_INVALID_ATTRIBUTE_VALUE,
                                 "Options are incorrect.");
        return FALSE;
    }
    else
    {
        GVariantIter iter;
        GVariant *value;
        gchar *key;
        guint16 uDeviceHandle;
        guchar cDirection;
        gboolean bIsDirectionFound = FALSE;
        gboolean bIsDeviceHandleFound = FALSE;
        gboolean bIsInputInvalid = FALSE;
        g_variant_iter_init(&iter,arg_options);
        while (g_variant_iter_loop (&iter, "{sv}", &key, &value))
        {
            if(!g_strcmp0 (key,"Direction"))
            {
                g_variant_get(value,"y",&cDirection);
                bIsDirectionFound = TRUE;
            }
            else if(!g_strcmp0(key,"DeviceHandle"))
            {
                g_variant_get(value,"q",&uDeviceHandle);
                bIsDeviceHandleFound = TRUE;
            }
            else
            {
                bIsInputInvalid = TRUE;
            }
        }

        if(bIsDirectionFound == FALSE || bIsDeviceHandleFound == FALSE)
        {
            vPrepareCallHistoryError(
                        AP_TELEPHONY_ERROR_INVALID_ATTRIBUTE_VALUE,
                        "DeviceHandle or Direction is not available.");
        }
        else if(bIsInputInvalid)
        {
            char sError[] = "Options are incorrect.";
            vPrepareContactListError(AP_CONTACT_ERROR_INVALID_ATTRIBUTE_VALUE,
                                     sError);
        }
        else
        {
            vPrepareCallHistoryList((ApTelephonyCallDirection)cDirection,
                                    uDeviceHandle);
        }
    }

    return TRUE;
}

/******************************************************************************
 * Function:    vPrepareCallHistoryResult
 * Description: sends the result of the prepare call history method.
 * Parameters:  gint32, gint32
 * Return:      void
 ******************************************************************************/
void vPrepareCallHistoryResult(gint32 iNumOfCallHist,
                               gint32 iListHandle)
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,
            DLT_STRING("+vPrepareCallHistoryResult List size:"),
            DLT_INT(iNumOfCallHist),DLT_STRING(" List handle:"),
            DLT_INT(iListHandle));


    CallManager* poCallManager = poGetCallManager();
    if(g_invocation_prepare_call_history == NULL || poCallManager == NULL)
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,DLT_STRING( __FUNCTION__),
                DLT_STRING(" invocation pointer or object is NULL"));
        return;
    }

    call_manager_complete_prepare_call_history_list(
                poCallManager,
                g_invocation_prepare_call_history,
                iNumOfCallHist,
                iListHandle);

    g_clear_object(&g_invocation_prepare_call_history);
}    

/******************************************************************************
 * Function:    vPrepareCallHistoryError
 * Description: sends the error for the prepare call history method.
 * Parameters:  APTelephonyError , error_msg
 * Return:      void
 ******************************************************************************/
void vPrepareCallHistoryError(APTelephonyError enError, const gchar* error_msg)
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING("+vPrepareCallHistoryError()"));
    if(g_invocation_prepare_call_history == NULL)
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,DLT_STRING( __FUNCTION__),
                DLT_STRING(" invocation pointer is NULL"));
        return;
    }

    if(error_msg == NULL)
    {
        g_dbus_method_invocation_return_error(g_invocation_prepare_call_history,
                                              AP_TELEPHONY_ERROR,
                                              enError,
                                              "Error in prepare call history");
    }
    else
    {
        g_dbus_method_invocation_return_error(g_invocation_prepare_call_history,
                                              AP_TELEPHONY_ERROR,
                                              enError,
                                              error_msg);
    }

    g_clear_object(&g_invocation_prepare_call_history);
} 

/*******************************************************************************
 * Function:    handle_get_call_history_in_slice
 * Description: Handler method for call history retrival in slices.
 * Parameters:  *invocation, options, list handle, user_data
 * Return:      boolean
 ******************************************************************************/
gboolean
handle_get_call_history_in_slice (CallManager *object,
                                  GDBusMethodInvocation *invocation,
                                  GVariant *arg_options,
                                  gint arg_handle,
                                  gpointer user_data)
{

    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

    g_invocation_get_call_history = g_object_ref(invocation);
    if(arg_options == NULL)
    {
        vGetCallHistoryInSliceError(AP_TELEPHONY_ERROR_INVALID_ATTRIBUTE_VALUE,
                                    "Options cannot be NULL");
        return TRUE;
    }
    else
    {
        GVariantIter iter;
        GVariant *value;
        gchar *key;
        guint32 uWindowStart = 0;
        guint16 uWindowSize = 0;
        gboolean bIsWindowStartFound = FALSE;
        gboolean bIsWindowSizeFound = FALSE;

        g_variant_iter_init(&iter,arg_options);
        while (g_variant_iter_loop (&iter, "{sv}", &key, &value))
        {
            if(!g_strcmp0 (key,"WindowStart"))
            {
                g_variant_get(value,"u",&uWindowStart);
                DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_UINT32( uWindowStart));
                bIsWindowStartFound = TRUE;
            }
            else if(!g_strcmp0 (key,"WindowSize"))
            {
                g_variant_get(value,"q",&uWindowSize);
                DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_UINT16( uWindowSize));
                bIsWindowSizeFound = TRUE;
            }
            else
            {
                continue;
            }
        }

        if(bIsWindowStartFound && bIsWindowSizeFound)
        {
            vGetCallHistorySlice(arg_handle,uWindowStart,uWindowSize);
        }
        else
        {
            vGetCallHistoryInSliceError(
                        AP_TELEPHONY_ERROR_INVALID_ATTRIBUTE_VALUE,
                        "Window start or window size is not available.");
        }
    }

    return TRUE;
}

/******************************************************************************
 * Function:    vGetCallHistoryInSliceResult
 * Description: sends the result of the get call history in slice method.
 * Parameters:  GVariant
 * Return:      void
 ******************************************************************************/
void vGetCallHistoryInSliceResult(CallHistoryInfo* call_history_entries,
                                  int list_size)
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING("+vGetCallHistoryInSliceResult"));

    if(0 != list_size)
    {
        GVariantBuilder builder;
		g_variant_builder_init ( &builder, G_VARIANT_TYPE ("a(ia{sv})"));

        int iCount = 0;
        while (iCount < list_size)
        {
            CallHistoryInfo l_CallHistoryItem = call_history_entries[iCount];

            GVariantBuilder inner_builder;
            g_variant_builder_init (&inner_builder, G_VARIANT_TYPE ("a{sv}"));

            g_variant_builder_add (
                        &inner_builder,
                        "{sv}", "RemoteParty",
                        g_variant_new("s", l_CallHistoryItem.sRemoteParty));

            g_variant_builder_add (
                        &inner_builder,
                        "{sv}", "StartTime",
                        g_variant_new("s", l_CallHistoryItem.sStartTime));

            g_variant_builder_add (
                        &inner_builder, "{sv}", "Direction",
                        g_variant_new_byte(l_CallHistoryItem.uDirection));

            g_variant_builder_add(&builder, "(ia{sv})",
                                  l_CallHistoryItem.iContactHandle,
                                  &inner_builder);
								    
            vCleanCallHistoryData(&l_CallHistoryItem);
            iCount++;
        }

        GVariant *call_history = g_variant_builder_end(&builder);
		GVariant* call_history_ref = g_variant_ref_sink(call_history);

        CallManager* poCallManager = poGetCallManager();
        if(g_invocation_get_call_history == NULL || poCallManager == NULL)
        {
            DLT_LOG(AGW_Phone,DLT_LOG_ERROR,
                    DLT_STRING( __FUNCTION__),
                    DLT_STRING(" invocation pointer or object is NULL"));
        }
        else
        {
            call_manager_complete_get_call_history_in_slice(
                        poCallManager,
                        g_invocation_get_call_history,
                        call_history_ref);
        }
		g_variant_unref(call_history_ref);
    }
    else
    {
        vGetCallHistoryInSliceError(
                    AP_TELEPHONY_ERROR_ILLEGAL_BEHAVIOUR,
                    "Internal error: Call history list is null");
    }
    free(call_history_entries);
    g_clear_object(&g_invocation_get_call_history);
}

/******************************************************************************
 * Function:    vGetCallHistoryInSliceError
 * Description: sends the error for the get call history in slice method.
 * Parameters:  APTelephonyError, error_msg
 * Return:      void
 ******************************************************************************/
void vGetCallHistoryInSliceError(APTelephonyError enError,
                                 const gchar* error_msg)
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,
            DLT_STRING("+vGetCallHistoryInSliceError()"));

    if(g_invocation_get_call_history == NULL)
    {
        DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( __FUNCTION__),
                DLT_STRING(" invocation pointer is NULL"));
        return;
    }

    if(error_msg == NULL)
    {
        g_dbus_method_invocation_return_error(g_invocation_get_call_history,
                                              AP_TELEPHONY_ERROR,
                                              enError,
                                              "Error in prepare call history");
    }
    else
    {
        g_dbus_method_invocation_return_error(g_invocation_get_call_history,
                                              AP_TELEPHONY_ERROR,
                                              enError,
                                              error_msg);
    }

    g_clear_object(&g_invocation_get_call_history);
} 

/*******************************************************************************
 * Function:    vSendOnCallStateChangeSignal
 * Description: Emits the callstate change signals.
 * Parameters:  i_duration, remote_party, state
 * Return:      boolean
 ******************************************************************************/

void vSendOnCallStateChangeSignal(btphn_callStatus l_callStatus)
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

    //wrap the data in a gvariant.
    GVariantBuilder builder;
    GVariant *result = NULL;
    g_variant_builder_init (&builder, G_VARIANT_TYPE ("a{sv}"));

    g_variant_builder_add (&builder, "{sv}", "CallId",
                           g_variant_new_uint32(l_callStatus.u16CallInstance));
    g_variant_builder_add (&builder, "{sv}", "RemoteParty",
                           g_variant_new_string(l_callStatus.pPhoneNum));
    g_variant_builder_add (&builder, "{sv}", "State",
                           g_variant_new_byte (l_callStatus.enCallStatus));

    guint seconds = 	(l_callStatus.u8CallDurationHr*3600)+
            (l_callStatus.u8CallDurationMin*60)+
            l_callStatus.u8CallDurationSec;

    g_variant_builder_add (&builder, "{sv}", "Duration",
                           g_variant_new_uint32 (seconds));
    result = g_variant_builder_end (&builder);
    GVariant *result_ref = g_variant_ref_sink(result);
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( "GVariant Building Ends"));

    //push the signal to D-Bus.
    CallManager* poCallManager = poGetCallManager();
    if(poCallManager == NULL)
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,DLT_STRING( __FUNCTION__),
                DLT_STRING(" object is NULL"));
    }
    call_manager_emit_on_call_state_change(poCallManager,
                                           result_ref);
    g_variant_unref(result_ref);
}


/******************************************************************************
* Function:    vTriggerOnSMSReceivedSignal
* Description: Triggers the SMS status signal.
* Parameters:  SMSNotifyType, SMSData
* Return:      boolean
******************************************************************************/
void vTriggerOnSMSReceivedSignal(SMSData l_SMSData)
{
    //wrap the data in a gvariant.
    GVariantBuilder builder;
    GVariant *result = NULL;
    g_variant_builder_init(&builder, G_VARIANT_TYPE("a{sv}"));

    g_variant_builder_add(&builder, "{sv}", "DeviceHandle",
                          g_variant_new_int32(l_SMSData.u16DeviceHandle));
    g_variant_builder_add(&builder, "{sv}", "MessageID",
                          g_variant_new_int32(l_SMSData.u16MessageHandle));
    g_variant_builder_add(&builder, "{sv}", "ListID",
                          g_variant_new_int32(l_SMSData.u16ListHandle));
    g_variant_builder_add(&builder, "{sv}", "From",
                          g_variant_new_string(l_SMSData.poFromPhoneNumber));
    g_variant_builder_add(&builder, "{sv}", "Timestamp",
                          g_variant_new_string(l_SMSData.poTimeStamp));
    g_variant_builder_add(&builder, "{sv}", "Read",
                          g_variant_new_boolean(l_SMSData.bRead));
    g_variant_builder_add(&builder, "{sv}", "Body",
                          g_variant_new_string(l_SMSData.poMessage));
    g_variant_builder_add(&builder, "{sv}", "State",
                          g_variant_new_byte(l_SMSData.eSMSState));
    g_variant_builder_add(&builder, "{sv}", "DeliveryStatus",
                          g_variant_new_byte(l_SMSData.eDeliveryState));
    g_variant_builder_add(&builder, "{sv}", "DeliveryTimestamp",
                          g_variant_new_string("not supported"));
    
    result = g_variant_builder_end(&builder);

    GVariant *result_ref = g_variant_ref_sink(result);
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( "GVariant Building Ends"));

    SmsManager* poSMSManager = poGetSMSManager();
    if(poSMSManager != NULL)
    {
        sms_manager_emit_on_received(poSMSManager, result_ref);
    }
    else
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,
                DLT_STRING("SmsManager object IS NULL"));
    }
    g_variant_unref(result_ref);
}


/******************************************************************************
* Function:    vTriggerOnDeliverySuccessSignal
* Description: Triggers the SMS status signal.
* Parameters:  SMSNotifyType, SMSData
* Return:      boolean
******************************************************************************/
void vTriggerOnDeliverySuccessSignal(SMSData l_SMSData)
{
    //wrap the data in a gvariant.
    GVariantBuilder builder;
    GVariant *result = NULL;
    g_variant_builder_init(&builder, G_VARIANT_TYPE("a{sv}"));

    g_variant_builder_add(&builder, "{sv}", "DeviceHandle",
                          g_variant_new_int32(l_SMSData.u16DeviceHandle));
    g_variant_builder_add(&builder, "{sv}", "MessageID",
                          g_variant_new_int32(l_SMSData.u16MessageHandle));
    g_variant_builder_add(&builder, "{sv}", "ListID",
                          g_variant_new_int32(l_SMSData.u16ListHandle));
    g_variant_builder_add(&builder, "{sv}", "Recipients",
                          g_variant_new_string(l_SMSData.poToPhoneNumber));
    g_variant_builder_add(&builder, "{sv}", "DeliveryTimestamp",
                          g_variant_new("s","not supported"));

    result = g_variant_builder_end(&builder);

    GVariant *result_ref = g_variant_ref_sink(result);
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( "GVariant Building Ends"));

    DLT_LOG(AGW_Phone,DLT_LOG_INFO,
            DLT_STRING( "emits delivery success message"));
    SmsManager* poSMSManager = poGetSMSManager();
    if(poSMSManager != NULL)
    {
        sms_manager_emit_on_delivery_success(poSMSManager,
                                             result_ref);
    }
    else
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,
                DLT_STRING( "SmsManager object IS NULL"));
    }
    
    g_variant_unref(result_ref);
}


/******************************************************************************
* Function:    vTriggerOnDeliveryFailureSignal
* Description: Triggers the SMS status signal.
* Parameters:  SMSNotifyType, SMSData
* Return:      boolean
******************************************************************************/
void vTriggerOnDeliveryFailureSignal(SMSData l_SMSData)
{
    //wrap the data in a gvariant.
    GVariantBuilder builder;
    GVariant *result = NULL;
    g_variant_builder_init(&builder, G_VARIANT_TYPE("a{sv}"));

    g_variant_builder_add(&builder, "{sv}", "DeviceHandle",
                          g_variant_new_int32(l_SMSData.u16DeviceHandle));
    g_variant_builder_add(&builder, "{sv}", "MessageID",
                          g_variant_new_int32(l_SMSData.u16MessageHandle));
    g_variant_builder_add(&builder, "{sv}", "ListID",
                          g_variant_new_int32(l_SMSData.u16ListHandle));
    g_variant_builder_add(&builder, "{sv}", "Recipients",
                          g_variant_new_string(l_SMSData.poToPhoneNumber));
    g_variant_builder_add(&builder, "{sv}", "DeliveryTimestamp",
                          g_variant_new("s","not supported"));

    result = g_variant_builder_end(&builder);
    
    GVariant *result_ref = g_variant_ref_sink(result);
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( "GVariant Building Ends"));

 
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,
            DLT_STRING( "emits delivery failure message"));
    SmsManager* poSMSManager = poGetSMSManager();
    if(poSMSManager != NULL)
    {
        sms_manager_emit_on_delivery_error(poSMSManager,
                                           result_ref);
    }
    else
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,
                DLT_STRING( "SmsManager object IS NULL"));
    }
    
    g_variant_unref(result_ref);
}


/******************************************************************************
* Function:    vTriggerOnSentSuccessSignal
* Description: Triggers the SMS status signal.
* Parameters:  SMSNotifyType, SMSData
* Return:      boolean
******************************************************************************/
void vTriggerOnSentSuccessSignal(SMSData l_SMSData)
{
    //wrap the data in a gvariant.
    GVariantBuilder builder;
    GVariant *result = NULL;
    g_variant_builder_init(&builder, G_VARIANT_TYPE("a{sv}"));

    g_variant_builder_add(&builder, "{sv}", "DeviceHandle",
                          g_variant_new_int32(l_SMSData.u16DeviceHandle));
    g_variant_builder_add(&builder, "{sv}", "MessageID",
                          g_variant_new_int32(l_SMSData.u16MessageHandle));
    g_variant_builder_add(&builder, "{sv}", "ListID",
                          g_variant_new_int32(l_SMSData.u16ListHandle));
    g_variant_builder_add(&builder, "{sv}", "To",
                          g_variant_new_string(l_SMSData.poToPhoneNumber));
    g_variant_builder_add(&builder, "{sv}", "Timestamp",
                          g_variant_new_string(l_SMSData.poTimeStamp));
    g_variant_builder_add(&builder, "{sv}", "Read",
                          g_variant_new_boolean(l_SMSData.bRead));
    g_variant_builder_add(&builder, "{sv}", "Body",
                          g_variant_new_string(l_SMSData.poMessage));
    g_variant_builder_add(&builder, "{sv}", "State",
                          g_variant_new_byte(l_SMSData.eSMSState));
    g_variant_builder_add(&builder, "{sv}", "DeliveryStatus",
                          g_variant_new_byte(l_SMSData.eDeliveryState));
    g_variant_builder_add(&builder, "{sv}", "DeliveryTimestamp",
                          g_variant_new_string("not supported"));

    result = g_variant_builder_end(&builder);
    GVariant *result_ref = g_variant_ref_sink(result);
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( "GVariant Building Ends"));


    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( "emits sent message"));
    SmsManager* poSMSManager = poGetSMSManager();
    if(poSMSManager != NULL)
    {
        sms_manager_emit_on_sent(poSMSManager,
                                 result_ref);
    }
    else
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,
                DLT_STRING( "SmsManager object IS NULL"));
    }

    g_variant_unref(result_ref);
}

/******************************************************************************
 * Function:    handle_prepare_contact_list
 * Description: Handler method for prepare contact list
 * Parameters:  *object, *invocation, options
 * Return:      boolean
 *****************************************************************************/
gboolean handle_prepare_contact_list (ContactManager *object,
                                      GDBusMethodInvocation *invocation,
                                      GVariant *arg_options)
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,
            DLT_STRING("Telephone :+handle_prepare_contact_list"));
    g_invocation_prepare_contacts_list = g_object_ref(invocation);
    if(arg_options == NULL)
    {
        char sError[] = "Options are invalid: NULL";
        vPrepareContactListError(AP_CONTACT_ERROR_INVALID_ATTRIBUTE_VALUE,
                                 sError);
        return TRUE;
    }
    else
    {
        GVariantIter iter;
        GVariant *value = NULL;
        gchar *key = NULL;
        guchar cSortBy;
        guint16 uDeviceHandle;
        gboolean bIsSortByFound = FALSE;
        gboolean bIsDeviceHandleFound = FALSE;
        gboolean bIsInputInvalid = FALSE;

        g_variant_iter_init(&iter,arg_options);
        while (g_variant_iter_loop (&iter, "{sv}", &key, &value))
        {
            if(!g_strcmp0 (key,"SortBy"))
            {
                g_variant_get(value,"y",&cSortBy);
                bIsSortByFound = TRUE;
            }
            else if(!g_strcmp0(key,"DeviceHandle"))
            {
                g_variant_get(value,"q",&uDeviceHandle);
                bIsDeviceHandleFound = TRUE;
            }
            else
            {
                bIsInputInvalid = TRUE;
            }
        }

        if(bIsSortByFound == FALSE || bIsDeviceHandleFound == FALSE)
        {
            char sError[] = "Options are unavailable";
            vPrepareContactListError(AP_CONTACT_ERROR_INVALID_ATTRIBUTE_VALUE,
                                     sError);
        }
        else if(bIsInputInvalid)
        {
            char sError[] = "Options are incorrect.";
            vPrepareContactListError(AP_CONTACT_ERROR_INVALID_ATTRIBUTE_VALUE,
                                     sError);
        }
        else
        {
            vPrepareContactList((ApTelephonySortBy)cSortBy,uDeviceHandle);
        }
    }
    return TRUE;
}

/******************************************************************************
 * Function:    vPrepareContactListResult
 * Description: sends the result for prepare contact list method.
 * Parameters:  int, int
 * Return:      boolean
 ******************************************************************************/
void vPrepareContactListResult(int iListHandle, int iNumOfContacts)
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,
            DLT_STRING("Telephone :+vPrepareContactListResult"),
            DLT_INT(iListHandle),DLT_INT(iNumOfContacts));


    ContactManager* poContactManager = poGetContactManager();
    if(poContactManager != NULL && g_invocation_prepare_contacts_list != NULL)
    {
        contact_manager_complete_prepare_contact_list(
                    poContactManager,
                    g_invocation_prepare_contacts_list,
                    iNumOfContacts, iListHandle);
		g_clear_object(&g_invocation_prepare_contacts_list);
    }
    else
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING(" invocation pointer or object is NULL"));
        return;
    }
}

/******************************************************************************
 * Function:    vPrepareContactListError
 * Description: sends the error for prepare contact list method.
 * Parameters:  APContactError, gchar*
 * Return:      boolean
 ******************************************************************************/
void vPrepareContactListError(APContactError enError, const gchar* error_msg)
{
    if(g_invocation_prepare_contacts_list == NULL)
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,DLT_STRING( __FUNCTION__),
                DLT_STRING(" invocation pointer is NULL"));
        return;
    }
    else
    {
        if(error_msg != NULL)
        {
            g_dbus_method_invocation_return_error(
                        g_invocation_prepare_contacts_list,
                        AP_CONTACT_ERROR,
                        enError,
                        error_msg);
        }
        else
        {
            g_dbus_method_invocation_return_error(
                        g_invocation_prepare_contacts_list,
                        AP_CONTACT_ERROR,
                        enError,
                        "Error in calling prepare contact list");
        }
		g_clear_object(&g_invocation_prepare_contacts_list);
    }
}

/******************************************************************************
 * Function:    handle_get_contacts_in_slice
 * Description: Handler method for destroy contact list
 * Parameters:  *object, *invocation, gvariant*, gint
 * Return:      boolean
 *****************************************************************************/
gboolean handle_get_contacts_in_slice (ContactManager *object,
                                       GDBusMethodInvocation *invocation,
                                       GVariant *arg_options,
                                       gint arg_handle)
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,
            DLT_STRING("Telephone :+handle_get_contacts_in_slice"));

    g_invocation_get_contacts_list = g_object_ref(invocation);
    if(arg_options == NULL)
    {
        char sError[] = "Parameter option is NULL";
        vGetContactsInSliceError(AP_CONTACT_ERROR_ATTRIBUTE_UNAVAILABLE,
                                 sError);
        return TRUE;
    }
    else
    {

        DLT_LOG(AGW_Phone,DLT_LOG_INFO,
                DLT_STRING("handle "),DLT_INT(arg_handle));


        GVariantIter iter;
        GVariant *value;
        gchar *key;
        guint32 uWindowStart = 0;
        guint16 uWindowSize = 0;
        gboolean bIsWindowStartFound = FALSE;
        gboolean bIsWindowSizeFound = FALSE;

        g_variant_iter_init(&iter,arg_options);
        while (g_variant_iter_loop (&iter, "{sv}", &key, &value))
        {
            if(!g_strcmp0 (key,"WindowStart"))
            {
                g_variant_get(value,"u",&uWindowStart);
                DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( "uWindowStart:"),
                        DLT_UINT32( uWindowStart));
                bIsWindowStartFound = TRUE;
            }
            else if(!g_strcmp0 (key,"WindowSize"))
            {
                g_variant_get(value,"q",&uWindowSize);
                DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( "WindowSize:"),
                        DLT_UINT16( uWindowSize));
                bIsWindowSizeFound = TRUE;
            }
            else
            {
                continue;
            }
        }

        if(bIsWindowStartFound && bIsWindowSizeFound)
        {
            vGetContactsInSlice(arg_handle,uWindowStart,uWindowSize);
        }
        else
        {
            char sError[] = "Parameters window start and size not available";
            vGetContactsInSliceError(AP_CONTACT_ERROR_ATTRIBUTE_UNAVAILABLE,
                                     sError);
        }
    }

    return TRUE;
}

/******************************************************************************
 * Function:    vGetContactsInSliceResult
 * Description: sends the result for get contact list method.
 * Parameters:  ContactListInfo contacts_list[], int list_size
 * Return:      void
 ******************************************************************************/
void vGetContactsInSliceResult(ContactInfo* contacts_list, int list_size)
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING("list_size : = "),
            DLT_INT(list_size));

    if(contacts_list == NULL || 0 == list_size)
    {
        char sError[] = "Contact List is NULL";
        vGetContactsInSliceError(AP_CONTACT_ERROR_ILLEGAL_BEHAVIOUR,
                                 sError);
    }
    else
    {
        GVariantBuilder get_contacts_builder;
        g_variant_builder_init (&get_contacts_builder, G_VARIANT_TYPE ("a(sa{sv})"));

        int iCount = 0;
        while (iCount < list_size)
        {
            ContactInfo st_ContactInfoItem = contacts_list[iCount];

            GVariantBuilder sub_builder;
                    g_variant_builder_init (&sub_builder, G_VARIANT_TYPE ("a{sv}"));

            //ContactName Array
            GVariantBuilder name_builder;
                    g_variant_builder_init (&name_builder, G_VARIANT_TYPE ("a{sv}"));

            g_variant_builder_add (&name_builder, "{sv}", "FirstName",
                                   g_variant_new_string(
                                       st_ContactInfoItem.contactNameArray[0]));
            g_variant_builder_add (&name_builder, "{sv}", "LastName",
                                   g_variant_new_string(
                                       st_ContactInfoItem.contactNameArray[1]));
            GVariant* nameArray = g_variant_builder_end (&name_builder);

            g_variant_builder_add (&sub_builder, "{sv}", "Name",
                                   g_variant_new("v", nameArray));

            DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING(" iNumOfPhns :"),
                    DLT_INT(st_ContactInfoItem.ui_PhoneNumberCount));

            //PhoneNumbers Array
            GVariantBuilder phnNum_builder;
            g_variant_builder_init ( &phnNum_builder, G_VARIANT_TYPE ("a(s)"));

            for(int itr = 0;
                itr != st_ContactInfoItem.ui_PhoneNumberCount;
                itr++)
            {
                g_variant_builder_add(&phnNum_builder, "(s)",
                                      st_ContactInfoItem.phoneNumberArray[itr]);
            }

            GVariant* phoneNumArray = g_variant_builder_end (&phnNum_builder);
            g_variant_builder_add (&sub_builder, "{sv}",
                                   "PhoneNumbers", phoneNumArray);

            //Email Array
            GVariantBuilder email_builder;
            g_variant_builder_init (& email_builder, G_VARIANT_TYPE ("a(s)"));
            DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING(" NumOfEmails :"),
                    DLT_INT(st_ContactInfoItem.ui_EmailAddressCount));

            for(int itr = 0;
                itr != st_ContactInfoItem.ui_EmailAddressCount;
                itr++)
            {
                g_variant_builder_add(
                            &email_builder,"(s)",
                            st_ContactInfoItem.emailAddressArray[itr]);
            }
            GVariant* emailArray = g_variant_builder_end (&email_builder);
            g_variant_builder_add (&sub_builder, "{sv}", "Emails", emailArray);
            
            g_variant_builder_add (&get_contacts_builder, "(sa{sv})",
                                   st_ContactInfoItem.strContactHandle,
                                   &sub_builder);
            vCleanContactInfoData(&st_ContactInfoItem);
            iCount++;
        }

        GVariant* varContactList = g_variant_builder_end(&get_contacts_builder);
		GVariant* varContactListRef = g_variant_ref_sink(varContactList);

        ContactManager* poContactManager = poGetContactManager();
        if(poContactManager != NULL && g_invocation_get_contacts_list != NULL)
        {
            contact_manager_complete_get_contacts_in_slice (
                        poContactManager,
                        g_invocation_get_contacts_list,
                        varContactListRef);
			g_clear_object(&g_invocation_get_contacts_list);
        }
        else
        {
            DLT_LOG(AGW_Phone,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                    DLT_STRING(" invocation pointer or object is NULL"));
        }
		g_variant_unref(varContactListRef);
    }
    free(contacts_list);
}

/******************************************************************************
 * Function:    vGetContactsInSliceError
 * Description: sends the error for get contact list method.
 * Parameters:  GVariant*
 * Return:      void
 ******************************************************************************/
void vGetContactsInSliceError(APContactError enError, const gchar* error_msg)
{
    if(g_invocation_get_contacts_list == NULL)
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,DLT_STRING( __FUNCTION__),
                DLT_STRING(" invocation pointer is NULL"));
        return;
    }

    if(error_msg != NULL)
    {
        g_dbus_method_invocation_return_error(g_invocation_get_contacts_list,
                                              AP_CONTACT_ERROR,
                                              enError,
                                              error_msg);
    }
    else
    {
        g_dbus_method_invocation_return_error(g_invocation_get_contacts_list,
                                              AP_CONTACT_ERROR,
                                              enError,
                                              "Error in getting contact list");
    }
    g_clear_object(&g_invocation_get_contacts_list);

}

/******************************************************************************
 * Function:    handle_hangup_call
 * Description: Handles termination of the active call
 * Parameters:  object,invocation,user_data
 * Return:      gboolean
 ******************************************************************************/
gboolean
handle_hangup_call (CallManager *object,
                    GDBusMethodInvocation *invocation,
                    unsigned int callinstance)
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    g_invocation_hangup_call = g_object_ref(invocation); //store the invocation pointer.
    vHangUpCall_Method(callinstance);
    return TRUE;

}

/******************************************************************************
* Function:    vHangUpMResult
* Description: Hang up method result
* Parameters:  void 
* Return:      void
******************************************************************************/
void vHangUpMResult()  
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    //push the complete to D-Bus.
    CallManager* poCallManager = poGetCallManager();
    if(g_invocation_hangup_call != NULL && poCallManager != NULL)
    {
        call_manager_complete_hang_up (poCallManager,
                                       g_invocation_hangup_call);
    }
    else
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,
                DLT_STRING( "g_invocation_hangup_call or object IS NULL"));
    }
	g_clear_object(&g_invocation_hangup_call);
} 

/******************************************************************************
* Function:    vHangUpMError
* Description: Hang up method error
* Parameters:  APTelephonyError enError 
* Return:      void
******************************************************************************/
void vHangUpMError(APTelephonyError enError)  
{
    DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

    if(g_invocation_hangup_call != NULL)
    {
        g_dbus_method_invocation_return_error (g_invocation_hangup_call,
                                               AP_TELEPHONY_ERROR,
                                               enError,
                                               "Error in Hang up call");
    }
    else
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,
                DLT_STRING( "g_invocation_hangup_call or object IS NULL"));
    }
    g_clear_object(&g_invocation_hangup_call);
}

/*******************************************************************************
 * Function:    handle_get_message_list_in_slice
 * Description: Handler for get message list in slice.
 * Parameters:  SmsManager *object, GDBusMethodInvocation *invocation,
                GVariant *arg_options, guint16 arg_handle
 * Return:      boolean
 ******************************************************************************/
gboolean handle_get_message_list_in_slice(SmsManager *object,
                                          GDBusMethodInvocation *invocation,
                                          GVariant *arg_options,
                                          guint16 arg_handle)
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));
    g_invocation_get_messages_list = g_object_ref(invocation);
    if(NULL == arg_options  || 0 == arg_handle)
    {
        vGetMessagesListSlice_Error(AP_TELEPHONY_ERROR_INVALID_ATTRIBUTE_VALUE,
                                    "Options are incorrect.");
        return TRUE;
    }
    else
    {

        DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING("handle : "),
                DLT_UINT16(arg_handle));
        
        GVariantIter iter;
        GVariant *value = NULL;
        gchar *key = NULL;
        guint32 uWindowStart = 0;
        guint16 uWindowSize = 0;
        gboolean bIsWindowStartFound = FALSE;
        gboolean bIsWindowSizeFound = FALSE;

        g_variant_iter_init(&iter, arg_options);
        while (g_variant_iter_loop (&iter, "{sv}", &key, &value))
        {
            if(!g_strcmp0 (key, "WindowStart"))
            {
                g_variant_get(value, "u", &uWindowStart);
                DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( "uWindowStart:"),
                        DLT_UINT32( uWindowStart));
                bIsWindowStartFound = TRUE;
            }
            else if(!g_strcmp0 (key, "WindowSize"))
            {
                g_variant_get(value, "q", &uWindowSize);
                DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( "WindowSize:"),
                        DLT_UINT16(uWindowSize));
                bIsWindowSizeFound = TRUE;
            }
            else
            {
                continue;
            }
        }

        if(bIsWindowStartFound && bIsWindowSizeFound)
        {
            vGetMessagesListSlice(arg_handle, uWindowStart, uWindowSize);
        }
        else
        {
            char sError[] = "Window start or window size is not available";
            vGetMessagesListSlice_Error(
                        AP_TELEPHONY_ERROR_INVALID_ATTRIBUTE_VALUE,
                        sError);
        }
    }

    return TRUE;
}

/******************************************************************************
 * Function:    vGetMessagesListSliceResult
 * Description: sends the result of the get message in slice method.
 * Parameters:  void
 * Return:      void
 ******************************************************************************/
void vGetMessagesListSliceResult(SMSData* messages_list, gint32 list_size)
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));
    if(0 != list_size)
    {
        GVariantBuilder builder;
        g_variant_builder_init (&builder, G_VARIANT_TYPE ("a(a{sv})"));

        int iCount = 0;
        while (iCount < list_size)
        {
            SMSData l_SMSData = messages_list[iCount];

            GVariantBuilder inner_builder;
            g_variant_builder_init (&inner_builder, G_VARIANT_TYPE ("a{sv}"));

            g_variant_builder_add(&inner_builder, "{sv}", "DeviceHandle",
                                  g_variant_new_int32(
                                      l_SMSData.u16DeviceHandle));

            g_variant_builder_add(&inner_builder, "{sv}", "MessageID",
                                  g_variant_new_int32(
                                      l_SMSData.u16MessageHandle));

            g_variant_builder_add(&inner_builder, "{sv}", "ListID",
                                  g_variant_new_int32(l_SMSData.u16ListHandle));

            g_variant_builder_add(&inner_builder, "{sv}", "From",
                                  g_variant_new("s",
                                                l_SMSData.poFromPhoneNumber));

            g_variant_builder_add(&inner_builder, "{sv}", "Timestamp",
                                  g_variant_new("s", l_SMSData.poTimeStamp));

            g_variant_builder_add(&inner_builder, "{sv}", "Read",
                                  g_variant_new_boolean(l_SMSData.bRead));

            g_variant_builder_add(&inner_builder, "{sv}", "Message",
                                  g_variant_new("s", l_SMSData.poMessage));

            g_variant_builder_add(&inner_builder, "{sv}", "State",
                                  g_variant_new_byte(l_SMSData.eSMSState));

            g_variant_builder_add(&inner_builder, "{sv}", "DeliveryStatus",
                                  g_variant_new_byte(l_SMSData.eDeliveryState));

            g_variant_builder_add(&inner_builder, "{sv}", "DeliveryTimestamp",
                                  g_variant_new("s",
                                                l_SMSData.poDeliveryTimeStamp));

            g_variant_builder_add(&builder, "(a{sv})", &inner_builder);

            vCleanSMSData(&l_SMSData);
            iCount++;
        }
        GVariant *unread_messages_list = g_variant_builder_end(&builder);
        GVariant *varUnreadMsgListRef = g_variant_ref_sink(unread_messages_list);

        SmsManager* poSMSManager = poGetSMSManager();
        if(g_invocation_get_messages_list == NULL || poSMSManager == NULL)
        {
            DLT_LOG(AGW_Phone, DLT_LOG_ERROR,
                    DLT_STRING( __FUNCTION__),
                    DLT_STRING(" invocation pointer or object is NULL"));
        }
        else
        {
            sms_manager_complete_get_message_list_in_slice(
                        poSMSManager,
                        g_invocation_get_messages_list,
                        varUnreadMsgListRef);
        }
		g_variant_unref(varUnreadMsgListRef);
		
    }
    else
    {
        vGetMessagesListSlice_Error(AP_TELEPHONY_ERROR_INTERNAL,
                                    "Message list is null");
    }
    free(messages_list);
    g_clear_object(&g_invocation_get_messages_list);
}

/******************************************************************************
 * Function:    vGetMessagesListSlice_Error
 * Description: sends the error for get message list in slice method.
 * Parameters:  APTelephonyError, error_msg
 * Return:      void
 ******************************************************************************/
void vGetMessagesListSlice_Error(APTelephonyError enError,
                                 const gchar* error_msg)
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));

    if(g_invocation_get_messages_list == NULL)
    {
        DLT_LOG(AGW_Phone, DLT_LOG_ERROR,
                DLT_STRING( __FUNCTION__),
                DLT_STRING(" invocation pointer is NULL"));
        return;
    }

    if(error_msg == NULL)
    {
        g_dbus_method_invocation_return_error(g_invocation_get_messages_list,
                                              AP_TELEPHONY_ERROR,
                                              enError,
                                              "Error in Message List Slice");
    }
    else
    {
        g_dbus_method_invocation_return_error(g_invocation_get_messages_list,
                                              AP_TELEPHONY_ERROR,
                                              enError,
                                              error_msg);
    }

    g_clear_object(&g_invocation_get_messages_list);
}

/*******************************************************************************
 * Function:	handle_prepare_get_messages_list
 * Description: Handler for prepare get message list
 * Parameters:  SmsManager *object, GDBusMethodInvocation *invocation,
                guint16 arg_device_handle, GVariant *arg_options
 * Return:	    boolean
 ******************************************************************************/
gboolean handle_prepare_get_messages_list(SmsManager *object,
                                          GDBusMethodInvocation *invocation,
                                          guint16 arg_device_handle,
                                          GVariant *arg_options)
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__)
            ,DLT_STRING("arg_device_handle : "), DLT_INT(arg_device_handle));
    g_invocation_prepare_get_messages_list = g_object_ref(invocation);
    if(NULL != arg_options)
    {
        GVariantIter iter;
        gchar *key = NULL;
        GVariant *value = NULL;
        guchar cSortBy;
        gboolean bIsSortByFound = FALSE;

        g_variant_iter_init(&iter, arg_options);
        while (g_variant_iter_loop (&iter, "{sv}", &key, &value))
        {
            if(!g_strcmp0 (key,"SortBy"))
            {
                g_variant_get(value,"y",&cSortBy);
                bIsSortByFound = TRUE;
            }
            else
            {
                char sError[] =
                        "Invalid Parameteres: Only SortBy option is supported";

                vPrepareGetMessagesList_Error(
                            AP_TELEPHONY_ERROR_INVALID_ATTRIBUTE_VALUE, sError);
                return TRUE;
            }
        }

        if(bIsSortByFound)
        {
            if((ApTelephonySortBy)cSortBy != AP_TELEPHONY_SORTBY_TIMESTAMP)
            {
                char sError[] =
                        "Invalid Parameteres: Only SortBy timestamp is supported";
                vPrepareGetMessagesList_Error(
                            AP_TELEPHONY_ERROR_INVALID_ATTRIBUTE_VALUE, sError);
                return TRUE;
            }
        }
        vPrepareMessagesList(arg_device_handle);
    }
    else
    {
        char sError[] = "Invalid Parameteres: Options is NULL or Invalid";
        vPrepareGetMessagesList_Error(
                    AP_TELEPHONY_ERROR_INVALID_ATTRIBUTE_VALUE, sError);
    }

    return TRUE;
}

/******************************************************************************
 * Function:    vGetMessagesListResult
 * Description: sends the result of the get message list method.
 * Parameters:  unsigned int iListHandle, unsigned int iNumOfMessages
 * Return:      void
 ******************************************************************************/
void vGetMessagesListResult(unsigned short iListHandle,
                            unsigned short iNumOfMessages)
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__),
            DLT_STRING("ListHandle : "), DLT_UINT16(iListHandle),
            DLT_STRING("NumOfMessages : "), DLT_UINT16(iNumOfMessages));

    SmsManager* poSMSManager = poGetSMSManager();
    if(g_invocation_prepare_get_messages_list == NULL || poSMSManager == NULL)
    {
        DLT_LOG(AGW_Phone, DLT_LOG_ERROR,
                DLT_STRING( __FUNCTION__),
                DLT_STRING(" invocation pointer or object is NULL"));
        return;
    }
    else
    {
        sms_manager_complete_prepare_get_messages_list(
                    poSMSManager,
                    g_invocation_prepare_get_messages_list,
                    iListHandle,
                    iNumOfMessages);
    }
    g_clear_object(&g_invocation_prepare_get_messages_list);
}

/******************************************************************************
 * Function:    vPrepareGetMessagesList_Error
 * Description: sends the error for prepare get message method.
 * Parameters:  APTelephonyError, error_msg
 * Return:      void
 ******************************************************************************/
void vPrepareGetMessagesList_Error(APTelephonyError enError,
                                   const gchar* error_msg)
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));

    if(g_invocation_prepare_get_messages_list == NULL)
    {
        DLT_LOG(AGW_Phone, DLT_LOG_ERROR, DLT_STRING( __FUNCTION__),
                DLT_STRING(" invocation pointer is NULL"));
        return;
    }

    if(error_msg == NULL)
    {
        g_dbus_method_invocation_return_error(
                    g_invocation_prepare_get_messages_list,
                    AP_TELEPHONY_ERROR,
                    enError,
                    "Error in Message List");
    }
    else
    {
        g_dbus_method_invocation_return_error(
                    g_invocation_prepare_get_messages_list,
                    AP_TELEPHONY_ERROR,
                    enError,
                    error_msg);
    }

    g_clear_object(&g_invocation_prepare_get_messages_list);
} 

/******************************************************************************
 * Function:    handle_get_message
 * Description: Handler method for get message method.
 * Parameters:  *object, *invocation, guint16
 * Return:      boolean
 *****************************************************************************/
gboolean 
handle_read_message(SmsManager *object,
                    GDBusMethodInvocation *invocation,
                    guint16 arg_device_handle,
                    guint16 arg_message_id,
                    guint16 arg_list_id)
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));
    g_invocation_read_message = g_object_ref(invocation);
    if(0 == arg_message_id || 0 == arg_device_handle || 0 == arg_list_id)
    {
        vReadMessageError(AP_TELEPHONY_ERROR_INVALID_ATTRIBUTE_VALUE,
                          "Parameter received was invalid");
    }
    else
    {
        vReadMessage(arg_device_handle, arg_message_id, arg_list_id);
    }
    return TRUE;
}

/******************************************************************************
 * Function:    vReadMessageResult
 * Description: sends the result of the get message list method.
 * Parameters:  unsigned int iListHandle, unsigned int iNumOfMessages
 * Return:      void
 ******************************************************************************/
void vReadMessageResult(SMSData* message_details)
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));

    SmsManager* poSMSManager = poGetSMSManager();
    if(g_invocation_read_message == NULL || poSMSManager == NULL)
    {
        DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__),
                DLT_STRING(" invocation pointer or object is NULL"));
        return;
    }
    else
    {
        if(message_details == NULL)
        {
            vReadMessageError(AP_TELEPHONY_ERROR_ILLEGAL_BEHAVIOUR,
                              "Message Details are corrupted");
            return;
        }
        
        GVariantBuilder message_builder;
        g_variant_builder_init (&message_builder, G_VARIANT_TYPE ("a{sv}"));

        g_variant_builder_add(
                    &message_builder, "{sv}", "MessageID",
                    g_variant_new_int32(message_details->u16MessageHandle));
        if( message_details->poFromPhoneNumber != NULL)
        {
            g_variant_builder_add(
                        &message_builder, "{sv}", "From",
                        g_variant_new("s", message_details->poFromPhoneNumber));
        }
        else if(message_details->poToPhoneNumber != NULL)
        {
            g_variant_builder_add(
                        &message_builder, "{sv}", "To",
                        g_variant_new("s", message_details->poToPhoneNumber));
        }
        g_variant_builder_add(
                    &message_builder, "{sv}", "Timestamp",
                    g_variant_new("s", message_details->poTimeStamp));

        g_variant_builder_add(
                    &message_builder, "{sv}", "Read",
                    g_variant_new_boolean(message_details->bRead));

        if( message_details->poMessage != NULL)
        {
            g_variant_builder_add(
                        &message_builder, "{sv}", "Message",
                        g_variant_new("s", message_details->poMessage));
        }
        
        GVariant* varSMSData = g_variant_builder_end(&message_builder);
        
        vCleanSMSData(message_details);
        GVariant* varSMSDataRef = g_variant_ref_sink(varSMSData);
		
        sms_manager_complete_read_message(poSMSManager,
                                          g_invocation_read_message,
                                          varSMSDataRef);
	    g_variant_unref(varSMSDataRef);
    }
    g_clear_object(&g_invocation_read_message);
}

/******************************************************************************
 * Function:    vReadMessageError
 * Description: sends the error for get message method.
 * Parameters:  APTelephonyError, error_msg
 * Return:      void
 ******************************************************************************/
void vReadMessageError(APTelephonyError enError, const gchar* error_msg)
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));

    if(g_invocation_read_message == NULL)
    {
        DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__),
                DLT_STRING(" invocation pointer is NULL"));
        return;
    }

    if(error_msg == NULL)
    {
        g_dbus_method_invocation_return_error(g_invocation_read_message,
                                              AP_TELEPHONY_ERROR,
                                              enError,
                                              "Error in get message");
    }
    else
    {
        g_dbus_method_invocation_return_error(g_invocation_read_message,
                                              AP_TELEPHONY_ERROR,
                                              enError,
                                              error_msg);
    }

    g_clear_object(&g_invocation_read_message);
} 

/******************************************************************************
 * Function:    vUpdateSignalStrengthStatus
 * Description: Wrapper function to update CellSignalStrength.
 * Parameters:  unsigned int u8SignalStrengthStatus
 * Return:      void
 ******************************************************************************/
void vUpdateSignalStrengthStatus(unsigned short u8DeviceHandle, 
                                 unsigned short u8SignalStrengthStatus)
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));
    GVariantBuilder builder;
    GVariantBuilder inner_builder;
    GVariant *result = NULL;

    g_variant_builder_init(&inner_builder, G_VARIANT_TYPE("a{sv}"));

    g_variant_builder_add(&inner_builder, "{sv}", "SignalStrength",
                          g_variant_new_uint16(u8SignalStrengthStatus));

    GVariant* varProperties = g_variant_builder_end (&inner_builder);

    g_variant_builder_init(&builder, G_VARIANT_TYPE("a{sv}"));

    g_variant_builder_add(&builder, "{sv}", "DeviceHandle",
                          g_variant_new_uint16(u8DeviceHandle));
    g_variant_builder_add(&builder, "{sv}", "TelephonyProvider",
                          varProperties);

    result = g_variant_builder_end(&builder);
    GVariant *result_ref = g_variant_ref_sink(result);
    CallManager* poCallManager = poGetCallManager();
    if(poCallManager == NULL)
    {
        DLT_LOG(AGW_Phone,DLT_LOG_ERROR,DLT_STRING( __FUNCTION__),
                DLT_STRING("CallManage object is NULL"));
    }
    call_manager_emit_on_device_properties_change(poCallManager,
                                                  result_ref);
    g_variant_unref(result_ref);
}

/******************************************************************************
* Function:    vExportSMSManagerInterface
* Description: function to expose SMS manager interface
* Parameters:  void
* Return:      void
******************************************************************************/
void vExportSMSManagerInterface()
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));
    GDBusConnection* poConn = poGetGDBusConnection();
    GError *error = NULL;
    SmsManager* poSMSManager = poGetSMSManager();
    if((NULL == poSMSManager) || (NULL == poConn))
    {
        DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING(
                    "SMS manager : - conn ptr/object ptr is NULL"));
        return;
    }
    if(!g_dbus_interface_skeleton_export (
                G_DBUS_INTERFACE_SKELETON (poSMSManager),
                poConn,
                "/com/bosch/AutomotiveProxy/SmsManager",
                &error))
    {
        if(error != NULL)
        {
            DLT_LOG(AGW_Phone,DLT_LOG_ERROR,
                    DLT_STRING("SMS manager:"
                               "- ERROR-- Interface skeleton export failed - "),
                    DLT_STRING(error->message));

            g_clear_error (&error);
        }
        return;
    }
    else
    {
        DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING(
                    "SMS manager : - Object created & exported"));
    }

    /* method signal connection for messages */
    g_signal_connect (poSMSManager,
                      "handle-prepare-get-messages-list",
                      G_CALLBACK (handle_prepare_get_messages_list), NULL);

    g_signal_connect (poSMSManager,
                      "handle-get-message-list-in-slice",
                      G_CALLBACK (handle_get_message_list_in_slice), NULL);
    
    g_signal_connect (poSMSManager,
                      "handle-read-message",
                      G_CALLBACK (handle_read_message), NULL);
}


/******************************************************************************
* Function:    vUnexportSMSManagerInterface
* Description: function to unexport SMS manager interface
* Parameters:  void
* Return:      void
******************************************************************************/
void vUnexportSMSManagerInterface()
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));
    SmsManager* poSMSManager = poGetSMSManager();
    if(poSMSManager)
    {
        DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING(
                    "SMS manager : - unexporting interface"));
        g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (poSMSManager));
    }
}


/******************************************************************************
* Function:    vExportCallManagerInterface
* Description: function to expose call manager interface
* Parameters:  void
* Return:      void
******************************************************************************/
void vExportCallManagerInterface()
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));
    GDBusConnection* poConn = poGetGDBusConnection();
    CallManager* poCallManager = poGetCallManager();
    if((NULL == poCallManager) || (NULL == poConn))
    {
        DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING(
                    "Call manager : - conn ptr/object ptr is NULL"));
        return;
    }
    GError *error = NULL;
    if (!g_dbus_interface_skeleton_export (
                G_DBUS_INTERFACE_SKELETON (poCallManager),
                poConn,
                "/com/bosch/AutomotiveProxy/CallManager",
                &error))
    {
        if(error != NULL)
        {
            DLT_LOG(AGW_Phone,DLT_LOG_ERROR,
                    DLT_STRING("Call manager:"
                               " -ERROR- Interface skeleton export failed - "),
                    DLT_STRING(error->message));
            g_clear_error (&error);
        }
        return;
    }
    else
    {
        DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING(
                    "Call manager : - Object created & exported"));
    }

    /* method signal connection for telephone */
    g_signal_connect (poCallManager,
                      "handle-prepare-call-history-list",
                      G_CALLBACK (handle_prepare_call_history_list), NULL);

    g_signal_connect (poCallManager,
                      "handle-get-call-history-in-slice",
                      G_CALLBACK (handle_get_call_history_in_slice), NULL);

    g_signal_connect (poCallManager,
                      "handle-hang-up",
                      G_CALLBACK (handle_hangup_call), NULL);

}


/******************************************************************************
* Function:    vUnexportCallManagerInterface
* Description: function to unexport call manager interface
* Parameters:  void
* Return:      void
******************************************************************************/
void vUnexportCallManagerInterface()
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));
    CallManager* poCallManager = poGetCallManager();
    if(poCallManager)
    {
        DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING(
                    "Call manager : - unexporting interface"));
        g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (poCallManager));
    }
}

/******************************************************************************
* Function:    vExportContactManagerInterface
* Description: function to expose call manager interface
* Parameters:  void
* Return:      void
******************************************************************************/
void vExportContactManagerInterface()
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));
    GDBusConnection* poConn = poGetGDBusConnection();
    ContactManager* poContactManager = poGetContactManager();
    if((NULL == poContactManager) || (NULL == poConn))
    {
        DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING(
                    "Contact manager : - conn ptr/object ptr is NULL"));
        return;
    }
    GError *error = NULL;
    if (!g_dbus_interface_skeleton_export (
                G_DBUS_INTERFACE_SKELETON (poContactManager),
                poConn,
                "/com/bosch/AutomotiveProxy/ContactManager",
                &error))
    {
        if(error != NULL)
        {
            DLT_LOG(AGW_Phone,DLT_LOG_ERROR,
                    DLT_STRING("Contact manager:"
                               " -ERROR- Interface skeleton export failed - "),
                    DLT_STRING(error->message));
            g_clear_error (&error);
        }
        return;
    }
    else
    {
        DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING(
                    "Contact manager : - Object created & exported"));
    }

    /* method signal connection for contact */
    g_signal_connect (poContactManager,
                      "handle-get-contacts-in-slice",
                      G_CALLBACK (handle_get_contacts_in_slice), NULL);

    g_signal_connect (poContactManager,
                      "handle-prepare-contact-list",
                      G_CALLBACK (handle_prepare_contact_list), NULL);

}


/******************************************************************************
* Function:    vUnexportContactManagerInterface
* Description: function to unexport call manager interface
* Parameters:  void
* Return:      void
******************************************************************************/
void vUnexportContactManagerInterface()
{
    DLT_LOG(AGW_Phone, DLT_LOG_INFO, DLT_STRING( __FUNCTION__));

    ContactManager* poContactManager = poGetContactManager();
    if(poContactManager)
    {
        DLT_LOG(AGW_Phone,DLT_LOG_INFO,DLT_STRING(
                    "Contact manager : - unexporting interface"));
        g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (poContactManager));
    }
}
#endif//VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC