/*!
*******************************************************************************
* @file             : NaviDBusHandler.h
*******************************************************************************
*  - PROJECT:       : Main Navigation handler file for Automotive D-Bus server
*  - SW-COMPONENT   : Gateway
*  - DESCRIPTION    : D-Bus server for automotive API's
*  - COPYRIGHT      : &copy; 2015 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|------------------------
*  16.03.2016 | Ashwini Savadi (RBEI/ECP2)	| 1.0.0 | methods for Automotive
                                                     D-Bus Server
*  19.01.2017 | Ashwini Savadi(RBEI/ECO22)	| 2.0.0 | Route information propety update to 
*														DBus client handled
*  24.01.2017 | Ashwini Savadi (RBEI/ECO22) | 3.0.0 | Adapted Review comments
*  14.02.2017 | Ashwini Savadi (RBEI/ECO22) | 4.0.0 | Implmented GeoCode feature
*  10.04.2017 | Ashwini Savadi (RBEI/ECO22) | 5.0.0 |Route Guidance Feature implementation
*  17.07.2017 | Deepa Jose     (RBEI/ECO22) | 6.0.0 |Route List Waypoint implementation,
*                                                    Navigation.IsAvailable, TMCStatus
******************************************************************************/

#ifndef NAVI_DBUS_MAIN_HANDLER_H
#define NAVI_DBUS_MAIN_HANDLER_H

#include "PositionProvider_generated.h"
#include "Navigation_RouteGuidance_generated.h"
#include "Navigation_RouteList_generated.h"
#include "Geocode_generated.h"
#include "Navigation_generated.h"

#include <stdbool.h>

typedef enum
{
    NAV_STATE_UNKNOWN = 0,
    NAV_STATE_BUSY,
    NAV_STATE_NAVDATA_UPDATING,
    NAV_STATE__MAP_RECOVERY_MODE,
    NAV_STATE_IDLE,
    NAV_STATE_GUIDANCE_ACTIVE,
    NAV_STATE_CALCULATING_ROUTE,
    NAV_STATE_ACTIVE_OFFBOARD
} ApNavStatus;

typedef struct
{
    gboolean bIsStartGuidance;
    gsize l_iInsertWaypointFirst;
    GVariant *g_gvWaypointList;
} structRouteGuidanceInfo;


/**
@brief     vInitWaypointOp : For waypoint insertion through URI
@param     GVariant*
@return    void
*/
void vInitWaypointOp(GVariant* poWaypointList);

/**
@brief     bRegisterNaviProperties : for getting property updates
           from AppHMI Navigaion service
@param     void
@return    gboolean
*/
gboolean bRegisterNaviProperties();

/**
@brief     vUpdateRouteGuidanceStatus: For updating Guidance status by Framework
           and Navigation handler
@param     void
@return    void
*/
void vUpdateRouteGuidanceStatus();

/**
@brief     navigation_properties_changed_cb: For getting property update of
           AGW: TMCStatus
@param     GDBusProxy *, GVariant *, const gchar* const *, gpointer
@return    void
*/
void navigation_properties_changed_cb(
        GDBusProxy *proxy, GVariant *changed_properties,
        const gchar* const  *invalidated_properties, gpointer user_data);

/**
@brief     bInitializeNavi: Initializing Navigation through framework
@param     void
@return    void
*/
gboolean bInitializeNavi();

/**
@brief     vUninitializeNavi: Uninitializing Navigation through framework
@param     void
@return    void
*/
void vUninitializeNavi();

/**
@brief     vSetAGWNavigationAvailabilty
@param     void
@return    void
*/
void vSetAGWNavigationAvailabilty();


// ************************ GEOCODE method handlers ************************//

/**
@brief     handle_reverse_resolve_method
@param     Geocode *, GDBusMethodInvocation *, GVariant *
@return    gboolean
*/
gboolean handle_reverse_resolve_method(Geocode *object
                                       , GDBusMethodInvocation *invocation
                                       , GVariant *arg_geocode_entity);
/**
@brief     handle_forward_resolve_method
@param     Geocode *, GDBusMethodInvocation *, const gchar *, guint , guint
@return    gboolean
*/
gboolean handle_forward_resolve_method(Geocode *object
                                       , GDBusMethodInvocation *invocation
                                       , const gchar *arg_address
                                       , guint arg_startIndex
                                       , guint arg_numElement);

// ********************* POSITIONPROVIDER method handlers ********************//

/**
@brief     handle_get_position_method
@param     PositionProvider *, GDBusMethodInvocation *, gpointer
@return    gboolean
*/
gboolean handle_get_position_method (PositionProvider *object
                                     , GDBusMethodInvocation *invocation
                                     , gpointer user_data);

// ********************* ROUTEGUIDANCE method handlers ***********************//

/**
@brief     handle_stop_guidance_method
@param     NavigationRouteGuidance *, GDBusMethodInvocation *
@return    gboolean
*/
gboolean handle_stop_guidance_method (NavigationRouteGuidance *object
                                      , GDBusMethodInvocation *invocation);

/**
@brief     handle_start_guidance_method
@param     NavigationRouteGuidance *, GDBusMethodInvocation *
                                    , gdouble, gdouble , GVariant *, guint
@return    gboolean
*/
gboolean handle_start_guidance_method (NavigationRouteGuidance *object
                                       , GDBusMethodInvocation *invocation
                                       , gdouble arg_latitude
                                       , gdouble arg_longitude
                                       , GVariant *arg_way_points
                                       , guint arg_route_criteria);
/**
@brief     handle_get_current_destination_method
@param     NavigationRouteGuidance * , GDBusMethodInvocation *
@return    gboolean
*/
gboolean handle_get_current_destination_method(NavigationRouteGuidance *object
                                               , GDBusMethodInvocation *invocation);

// ************************* ROUTELIST method handlers ***********************//

/**
@brief     handle_insert_way_point_method
@param     NavigationRouteList *, GDBusMethodInvocation *
                                , gdouble , gdouble, guint , guint
@return    gboolean
*/
gboolean handle_insert_way_point_method(NavigationRouteList *object
                                        , GDBusMethodInvocation *invocation
                                        , gdouble arg_latitude
                                        , gdouble arg_longitude
                                        , guint arg_route_index
                                        , guint arg_way_point_index);
/**
@brief     handle_replace_way_point_method
@param     NavigationRouteList *, GDBusMethodInvocation *
                                , gdouble, gdouble , guint , guint
@return    gboolean
*/
gboolean handle_replace_way_point_method(NavigationRouteList *object
                                         , GDBusMethodInvocation *invocation
                                         , gdouble arg_latitude
                                         , gdouble arg_longitude
                                         , guint arg_route_index
                                         , guint arg_way_point_index);
/**
@brief     handle_delete_way_points_method
@param     NavigationRouteList *, GDBusMethodInvocation *
                                , guint , guint , guint
@return    gboolean
*/
gboolean handle_delete_way_points_method(NavigationRouteList *object
                                         , GDBusMethodInvocation *invocation
                                         , guint arg_route_index
                                         , guint arg_start_index
                                         , guint arg_end_index);
/**
@brief     handle_read_favorites_list
@param     Navigation *, GDBusMethodInvocation *
@return    gboolean
*/
gboolean handle_read_favorites_list(Navigation *object
                                    , GDBusMethodInvocation *invocation);

/**
@brief     handle_add_to_favorites_list
@param     Navigation *, GDBusMethodInvocation *, GVariant *
@return    gboolean
*/
gboolean handle_add_to_favorites_list(Navigation *object
                                      , GDBusMethodInvocation *invocation
                                      , GVariant *arg_favorite_location);

/**
@brief   Gets the navi position provider proxy object
@param   void
@return  PositionProvider*
*/
PositionProvider* poNavGetPositionProxyObj();

/**
@brief     Gets the navi route list proxy object
@param     void
@return:   NavigationRouteList*
*/
NavigationRouteList* poNavGetRouteListProxyObj();

/**
@brief      Gets the navi route guidance proxy object
@param      void
@return     NavigationRouteGuidance*
*/
NavigationRouteGuidance* poNavGetRouteGuidanceProxyObj();

/**
@brief      Gets the navigation proxy object
@param      void
@return     Navigation*
*/
Navigation* poGetNavigationProxyObj();

/**
@brief      Gets the navi route list proxy object
@param      void
@return     NavigationRouteList*
*/
Geocode* poNavGetGeocodeProxyObj();

/**
@brief  	Get Proxy pointer.
@param      void
@return		GDBusProxy
*/
GDBusProxy* poGetNaviProxy();

/**
@brief      sets Navi proxy
@param      GDBusProxy
@return     void
*/
void vSetNaviProxy(GDBusProxy* poProxy);

/**
@brief      sets bUpdateRoute variable
@param      gboolean
@return     void
*/
void set_bUpdateRoute(gboolean arg_value);

/**
@brief      return the structRouteGuidanceInfo
@param      void
@return     structRouteGuidanceInfo
*/
structRouteGuidanceInfo get_structRouteGuidanceInfo();

/**
@brief      setsstructRouteGuidanceInfo
@param      gboolean , gsize , GVariant *
@return     void
*/
void set_structRouteGuidanceInfo(gboolean arg_bValue, gsize arg_uSize
                                 , GVariant *arg_vWaypointList);

#endif //NAVI_DBUS_MAIN_HANDLER_H
