/******************************************************************************
* FILE:         FC_Gateway_MediaPlayerClientHandler.cpp
* PROJECT:      G3G project
* SW-COMPONENT: generic gateway
* DESCRIPTION:  CCA client handler for MediaPlayer
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2016 Robert Bosch GmbH, Hildesheim
******************************************************************************/

#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#include "FC_Gateway_MediaPlayerClientHandler.h"
#include "ap-media-player-error-enums.h"
#include "ap-media-player-enums.h"

// ETG defines
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#include "mplay_fi_if.h"

//Dlt
#include "dlt/dlt.h"
//Dlt Context
DLT_DECLARE_CONTEXT(AGW_MediaPlayer);

#define MPLAY_FI_S_IMPORT_INTERFACE_MPLAY_MEDIAPLAYERFI_TYPES
#define MPLAY_FI_S_IMPORT_INTERFACE_MPLAY_MEDIAPLAYERFI_FUNCTIONIDS
#define MPLAY_FI_S_IMPORT_INTERFACE_MPLAY_MEDIAPLAYERFI_SERVICEINFO
#define MPLAY_FI_S_IMPORT_INTERFACE_MPLAY_MEDIAPLAYERFI_DBGVISITORS
#define MPLAY_FI_S_IMPORT_INTERFACE_MPLAY_DEVICEMANAGERFI_ERRORCODES

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_INTERNETAPPLICATIONS_APPLICATION
#include "trcGenProj/Header/FC_Gateway_MediaPlayerClientHandler.cpp.trc.h"
#endif

/*****************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/*****************************************************************************/
#define MEDIAPLAYER_SERVICE_ID              558
#define MEDIAPLAYER_FI_MAJOR_VERSION        4
#define MEDIAPLAYER_FI_MINOR_VERSION        5

FC_Gateway_MediaPlayerClientHandler* 
FC_Gateway_MediaPlayerClientHandler::m_poFC_Gateway_MediaPlayerClientHandler =
                                                                           NULL;

/*****************************************************************************/
/*                                                                            */
/* CCA MESSAGE MAP                                                            */
/*                                                                            */
/*****************************************************************************/

BEGIN_MSG_MAP(FC_Gateway_MediaPlayerClientHandler, ahl_tclBaseWork)

	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_REQUESTPLAYBACKACTION, 
	                    AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,
						vHandleRequestPlaybackActionOpMResult)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_SEEKTO,
                  	    AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,
						vHandleSeekToOpMResult)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_PLAYTIME,
            	        AMT_C_U8_CCAMSG_OPCODE_STATUS,
 						vHandlePlaytimeOpStatus)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKSTATE,
          	            AMT_C_U8_CCAMSG_OPCODE_STATUS,
						vHandlePlaybackStateOpStatus)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKMODE,
  	                    AMT_C_U8_CCAMSG_OPCODE_STATUS,
						vHandlePlaybackModeOpStatus)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_REPEATMODE,
            	        AMT_C_U8_CCAMSG_OPCODE_STATUS,
						vHandleRepeatModeOpStatus)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_CURRENTFOLDERPATH,
 	                    AMT_C_U8_CCAMSG_OPCODE_STATUS,
						vHandleCurrentFolderPathOpStatus)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_NOWPLAYING,
	                    AMT_C_U8_CCAMSG_OPCODE_STATUS,
						vHandleNowPlayingOpStatus)
	
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_PLAYTIME,
	                    AMT_C_U8_CCAMSG_OPCODE_GET,
						vHandlePlaytimeOpStatus)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKSTATE,
 	                    AMT_C_U8_CCAMSG_OPCODE_GET,
						vHandlePlaybackStateOpStatus)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKMODE,
	                    AMT_C_U8_CCAMSG_OPCODE_GET,
					    vHandlePlaybackModeOpStatus)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_REPEATMODE,
    	                AMT_C_U8_CCAMSG_OPCODE_GET,
						vHandleRepeatModeOpStatus)

	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_REQUESTPLAYBACKACTION,
	                    AMT_C_U8_CCAMSG_OPCODE_ERROR,
						vHandleRequestPlaybackActionOpMError)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_SEEKTO,
	                    AMT_C_U8_CCAMSG_OPCODE_ERROR,
						vHandleSeekToOpMError)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_PLAYTIME,
 	                    AMT_C_U8_CCAMSG_OPCODE_ERROR,
						vHandlePlaytimeOpMError)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKSTATE,
	                    AMT_C_U8_CCAMSG_OPCODE_ERROR,
						vHandlePlaybackStateOpMError)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKMODE,
	                    AMT_C_U8_CCAMSG_OPCODE_ERROR,
						vHandlePlaybackModeOpMError)
	ON_MESSAGE_SVCDATA( MPLAY_MEDIAPLAYERFI_C_U16_REPEATMODE,
	                    AMT_C_U8_CCAMSG_OPCODE_ERROR,
						vHandleRepeatModeOpMError)
END_MSG_MAP()

/*****************************************************************************/
/* METHODS                                                                    */
/*****************************************************************************/

/******************************************************************************
*
* FUNCTION: FC_Gateway_MediaPlayerClientHandler::
*            FC_Gateway_MediaPlayerClientHandler(gateway_tclAppMain* poMainAppl)
*
* DESCRIPTION: Constructor.
*
*              Create an object of the base class
*              ahl_tclBaseOneThreadClientHandler with a pointer to this
*              application, the to be used service identifier and the service
*              version as parameters.
*
* PARAMETER: [IN] poMainAppl = Pointer to the object of this application.
*
* RETURNVALUE: None.
*
******************************************************************************/
FC_Gateway_MediaPlayerClientHandler::FC_Gateway_MediaPlayerClientHandler(
                               procgenericgateway_tclMainApp* poMainAppl)
                          : ahl_tclBaseOneThreadClientHandler(
                                (ahl_tclBaseOneThreadApp*)poMainAppl, 
                                MEDIAPLAYER_SERVICE_ID, 
	                        MEDIAPLAYER_FI_MAJOR_VERSION,
			        MEDIAPLAYER_FI_MINOR_VERSION )
{
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
	                                        DLT_STRING(" entered."));
   m_poMainApp = poMainAppl;
   vRegisterPropertyOnStartup();

   m_lastPlaybackAction = kPlay;
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
	                                        DLT_STRING(" left."));
}

/******************************************************************************
*
* FUNCTION: 	~FC_Gateway_MediaPlayerClientHandler
*
* DESCRIPTION: Destructor.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
******************************************************************************/
FC_Gateway_MediaPlayerClientHandler::~FC_Gateway_MediaPlayerClientHandler(
										tVoid)
{
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
   DLT_UNREGISTER_CONTEXT(AGW_MediaPlayer);
	
   if(m_poFC_Gateway_MediaPlayerClientHandler!= NULL)
   {
        delete m_poFC_Gateway_MediaPlayerClientHandler;
        m_poFC_Gateway_MediaPlayerClientHandler = NULL;
   }
  DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
										   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
* Function:    	vRegisterPropertyOnStartup
*
* Description: 	singleton implementation. Gets client handler instance pointer
*
* Parameters:  	tVoid.
*
* Return:      	tVoid.
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::vRegisterPropertyOnStartup()
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
    vAddAutoRegisterForProperty(MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKSTATE);
    vAddAutoRegisterForProperty(MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKMODE);
    vAddAutoRegisterForProperty(MPLAY_MEDIAPLAYERFI_C_U16_REPEATMODE);	
    vAddAutoRegisterForProperty(MPLAY_MEDIAPLAYERFI_C_U16_PLAYTIME);
    vAddAutoRegisterForProperty(MPLAY_MEDIAPLAYERFI_C_U16_CURRENTFOLDERPATH);
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
										   DLT_STRING(__FUNCTION__));
	
}

/******************************************************************************
* Function:    getInstance
*
* Description: 	singleton implementation. Gets client handler instance pointer
*	
* Parameters:  Main app pointer
*
* Return:      pointer
******************************************************************************/
FC_Gateway_MediaPlayerClientHandler* 
FC_Gateway_MediaPlayerClientHandler::getInstance(
							procgenericgateway_tclMainApp* poMainAppl)
{
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   if(m_poFC_Gateway_MediaPlayerClientHandler == NULL)
   {
      m_poFC_Gateway_MediaPlayerClientHandler = 
                            new FC_Gateway_MediaPlayerClientHandler(poMainAppl);
		
		DLT_REGISTER_CONTEXT(AGW_MediaPlayer,"GWMP","Media Player"
									"context for DLT Logging");
   }
   
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
								
   return m_poFC_Gateway_MediaPlayerClientHandler;
}

/******************************************************************************
*
* FUNCTION: 	vOnServiceAvailable
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              this client-handler has registered for has become available.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************
* Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceAvailable().
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::vOnServiceAvailable()
{
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   vExportAGWMediaPlayerInterface();
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
*
* FUNCTION: 	vOnServiceUnavailable
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              this client-handler has registered for has become unavailable.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************
* Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceUnavailable().
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::vOnServiceUnavailable()
{
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   vUnexportAGWMediaPlayerInterface();
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
*
* FUNCTION: 	sendRequestPlaybackActionMStart
*											  
*
* DESCRIPTION:
*
* PARAMETER: 	mplay_fi_tcl_e8_MPlayAction,tU8
*
* RETURNVALUE: 	tBool
*
*******************************************************************************
*******************************************************************************
* Handling of property 
******************************************************************************/
tBool FC_Gateway_MediaPlayerClientHandler::sendRequestPlaybackActionMStart(
             mplay_fi_tcl_e8_MPlayAction e8Action, tU8 u8NextPrevSkipCount) 
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
    tBool bRetVal = FALSE;
    if (u16GetRegID() != AMT_C_U16_REGID_INVALID) 
    {
	   DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
										DLT_STRING("Valid Registered Id."));
       mplay_mediaplayerfi_tclMsgRequestPlaybackActionMethodStart  oMStart;
       //the data content
       oMStart.e8Action = e8Action;
       oMStart.u8NextPrevSkipCount = u8NextPrevSkipCount;


       fi_tclVisitorMessage oOutMessage(oMStart);
       //Initialize the message with register id and so on

	   vInitServiceData( oOutMessage,
     	                 (tU16) 0,
						 (tU16) MPLAY_MEDIAPLAYERFI_C_U16_REQUESTPLAYBACKACTION,
						 (tU8) AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

      /* Post message - since the second parameter is true
	     we need not explicitly delete the message */
      ail_tenCommunicationError enResult =
        	         _poMainAppl->enPostMessage(&oOutMessage, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR) {
         // Error situation:
		 DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
		                                         DLT_STRING(" error"));
		 DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
        	 DLT_STRING(" error: "), DLT_INT(m_lastPlaybackAction));

		  // Always destroy the FI data object before leaving its creation scope
		  oMStart.vDestroy();
		  bRetVal =  FALSE;

      } else {
         bRetVal = TRUE;
      }
   }
   else
   {
		DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
		        DLT_STRING(__FUNCTION__),DLT_STRING("InValid Registered Id."));
        return FALSE;
   }

   if(e8Action.enType == mplay_fi_tcl_e8_MPlayAction::FI_EN_E8PBA_PLAY )
		m_lastPlaybackAction = kPlay;
   if(e8Action.enType == mplay_fi_tcl_e8_MPlayAction::FI_EN_E8PBA_PAUSE )
		m_lastPlaybackAction = kPause;
   if(e8Action.enType == mplay_fi_tcl_e8_MPlayAction::FI_EN_E8PBA_STOP )
		m_lastPlaybackAction = kStop;
   if(e8Action.enType == mplay_fi_tcl_e8_MPlayAction::FI_EN_E8PBA_NEXT )
		m_lastPlaybackAction = kNext;
   if(e8Action.enType == mplay_fi_tcl_e8_MPlayAction::FI_EN_E8PBA_PREV )
		m_lastPlaybackAction = kPrev;
   DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
		DLT_STRING("The last playback action:"),DLT_INT(m_lastPlaybackAction));
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
   return TRUE;
}

/******************************************************************************
*
* FUNCTION: 	sendSeekToMStart
*
* DESCRIPTION:
*
* PARAMETER: 	tU32,mplay_fi_tcl_e8_MPlayPlaypointFormat.
*
* RETURNVALUE: 	tBool.
*
******************************************************************************/
tBool FC_Gateway_MediaPlayerClientHandler::sendSeekToMStart(
                     tU32                                 u32PlaypointPosition,
					 mplay_fi_tcl_e8_MPlayPlaypointFormat e8PlaypointFormat) 
{
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   tBool bRetVal = FALSE;
   if (u16GetRegID() != AMT_C_U16_REGID_INVALID) 
   {
	    DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
											DLT_STRING("Valid Registered Id"));
      mplay_mediaplayerfi_tclMsgSeekToMethodStart  oMStart;
      //the data content
      oMStart.u32PlaypointPosition = u32PlaypointPosition;
      oMStart.e8PlaypointFormat = e8PlaypointFormat;


      fi_tclVisitorMessage oOutMessage(oMStart);
      //Initialize the message with register id and so on
      vInitServiceData( oOutMessage,
      	                (tU16) 0, 
	                    (tU16) MPLAY_MEDIAPLAYERFI_C_U16_SEEKTO,
						(tU8) AMT_C_U8_CCAMSG_OPCODE_METHODSTART);
      /* Post message - since the second parameter is true
  	     we need not explicitly delete the message */
      ail_tenCommunicationError enResult = 
	                  _poMainAppl->enPostMessage(&oOutMessage, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR) {
         // Error situation:
      } else {
         bRetVal = TRUE;
      }

      // Always destroy the FI data object before leaving its creation scope
      oMStart.vDestroy();
   }
   else
   {
	  DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
										DLT_STRING("InValid Registered Id"));
      return FALSE;
   }
   return bRetVal;
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION: 	vHandleRequestPlaybackActionOpMResult
*                                     
*
* DESCRIPTION:
*
* PARAMETER: 	amt_tclServiceData*.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
/******************************************************************************
* Handling of property MPLAY_MEDIAPLAYERFI_C_U16_REQUESTPLAYBACKACTION
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::
										vHandleRequestPlaybackActionOpMResult(	
											  amt_tclServiceData* poMessage) 
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	 
   if (poMessage != NULL) 
   {
         fi_tclVisitorMessage            oInMsg( poMessage );
         mplay_mediaplayerfi_tclMsgRequestPlaybackActionMethodResult  oMResult;

		   tS32 s32Result = oInMsg.s32GetData(
										 oMResult,
										 MEDIAPLAYER_FI_MAJOR_VERSION
										 );
		   if(OSAL_ERROR != s32Result)
		   {
				DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
				    "FC_Gateway_MediaPlayerClientHandler::"
					 "vHandleRequestPlaybackActionOpMResult(): Method result"
					 "update message received for"
					 "'MPLAY_MEDIAPLAYERFI_C_U16_REQUESTPLAYBACKACTION'"
					 "(FID: 0x212)!"));
				onRequestPlaybackActionMResult();
		   }
		   else
		   {
				DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
				   "FC_Gateway_MediaPlayerClientHandler::"
				   "vHandleRequestPlaybackActionOpMResult: Received message"
				   "is invalid!"));
		   }

         oMResult.vDestroy();
   } 
    DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	       "FC_Gateway_MediaPlayerClientHandler::"
		   "vHandleRequestPlaybackActionOpMResult left."));
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
*
* FUNCTION: 	onRequestPlaybackActionMResult
*                                  
*
* DESCRIPTION:
*
* PARAMETER: 	void
*
* RETURNVALUE: 	tVoid.
*******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::onRequestPlaybackActionMResult() 
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   // onRequestPlaybackActionMResult: user specific code start
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),DLT_STRING(
	       "The last playback action: "),DLT_INT( m_lastPlaybackAction));
	if(m_lastPlaybackAction == kPlay)
		vRequestPlayMResult();

	if(m_lastPlaybackAction == kPause )
		vRequestPauseMResult();

	if(m_lastPlaybackAction == kStop )
		vRequestStopMResult();

	if(m_lastPlaybackAction == kNext )
		vRequestNextMResult();

	if(m_lastPlaybackAction == kPrev )
		vRequestPrevMResult();
	
   // onRequestPlaybackActionMResult: user specific code end
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));

}
/******************************************************************************
*
* FUNCTION:   	vHandleSeekToOpMResult
*
* DESCRIPTION:
*
* PARAMETER: 	amt_tclServiceData*.
*	
* RETURNVALUE: 	tVoid.
*******************************************************************************
*******************************************************************************
* Handling of property MPLAY_MEDIAPLAYERFI_C_U16_SEEKTO
******************************************************************************/

tVoid FC_Gateway_MediaPlayerClientHandler::vHandleSeekToOpMResult(
                                                 amt_tclServiceData* poMessage) 
{
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   if (poMessage != NULL) 
   {
         fi_tclVisitorMessage            oInMsg( poMessage );
         mplay_mediaplayerfi_tclMsgSeekToMethodResult  oMResult;

		   tS32 s32Result = oInMsg.s32GetData(
										 oMResult,
										 MEDIAPLAYER_FI_MAJOR_VERSION
										 );
		   if(OSAL_ERROR != s32Result)
		   {
				DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
				   "FC_Gateway_MediaPlayerClientHandler::"
				   "vHandleSeekToOpMResult(): Method result update message" 
				   "received for 'MPLAY_MEDIAPLAYERFI_C_U16_SEEKTO'"
				   "(FID: 0x213)!"));

				onSeekToMResult();
		   }
		   else
		   {
			  DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
		         "FC_Gateway_MediaPlayerClientHandler::vHandleSeekToOpMResult:"
				 "Received message is invalid!"));
		   }
         oMResult.vDestroy();
   } 
  DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION: 	onSeekToMResult
*
* DESCRIPTION:
*
* PARAMETER: 	void.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::onSeekToMResult() 
{
	 DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   // onSeekToMResult: user specific code start
	vSeekToMResult();
   // onSeekToMResult: user specific code end
	 DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION:	 	vHandlePlaytimeOpStatus
*
* DESCRIPTION:
*
* PARAMETER: 	amt_tclServiceData*.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
/******************************************************************************
* Handling of property MPLAY_MEDIAPLAYERFI_C_U16_PLAYTIME
******************************************************************************/

tVoid FC_Gateway_MediaPlayerClientHandler::vHandlePlaytimeOpStatus(
                                                amt_tclServiceData* poMessage) 
{
    #if 0
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
    #endif
   if (poMessage != NULL) {
         fi_tclVisitorMessage            oInMsg( poMessage );
         mplay_mediaplayerfi_tclMsgPlaytimeStatus  oStatus;

         if ( oInMsg.s32GetData(oStatus, MEDIAPLAYER_FI_MAJOR_VERSION) != 
		      OSAL_ERROR)
 	     {

            #if 0
            DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
			DLT_STRING("Status update message received for"
			   " MPLAY_MEDIAPLAYERFI_C_U16_PLAYTIME"));
            #endif

            onPlaytimeStatus( oStatus.u32ElapsedPlaytime,
      			              oStatus.u32TotalPlaytime );

         } else {
            #if 0
             DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
						DLT_STRING("Received message is invalid!"));
            #endif
         }

         oStatus.vDestroy();
   }  
  DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION: 	onPlaytimeStatus
*
* DESCRIPTION:
*
* PARAMETER: 	tU32,tU32
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::onPlaytimeStatus(
                                     tU32 u32ElapsedPlaytime,
 									 tU32 u32TotalPlaytime )
{
    DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   tU32 currentPlaytime = u32TotalPlaytime - u32ElapsedPlaytime;
#if 0
   DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING("Call sendPlayTimeStatus"));
#endif
   vSendPlayTimeStatus(currentPlaytime);
    DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
*
* FUNCTION: 	vHandlePlaybackStateOpStatus
*											
* DESCRIPTION:
*
* PARAMETER: 	amt_tclServiceData*
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
/******************************************************************************
* Handling of property MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKSTATE
******************************************************************************/

tVoid FC_Gateway_MediaPlayerClientHandler::vHandlePlaybackStateOpStatus(
                                           amt_tclServiceData* poMessage) 
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   if (poMessage != NULL)
   {
         fi_tclVisitorMessage            oInMsg( poMessage );
         mplay_mediaplayerfi_tclMsgPlaybackStateStatus  oStatus;

         if (oInMsg.s32GetData(oStatus, MEDIAPLAYER_FI_MAJOR_VERSION) != 
		                  OSAL_ERROR)
 	     {

            #if 0
             DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
			    "FC_Gateway_MediaPlayerClientHandler::"
				"vHandlePlaybackStateOpStatus(): Status update message received"
				"for 'MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKSTATE' (FID: 0x209)!"));
			DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
			                              "Call onPlaybackStateStatus"));
            #endif
            onPlaybackStateStatus(oStatus.e8State);

         } else {
            DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
			    "FC_Gateway_MediaPlayerClientHandler::"
				"vHandlePlaybackStateOpStatus: Received message is invalid!"));
         }

         oStatus.vDestroy();
   }  
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
*
* FUNCTION: 	onPlaybackStateStatus
*
* DESCRIPTION:
*
* PARAMETER:  	mplay_fi_tcl_e8_MPlayState
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::onPlaybackStateStatus(
                              mplay_fi_tcl_e8_MPlayState e8State) 
{
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   switch(e8State.enType)
   {
   case mplay_fi_tcl_e8_MPlayState::FI_EN_E8PBS_PLAYING:
      vSendPlaybackStatus(AP_MPLAYER_STATE_PLAYING);
      break;
   case mplay_fi_tcl_e8_MPlayState::FI_EN_E8PBS_PAUSED:
	  vSendPlaybackStatus(AP_MPLAYER_STATE_PAUSED);
      break;
   case mplay_fi_tcl_e8_MPlayState::FI_EN_E8PBS_STOPPED:
	  vSendPlaybackStatus(AP_MPLAYER_STATE_STOPPED);
      break;

   default:
      DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	              "onPlaybackStateStatus invalid enum value."));
   }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
*
* FUNCTION: 	vHandlePlaybackModeOpStatus
*
* DESCRIPTION:
*
* PARAMETER:  	amt_tclServiceData*
*
* RETURNVALUE: 	tVoid.
*******************************************************************************/
/*******************************************************************************
* Handling of property MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKMODE
*******************************************************************************/

tVoid FC_Gateway_MediaPlayerClientHandler::vHandlePlaybackModeOpStatus(
                                         amt_tclServiceData* poMessage) 
{
    DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   if (poMessage != NULL) {
         fi_tclVisitorMessage            oInMsg( poMessage );
         mplay_mediaplayerfi_tclMsgPlaybackModeStatus  oStatus;

         if (oInMsg.s32GetData(oStatus, MEDIAPLAYER_FI_MAJOR_VERSION) !=
      		 OSAL_ERROR)
		 {

            #if 0
             DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
			        "FC_Gateway_MediaPlayerClientHandler::"
					"vHandlePlaybackModeOpStatus(): Status update message"
					" received for 'MPLAY_MEDIAPLAYERFI_C_U16_PLAYBACKMODE'"
					" (FID: 0x210)!"));
			DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
			         DLT_STRING("Call onPlaybackModeStatus"));
           #endif
            onPlaybackModeStatus(oStatus.e8Mode);

         } else {
            DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
			    "FC_Gateway_MediaPlayerClientHandler::"
				"vHandlePlaybackModeOpStatus: Received message is invalid!"));
         }

         oStatus.vDestroy();
   }  
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION:   	onPlaybackModeStatus
*
* DESCRIPTION:
*
* PARAMETER: 	mplay_fi_tcl_e8_MPlayMode
*	
* RETURNVALUE: 	tVoid.
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::onPlaybackModeStatus(
                               mplay_fi_tcl_e8_MPlayMode e8Mode) 
{
    DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__)); 

   // onPlaybackModeStatus: user specific code start
   switch(e8Mode.enType)
   {
   case  mplay_fi_tcl_e8_MPlayMode::FI_EN_E8PBM_NORMAL:
      vSendShuffleStatus(FALSE);
      break;
   case mplay_fi_tcl_e8_MPlayMode::FI_EN_E8PBM_RANDOM:
	  vSendShuffleStatus(TRUE);
      break;
   default:
      vSendShuffleStatus(FALSE);
   }
   // onPlaybackModeStatus: user specific code end
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
*
* FUNCTION:   	vHandleRepeatModeOpStatus
*
* DESCRIPTION:
*
* PARAMETER:	amt_tclServiceData*
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
/******************************************************************************
* Handling of property MPLAY_MEDIAPLAYERFI_C_U16_REPEATMODE
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::vHandleRepeatModeOpStatus(
                                       amt_tclServiceData* poMessage) 
{
#if 0
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
#endif
   if (poMessage != NULL) {
         fi_tclVisitorMessage            oInMsg( poMessage );
         mplay_mediaplayerfi_tclMsgRepeatModeStatus  oStatus;

         if (oInMsg.s32GetData(oStatus, MEDIAPLAYER_FI_MAJOR_VERSION) != 
		     OSAL_ERROR)
		 {
            DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
			    "FC_Gateway_MediaPlayerClientHandler::"
				"vHandleRepeatModeOpStatus(): Status update message received"
				" for 'MPLAY_MEDIAPLAYERFI_C_U16_REPEATMODE' (FID: 0x211)!"));
			DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
			                           "Call onRepeatModeStatus"));
            onRepeatModeStatus(oStatus.e8Repeat);

         } else {
            DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
		       "FC_Gateway_MediaPlayerClientHandler::vHandleRepeatModeOpStatus:"
			   "Received message is invalid!"));
         }

         oStatus.vDestroy();
   }  
    DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION:  	onRepeatModeStatus
*
* DESCRIPTION:
*
* PARAMETER:  	mplay_fi_tcl_e8_MPlayRepeat
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::onRepeatModeStatus(
                         mplay_fi_tcl_e8_MPlayRepeat e8Repeat) 
{
     #if 0
    DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
    #endif

   // onRepeatModeStatus: user specific code start
   switch(e8Repeat.enType)
   {
   case mplay_fi_tcl_e8_MPlayRepeat::FI_EN_E8RPT_NONE:
	  vSendLoopStatus(AP_MPLAYER_LOOP_NONE);
      break;
   case mplay_fi_tcl_e8_MPlayRepeat::FI_EN_E8RPT_ONE:
	   vSendLoopStatus(AP_MPLAYER_LOOP_TRACK);
      break;
   case mplay_fi_tcl_e8_MPlayRepeat::FI_EN_E8RPT_LIST:
	   vSendLoopStatus(AP_MPLAYER_LOOP_PLAYLIST);
      break;
   case mplay_fi_tcl_e8_MPlayRepeat::FI_EN_E8RPT_LIST_WITH_SUBLISTS:
	   vSendLoopStatus(AP_MPLAYER_LOOP_PLAYLIST);
      break;
   case mplay_fi_tcl_e8_MPlayRepeat::FI_EN_E8RPT_ALL_LISTS:
	   vSendLoopStatus(AP_MPLAYER_LOOP_PLAYLIST);
      break;
   case mplay_fi_tcl_e8_MPlayRepeat::FI_EN_E8RPT_ALL:
	   vSendLoopStatus(AP_MPLAYER_LOOP_PLAYLIST);
      break;
   default:
     vSendLoopStatus(AP_MPLAYER_LOOP_NONE);
   }
   // onRepeatModeStatus: user specific code end
    #if 0
    DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
    #endif
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}


/******************************************************************************
*
* FUNCTION:  	getCurrentPlayingTrack
*
* DESCRIPTION:
*
* PARAMETER:  	tVoid.
*
* RETURNVALUE: 	tBool.
******************************************************************************/
tBool FC_Gateway_MediaPlayerClientHandler::getCurrentPlayingTrack() 
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   tBool bRetVal = FALSE;
   if (u16GetRegID() != AMT_C_U16_REGID_INVALID) 
   {
	  DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),DLT_STRING("Valid Registered Id."));
      mplay_mediaplayerfi_tclMsgSeekToMethodStart  oMStart;

      fi_tclVisitorMessage oOutMessage(oMStart);
      //Initialize the message with register id and so on
      vInitServiceData( oOutMessage, 
	                    (tU16) 0,
						(tU16) MPLAY_MEDIAPLAYERFI_C_U16_NOWPLAYING,
						(tU8) AMT_C_U8_CCAMSG_OPCODE_GET );
      /* Post message - since the second parameter is true 
	     we need not explicitly delete the message */
      ail_tenCommunicationError enResult = 
	                    _poMainAppl->enPostMessage(&oOutMessage, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR) {
         // Error situation:
      } else {
         bRetVal = TRUE;
      }

      // Always destroy the FI data object before leaving its creation scope
      oMStart.vDestroy();
   }
   else
   {
	  DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),DLT_STRING("InValid Registered Id"));
      return FALSE;
   }
   return bRetVal;
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/*******************************************************************************
*
* FUNCTION:   	vHandleNowPlayingOpStatus
*
* DESCRIPTION:
*
* PARAMETER:  	amt_tclServiceData*.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
/******************************************************************************
* Handling of property MPLAY_MEDIAPLAYERFI_C_U16_NOWPLAYING
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::vHandleNowPlayingOpStatus(
                                       amt_tclServiceData* poMessage) 
{
   #if 0
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
   #endif
   if (poMessage != NULL) {
         fi_tclVisitorMessage            oInMsg( poMessage );
         mplay_mediaplayerfi_tclMsgNowPlayingStatus  oStatus;

         if (oInMsg.s32GetData(oStatus, MEDIAPLAYER_FI_MAJOR_VERSION) != 
		     OSAL_ERROR) 
		 {

            DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
			  "FC_Gateway_MediaPlayerClientHandler::"
			  "vHandleNowPlayingOpStatus(): "
			  "Status update message received for"
			  "'MPLAY_MEDIAPLAYERFI_C_U16_NOWPLAYING' (FID: 0x207)!"));

            onNowPlayingStatus( oStatus.u32ListHandle, 
			                    oStatus.e8NowPlayingState,
								oStatus.u32Position, 
								oStatus.u32Tag,
								&(oStatus.oMediaObject) );

         } else {
            DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
			   "FC_Gateway_MediaPlayerClientHandler::vHandleNowPlayingOpStatus:"
			   "Received message is invalid!"));
         }

         oStatus.vDestroy();
   }  
#if 0
   DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
     "FC_Gateway_MediaPlayerClientHandler::vHandleRepeatModeOpStatus entered."));
#endif
DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION:   	onNowPlayingStatus
*
* DESCRIPTION:
*
* PARAMETER:  	tU32,mplay_fi_tcl_e8_MPlayNowPlayingState,tU32,tU32,
*             	mplay_fi_tcl_MPlayMediaObject*.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
/******************************************************************************
* Handling of property onNowPlayingStatus
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::onNowPlayingStatus(
                      tU32                                 u32ListHandle, 
					  mplay_fi_tcl_e8_MPlayNowPlayingState e8NowPlayingState, 
					  tU32                                 u32Position, 
					  tU32                                 u32Tag,					  
					  mplay_fi_tcl_MPlayMediaObject*        oMediaObject) 
{
   // onNowPlayingStatus: user specific code start
#if 0
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
#endif
	GVariantBuilder builder1;
	GVariantBuilder builder2;
	GVariantBuilder builder_sub;
	ApMPlayerMediaType eMediaType = AP_MPLAYER_MEDIA_TYPE_UNKNOWN;
	char strTrackId[5];
	char * sMetadata1;
	char * sMetadata2;
	char * sMetadata3;
	char * sMetadata4;

	tU16 u16TrackId = oMediaObject->u16ExtendedTrackNumber;

	tU8 u8SourceTag = oMediaObject->u8DeviceTag;

	tU32 u32TotalDuration = oMediaObject->u32TotalPlaytime;

	sMetadata1 = oMediaObject->sMetaDataField1;

	sMetadata2 = oMediaObject->sMetaDataField2;

	sMetadata3 = oMediaObject->sMetaDataField3;

	sMetadata4 = oMediaObject->sMetaDataField4;
#if 0
    DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
              "onNowPlayingStatus : TrackNumber "),DLT_INT(u16TrackId));
    DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
              "onNowPlayingStatus : SourceTag "),DLT_INT(u8SourceTag));

    DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
             "onNowPlayingStatus : TotalDuration "),DLT_INT(u32TotalDuration));
    DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
              "onNowPlayingStatus : sMetadata1 "),DLT_STRING(sMetadata1));

    DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
              "onNowPlayingStatus : sMetadata2 "),DLT_STRING(sMetadata2));
    DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
              "onNowPlayingStatus : sMetadata3 "),DLT_STRING(sMetadata3));
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
              "onNowPlayingStatus : sMetadata4 "),DLT_STRING(sMetadata4));
#endif

	if( (u8SourceTag == 255) && (strcmp(sMetadata1, "") == 0) && 
	    (strcmp(sMetadata2, "") == 0) && 
	    (strcmp(sMetadata3, "") == 0) && (strcmp(sMetadata4, "") == 0))
	{
		DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING("Metadata Empty"));
		vCurrentPlayingTrackInfoError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR);
	}

	vNotifyTrackState(e8NowPlayingState,u16TrackId);
	
	switch(oMediaObject->e8MediaType.enType)
	{
      #if 0
        DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	      "onNowPlayingStatus : Mediatype "),
		  DLT_INT(oMediaObject->e8MediaType.enType));
       #endif
      case mplay_fi_tcl_e8_MPlayMediaType::FI_EN_E8MTY_UNKNOWN:
         eMediaType = AP_MPLAYER_MEDIA_TYPE_UNKNOWN;
           break;
      case mplay_fi_tcl_e8_MPlayMediaType::FI_EN_E8MTY_MUSIC_FILE:
         eMediaType = AP_MPLAYER_MEDIA_TYPE_AUDIO;
           break;
      case mplay_fi_tcl_e8_MPlayMediaType::FI_EN_E8MTY_PODCAST:
         //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
         //                               "MediaType is Podcast"));
           break;
      case mplay_fi_tcl_e8_MPlayMediaType::FI_EN_E8MTY_AUDIOBOOK:
         //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
         //                             "MediaType is Audiobook"));
           break;
      case mplay_fi_tcl_e8_MPlayMediaType::FI_EN_E8MTY_VIDEO:
         eMediaType = AP_MPLAYER_MEDIA_TYPE_VIDEO;
           break;
      case mplay_fi_tcl_e8_MPlayMediaType::FI_EN_E8MTY_PLAYLIST:
         //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
          //                             "MediaType is Playlist"));
           break;
      case mplay_fi_tcl_e8_MPlayMediaType::FI_EN_E8MRY_AUDIO_STREAM:
         //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
         //                          "MediaType is Audio Stream"));
           break;
      case mplay_fi_tcl_e8_MPlayMediaType::FI_EN_E8MTY_PLAYLIST_INTERNAL:
         //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
         //                    "MediaType is Internal playlist"));
           break;
      case mplay_fi_tcl_e8_MPlayMediaType::FI_EN_E8MTY_IMAGE:
         eMediaType = AP_MPLAYER_MEDIA_TYPE_IMAGE;
           break;
      default :
         eMediaType = AP_MPLAYER_MEDIA_TYPE_UNKNOWN;
	}
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	              "onNowPlayingStatus : Mediatype "),DLT_INT(eMediaType));
	
	memset(strTrackId, 0, 5);
	sprintf(strTrackId, "%d", u16TrackId);

    //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
    //                      "Add meta information to GVariant"));

	/* For meta_data_keys */
	g_variant_builder_init (&builder1, G_VARIANT_TYPE("a{sv}"));
	
	g_variant_builder_add (&builder1, "{sv}","AP_MPLAYER_METADATA_KEY_TITLE",
		                             g_variant_new_string(sMetadata4));
	g_variant_builder_add (&builder1, "{sv}","AP_MPLAYER_METADATA_KEY_ARTIST",
		                             g_variant_new_string(sMetadata2));
	g_variant_builder_add (&builder1, "{sv}","AP_MPLAYER_METADATA_KEY_DURATION",
		                             g_variant_new_int32(u32TotalDuration));
	g_variant_builder_add (&builder1, "{sv}","AP_MPLAYER_METADATA_KEY_AUTHOR",
		                             g_variant_new_string(sMetadata3));

	GVariant *metadata = g_variant_builder_end (&builder1);

	g_variant_builder_init (&builder2, G_VARIANT_TYPE ("a{sv}"));
	g_variant_builder_add (&builder2, "{sv}",
            	"source_tag",g_variant_new_int32(u8SourceTag));
	g_variant_builder_add (&builder2, "{sv}", 
	            "source_name",g_variant_new_string(""));
	g_variant_builder_add (&builder2, "{sv}", 
	            "media_type",g_variant_new_int32(eMediaType));
	
	g_variant_builder_add (&builder2, "{sv}", 
	            "meta_data_keys",g_variant_new_variant(metadata));
   
   GVariant *trackInfo = g_variant_builder_end (&builder2);
   
   g_variant_builder_init (&builder_sub, G_VARIANT_TYPE ("a{sv}"));
   g_variant_builder_add (&builder_sub, "{sv}",
                       strTrackId,g_variant_new_variant(trackInfo));
   
   
   
   GVariant *completeTrackInfo = g_variant_builder_end (&builder_sub);
   gchar* sCompleteTrackInfo = g_variant_print (completeTrackInfo, TRUE);
   DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING("sCompleteTrackInfo ::"),
                                            DLT_STRING(sCompleteTrackInfo));
   g_free(sCompleteTrackInfo);
	
    completeTrackInfo = g_variant_take_ref(completeTrackInfo);
			
    //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
    //                       "Invoke vNotifyCurrentPlayingInfo"));
	vNotifyCurrentPlayingInfo(completeTrackInfo);
	g_variant_unref(completeTrackInfo);

    //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
    //       "FC_Gateway_MediaPlayerClientHandler::onNowPlayingStatus left."));
   // onNowPlayingStatus: user specific code end
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/*******************************************************************************
*
* FUNCTION:   	vNotifyTrackState
*
* DESCRIPTION:
*
* PARAMETER:  	mplay_fi_tcl_e8_MPlayNowPlayingState,tU16.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
/******************************************************************************
* Handling of property vNotifyTrackState
******************************************************************************/
tVoid FC_Gateway_MediaPlayerClientHandler::vNotifyTrackState(
         mplay_fi_tcl_e8_MPlayNowPlayingState e8NowPlayingState,tU16 u16TrackId)
{
#if 0
 DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
#endif
   switch(e8NowPlayingState.enType)
   {
       #if 0
        DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	             "vNotifyTrackState: Nowplayingstate "),
			         DLT_INT( e8NowPlayingState.enType));
        #endif

   case mplay_fi_tcl_e8_MPlayNowPlayingState::FI_EN_E8NP_NEW_TRACK:
	   {
        //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
         //             DLT_STRING("Invoke vNotifyTrackChanged"));
		vNotifyTrackChanged(u16TrackId);
	   }
      break;
   case mplay_fi_tcl_e8_MPlayNowPlayingState::FI_EN_E8NP_SAME_TRACK:
       //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
        //             DLT_STRING("NowPlayingState is Same track"));
      break;
   case mplay_fi_tcl_e8_MPlayNowPlayingState::FI_EN_E8NP_TRACK_START:
       //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
       //             DLT_STRING("NowPlayingState is Track start"));
      break;
   case mplay_fi_tcl_e8_MPlayNowPlayingState::FI_EN_E8NP_TRACK_COMPLETE:
       //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
        //        DLT_STRING("NowPlayingState is track complete"));
      break;
   case mplay_fi_tcl_e8_MPlayNowPlayingState::FI_EN_E8NP_LIST_START:
       //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
       //            DLT_STRING("NowPlayingState is list start"));
      break;
   case mplay_fi_tcl_e8_MPlayNowPlayingState::FI_EN_E8NP_LIST_COMPLETE:
       //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
       //         DLT_STRING("NowPlayingState is list complete"));
      break;
   case mplay_fi_tcl_e8_MPlayNowPlayingState::FI_EN_E8NP_NO_DEVICE:
       //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
       //             DLT_STRING("NowPlayingState is No device"));
	   break;
   case mplay_fi_tcl_e8_MPlayNowPlayingState::FI_EN_E8NP_INVALID:
       //DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
       //              DLT_STRING("NowPlayingState is Invalid"));
	   break;
   default:
    DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
	                   DLT_STRING("NowPlayingState is New track"));
   }
 
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
 
}
/******************************************************************************
*
* FUNCTION:  	vHandleCurrentFolderPathOpStatus
*
* DESCRIPTION:
*
* PARAMETER:  	amt_tclServiceData*.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
/******************************************************************************
* Handling of property MPLAY_MEDIAPLAYERFI_C_U16_CURRENTFOLDERPATH
******************************************************************************/

tVoid FC_Gateway_MediaPlayerClientHandler::vHandleCurrentFolderPathOpStatus(
                                              amt_tclServiceData* poMessage) 
{
#if 0
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
#endif
   if (poMessage != NULL) 
   {
         fi_tclVisitorMessage            oInMsg( poMessage );
         mplay_mediaplayerfi_tclMsgCurrentFolderPathStatus  oStatus;

         if ( oInMsg.s32GetData(oStatus, MEDIAPLAYER_FI_MAJOR_VERSION) != 
		      OSAL_ERROR) 
		 {

            DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
			  "gateway_tclClientMediaplayer::vHandleCurrentFolderPathOpStatus():"
              "Status update message received for"
			  " 'MPLAY_MEDIAPLAYERFI_C_U16_CURRENTFOLDERPATH' (FID: 0x235)!"));

            onCurrentFolderPathStatus( oStatus.sCurrentFolderPath,
                         			   oStatus.u32ListHandle);
         } else {
            DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
			   "gateway_tclClientMediaplayer::vHandleCurrentFolderPathOpStatus:"
			   "Received message is invalid!"));
         }

         oStatus.vDestroy();
   }
#if 0
    DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
        
#endif

}
/******************************************************************************
*
* FUNCTION:   	onCurrentFolderPathStatus
*
* DESCRIPTION:
*
* PARAMETER: 	mplay_fi_tclString,tU32.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/

tVoid FC_Gateway_MediaPlayerClientHandler::onCurrentFolderPathStatus(
           mplay_fi_tclString sCurrentFolderPath, tU32 u32ListHandle) 
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));

   // onCurrentFolderPathStatus: user specific code start
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	         "FC_Gateway_MediaPlayerClientHandler::"
		       "onCurrentFolderPathStatus entered."));

	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	         "onCurrentFolderPathStatus: The current folder path is "),
			 DLT_STRING(sCurrentFolderPath.szValue));
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	      "onCurrentFolderPathStatus: The List handle is "),
		                      DLT_INT(u32ListHandle));
	vNotifyCurrentFolderPath(sCurrentFolderPath.szValue, u32ListHandle);

   // onCurrentFolderPathStatus: user specific code end
	 DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
	   
}
/******************************************************************************
*
* FUNCTION: 	vHandleRequestPlaybackActionOpMError
*
* DESCRIPTION:
*
* PARAMETER:  	amt_tclServiceData*.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
/******************************************************************************
* Handling of Error Methods
******************************************************************************/
void FC_Gateway_MediaPlayerClientHandler::vHandleRequestPlaybackActionOpMError(
                                                 amt_tclServiceData* poMessage)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	amt_tclServiceDataError oErrorMsg(poMessage);
	tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING("Error code ::"),
	                                          DLT_INT( u16ErrorCode));

	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),DLT_STRING(
	         "The last playback action: "),DLT_INT( m_lastPlaybackAction));
	if(m_lastPlaybackAction == kPlay)
	{
		vHandlePlayOpMError(u16ErrorCode);
	}
	if(m_lastPlaybackAction == kPause )
	{
		vHandlePauseOpMError(u16ErrorCode);
	}
	if(m_lastPlaybackAction == kStop )
	{
		vHandleStopOpMError(u16ErrorCode);
	}
	if(m_lastPlaybackAction == kNext )
	{
		vHandleNextOpMError(u16ErrorCode);
	}
	if(m_lastPlaybackAction == kPrev )
	{
		vHandlePauseOpMError(u16ErrorCode);
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));

}

/******************************************************************************
*
* FUNCTION:  	vHandlePlayOpMError
*
* DESCRIPTION:
*
* PARAMETER:  	tInt.
*
* RETURNVALUE:	tVoid.
*******************************************************************************/
void FC_Gateway_MediaPlayerClientHandler::vHandlePlayOpMError(int arg_errorCode)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
	          "The received error code ::"),DLT_INT( arg_errorCode));

	 switch(arg_errorCode)
	 {
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UNSPECIFIED:
		vRequestPlayMError(AP_MPLAYER_ERROR_INVALID_HANDLE); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERMISSING:
		vRequestPlayMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERTOOMANY:
		 vRequestPlayMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_INTERNALFAILURE:
		 vRequestPlayMError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_OPCODENOTSUPPORTED:
		 vRequestPlayMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED); 
		 break;
	 default:
		vRequestPlayMError(AP_MPLAYER_ERROR_UNKNOWN); 
		break;
	 }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION: 	vHandlePauseOpMError
*
* DESCRIPTION:
*
* PARAMETER:  	tInt.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
void FC_Gateway_MediaPlayerClientHandler::vHandlePauseOpMError(
                                                  int arg_errorCode)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
	            "The received error code ::"),DLT_INT( arg_errorCode));

	 switch(arg_errorCode)
	 {
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UNSPECIFIED:
		vRequestPauseMError(AP_MPLAYER_ERROR_INVALID_HANDLE); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERMISSING:
		vRequestPauseMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERTOOMANY:
		 vRequestPauseMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_INTERNALFAILURE:
		 vRequestPauseMError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_OPCODENOTSUPPORTED:
		 vRequestPauseMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED); 
		 break;
	 default:
		vRequestPauseMError(AP_MPLAYER_ERROR_UNKNOWN); 
		break;
	 }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION: 	vHandleStopOpMError
*
* DESCRIPTION:
*
* PARAMETER:  	tInt.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
void FC_Gateway_MediaPlayerClientHandler::vHandleStopOpMError(
															int arg_errorCode)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
	        "The received error code ::"),DLT_INT( arg_errorCode));

	 switch(arg_errorCode)
	 {
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UNSPECIFIED:
		vRequestStopMError(AP_MPLAYER_ERROR_INVALID_HANDLE); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERMISSING:
		vRequestStopMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERTOOMANY:
		 vRequestStopMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_INTERNALFAILURE:
		 vRequestStopMError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_OPCODENOTSUPPORTED:
		 vRequestStopMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED); 
		 break;
	 default:
		vRequestStopMError(AP_MPLAYER_ERROR_UNKNOWN); 
		break;
	 }
	 DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/*******************************************************************************
*
* FUNCTION: 	vHandleNextOpMError
*
* DESCRIPTION:
*
* PARAMETER:  	tInt.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
void FC_Gateway_MediaPlayerClientHandler::vHandleNextOpMError(
															int arg_errorCode)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
	              "The received error code ::"),DLT_INT( arg_errorCode));

	 switch(arg_errorCode)
	 {
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UNSPECIFIED:
		vRequestNextMError(AP_MPLAYER_ERROR_INVALID_HANDLE); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERMISSING:
		vRequestNextMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERTOOMANY:
		 vRequestNextMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_INTERNALFAILURE:
		 vRequestNextMError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_OPCODENOTSUPPORTED:
		 vRequestNextMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED); 
		 break;
	 default:
		vRequestNextMError(AP_MPLAYER_ERROR_UNKNOWN); 
		break;
	 }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION:		vHandlePrevOpMError
*
* DESCRIPTION:
*
* PARAMETER:  	tInt.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
void FC_Gateway_MediaPlayerClientHandler::vHandlePrevOpMError(
															int arg_errorCode)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
	            "The received error code ::"),DLT_INT( arg_errorCode));

	 switch(arg_errorCode)
	 {
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UNSPECIFIED:
		vRequestPrevMError(AP_MPLAYER_ERROR_INVALID_HANDLE); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERMISSING:
		vRequestPrevMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERTOOMANY:
		 vRequestPrevMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_INTERNALFAILURE:
		 vRequestPrevMError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_OPCODENOTSUPPORTED:
		 vRequestPrevMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED); 
		 break;
	 default:
		vRequestPrevMError(AP_MPLAYER_ERROR_UNKNOWN); 
		break;
	 }
	 DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
*
* FUNCTION:		vHandleSeekToOpMError
*
* DESCRIPTION:
*
* PARAMETER:  	amt_tclServiceData*.
*
* RETURNVALUE:	tVoid.
******************************************************************************/
void FC_Gateway_MediaPlayerClientHandler::vHandleSeekToOpMError(
                                  amt_tclServiceData* poMessage)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	amt_tclServiceDataError oErrorMsg(poMessage);
	tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
	               "Error code ::"),DLT_INT( u16ErrorCode));

	 switch(u16ErrorCode)
	 {
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UNSPECIFIED:
		vSeekToMError(AP_MPLAYER_ERROR_INVALID_HANDLE); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERMISSING:
		vSeekToMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_PARAMETERTOOMANY:
		 vSeekToMError(AP_MPLAYER_ERROR_BAD_PARAMS); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_BUSY:
		 vSeekToMError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_INTERNALFAILURE:
		 vSeekToMError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR); 
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_OPCODENOTSUPPORTED:
		 vSeekToMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED); 
		 break;
	 default:
		vRequestPrevMError(AP_MPLAYER_ERROR_UNKNOWN); 
		break;
	 }
	 DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
*
* FUNCTION:		vHandlePlaytimeOpMError
*
* DESCRIPTION:
*
* PARAMETER:  	tInt.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
void FC_Gateway_MediaPlayerClientHandler::vHandlePlaytimeOpMError(
                                     amt_tclServiceData* poMessage)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	amt_tclServiceDataError oErrorMsg(poMessage);
	tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
	            "Error code ::"),DLT_INT( u16ErrorCode));

	 switch(u16ErrorCode)
	 {
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UNSPECIFIED:
		vPlaytimeOpMError(AP_MPLAYER_ERROR_INVALID_HANDLE); 
		break;  
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_INTERNALFAILURE:
		 vPlaytimeOpMError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR);
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UPREGFAILURE:
		 vPlaytimeOpMError(AP_MPLAYER_ERROR_INVALID_HANDLE);
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_OPCODENOTSUPPORTED:
		 vPlaytimeOpMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED);
		 break;
	 default:
		vPlaytimeOpMError(AP_MPLAYER_ERROR_UNKNOWN); 
		break;
	 }
	 DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION:   	vHandlePlaybackStateOpMError
*
* DESCRIPTION:
*
* PARAMETER:  	amt_tclServiceData*.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
void FC_Gateway_MediaPlayerClientHandler::vHandlePlaybackStateOpMError(
                                         amt_tclServiceData* poMessage)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	amt_tclServiceDataError oErrorMsg(poMessage);
	tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
	               "Error code ::"),DLT_INT( u16ErrorCode));

	 switch(u16ErrorCode)
	 {
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UNSPECIFIED:
		vPlaybackStateOpMError(AP_MPLAYER_ERROR_INVALID_HANDLE); 
		break;  
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_INTERNALFAILURE:
		 vPlaybackStateOpMError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR);
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UPREGFAILURE:
		 vPlaybackStateOpMError(AP_MPLAYER_ERROR_INVALID_HANDLE);
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_OPCODENOTSUPPORTED:
		 vPlaybackStateOpMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED);
		 break;
	 default:
		vPlaybackStateOpMError(AP_MPLAYER_ERROR_UNKNOWN); 
		break;
	 }
	 DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
*
* FUNCTION:   	vHandlePlaybackModeOpMError
*
* DESCRIPTION:
*
* PARAMETER: 	amt_tclServiceData*.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
void FC_Gateway_MediaPlayerClientHandler::vHandlePlaybackModeOpMError(
                                        amt_tclServiceData* poMessage)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	amt_tclServiceDataError oErrorMsg(poMessage);
	tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING("Error code ::"),
	                                         DLT_INT( u16ErrorCode));

	 switch(u16ErrorCode)
	 {
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UNSPECIFIED:
		vPlaybackModeOpMError(AP_MPLAYER_ERROR_INVALID_HANDLE); 
		break;  
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_INTERNALFAILURE:
		 vPlaybackModeOpMError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR);
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UPREGFAILURE:
		 vPlaybackModeOpMError(AP_MPLAYER_ERROR_INVALID_HANDLE);
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_OPCODENOTSUPPORTED:
		 vPlaybackModeOpMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED);
		 break;
	 default:
		vPlaybackModeOpMError(AP_MPLAYER_ERROR_UNKNOWN); 
		break;
	 }

	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
*
* FUNCTION:  	vHandlePlaybackModeOpMError
*
* DESCRIPTION:
*
* PARAMETER:  	amt_tclServiceData*.
*
* RETURNVALUE: 	tVoid.
******************************************************************************/
void FC_Gateway_MediaPlayerClientHandler::vHandleRepeatModeOpMError(
                                       amt_tclServiceData* poMessage)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	amt_tclServiceDataError oErrorMsg(poMessage);
	tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
	         "Error code ::"),DLT_INT( u16ErrorCode));

	switch(u16ErrorCode)
	 {
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UNSPECIFIED:
		vRepeatModeOpMError(AP_MPLAYER_ERROR_INVALID_HANDLE); 
		break;  
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_INTERNALFAILURE:
		 vRepeatModeOpMError(AP_MPLAYER_ERROR_INTERNAL_ENGINE_ERR);
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_UPREGFAILURE:
		 vRepeatModeOpMError(AP_MPLAYER_ERROR_INVALID_HANDLE);
		 break;
	 case MPLAY_DEVICEMANAGERFI_C_U16_ERROR_OPCODENOTSUPPORTED:
		 vRepeatModeOpMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED);
		 break;
	 default:
		vRepeatModeOpMError(AP_MPLAYER_ERROR_UNKNOWN); 
		break;
	 }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

#ifdef __cplusplus
extern "C"
{
#endif

/******************************************************************************
*
* FUNCTION:		bMediaInterface_play
*
* DESCRIPTION:  Play the file
*
* PARAMETER:  	tVoid.
*
* RETURNVALUE: 	tBool.
******************************************************************************/
bool bMediaInterface_play()
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	mplay_fi_tcl_e8_MPlayAction::tenType o_enType = 
	                 mplay_fi_tcl_e8_MPlayAction::FI_EN_E8PBA_PLAY;
	mplay_fi_tcl_e8_MPlayAction enMAction;
	enMAction.enType = o_enType;// = o_enType;

    FC_Gateway_MediaPlayerClientHandler *poMediaPlayerClientHandlerObj =
	  procgenericgateway_tclMainApp::
							poGetInstance()->poGetMediaPlayerClientHandler();
    if(poMediaPlayerClientHandlerObj)
    {
        bool bRet = 
		poMediaPlayerClientHandlerObj->sendRequestPlaybackActionMStart(
																enMAction,0);
        if(bRet)
        {
            DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
			     "sendRequestPlaybackActionMStart invoked successfully"));
            return bRet;
        }else{
            DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
			      "sendRequestPlaybackActionMStart invocation failed"));
            return bRet;
        }
    }
    else{
        DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
		               "poMediaPlayerClientHandlerObj is NULL"));
        return false;
    }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
 * Function:		bmediaInterface_pause
 *
 * Description:		Pause the media file.
 *
 * Parameters:		tVoid.
 *
 * Return:			tBool.
 *****************************************************************************/
bool bMediaInterface_pause()
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));

	mplay_fi_tcl_e8_MPlayAction::tenType o_enType = 
	                mplay_fi_tcl_e8_MPlayAction::FI_EN_E8PBA_PAUSE;
	mplay_fi_tcl_e8_MPlayAction enMAction;
	enMAction.enType = o_enType;// = o_enType;

    FC_Gateway_MediaPlayerClientHandler *poMediaPlayerClientHandlerObj =
   	   procgenericgateway_tclMainApp::
							poGetInstance()->poGetMediaPlayerClientHandler();
    bool bRet = 
	 poMediaPlayerClientHandlerObj->sendRequestPlaybackActionMStart(
																  enMAction,0);
    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	           "sendRequestPlaybackActionMStart invoked successfully"));
	   return bRet;
	}else{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	              "sendRequestPlaybackActionMStart invocation failed"));
	   return bRet;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
 * Function:		mediaInterface_stop
 *
 * Description:		stops the playback of the audio file.
 *
 * Parameters:		tVoid.
 *
 * Return:			tBool.
 *****************************************************************************/
bool bMediaInterface_stop()
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));

	mplay_fi_tcl_e8_MPlayAction::tenType o_enType =
           	   mplay_fi_tcl_e8_MPlayAction::FI_EN_E8PBA_STOP;
	mplay_fi_tcl_e8_MPlayAction enMAction;
	enMAction.enType = o_enType;// = o_enType;

    FC_Gateway_MediaPlayerClientHandler *poMediaPlayerClientHandlerObj= 
	procgenericgateway_tclMainApp::
	poGetInstance()->poGetMediaPlayerClientHandler();
    bool bRet = 
	poMediaPlayerClientHandlerObj->sendRequestPlaybackActionMStart(
																enMAction,0);
    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	             "sendRequestPlaybackActionMStart invoked successfully"));
	   return bRet;
	}else{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	            "sendRequestPlaybackActionMStart invocation failed"));
	   return bRet;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
 * Function:		mediaInterface_next
 *
 * Description:		stops the playback of the audio file.
 *
 * Parameters:		tVoid.
 *
 * Return:			tBool.
 *****************************************************************************/
bool bMediaInterface_next()
{
		DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));

	mplay_fi_tcl_e8_MPlayAction::tenType o_enType = 
	             mplay_fi_tcl_e8_MPlayAction::FI_EN_E8PBA_NEXT;
	mplay_fi_tcl_e8_MPlayAction enMAction;
	enMAction.enType = o_enType;// = o_enType;

    FC_Gateway_MediaPlayerClientHandler *poMediaPlayerClientHandlerObj = 
	   procgenericgateway_tclMainApp::
							poGetInstance()->poGetMediaPlayerClientHandler();
    bool bRet = 
	  poMediaPlayerClientHandlerObj->sendRequestPlaybackActionMStart(
																enMAction,0);
    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	              "sendRequestPlaybackActionMStart invoked successfully"));
	   return bRet;
	}else{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	                 "sendRequestPlaybackActionMStart invocation failed"));
	   return bRet;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/*****************************************************************************
 * Function:		bMediaInterface_prev
 *
 * Description:		stops the playback of the audio file.
 *
 * Parameters:		tVoid.
 *
 * Return:			tBool.
 *****************************************************************************/
bool bMediaInterface_prev()
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));

	mplay_fi_tcl_e8_MPlayAction::tenType o_enType = 
	                  mplay_fi_tcl_e8_MPlayAction::FI_EN_E8PBA_PREV;
	mplay_fi_tcl_e8_MPlayAction enMAction;
	enMAction.enType = o_enType;// = o_enType;

    FC_Gateway_MediaPlayerClientHandler *poMediaPlayerClientHandlerObj = 
	   procgenericgateway_tclMainApp::	
							poGetInstance()->poGetMediaPlayerClientHandler();
    bool bRet =
  	  poMediaPlayerClientHandlerObj->sendRequestPlaybackActionMStart(
																enMAction,0);
    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	                 "sendRequestPlaybackActionMStart invoked successfully"));
	   return bRet;
	}else{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	             "sendRequestPlaybackActionMStart invocation failed"));
	   return bRet;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}
/******************************************************************************
 * Function:		bmediaInterface_seekTo
 *
 * Description:		Play from the seek position.
 *
 * Parameters:		tInt.
 *
 * Return:			tBool.
 *****************************************************************************/
bool bMediaInterface_seekTo(int arg_iSeekPosition)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
	mplay_fi_tcl_e8_MPlayPlaypointFormat::tenType o_enType = 
           	mplay_fi_tcl_e8_MPlayPlaypointFormat::FI_EN_E8PPF_ABSOLUTE;
	mplay_fi_tcl_e8_MPlayPlaypointFormat enMPlayPlaypointFormat;
	enMPlayPlaypointFormat.enType = o_enType;

    FC_Gateway_MediaPlayerClientHandler *poMediaPlayerClientHandlerObj = 
	  procgenericgateway_tclMainApp::
							poGetInstance()->poGetMediaPlayerClientHandler();
    bool bRet = 
	   poMediaPlayerClientHandlerObj->sendSeekToMStart(arg_iSeekPosition,
	                                                   enMPlayPlaypointFormat);
    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	                      "mediaInterface_seekTo invoked successfully"));
	   return bRet;
	}else{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	                         "mediaInterface_seekTo invocation failed"));
	   return bRet;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
 * Function:		bMediaInterface_getCurrentPlayingTrack
 *
 * Description:		Get the current track info
 *
 * Parameters:		tVoid
 *
 * Return:			tBool
 ******************************************************************************/
bool bMediaInterface_getCurrentPlayingTrack()
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
								   DLT_STRING(__FUNCTION__));
    FC_Gateway_MediaPlayerClientHandler *poMediaPlayerClientHandlerObj =
	procgenericgateway_tclMainApp::
							poGetInstance()->poGetMediaPlayerClientHandler();
    bool bRet = poMediaPlayerClientHandlerObj->getCurrentPlayingTrack();
    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	                         "getCurrentPlayingTrack invoked successfully"));
	   return bRet;
	}else{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	                           "getCurrentPlayingTrack invocation failed"));
	   return bRet;
	}

	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

#ifdef __cplusplus
}
#endif
//EOF
#endif//VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
