 /******************************************************************************
 * FILE:          ManAppsListHandler.c
 * PROJECT:       Generic Gateway
 * DESCRIPTION:   Implementation for ManApps List Interface
 * AUTHOR:        IPD5KOR(RBEI/ECO2)
 * COPYRIGHT:     (c) 2017 Robert Bosch GmbH, Hildesheim
 *****************************************************************************/

 #include "ManAppsListHandler.h"
 #include "ap-managedapp-service-enums.h"
 #include "dlt/dlt.h"
 #include "AutomDBusServer.h"

 DLT_DECLARE_CONTEXT(AGW_Manapp);
 
 #define MAN_APP_BUS_NAME         "com.obigo.managedapps"
 #define MAN_APP_INTERFACE_NAME   "bosch.managedapps.AutomotiveProxy.ManagedAppService"
 #define MAN_APP_OBJECT_PATH      "/bosch/managedapps/AutomotiveProxy/ManagedAppService"
 #define GATEWAY_MAN_APP_OBJECT_PATH  "/com/bosch/AutomotiveProxy/ManagedAppService"
static GDBusObjectManagerServer *pManAppObjManager = NULL;
 static GDBusProxy*        pManAppClientProxy     = NULL;
 static ManagedAppService* pManAppServiceProxyObj = NULL;
 static guint              g_manAppWatcherID        = 0;
 GDBusObjectSkeleton *object = NULL;
 gboolean handle_send_app_intent(ManagedAppService *object,
                                 GDBusMethodInvocation *invocation,
                                 const gchar *arg_intent,
                                 const gchar *arg_argument);

 gboolean handle_is_app_intent_supported(ManagedAppService *object,
                                         GDBusMethodInvocation *invocation,
                                         const gchar *arg_intent);



 /******************************************************************************
 * Function:    bStartManAppService
 * Description: Starts the ManApps Service
 * Parameters:  void
 * Returns:     gboolean
 *****************************************************************************/
 gboolean bStartManAppService()
 {
      DLT_REGISTER_CONTEXT(AGW_Manapp, "GWMA",
	           "DLT Logging for ManagedApps Gateway");
      DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING("+"),
               DLT_STRING(__FUNCTION__));

      GDBusConnection* connection = NULL;
      if((connection = poGetGDBusConnection()) == NULL)
      {
          DLT_LOG(AGW_Manapp,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
              DLT_STRING("(): poGetGDBusConnection returned NULL"));
          return FALSE;
      } 
         
      g_manAppWatcherID = g_bus_watch_name_on_connection
                                                   ( connection,
                                               MAN_APP_BUS_NAME,
                            G_BUS_NAME_WATCHER_FLAGS_AUTO_START,
                                            vManAppNameAppeared,
                                            vManAppNameVanished,
                                                           NULL, 
                                                           NULL 
                                                   );
	
       if( !g_manAppWatcherID )
       {
           DLT_LOG(AGW_Manapp,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
              DLT_STRING("(): g_bus_watch_name failed"));
  	   return FALSE;
       } 

       DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING("-"),
               DLT_STRING(__FUNCTION__));
       return TRUE;
 }

 /*****************************************************************************
 * Function:    vStopManAppService
 * Description: Stops the ManApps Service
 * Parameters:  void
 * Return:      void
 *****************************************************************************/
 void vStopManAppService()
 {
     DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING("+"),
               DLT_STRING(__FUNCTION__));
	 gboolean bExported = FALSE;
     if(g_manAppWatcherID)
     {
        g_bus_unwatch_name(g_manAppWatcherID);
        g_manAppWatcherID = 0;
     } 
     if(pManAppClientProxy)
     {
        g_clear_object(&pManAppClientProxy);
     }  
     if(pManAppObjManager != NULL && pManAppServiceProxyObj != NULL && object != NULL)
	 {
	    bExported = g_dbus_interface_skeleton_has_connection(G_DBUS_INTERFACE_SKELETON(pManAppServiceProxyObj),poGetSessionBusConnection());
	    DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING("+"),
        DLT_STRING(__FUNCTION__),DLT_INT(bExported));
	    if(bExported == TRUE)
	   {
	      g_dbus_object_skeleton_remove_interface(object,
          G_DBUS_INTERFACE_SKELETON (pManAppServiceProxyObj));
	      g_clear_object(&pManAppServiceProxyObj);
	      g_object_unref(object);
	   }
	   g_dbus_object_manager_server_unexport(pManAppObjManager,
	   GATEWAY_MAN_APP_OBJECT_PATH);
       g_clear_object(&pManAppObjManager);
    }
    DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING("-"),
            DLT_STRING(__FUNCTION__));
    DLT_UNREGISTER_CONTEXT(AGW_Manapp);
}

 /****************************************************************************
 * Function:    poGetManAppServiceProxyObj
 * Description: Gives the ManagedApp proxy object
 * Parameters:  void
 * Returns:     ManagedAppService*
 *****************************************************************************/
 ManagedAppService* poGetManAppServiceProxyObj()
 {
     if(!pManAppServiceProxyObj)
     {
         pManAppServiceProxyObj = managed_app_service_skeleton_new ();
     }
     return pManAppServiceProxyObj;
 }
 /*****************************************************************************
 * Function:    vManAppNameAppeared
 * Description: Call back for ManApp service appeared
 * Parameters:  connection, bus name, 
                owner name, user_name
 * Returns:     void
 *****************************************************************************/
 void vManAppNameAppeared( GDBusConnection *connection, 
                           const gchar     *name,
                           const gchar     *name_owner, 
                           gpointer         user_data )
 {
      DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING("+"),
               DLT_STRING(__FUNCTION__));
      if(connection)
      {
           g_dbus_proxy_new ( connection,
                              G_DBUS_PROXY_FLAGS_NONE,
                              NULL,
                              MAN_APP_BUS_NAME,
                              MAN_APP_OBJECT_PATH,
                              MAN_APP_INTERFACE_NAME,
                              NULL,
                              vManAppProxyCB,
                              NULL );
      } 
      else
      {
          DLT_LOG(AGW_Manapp, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                                     DLT_STRING("Null connection"));
      } 
      DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
 }
      
 /****************************************************************************
 * Function:    vManAppNameVanished
 * Description: Name disappeared call back for ManApp Servcie
 * Parameters:  connection, service_name, user_data
 * Return:      void
 *****************************************************************************/
 void
 vManAppNameVanished( GDBusConnection *connection, 
                      const gchar *service_name,
                      gpointer user_data )
 {
	 gboolean bExported = FALSE;
    DLT_LOG(AGW_Manapp, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if(pManAppClientProxy)
    {
        g_clear_object(&pManAppClientProxy);
    }
    if(pManAppObjManager != NULL && pManAppServiceProxyObj != NULL && object !=NULL)
	{
	    bExported = g_dbus_interface_skeleton_has_connection(G_DBUS_INTERFACE_SKELETON(pManAppServiceProxyObj),poGetSessionBusConnection());
	    DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING("+"),
        DLT_STRING(__FUNCTION__),DLT_INT(bExported));
		if(bExported == TRUE)
		{
		   g_dbus_object_skeleton_remove_interface(object,
           G_DBUS_INTERFACE_SKELETON (pManAppServiceProxyObj));
           g_clear_object(&pManAppServiceProxyObj);
		   g_object_unref(object);
		}
     
		 g_dbus_object_manager_server_unexport(pManAppObjManager,
								GATEWAY_MAN_APP_OBJECT_PATH);
        g_clear_object(&pManAppObjManager);
    }
    DLT_LOG(AGW_Manapp, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
 }

  
 /******************************************************************
 * Function:     vManAppProxyCB
 * Description:  Call back for ManApp proxy
 * Parameters:   object, result, user_data
 * Return:       void
 ******************************************************************/
 void vManAppProxyCB( GObject      *object, 
                      GAsyncResult *result,
                      gpointer     user_data )
 {
     DLT_LOG(AGW_Manapp,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
     GError *error = NULL;
     pManAppClientProxy = g_dbus_proxy_new_finish (result, &error);

     if(pManAppClientProxy)
     {
           DLT_LOG(AGW_Manapp, DLT_LOG_INFO,DLT_STRING(__FUNCTION__), 
             DLT_STRING("():Mananged app proxy available"));
           vExportManagedAppGW();
     } 
     else 
     {
         if(error != NULL)
         {
             DLT_LOG(AGW_Manapp, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__), 
                DLT_STRING("():error = "), DLT_STRING(
                                error->message?error->message: ""));
             g_clear_error (&error);
         }        
     }
     DLT_LOG(AGW_Manapp,DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
 }  

 /******************************************************************************
 * Function:    vExportManagedAppGW
 * Description: Exports gateway Managed app interface
 * Parameters:  void
 * Return:      void
 *****************************************************************************/
 void vExportManagedAppGW()
 {
     DLT_LOG(AGW_Manapp,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     

     if( (!poGetManAppServiceProxyObj()) || (!poGetSessionBusConnection()) )
     {
         DLT_LOG(AGW_Manapp, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
             DLT_STRING("():NULL Proxy or connection object"));
         return;
     } 
   
    
    gchar *sObjPath = g_strdup_printf
							  (GATEWAY_MAN_APP_OBJECT_PATH);
	pManAppObjManager =
            g_dbus_object_manager_server_new("/com/bosch/AutomotiveProxy");
    g_dbus_object_manager_server_set_connection(pManAppObjManager,
												   poGetSessionBusConnection());						  
   if( pManAppObjManager == NULL)
	{
		DLT_LOG(AGW_Manapp, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
             DLT_STRING("():NULL Object manager server"));
			 if(sObjPath!=NULL)
				g_free(sObjPath);
         return;		
	}	
	else if(sObjPath == NULL)
	{		
	DLT_LOG(AGW_Manapp, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
             DLT_STRING("():NULL Object path"));
        return;
     }
	else
	{	
 	object = g_dbus_object_skeleton_new (GATEWAY_MAN_APP_OBJECT_PATH);
     g_dbus_object_skeleton_add_interface (
                object,
                (GDBusInterfaceSkeleton *) poGetManAppServiceProxyObj());  
	g_dbus_object_manager_server_export(pManAppObjManager,
												G_DBUS_OBJECT_SKELETON(object));
     vUpdateApplications();
     g_signal_connect (pManAppClientProxy,
                       "g-properties-changed",
                       G_CALLBACK(vHandleManAppPropertyChange),
                       NULL);
    
     g_signal_connect (pManAppServiceProxyObj,
                       "handle-is-app-intent-supported",
                       G_CALLBACK(handle_is_app_intent_supported),
                       NULL);

     g_signal_connect (pManAppServiceProxyObj,
                       "handle-send-app-intent",
                       G_CALLBACK(handle_send_app_intent),
                       NULL);
	g_free(sObjPath);				   
	}
     DLT_LOG(AGW_Manapp,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
 }
 /*****************************************************************************
 * Function:    vHandleManAppPropertyChange
 * Description: Call back for Manapps properties change
 * Parameters:  Proxy, changed properties,
                invalidated properties, user_data
 * Returns:     void
 ******************************************************************************/
 void  vHandleManAppPropertyChange( GDBusProxy          *proxy,
	                            GVariant            *changed_properties,
	                            const gchar* const  *invalidated_properties,
	                            gpointer            user_data )
 {
    DLT_LOG(AGW_Manapp, DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING(__FUNCTION__));
	if (g_variant_n_children (changed_properties) > 0)
	{
		GVariantIter *iter;
		const gchar *key;
		GVariant *value;

		g_variant_get (changed_properties,
			"a{sv}",
			&iter);
		while (g_variant_iter_loop (iter, "{&sv}", &key, &value))
		{
		     g_autofree gchar *value_str;
		     value_str = g_variant_print (value, TRUE);
		     DLT_LOG(AGW_Manapp,DLT_LOG_INFO, DLT_STRING("Key - Value: "),
                      DLT_STRING(key),DLT_STRING(" ->"),DLT_STRING(value_str));
                     if(strcmp(key, "Applications") == 0)
                     {
                        vUpdateApplications();
                     }   
		}
		g_variant_iter_free (iter);
	}

	if (g_strv_length ((GStrv) invalidated_properties) > 0)
	{
		guint ucount;
		DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING("Properties Invalidated:"));
		for (ucount = 0; invalidated_properties[ucount] != NULL; ucount++)
		{
			const gchar *key = invalidated_properties[ucount];
			DLT_LOG(AGW_Manapp ,DLT_LOG_INFO,DLT_STRING("  "),DLT_STRING( key));
		}
	}      
    DLT_LOG(AGW_Manapp, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
 }

 /*****************************************************************************
 * Function:    vUpdateApplications
 * Description: Updates the latest value of Applications
 * Parameters:  void
 * Returns:     void
 ******************************************************************************/
 void vUpdateApplications()
 {
     DLT_LOG(AGW_Manapp,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     GVariant* value = NULL;  
     if((!pManAppClientProxy) || (!poGetManAppServiceProxyObj()))
     {
         DLT_LOG(AGW_Manapp, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): NULL Proxy"));
         return;
     }
     value = g_dbus_proxy_get_cached_property( pManAppClientProxy, 
                                                      "Applications" ); 
     if((value == NULL))
     {
        DLT_LOG(AGW_Manapp, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): Recieved Null value, applications update failed"));
        return;
     }

     managed_app_service_set_applications(poGetManAppServiceProxyObj(), value);
     g_variant_unref(value);
     DLT_LOG(AGW_Manapp,DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
 }


 /*****************************************************************************
 * Function:    handle_is_app_intent_supported
 * Description: Dbus handler function to check if an intent is supported
 *              by managed apps
 * Parameters:  obj ptr, innvocation pointer, intent to be verified
 * Returns:     gboolean
 ******************************************************************************/
 gboolean handle_is_app_intent_supported(ManagedAppService *object,
                                         GDBusMethodInvocation *invocation,
                                         const gchar *arg_intent)
 {
     DLT_LOG(AGW_Manapp,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     GError *l_poError = NULL;
     gboolean b_result = FALSE;

     GVariant *l_poRet =   g_dbus_proxy_call_sync (G_DBUS_PROXY (pManAppClientProxy),
                                                   "isAppIntentSupported",
                                                    g_variant_new ("(s)",arg_intent),
                                                    G_DBUS_CALL_FLAGS_NONE,
                                                    -1,
                                                    NULL,
                                                    &l_poError);
     if(l_poError != NULL)
     {
         DLT_LOG(AGW_Manapp,DLT_LOG_ERROR,
                 DLT_STRING("Error in call isHandlingAppInstalled"
                 "error: "),DLT_STRING(l_poError->message));
         g_clear_error(&l_poError);
         g_dbus_method_invocation_return_error(invocation,
                                             G_DBUS_ERROR,
                                                     NULL,
                       "unable to get value from back-end ");

     }
     else
     {
         DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING(
                     "Success:isHandlingAppInstalled call"));

         g_variant_get (l_poRet, "(b)", &b_result);
         //DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING(
         //        "isHandlingAppInstalled returns: "),DLT_BOOL(b_result));
         managed_app_service_complete_is_app_intent_supported(object,
                                                              invocation,
                                                              b_result);
     }
     g_variant_unref(l_poRet);
     DLT_LOG(AGW_Manapp,DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
     return TRUE;
 }

 /*****************************************************************************
 * Function:    handle_send_app_intent
 * Description: Dbus handler function to request manapps framework
 *              to handle the intent
 * Parameters:  objptr,innvocationptr,intent to be handled,app associated arg
 * Returns:     gboolean
 ******************************************************************************/
 gboolean handle_send_app_intent(ManagedAppService *object,
                                 GDBusMethodInvocation *invocation,
                                 const gchar *arg_intent,
                                 const gchar *arg_argument)
 {
     DLT_LOG(AGW_Manapp,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

     GError *l_poError = NULL;
     ApSendIntentResult l_eRet = AP_MANAPP_INTENT_UNKNOWN;

     GVariant *l_poRet =   g_dbus_proxy_call_sync (G_DBUS_PROXY (pManAppClientProxy),
                                                   "sendAppIntent",
                                                   g_variant_new("(ss)",arg_intent,arg_argument),
                                                   G_DBUS_CALL_FLAGS_NONE,
                                                   -1,
                                                   NULL,
                                                   &l_poError);
     if(l_poError != NULL)
     {
         DLT_LOG(AGW_Manapp,DLT_LOG_ERROR,
                 DLT_STRING("Error in call sendAppIntent"
                 "error: "),DLT_STRING(l_poError->message));
         g_clear_error(&l_poError);
         g_dbus_method_invocation_return_error(invocation,
                                             G_DBUS_ERROR,
                                                     NULL,
                       "unable to get value from back-end ");

     }
     else
     {
         DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING(
                     "Success:sendAppIntent call"));

         g_variant_get (l_poRet, "(y)", &l_eRet);
         DLT_LOG(AGW_Manapp,DLT_LOG_INFO,DLT_STRING(
                 "sendAppIntent returns: "),DLT_UINT(l_eRet));
         managed_app_service_complete_send_app_intent(object,
                                                      invocation,
                                                      l_eRet);
     }
     g_variant_unref(l_poRet);
     DLT_LOG(AGW_Manapp,DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
     return TRUE;
 }
