/*!
********************************************************************************
* @file             : ECMClientHandler.c
********************************************************************************
*  - PROJECT:       : Methods to connect to ECM
*                       and call API's
*  - SW-COMPONENT   : Gateway
*  - DESCRIPTION    : D-Bus client to connect to ECM
*  - COPYRIGHT      : &copy; 2017 Robert Bosch Engineering & Business Solutions
*  - Documents      : -
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|------------------------
*3.23.2017 | IPD5KOR(RBEI/ECO2) | 1.0.0 | Dbus client impl. to connect to ECM
*******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "ECMClientHandler.h"
#include "ECM_ClientInterface.h"
#include "dlt/dlt.h"

DLT_IMPORT_CONTEXT(AGW_ECM);

static GDBusProxy*  pECMClientProxyObj = NULL;
static guint        iECMWatcherID      = 0;

/******************************************************************************
 * Function:      bCheckCertificate
 * Description:   Function to check the certificate
 * Parameters:    certBuffer, certSize, user_data
 * Return:        gboolean
 *****************************************************************************/
gboolean bCheckCertificate(GVariant* certBuffer,
                           guint32      certSize,
                           gpointer  user_data)
                       
{
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
  
   if(pECMClientProxyObj)
   {
      g_dbus_proxy_call ( pECMClientProxyObj, 
                          "checkCert",
                          g_variant_new ("(@ayu)",
                                   certBuffer, certSize),
                          G_DBUS_CALL_FLAGS_NONE, 
                          -1, 
                          NULL,
                          vCheckCertResult, 
                          user_data );       
   }
   else
   {
      DLT_LOG(AGW_ECM, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__), DLT_STRING(
	  "():No connection established with ECM"));
      return FALSE;
   }
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
   return TRUE; 
}

/******************************************************************************
 * Function:      bPushCertificate
 * Description:   Function to Push Ceritificate
 * Parameters:    certBuffer , certSize, user_data
 * Return:        gboolean
 *****************************************************************************/

gboolean bPushCertificate(GVariant* certBuffer,
                          guint32 certSize ,
                          gpointer user_data)
{
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 

   if(pECMClientProxyObj)
   {
      g_dbus_proxy_call (      pECMClientProxyObj,
                                       "pushCert",
                          g_variant_new ("(@ayu)",
                            certBuffer, certSize),
                           G_DBUS_CALL_FLAGS_NONE,
                                               -1,
                                             NULL,
                                  vPushCertResult,
                                       user_data );
   }
   else
   {
      DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), DLT_STRING(
	      "():No connection established with ECM"));
      return FALSE;
   }
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
   return TRUE;
}

/******************************************************************************
 * Function:      bPushCrl
 * Description:   Function to Push Crl
 * Parameters:    revocationList , certSize, user_data
 * Return:        gboolean
 *****************************************************************************/
gboolean bPushCrl ( GVariant*  revocationList,
                    guint32   certSize,
                    gpointer user_data )
{
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 

   if(pECMClientProxyObj)
   {
      g_dbus_proxy_call (      pECMClientProxyObj,
                                       "pushCrl",
                          g_variant_new ("(@ayu)",
                            revocationList, certSize),
                           G_DBUS_CALL_FLAGS_NONE,
                                               -1,
                                             NULL,
                                   vPushCrlResult,
                                       user_data );
   }
   else
   {
      DLT_LOG(AGW_ECM, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__), DLT_STRING(
	      "():No connection established with ECM"));
      return FALSE;
   }
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
   return TRUE;
}

/******************************************************************************
 * Function:      bGetCrlNextUpd
 * Description:   Function to get next crl update
 * Parameters:    identifier, user_data
 * Return:        gboolean
 *****************************************************************************/
gboolean bGetCrlNextUpd ( gchar* identifier,
                          gpointer user_data ) 
{
  DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 


   if(pECMClientProxyObj)
   {
      g_dbus_proxy_call (      pECMClientProxyObj,
                                  "getCrlNextUpd",
                             g_variant_new ("(s)",
                                      identifier),
                           G_DBUS_CALL_FLAGS_NONE,
                                               -1,
                                             NULL,
                             vGetCrlNextUpdResult,
                                       user_data );
   }
   else
   {
      DLT_LOG(AGW_ECM, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__), DLT_STRING(
	  "():No connection found with ECM"));
      return FALSE;
   }
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
   return TRUE;
}

/******************************************************************************
 * Function:      vCheckCertResult
 * Description:   This function handles the result of CheckCert
                  non-blocking DBus call
 * Parameters:    resObject, result sync pointer, user_data
 * Return:        void
 ******************************************************************************/
void vCheckCertResult( GObject* pObj,
                       GAsyncResult *res,
                       gpointer user_data )
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
    GError* error =   NULL;
    gchar* errorMsg = NULL;
    GVariant *_ret;
    guint certValidity = 0;

    _ret = g_dbus_proxy_call_finish (G_DBUS_PROXY(pObj), res, &error);
    if (_ret == NULL)
    {
		if (error != NULL)
		{
			DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
			  DLT_STRING("():Failed getting result, error = "),
 			  DLT_STRING(error->message));
			g_clear_error(&error);
		}
	   send_check_cert_result((guint16)certValidity, TRUE, errorMsg, user_data);
	   return;
    }
    else
    {
       g_variant_get (_ret, "(us)", &certValidity, &errorMsg);
       g_variant_unref (_ret);
    }
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
	      DLT_STRING("():certValidity = "), 
          DLT_UINT(certValidity), DLT_STRING("error = "), 
		  DLT_STRING(errorMsg ? errorMsg: ""));
		   
	DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
    send_check_cert_result((guint16)certValidity,  FALSE, errorMsg, user_data);
}


/******************************************************************************
 * Function:      vPushCertResult
 * Description:   This function handles the result of pushCert
                  non-blocking DBus call
 * Parameters:    resObject, result sync pointer, user_data
 * Return:        void
 ******************************************************************************/
void vPushCertResult( GObject* pObj,
                      GAsyncResult *res,
                      gpointer user_data )
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
    GError* error = NULL;
    gchar* errorDesc = NULL;
    GVariant *_ret = NULL;
    guint32 retCode = 0;

    _ret = g_dbus_proxy_call_finish (G_DBUS_PROXY(pObj), res, &error);
    if (_ret == NULL)
    {
		if (error != NULL)
		{
			DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
			DLT_STRING("():Failed getting result, error = "), 
			DLT_STRING(error->message));
			g_clear_error(&error);
		}
	   send_push_cert_result((guint16)retCode,  TRUE, errorDesc, user_data);
	   return;
    }
    else
    {
        g_variant_get (_ret,
                     "(us)",
                    &retCode,
                 &errorDesc);
        g_variant_unref (_ret);
    }
	DLT_LOG(AGW_ECM, DLT_LOG_INFO,DLT_STRING(__FUNCTION__), DLT_STRING("():"
	   "errorDescription = "), DLT_STRING(errorDesc ? errorDesc : ""));
	   
	DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
    send_push_cert_result((guint16)retCode,  FALSE, errorDesc, user_data);
}

/******************************************************************************
 * Function:      vPushCrlResult
 * Description:   This function handles the result of pushCrl
                  non-blocking DBus call
 * Parameters:    resObject, result sync pointer, user_data
 * Return:        void
 ******************************************************************************/
void vPushCrlResult( GObject* pObj,
                      GAsyncResult *res,
                      gpointer user_data )
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
    GError* error = NULL;
	gchar* errorDesc = NULL;
    GVariant *_ret = NULL;
    guint32 retCode = 0;

    _ret = g_dbus_proxy_call_finish (G_DBUS_PROXY(pObj), res, &error);
    if (_ret == NULL)
    {
		if (error != NULL)
		{
			DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
			   DLT_STRING("():Failed getting result, error = "), 
			                        DLT_STRING(error->message));
			g_clear_error(&error);
		}
	   send_push_crl_result((guint16)retCode,  TRUE, errorDesc, user_data);
	   return;
    }
    else
    {
        g_variant_get (_ret,
                     "(us)",
                    &retCode,
                 &errorDesc);
        g_variant_unref (_ret);
    }
	DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
    send_push_crl_result((guint16)retCode,  FALSE, errorDesc, user_data);
}

/******************************************************************************
 * Function:      vGetCrlNextUpdtResult
 * Description:   This function handles the result of getCrlNextUpd
                  non-blocking DBus call
 * Parameters:    resObject, result sync pointer, user_data
 * Return:        void
 *****************************************************************************/
void vGetCrlNextUpdResult( GObject* pObj,
                      GAsyncResult *res,
                      gpointer user_data )
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
    GError* error = NULL;
    gchar* errorDesc = NULL;
    guint64 nextUpdTime = 0;
    GVariant *_ret = NULL;
    guint32 retCode = 0;

    _ret = g_dbus_proxy_call_finish (G_DBUS_PROXY(pObj), res, &error);
    if (_ret == NULL)
    {
		if (error != NULL)
		{
			DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
			  DLT_STRING("():Failed getting result, error = "),
                       			   DLT_STRING(error->message));
			g_clear_error(&error);
		}
			 
	   send_get_crl_validity_result( retCode,  
	                                 nextUpdTime,
         							 TRUE, 
									 errorDesc, 
									 user_data );
	   return;
    }
    else
    {
        g_variant_get (_ret,
                   "(ust)",
                   &retCode,
                 &errorDesc,
                 &nextUpdTime);
       g_variant_unref (_ret);
    }
	DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
    send_get_crl_validity_result( retCode,
            	                  nextUpdTime,
								  FALSE, 
								  errorDesc, 
								  user_data );
}

/******************************************************************
 * Function:     vECMClientProxyCallBack
 * Description:  This is call back function for ECM proxy
                 object creation
 * Parameters:   source_object, res, user_data
 * Return:       void
 ******************************************************************/
void vECMClientProxyCallBack( GObject *source_object, 
                        GAsyncResult *res,
                        gpointer user_data )
{
  DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
  GError *error = NULL;
  pECMClientProxyObj = g_dbus_proxy_new_finish (res, &error);

  if(pECMClientProxyObj)
  {
      DLT_LOG(AGW_ECM, DLT_LOG_INFO,DLT_STRING(__FUNCTION__), DLT_STRING("():" 
	     "ECM Proxy object created"));
      vExportAGWECMInterface();
  } 
  else 
  {
      if(error != NULL)
      {
         DLT_LOG(AGW_ECM,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),DLT_STRING("():"
   		    "error = "), DLT_STRING(error->message?error->message: ""));
         g_clear_error (&error);
      }        
  }
  DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
}  

/****************************************************************************
 * Function:    ecm_service_appeared
 * Description: Name appeared call back for ECM
 * Parameters:  connection , service_name , owner_name, userdata
 * Return:      void
 *****************************************************************************/
void
ecm_service_appeared (GDBusConnection *connection, const gchar *name,
               const gchar *name_owner, gpointer user_data)
{
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
   if(connection)
   {
        g_dbus_proxy_new ( connection,
                           G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
                           NULL,
                           "com.escrypt.ecm",
                           "/com/escrypt/ecm",
                           "com.escrypt.ecm",
                           NULL,
                           vECMClientProxyCallBack,
                           NULL );
   } 
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
}

/****************************************************************************
 * Function:    ecm_service_vanished
 * Description: Name disappeared call back for ECM service
 * Parameters:  connection, service_name, user_data
 * Return:      void
 *****************************************************************************/
void
ecm_service_vanished (GDBusConnection *connection, 
                      const gchar *service_name,
                      gpointer user_data)
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    clearGatewayECMClientProxy();
    vUnexportAGWECMInterface();
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/*******************************************************************************
 * Function:    ecm_service_watch
 * Description: This methods intiates the watch for ECM service
 * Parameters:  void
 * Return:      gboolean
 ******************************************************************************/
gboolean ecm_service_watch()
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    iECMWatcherID = g_bus_watch_name ( G_BUS_TYPE_SESSION,
                                    "com.escrypt.ecm",
                                    G_BUS_NAME_WATCHER_FLAGS_NONE,
                                    ecm_service_appeared,
                                    ecm_service_vanished,
                                    NULL,
                                    NULL );

     DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
     return (iECMWatcherID ? TRUE : FALSE );
}

/*******************************************************************************
 * Function:    ecm_service_unwatch
 * Description: Function to unwatch ECM bus
 * Parameters:  void
 * Return:      void
 ******************************************************************************/
void ecm_service_unwatch()
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if(iECMWatcherID)
    {
       g_bus_unwatch_name(iECMWatcherID);
       iECMWatcherID = 0;
    }
    clearGatewayECMClientProxy();
    vUnexportAGWECMInterface();
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/******************************************************************************
 * Function:      clearGatewayECMClientProxy
 * Description:   This function clears the AGW ECM Proxy
 * Parameters:    void
 * Return:        void
 ******************************************************************************/
void clearGatewayECMClientProxy()
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
   
    if(pECMClientProxyObj)
    {
       g_clear_object(&pECMClientProxyObj);  
    }

    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
}

/******************************************************************************
 * Function:      getECMClientProxy
 * Description:   This function returns the proxy for ECM service
 * Parameters:    void
 * Return:        GDBusProxy pointer
 ******************************************************************************/
GDBusProxy* getECMClientProxy()
{
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
   return (pECMClientProxyObj? pECMClientProxyObj : NULL);
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}
