/***********************************************************************
*FILE:         ConfigDataDBusHandler.c
*SW-COMPONENT: Internet
*DESCRIPTION:  This file provides the method definitions for
*              the config data interface
*COPYRIGHT: Ã© 2017 Robert Bosch GmbH
*
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
***********************************************************************/

#include <glib.h>
#include <gio/gio.h>
#include "dlt/dlt.h"
#include "ConfigDataDBusHandler.h"
#include "ConfigDataClientInterface.h"
#include "AutomDBusServer.h"

DLT_IMPORT_CONTEXT(AGW_CONFIGMGR);

#define CONFIG_DATA_OBJECT_PATH "/de/bosch/cm/ConfigManager/ConfigDataInterface"

static guint             iConfigMgrDbusID = 0;
static ConfigData*       poConfigDataProxy = NULL;

/* Function  Definitions */
/******************************************************************************
 * Function:      poGetConfigDataProxy
 * Description:   Gets the configdata proxy object.
 * Parameters:    void
 * Return:        ConfigData*
 *****************************************************************************/
ConfigData*  poGetConfigDataProxy()
{
   if(poConfigDataProxy == NULL)
   {
      poConfigDataProxy = config_data_skeleton_new();
   }
   return poConfigDataProxy;
}

/***************************************************************************
 * Function:    handle_set_private_item
 * Description: Handler function to set the private configuration item
 * Parameters:  proxyObj, invocation, itemName,
                item, user_data
 * Returns:     gboolean
 **************************************************************************/
gboolean handle_set_private_item( ConfigData*            proxyObj,
                                  GDBusMethodInvocation* invocation,
                                  gchar*    itemName,
                                  GVariant* item,
                                  gpointer  user_data )
{
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                             DLT_STRING(__FUNCTION__));
    user_data = invocation;
    if( (!itemName) || (!item))
    {
        DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                           DLT_STRING("(): Null inputs"));
        config_data_complete_set_private_item ( proxyObj,
                                              invocation,
            (guint)  AP_CONFIG_ACTION_STATUS_INVALID_INPUT );
        return TRUE;
    }
    
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
      DLT_STRING("(): Set private request. Itemname: "), DLT_STRING(itemName));
      
    vSetPrivateItem(itemName, item, user_data);
    
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("-"),
                             DLT_STRING(__FUNCTION__));
    return TRUE; 
}

/***************************************************************************
 * Function:    vHandleSetPrivateItemResult
 * Description: Handles the result of SetPrivateConfigItem
 * Parameters:  status, user_data
 * Returns:     void
 **************************************************************************/
void vHandleSetPrivateItemResult( guint    status,
                                  gpointer user_data )
{
    DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                               DLT_STRING(__FUNCTION__));
     if(!user_data)
     {
         DLT_LOG( AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Null invocation") );
        return;  
     }
     if(poGetConfigDataProxy() == NULL)
     {
         DLT_LOG( AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Null Proxy") );
         return;
     }
     if( status > AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR )
     {
          DLT_LOG(AGW_CONFIGMGR,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
               DLT_STRING("():Status is not a valid ApConfigDataActionStatus"));
          status = AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR;
     }

     config_data_complete_set_private_item ( 
                      poGetConfigDataProxy(),
          (GDBusMethodInvocation*)user_data,
                                     status 
                                           );
 
    DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("-"),
                               DLT_STRING(__FUNCTION__));      
}

/***************************************************************************
 * Function:    handle_get_private_item
 * Description: Gets the private configuration item
 * Parameters:  proxyObj, invocation, itemName, user_data
 * Returns:     gboolean
 ***************************************************************************/
gboolean handle_get_private_item( ConfigData*            proxyObj,
                                  GDBusMethodInvocation* invocation,
                                  gchar*    itemName,
                                  gpointer  user_data )
{
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                             DLT_STRING(__FUNCTION__));
    user_data = invocation;
    if(!itemName)
    {
        DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Null input"));
        GVariantBuilder builder;
        g_variant_builder_init (&builder, G_VARIANT_TYPE ("a(ss)"));
        g_variant_builder_add (&builder, "ss", "", "");
        GVariant* item = NULL;

        item = g_variant_builder_end(&builder);

        config_data_complete_get_private_item ( proxyObj,
                                                invocation,
                                                item,
           (guint)  AP_CONFIG_ACTION_STATUS_INVALID_INPUT );
        return TRUE;
    }

    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
      DLT_STRING("(): Get private request. Itemname: "), DLT_STRING(itemName));

    vGetPrivateItem(itemName, user_data);

    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("-"),
                             DLT_STRING(__FUNCTION__));
    return TRUE;
}

/***************************************************************************
 * Function:    vHandleGetPrivateItemResult
 * Description: Handles the result of GetPrivateConfigItem
 * Parameters:  item, status, user_data
 * Returns:     void
 **************************************************************************/
void vHandleGetPrivateItemResult( GVariant* item,
                                  guint     status,
                                  gpointer  user_data )
{
    DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                               DLT_STRING(__FUNCTION__));
     if((!user_data) || (poGetConfigDataProxy() == NULL))
     {
         DLT_LOG( AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Null proxy or invocation") );
        return;
     }
     if(!item)
     {
         DLT_LOG( AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                 DLT_STRING("(): Null item data") );
        GVariantBuilder builder;
        g_variant_builder_init (&builder, G_VARIANT_TYPE ("a(ss)"));

        g_variant_builder_add(&builder, "(ss)", "", "");
        item = g_variant_builder_end(&builder);
    }
    if( status > AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR )
    {
        DLT_LOG(AGW_CONFIGMGR,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("():Status is not a valid ApConfigDataActionStatus"));
        status = AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR;
    }

     config_data_complete_get_private_item (
                     poGetConfigDataProxy(),
          (GDBusMethodInvocation*)user_data,
                                       item,
                                     status
                                           );

    DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("-"),
                               DLT_STRING(__FUNCTION__));
}

/**************************************************************************
 * Function:    handle_exchange_descmo
 * Description: Handles private config set / get in XML format
 * Parameters:  proxyObj, invocation, itemName,
                itemXML, signatureXML, user_data
 * Returns:     gboolean
 **************************************************************************/
gboolean handle_exchange_descmo(  ConfigData*             proxyObj,
                                  GDBusMethodInvocation* invocation,
                                  gchar*    itemName,
                                  gchar*    itemXML,
                                  gchar*    signatureXML,
                                  gpointer  user_data )
{
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                             DLT_STRING(__FUNCTION__));

    user_data = invocation;
    if( (!itemName) || (!itemXML) || (!signatureXML) )
    {
        DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                           DLT_STRING("(): Null input"));
        config_data_complete_exchange_descmo( proxyObj,
                                              invocation,
                                              "",
                       AP_CONFIG_ACTION_STATUS_INVALID_INPUT );
        return TRUE;
    }

    DLT_LOG( AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
         DLT_STRING("():Item name= "), DLT_STRING(itemName), 
              DLT_STRING(" item XML= "), DLT_STRING(itemXML));

    vExchangeDescmo(itemName, itemXML, signatureXML, user_data);

    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("-"),
                             DLT_STRING(__FUNCTION__));
    return TRUE;
}

/***************************************************************************
 * Function:    vHandleExchangeDescmoResult
 * Description: Handles the result of ExchangeDescmo
 * Parameters:  outXML, status, user_data
 * Returns:     void
 **************************************************************************/
void vHandleExchangeDescmoResult( gchar*    outXML,
                                  guint     status,
                                  gpointer  user_data )
{
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                               DLT_STRING(__FUNCTION__));
     if((!user_data) || (poGetConfigDataProxy() == NULL) )
     {
         DLT_LOG( AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Null params") );
        return;
     }

     if( status > AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR )
     {
          DLT_LOG(AGW_CONFIGMGR,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
             DLT_STRING("():Status is not a valid ApConfigDataActionStatus"));
          status = AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR;
     }

     if(!outXML)
     {
         DLT_LOG( AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Null Proxy") );

         config_data_complete_exchange_descmo  (
                         poGetConfigDataProxy(),
             (GDBusMethodInvocation*)user_data,
                                             "",
                                         status
                                               );
         return;
     }

     config_data_complete_exchange_descmo  (
                     poGetConfigDataProxy(),
          (GDBusMethodInvocation*)user_data,
                                     outXML,
                                     status
                                           );

     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("-"),
                               DLT_STRING(__FUNCTION__));
}

/*****************************************************************
 * Function:    handle_update_public_config_item
 * Description: Handles update of public config items
 * Parameters:  proxyObj, invocation, itemName,
                itemData, updateType, user_data
 * Returns:     gboolean
 *****************************************************************/
gboolean handle_update_public_config_item( ConfigData* proxyObj,
                                  GDBusMethodInvocation* invocation,
                                  gchar*    itemName,
                                  GVariant* itemData,
                                  guint32   updateType,
                                  gpointer  user_data )
{
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                             DLT_STRING(__FUNCTION__));
    if(!invocation)
    {
        DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                           DLT_STRING("(): Null invocation"));
        return FALSE;
    }
    user_data = invocation;
    if((!itemName) || (!itemData))
    {
        DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                           DLT_STRING("(): Null input"));
        config_data_complete_update_public_config_item (
                                               proxyObj,
                                            invocation );

    }
    else
      vUpdatePublicItem(itemName, itemData, updateType, user_data);
     return TRUE;
}

/***************************************************************************
 * Function:    vHandleUpdatePublicItemResult
 * Description: Handles the result of UpdatePublicConfigItem
 * Parameters:  user_data
 * Returns:     void
 **************************************************************************/
void vHandleUpdatePublicItemResult(gpointer  user_data )
{
    DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                               DLT_STRING(__FUNCTION__));

     if( (!user_data) || (poGetConfigDataProxy() == NULL) )
     {
         DLT_LOG( AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Null invocation") );
        return;
     }

     config_data_complete_update_public_config_item(
                             poGetConfigDataProxy(),
                 (GDBusMethodInvocation*)user_data
                                                   );

     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("-"),
                               DLT_STRING(__FUNCTION__));
}

/*************************************************************************
 * Function:    initializeConfigData
 * Descriprion: Initializes the gateway config manager interface
 * Parameters:  void
 * Returns:     gboolean
 ************************************************************************/
gboolean initializeConfigData()
{
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                DLT_STRING(__FUNCTION__));
    GError *error = NULL;
    if( (poGetConfigDataProxy() == NULL)
           || (poGetSessionBusConnection() == NULL) )
    {
         DLT_LOG(AGW_CONFIGMGR,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
               DLT_STRING("(): null proxy or connection obj"));
         return FALSE;
    }

    if (!g_dbus_interface_skeleton_export  (
                                    G_DBUS_INTERFACE_SKELETON (
                                    (ConfigData*) poGetConfigDataProxy()),
                                    poGetSessionBusConnection(),
                                    CONFIG_DATA_OBJECT_PATH,
                                    &error) )
    {
        if(error != NULL)
        {
            DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("Skeleton export error:"),DLT_STRING(error->message));
            g_clear_error (&error);       
        }
        return FALSE;
    }

    else
    {
         g_signal_connect ( poGetConfigDataProxy(), "handle-set-private-item",
                     G_CALLBACK (handle_set_private_item), NULL );

         g_signal_connect ( poGetConfigDataProxy(), "handle-get-private-item",
                     G_CALLBACK (handle_get_private_item), NULL );

         g_signal_connect ( poGetConfigDataProxy(), "handle-exchange-descmo",
                     G_CALLBACK (handle_exchange_descmo), NULL );

         g_signal_connect( poGetConfigDataProxy(), 
		                   "handle-update-public-config-item",
                           G_CALLBACK (handle_update_public_config_item),
 						   NULL );
    }

    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"), 
	                           DLT_STRING(__FUNCTION__));
    return TRUE;
}

/*************************************************************************
 * Function:    uninitializeConfigData
 * Descriprion: uninitializes the gateway config manager interface
 * Parameters:  void
 * Returns:     void
 ************************************************************************/
void uninitializeConfigData()
{
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                            DLT_STRING(__FUNCTION__));
    if(poConfigDataProxy)
    {
        g_dbus_interface_skeleton_unexport(
            G_DBUS_INTERFACE_SKELETON(poConfigDataProxy));
        g_clear_object(&poConfigDataProxy);
    }

    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                            DLT_STRING(__FUNCTION__));
    DLT_UNREGISTER_CONTEXT(AGW_CONFIGMGR);
}

