/***********************************************************************
*FILE: ConfigMgmtClientHandler.c
*SW-COMPONENT: Internet
*DESCRIPTION: Client Handler for the 
*             config management interface
*COPYRIGHT: Ã© 2017 Robert Bosch GmbH
*
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
***********************************************************************/

#include "ConfigMgmtClientHandler.h"
#include "ConfigMgmtClientInterface.h"
#include "ap-cm-enums.h"
#include <stdlib.h>
#include <string.h>
#include "dlt/dlt.h"
#include "ConfigClientBusMap.h"

DLT_IMPORT_CONTEXT(AGW_CONFIGMGR);

#define  AGW_DBUS_NAME               "com.bosch.AutomotiveProxy"
#define  CONFIG_MGMT_BUS_NAME        "asf.ProcConfigManager"
#define  CONFIG_MGMT_OBJECT_PATH     "/de/bosch/cm/ConfigManager/ConfigManagement"
#define  CONFIG_MGMT_INTERFACE_NAME  "de.bosch.cm.ConfigManager.ConfigManagement"

static GDBusProxy* poConfigMgmtClientProxy = NULL;
static guint       configMgmtWatcherID     = 0;

 /***************************************************************
 * Function:    vRegisterConfigItem
 * Description: Registers with Config Manager for
                the requested config item
 * Parameters:  item name, dbus name, format,
                item type, user_data
 * Returns:     void
 ***************************************************************/
 void vRegisterConfigItem( gchar*         itemName,
                           gchar*         dbusName,
                           guint          format,
                           ConfigItemType type,
                           gpointer       user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));

      guint status = (guint) AP_CONFIG_REG_STATUS_INTERNAL_ERROR;
      GError* error = NULL;
      GVariant* ret = NULL;

      if( (!itemName) || (!dbusName))
      {
         DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                                  DLT_STRING("():Null params recieved"));
         vHandleRegisterConfigResult( status, type, user_data );
         return;
      }

      if(poConfigMgmtClientProxy)
      {
          ret = g_dbus_proxy_call_sync(
                   poConfigMgmtClientProxy,
                   ((type == CONFIG_PUBLIC_ITEM) ?
                      "RegisterPublicItem" : "RegisterPrivateItem"),
                   g_variant_new ("(ssu)",
                   itemName, AGW_DBUS_NAME, 
                   format),
                   G_DBUS_CALL_FLAGS_NONE,
                   -1,
                   NULL,
                   &error );  

         if(ret == NULL)
         {
             DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                                  DLT_STRING("():Registration Failed"));
             if(error)
             {
                 DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING("Error = "), 
                    DLT_STRING(error->message ? error->message : ""));
                 g_clear_error(&error);
             } 
         }
         else
         { 
             g_variant_get (ret, "(u)", &status);             

             DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
			    DLT_STRING("():Status = "), DLT_UINT(status));
             if(status == AP_CONFIG_REG_STATUS_SUCCESS)
             {
                 AddConfigClientBusPair( itemName,
                                         dbusName,
                                           type );
             }
         }  
     }
     else
     {
        DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
                DLT_STRING("(): Connection error with Config Manager"));
     }
     vHandleRegisterConfigResult( status, type, user_data );
     g_variant_unref (ret);
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                  DLT_STRING(__FUNCTION__));
 }

 /***************************************************************
 * Function:    vUnregisterConfigItem
 * Description: De-Registers the requested config item
 * Parameters:  item name, dbus name, 
                ConfigItemType, user_data
 * Returns:     void
 ***************************************************************/
 void vUnregisterConfigItem( gchar*         itemName,
                             gchar*         dbusName,
                             ConfigItemType type,
                             gpointer       user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));

	  
      GError*    error =  NULL;
      GVariant*  ret = NULL;
      guint status = (guint) AP_CONFIG_REG_STATUS_INTERNAL_ERROR;

      if((!itemName) || (!dbusName))
      {
         DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                                    DLT_STRING("Null params recieved"));
         vHandleUnregisterConfigResult( status, type, user_data );
         return;
      }
	  
	  gchar* busname_check = NULL;
      busname_check = FindClientBusName(itemName,type);

	  if(g_strcmp0(busname_check,dbusName) !=0 || busname_check == NULL)
	  {
		  DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                                    DLT_STRING("items not registered"));
         vHandleUnregisterConfigResult( (guint)AP_CONFIG_REG_STATUS_UNKNOWN_ITEM,type, user_data );
		 busname_check = NULL;
		 return;  	  
		 
	  }
	   busname_check = NULL;
	  if(poConfigMgmtClientProxy)
      {
           ret =  g_dbus_proxy_call_sync(
                      poConfigMgmtClientProxy,
                      ((type == CONFIG_PUBLIC_ITEM) ?
                        "UnregisterPublicItem" : "UnregisterPrivateItem"),
                      g_variant_new ("(ss)",
                      itemName, AGW_DBUS_NAME),
                      G_DBUS_CALL_FLAGS_NONE,
                      -1,
                      NULL,
                      &error );
           if(ret == NULL)
           {
               DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                                    DLT_STRING("():Unregistration failed") );
               if(error)
               {
                   DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING("Error = "),
                       DLT_STRING(error->message ? error->message : "")); 
                   g_clear_error(&error);
               }
           }   
           else
           { 
               g_variant_get (ret, "(u)", &status);
               g_variant_unref (ret);
               DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), 
			       DLT_STRING("():Status = "), DLT_UINT(status));
               if(status == AP_CONFIG_REG_STATUS_SUCCESS)
               {
                  RemoveConfigClientBusPair( itemName,
                                             dbusName,
                                             type );

               }
           }
     }
     else
     {
        DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Connection error with Config Manager"));
     }
	 
     vHandleUnregisterConfigResult( status, type, user_data );
	
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                 DLT_STRING(__FUNCTION__));
 }

 /***************************************************************
 * Function:    vSetPublicItem
 * Description: Sets public configuration item data
 * Parameters:  itemName, itemData, user_data
 * Returns:     void
 ***************************************************************/
 void vSetPublicItem( gchar*    itemName,
                      GVariant* item,
                      gpointer  user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));

      if(!item)
      {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                                    DLT_STRING("Null params recieved"));
          vHandleSetPublicItemResult( AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR, 
                                      user_data );
         return;
      }

      if(poConfigMgmtClientProxy)
      {
          g_dbus_proxy_call(
                  poConfigMgmtClientProxy,
                  "setPublicItem",
                  g_variant_new ("(s@a(ss))",
                     itemName, item ),
                  G_DBUS_CALL_FLAGS_NONE,
                  -1,
                  NULL,
                  vSetPublicItemCallBack,
                  user_data );
     }
     else
     {
        DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Connection error with Config Manager"));
        vHandleSetPublicItemResult( AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR,
                                 user_data );
     }
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                 DLT_STRING(__FUNCTION__));
 }

 /******************************************************************
 * Function:    vSetPublicItemCallBack
 * Description: Call back for SetPublicConfigItem result
 * Parameters:  GObject*, GAsyncResult, user_data
 * Returns:     void
 *******************************************************************/
 void vSetPublicItemCallBack( GObject*      pObject,
                              GAsyncResult* res,
                              gpointer      user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));

      GError*   error =   NULL;
      GVariant* ret = NULL;
      guint status = (guint) AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR;

      ret = g_dbus_proxy_call_finish (G_DBUS_PROXY(pObject), res, &error);
      if (ret == NULL)
      {
           if (error != NULL)
           {
              DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
  			     DLT_STRING("():Null return value, error= "), 
				 DLT_STRING(error->message ? error->message : ""));
               g_clear_error(&error);
           }
           vHandleSetPublicItemResult( status, user_data );
           return;
      }
      g_variant_get (ret, "(u)", &status);
      g_variant_unref (ret);
      DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING(
          "():Status = "), DLT_UINT(status));
      vHandleSetPublicItemResult( status, user_data );

      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                  DLT_STRING(__FUNCTION__));
}

 /******************************************************************************
 * Function:    vGetPublicItem
 * Description: Provides the public config item value
 * Parameters:  itemName, user_data
 * Returns:     void
 ******************************************************************************/
 void vGetPublicItem( gchar*   itemName,
                      gpointer user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));

      if(!itemName)
      {
         DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): NULL input param"));
         vHandleGetPublicItemResult( NULL,
                                     AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR,
                                     user_data );
      }
      if(poConfigMgmtClientProxy)
      {
          g_dbus_proxy_call(
                  poConfigMgmtClientProxy,
                  "getPublicItem",
                  g_variant_new ("(s)",
                             itemName),
                  G_DBUS_CALL_FLAGS_NONE,
                  -1,
                  NULL,
                  vGetPublicItemCallBack,
                  user_data );
     }
     else
     {
        DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Connection error with Config Manager"));
        vHandleGetPublicItemResult( NULL,
                                    AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR,
                                    user_data );
     }
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                 DLT_STRING(__FUNCTION__));
 } 

 /******************************************************************
 * Function:    vGetPublicItemCallBack
 * Description: Call back for GetPublicConfigItem result
 * Parameters:  GObject*, GAsyncResult, user_data
 * Returns:     void
 *******************************************************************/
 void vGetPublicItemCallBack( GObject*      pObject,
                              GAsyncResult* res,
                              gpointer      user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));

      GError* error = NULL;
      GVariant* ret = NULL;
      GVariant* item = NULL;
      guint status = (guint) AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR;

      ret = g_dbus_proxy_call_finish (G_DBUS_PROXY(pObject), res, &error);
      if (ret == NULL)
      {
           if (error != NULL)
           {
               DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
			      DLT_STRING("():Null ret value, error = "),
				  DLT_STRING(error->message ? error->message : ""));
               g_clear_error(&error);
           }
           vHandleGetPublicItemResult( item, status, user_data );
           return;
      }

      g_variant_get (ret, "(@a(ss)u)", &item, &status);
      g_variant_unref (ret);
      DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING(
          "():Status = "), DLT_UINT(status));
      vHandleGetPublicItemResult( item, status, user_data );

      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                  DLT_STRING(__FUNCTION__));
}

 /******************************************************************
 * Function:     vConfigMgmtProxyCallBack
 * Description:  Proxy Call back for Config Management
 * Parameters:   source_object, res, user_data
 * Return:       void
 ******************************************************************/
 void vConfigMgmtProxyCallBack( GObject    *source_object,
                              GAsyncResult *res,
                              gpointer     user_data )
 {
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));
    GError *error = NULL;

    poConfigMgmtClientProxy = g_dbus_proxy_new_finish (res, &error);
    if(!poConfigMgmtClientProxy)
    {
       DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
             DLT_STRING("(): Failed getting Config manager proxy"));
       if( error != NULL )
       {
           DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
             DLT_STRING("():Error = "),DLT_STRING(
			 error->message?error->message : "") );
           g_clear_error (&error);
       }
    }
    else
    {
       vExportConfigMgmtInterface();
    }
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                  DLT_STRING(__FUNCTION__));
 }

 /****************************************************************************
 * Function:    config_mgmt_service_appeared
 * Description: Service appeared call back of config management
 * Parameters:  connection , service name , owner name, userdata
 * Returns:     void
 *****************************************************************************/
 void
 config_mgmt_service_appeared ( GDBusConnection *connection,
                               const           gchar *name,
                               const     gchar *owner_name,
                               gpointer  user_data )
 {
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));

    if( connection )
    {
        g_dbus_proxy_new ( connection,
                           G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
                           NULL,
                           CONFIG_MGMT_BUS_NAME,
                           CONFIG_MGMT_OBJECT_PATH,
                           CONFIG_MGMT_INTERFACE_NAME,
                           NULL,
                           vConfigMgmtProxyCallBack,
                           NULL );
    }
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                  DLT_STRING(__FUNCTION__));
 }

/****************************************************************************
 * Function:    config_mgmt_service_vanished
 * Description: Name disappeared call back for Config Management
 * Parameters:  connection, service_name, user_data
 * Return:      void
 *****************************************************************************/
void
config_mgmt_service_vanished (GDBusConnection *connection,
                      const gchar *service_name,
                      gpointer user_data)
{
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));
    vCleanUpConfigMgmtService();
    vUnExportConfigMgmtInterface();
}

 /***************************************************************
 * Function:    watch_config_mgmt
 * Description: Connects to ConfigurationManager Dbus service
 * Parameters:  void
 * Returns:     gboolean
 ***************************************************************/
 gboolean watch_config_mgmt()
 {
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                        DLT_STRING(__FUNCTION__));

    configMgmtWatcherID = g_bus_watch_name ( G_BUS_TYPE_SESSION,
                                    CONFIG_MGMT_BUS_NAME,
                                    G_BUS_NAME_WATCHER_FLAGS_NONE,
                                    config_mgmt_service_appeared,
                                    config_mgmt_service_vanished,
                                    NULL,
                                    NULL );

    if(!configMgmtWatcherID)
    {
         DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                          DLT_STRING("():g_bus_watch_name failed"));
         return FALSE;
    }

    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                           DLT_STRING(__FUNCTION__));
    return TRUE;
 }

 /***************************************************************
 * Function:    unwatch_config_mgmt
 * Description: Disconnects  ConfigurationManager Dbus service
 * Parameters:  void
 * Returns:     void
 ***************************************************************/
 void unwatch_config_mgmt()
 {
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                        DLT_STRING(__FUNCTION__));

    if(configMgmtWatcherID)
    {
       g_bus_unwatch_name(configMgmtWatcherID);
       configMgmtWatcherID = 0;
    }
    vCleanUpConfigMgmtService();
    vUnExportConfigMgmtInterface();

    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                           DLT_STRING(__FUNCTION__));
 }


 /***************************************************************
 * Function:    vCleanUpConfigMgmtService
 * Description: Performs the clean up for Configuration 
                Management service
 * Parameters:  void
 * Returns:     void
 ***************************************************************/
 void vCleanUpConfigMgmtService()
 {
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                        DLT_STRING(__FUNCTION__));

    if(poConfigMgmtClientProxy)
    {
       g_clear_object(&poConfigMgmtClientProxy);
       poConfigMgmtClientProxy = NULL;
    }
    DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("-"),
                        DLT_STRING(__FUNCTION__));
 }

