/***********************************************************************
*FILE: ConfigDataClientHandler.c
*SW-COMPONENT: Internet
*DESCRIPTION: Client Handler for the
*             config data interface
*COPYRIGHT: Ã© 2017 Robert Bosch GmbH
*
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
***********************************************************************/

 #include "ConfigDataClientHandler.h"
 #include "ConfigDataClientInterface.h"
 #include "ConfigClientBusMap.h"
 #include "dlt/dlt.h"

 DLT_IMPORT_CONTEXT(AGW_CONFIGMGR);

 /****************************************************************
  * Function:    vSetPrivateItem
  * Description: Sets the requested private config item data
  * Parameters:  itemName, itemData, user_data
  * Return:      void
  ***************************************************************/
 void vSetPrivateItem( gchar*    itemName,
                       GVariant* itemData,
                       gpointer  user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                    DLT_STRING(__FUNCTION__));
       
      char* busName = NULL;

      if((!itemData) || (!itemName))
      {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                               DLT_STRING("(): Null parameter recieved"));
          vHandleSetPrivateItemResult( AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR,
                                       user_data );
      } 
      if((busName = FindClientBusName(itemName, CONFIG_PRIVATE_ITEM)) == NULL)
      {
           DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): FindClientBusName Failed"));
           vHandleSetPrivateItemResult( AP_CONFIG_ACTION_STATUS_UNKNOWN_ITEM,
                                       user_data );
          return;
      }

      if(poGetGDBusConnection() != NULL)
      {
          g_dbus_connection_call ( poGetGDBusConnection(),
                                   busName,
                                   "/com/bosch/CM/ConfigData",
                                   "com.bosch.CM.ConfigData",
                                   "SetPrivateItem",
                                   g_variant_new ("(s@a(ss))",
                                     itemName, itemData),
                                   NULL,
                                   G_DBUS_CALL_FLAGS_NONE,
                                   -1,
                                   NULL,
                                   (GAsyncReadyCallback)vSetPrivateItemCallBack,
                                   user_data );
      }
      else
      {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                         DLT_STRING("(): Gateway Bus connection is NULL"));
          vHandleSetPrivateItemResult( AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR,
                                       user_data );
      }
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                    DLT_STRING(__FUNCTION__));
 }     

 /********************************************************************
  * Function:    vSetPrivateItemCallBack
  * Description: Sets the requested private config item data
  * Parameters:  GDBusConnection*, GAsyncResult*, gpointer
  * Return:      void
  ********************************************************************/
 void vSetPrivateItemCallBack( GDBusConnection* connection,
                               GAsyncResult*    res,
                               gpointer         user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));

      GError*    error =   NULL;
      GVariant* ret =  NULL;
      guint status = (guint) AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR;

      ret = g_dbus_connection_call_finish (connection, res, &error);
      if (ret == NULL)
      {
           DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
		         DLT_STRING("SetPrivateItem failed" ));
           if (error != NULL)
           {
               DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
  			     DLT_STRING("Error = "), DLT_STRING(
				 error->message ? error->message : ""));
               g_clear_error(&error);
           }
           vHandleSetPrivateItemResult( status, user_data );
           return;
      }
      g_variant_get (ret, "(u)", &status);
      g_variant_unref (ret);
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING(__FUNCTION__), DLT_STRING(
          "():Status = "), DLT_UINT(status));

      vHandleSetPrivateItemResult( status, user_data );

      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                    DLT_STRING(__FUNCTION__));
}
      
 /****************************************************************
  * Function:    vGetPrivateItem
  * Description: Gets the requested private config item data
  * Parameters:  itemName, user_data
  * Return:      void
  ***************************************************************/
 void vGetPrivateItem( gchar*    itemName,
                       gpointer  user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                    DLT_STRING(__FUNCTION__));
      char* busName = NULL;

      if((!itemName))
      {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                               DLT_STRING("(): Null parameter recieved"));
          vHandleGetPrivateItemResult( NULL, 
                                       AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR,
                                       user_data );
          return;
      }

      if((busName = FindClientBusName(itemName, CONFIG_PRIVATE_ITEM)) == NULL)
      {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): FindClientBusName Failed"));
          vHandleGetPrivateItemResult( NULL,
                                       AP_CONFIG_ACTION_STATUS_UNKNOWN_ITEM,
                                       user_data );
          return;
      }
      if(poGetGDBusConnection() != NULL)
      {
          g_dbus_connection_call ( poGetGDBusConnection(),
                                   busName,
                                   "/com/bosch/CM/ConfigData",
                                   "com.bosch.CM.ConfigData",
                                   "GetPrivateItem",
                                   g_variant_new ("(s)",
                                            itemName),
                                   NULL,
                                   G_DBUS_CALL_FLAGS_NONE,
                                   -1,
                                   NULL,
                                   (GAsyncReadyCallback)vGetPrivateItemCallBack,
                                   user_data );
      }
      else
      {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                         DLT_STRING("(): Gateway Bus connection is NULL"));
          vHandleGetPrivateItemResult( NULL,
                                       AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR,
                                       user_data );
      }
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                    DLT_STRING(__FUNCTION__));
 }
      
 /********************************************************************
  * Function:    vGetPrivateItemCallBack
  * Description: Callback for GetPrivateItem result
  * Parameters:  GDBusConnection*, GAsyncResult*, gpointer
  * Return:      void
  ********************************************************************/
 void vGetPrivateItemCallBack( GDBusConnection*  connection,
                               GAsyncResult*     res,
                               gpointer          user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));

      GError*    error =  NULL;
      GVariant*  ret   =  NULL;
      GVariant*  item  =  NULL;
      guint status = (guint) AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR;

      ret = g_dbus_connection_call_finish(connection, res, &error);
      if (ret == NULL)
      {
           DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
 		           DLT_STRING("(): GetPrivateItem failed"));
           if (error != NULL)
           {
               DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
 			     DLT_STRING("Error= "), DLT_STRING(
				 error->message ? error->message : ""));
				 
               g_clear_error(&error);
           }
           vHandleGetPrivateItemResult(item, status, user_data );
           return;
      }
      g_variant_get (ret, "(@a(ss)u)", &item, &status);
      g_variant_unref (ret);	

      DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING(
          "():Status = "), DLT_UINT(status));
      vHandleGetPrivateItemResult( item, status, user_data );

      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                    DLT_STRING(__FUNCTION__));
}

 /****************************************************************
  * Function:    vUpdatePublicItem
  * Description: Updates the public config item 
  * Parameters:  itemName, item, updateType, user_data
  * Return:      void
  ***************************************************************/
 void vUpdatePublicItem( gchar*    itemName,
                         GVariant* item,
                         guint     updateType,
                         gpointer  user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                    DLT_STRING(__FUNCTION__));
      char* busName = NULL; 
       
      if((!item) || (!itemName))
      {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                               DLT_STRING("(): Null parameter recieved"));
          vHandleUpdatePublicItemResult( user_data );
          return;
      }

      if((busName = FindClientBusName(itemName, CONFIG_PUBLIC_ITEM)) == NULL)
      {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                    DLT_STRING("(): FindClientBusName failed"));
          vHandleSetPrivateItemResult( AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR,
                                       user_data );
          return;
      }

      if(poGetGDBusConnection() != NULL)
      {
          g_dbus_connection_call ( poGetGDBusConnection(),
                                   busName,
                                   "/com/bosch/CM/ConfigData",
                                   "com.bosch.CM.ConfigData",
                                   "UpdatePublicConfigItem",
                                   g_variant_new ("(s@a(ss)u)",
                                    itemName, item, updateType),
                                   NULL,
                                   G_DBUS_CALL_FLAGS_NONE,
                                   -1,
                                   NULL,
                                   (GAsyncReadyCallback)vUpdatePublicItemCallBack,
                                   user_data );
      }
      else
      {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                         DLT_STRING("(): Gateway Bus connection is NULL"));
          vHandleUpdatePublicItemResult( user_data );
      }
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                    DLT_STRING(__FUNCTION__));
 }

 /********************************************************************
  * Function:    vUpdatePublicItemCallBack
  * Description: Callback for UpdatePublicItem result
  * Parameters:  GDBusConnection*, GAsyncResult*, gpointer
  * Return:      void
  ********************************************************************/
 void  vUpdatePublicItemCallBack( GDBusConnection*  connection,
                                  GAsyncResult*     res,
                                  gpointer          user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));

      GError*    error =  NULL;
      GVariant*  ret   =  NULL;

      ret = g_dbus_connection_call_finish (connection, res, &error);
      if (ret == NULL)
      {
         DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
 		        DLT_STRING("(): UpdatePublicItem failed"));
           if (error != NULL)
           {
               DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
  			      DLT_STRING("Error= "),  DLT_STRING(
				  error->message ? error->message : ""));
               g_clear_error(&error);
           }
      }
      vHandleUpdatePublicItemResult( user_data );
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                    DLT_STRING(__FUNCTION__));
}

 /****************************************************************
  * Function:    vExchangeDescmo
  * Description: SET / GET of private config in XML format
  * Parameters:  itemName, itemXML, signatureXML, user_data
  * Return:      void
  ***************************************************************/
 void vExchangeDescmo( gchar*    itemName,
                       gchar*    itemXML,
                       gchar*    signatureXML,
                       gpointer  user_data )
 {
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));
     char* busName = NULL;

     if( (!itemName) || (!itemXML) || (!signatureXML) )
     {
         DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                               DLT_STRING("(): Null parameter recieved"));
         vHandleExchangeDescmoResult( "",
                                      AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR,
                                      user_data );
         return;
     }  
     
     if((busName = FindClientBusName(itemName, CONFIG_PRIVATE_ITEM)) == NULL)
     {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                       DLT_STRING("(): FindClientBusName failed"));
          vHandleSetPrivateItemResult( AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR,
                                       user_data );
          return;
     }
      
     if(poGetGDBusConnection() != NULL)
     {
          g_dbus_connection_call ( poGetGDBusConnection(),
                                   busName,
                                   "/com/bosch/CM/ConfigData",
                                   "com.bosch.CM.ConfigData",
                                   "ExchangeDESCMO",
                                   g_variant_new ("(sss)",
                                     itemName, itemXML, signatureXML),
                                   NULL,
                                   G_DBUS_CALL_FLAGS_NONE,
                                   -1,
                                   NULL,
                                   (GAsyncReadyCallback)vExchangeDescmoCallBack,
                                   user_data );
     }
     else
     {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                         DLT_STRING("(): Gateway Bus connection is NULL"));
          vHandleExchangeDescmoResult( "",
                                      AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR,
                                      user_data );
     }
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                    DLT_STRING(__FUNCTION__));
}                                      

 /********************************************************************
  * Function:    vExchangeDescmoCallBack
  * Description: Callback for ExchangeDescmo result
  * Parameters:  GDBusConnection*, GAsyncResult*, gpointer
  * Return:      void
  ********************************************************************/
 void vExchangeDescmoCallBack( GDBusConnection* connection,
                               GAsyncResult*    res,
                               gpointer         user_data )
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                                  DLT_STRING(__FUNCTION__));

      GError*    error =  NULL;
      GVariant*  ret   =  NULL;
      gchar*     outXML = NULL;     
      guint status = (guint) AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR;

      ret = g_dbus_connection_call_finish (connection, res, &error);
      if (ret == NULL)
      {
           DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
 		        DLT_STRING("(): ExchangeDescmo failed" ));
           if (error != NULL)
           {
               DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
      			   DLT_STRING("Error = "), DLT_STRING(
				   error->message ? error->message : ""));
               g_clear_error(&error);
           }
           vHandleExchangeDescmoResult( "", status, user_data );
           return;
      }

      g_variant_get (ret, "(su)", &outXML, &status);
      g_variant_unref (ret);

      DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING(
          "():Status = "), DLT_UINT(status));
      vHandleExchangeDescmoResult(outXML, status, user_data );

      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                                    DLT_STRING(__FUNCTION__));
 }

