/*!
*******************************************************************************
* @file             : FC_Gateway_BluetoothClientHandler.cpp
*******************************************************************************
*  - PROJECT:       : Automotive D-Bus server
*  - SW-COMPONENT   : Gateway
*  - DESCRIPTION    : Client Handler to FC_Bluetooth Component
*  - COPYRIGHT      : &copy; 2016 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|-----------------------
* 02.12.2016 | RHK6KOR      | 1.0.0    | methods for start Bluetooth
                                         client handler
******************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#include "FC_Gateway_BluetoothClientHandler.h"

#include "conn_most_fi_if.h"

#include "Utility.h"

//Dlt
#include "dlt/dlt.h"
//Dlt Context
DLT_IMPORT_CONTEXT(AGW_Bluetooth);



/******************************************************************************/
/*			   DEFINES                                            */
/******************************************************************************/
/******************************************************************************/
/*			CCA MESSAGE MAP					      */
/******************************************************************************/
BEGIN_MSG_MAP(fc_gateway_tcl_bluetoothclienthandler, ahl_tclBaseWork)
ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_DEVICELISTEXTENDED,
                    AMT_C_U8_CCAMSG_OPCODE_STATUS,
                    vHandle_DeviceListExtended_Status)
ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_DEVICELISTEXTENDED,
                    AMT_C_U8_CCAMSG_OPCODE_ERROR,
                    vOnServiceError)
END_MSG_MAP()

fc_gateway_tcl_bluetoothclienthandler* 
fc_gateway_tcl_bluetoothclienthandler::m_poBluetoothClientHandler = NULL;

/******************************************************************************
* Function:    fc_gateway_tcl_bluetoothclienthandler
* Description: single parameter constructor
* Parameters:  main app pointer
* Return:      void
******************************************************************************/
fc_gateway_tcl_bluetoothclienthandler::fc_gateway_tcl_bluetoothclienthandler(
        procgenericgateway_tclMainApp *poMainAppl)
    :ahl_tclBaseOneThreadClientHandler(
          (ahl_tclBaseOneThreadApp*)poMainAppl,
          CCA_C_U16_SRV_FB_BLUETOOTHSETTINGS, /* ID of used Service */
          MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION, /* MajorVersion of used Service */
          MOST_BTSETFI_C_U16_SERVICE_MINORVERSION  /* MinorVersion of used Service */
          )
{
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                "fc_gateway_tcl_bluetoothclienthandler : \n"),DLT_STRING( __FUNCTION__));
    m_bIsLastRequestIsMethodCall = FALSE;
    m_u8ChangedDeviceHandle = 0; /* set the handle to zero initially*/
    m_bIsBTServiceAvailable = FALSE;
    m_bIsGetSourcesMethodCall = FALSE;
    vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_DEVICELISTEXTENDED);
}

/******************************************************************************
* Function:    vGetServicesMethod
* Description: Gets the current active call.
* Parameters:  void
* Return:      bool
******************************************************************************/
bool bGetServicesMethod()
{
    fc_gateway_tcl_bluetoothclienthandler* btClientObj =
            fc_gateway_tcl_bluetoothclienthandler::getInstance(
                procgenericgateway_tclMainApp::poGetInstance());
    btClientObj->m_bIsLastRequestIsMethodCall = TRUE;
    bool bRet = btClientObj->bGetServices_MethodStart();
    if (!bRet)
    {
        DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,
                DLT_STRING("fc_gateway_tcl_bluetoothclienthandler \n"));
        btClientObj->m_bIsLastRequestIsMethodCall = FALSE;
        return FALSE;
    }
    return TRUE;
}


/******************************************************************************
* Function:    bGetServices_MethodStart
* Description: Post a get method request to CCA
* Parameters:  VOID
* Return:      VOID
******************************************************************************/

bool fc_gateway_tcl_bluetoothclienthandler::bGetServices_MethodStart()
{
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                "fc_gateway_tcl_bluetoothclienthandler : \n"),DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(" \n"),
            DLT_STRING( __FUNCTION__));

    most_btsetfi_tclMsgDeviceListExtendedGet oDeviceListExtendedGet;

    fi_tclVisitorMessage oMsg(oDeviceListExtendedGet,
                              MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(
                oMsg,                        // ServiceDataMsg
                101,                                  // s32_value
                MOST_BTSETFI_C_U16_DEVICELISTEXTENDED,  // Property ID
                AMT_C_U8_CCAMSG_OPCODE_GET           // Opcode
                );

    //Post message
    ail_tenCommunicationError enResult= _poMainAppl->enPostMessage(&oMsg, TRUE);

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,
                DLT_STRING("bGetServices_MethodStart Main error"));
        DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,
                DLT_STRING("bGetServices_MethodStart"));
        //Destroying the Fi object
        oDeviceListExtendedGet.vDestroy();
        return false;
    }

    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,
            DLT_STRING("-bGetServices_MethodStart : \t"),
            DLT_STRING("m_bIsLastRequestIsMethodCall = \t"),
            DLT_STRING("m_bIsGetSourcesMethodCall = \t"),
            DLT_INT( m_bIsLastRequestIsMethodCall),
            DLT_INT(m_bIsGetSourcesMethodCall));
    return true;
}

/******************************************************************************
* Function:    getInstance
* Description: singleton. Function to get bluetooth client-handler instance
* Parameters:  main app pointer
* Return:      fc_gateway_tcl_bluetoothclienthandler instance pointer
******************************************************************************/

fc_gateway_tcl_bluetoothclienthandler*
fc_gateway_tcl_bluetoothclienthandler::getInstance(procgenericgateway_tclMainApp
                                                   *poMainAppl)
{
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                "fc_gateway_tcl_bluetoothclienthandler : \n"),DLT_STRING( __FUNCTION__));
    if (m_poBluetoothClientHandler == NULL)
    {
        m_poBluetoothClientHandler =
                new fc_gateway_tcl_bluetoothclienthandler(poMainAppl);

        DLT_REGISTER_CONTEXT(AGW_Bluetooth,"GWBT","Bluetooth"
                             "context for DLT Logging");
    }

    return m_poBluetoothClientHandler;
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                "-fc_gateway_tcl_bluetoothclienthandler : \n"),DLT_STRING( __FUNCTION__));
}

/******************************************************************************
* Function:    ~fc_gateway_tcl_bluetoothclienthandler
* Description: destructor
* Parameters:  void
* Return:      void
******************************************************************************/
fc_gateway_tcl_bluetoothclienthandler::~fc_gateway_tcl_bluetoothclienthandler()
{
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                "fc_gateway_tcl_bluetoothclienthandler : \n"),DLT_STRING( __FUNCTION__));

    vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_DEVICELISTEXTENDED);
    vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_PAIRINGSTATUS);
    if (m_poBluetoothClientHandler != NULL)
    {
        delete m_poBluetoothClientHandler;
        m_poBluetoothClientHandler = NULL;
    }
}

/******************************************************************************
* Function:    vOnServiceAvailable
* Description: CCA callback function
* Parameters:  void
* Return:      void
******************************************************************************/
tVoid fc_gateway_tcl_bluetoothclienthandler::vOnServiceAvailable() 
{
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                "fc_gateway_tcl_bluetoothclienthandler : \n"),DLT_STRING( __FUNCTION__));
    vExportAGWBluetoothInterface(); //Export D-Bus service
    m_bIsBTServiceAvailable = TRUE;
}

/******************************************************************************
* Function:    vOnServiceUnavailable
* Description: CCA callback function
* Parameters:  void
* Return:      void
******************************************************************************/
tVoid fc_gateway_tcl_bluetoothclienthandler::vOnServiceUnavailable() 
{
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                "fc_gateway_tcl_bluetoothclienthandler : \n"),DLT_STRING( __FUNCTION__));
    vUnexportAGWBluetoothInterface(); //Unexport D-Bus service
    m_bIsBTServiceAvailable = FALSE;
}

/************************************************************************
*FUNCTION		: getCurrentBTDeviceHandle
*PARAMETER		:
*RETURNVALUE	: tU8
*
************************************************************************/
tU8 fc_gateway_tcl_bluetoothclienthandler::getCurrentBTDeviceHandle()
{
    return m_u8ChangedDeviceHandle;
}

/************************************************************************
*FUNCTION	: vOnServiceError
*DESCRIPTION: Receives the Error message from BT service
*PARAMETER:
*         poMessage:   (->I)
*         Message object
*
*RETURNVALUE:	 tVoid
************************************************************************/
tVoid fc_gateway_tcl_bluetoothclienthandler::vOnServiceError(
        amt_tclServiceData *poMessage)
{
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO
            , DLT_STRING("fc_gateway_tcl_bluetoothclienthandler : \n")
            , DLT_STRING( __FUNCTION__));
}
/******************************************************************************
* Function:    vGetSupportedProfileNames
* Description: utility function to conver the most DS to GVariant.
* Parameters:  void
* Return:      void
******************************************************************************/
GVariant* fc_gateway_tcl_bluetoothclienthandler::vGetSupportedProfileNames(
        const most_fi_tcl_BTSetSupportedProfiles& l_most_fi_tcl_BTSetSupportedProfiles)
{

    /* get UUIds */
    GVariantBuilder profile_builder;
    GVariant *profilevalue;

    g_variant_builder_init(&profile_builder, G_VARIANT_TYPE("as"));

    if (l_most_fi_tcl_BTSetSupportedProfiles.bSPPSupport)
    {
        g_variant_builder_add(&profile_builder, "s", "SPP");
    }

    if (l_most_fi_tcl_BTSetSupportedProfiles.bHFPSupport)
    {
        g_variant_builder_add(&profile_builder, "s", "HFP");
    }

    if (l_most_fi_tcl_BTSetSupportedProfiles.bA2DPSupport)
    {
        g_variant_builder_add(&profile_builder, "s", "A2DP");
    }

    if (l_most_fi_tcl_BTSetSupportedProfiles.bAVRCPSupport)
    {
        g_variant_builder_add(&profile_builder, "s", "AVRCP");
    }

    if (l_most_fi_tcl_BTSetSupportedProfiles.bMAPSupport)
    {
        g_variant_builder_add(&profile_builder, "s", "MAP");
    }

    if (l_most_fi_tcl_BTSetSupportedProfiles.bDUNSupport)
    {
        g_variant_builder_add(&profile_builder, "s", "DUN");
    }

    if (l_most_fi_tcl_BTSetSupportedProfiles.bPANSupport)
    {
        g_variant_builder_add(&profile_builder, "s", "PAN");
    }

    if (l_most_fi_tcl_BTSetSupportedProfiles.bPBAPSuport)
    {
        g_variant_builder_add(&profile_builder, "s", "PBAP");
    }

    if (l_most_fi_tcl_BTSetSupportedProfiles.bOPPSupport)
    {
        g_variant_builder_add(&profile_builder, "s", "OOP");
    }

    if (l_most_fi_tcl_BTSetSupportedProfiles.bDIPSupport)
    {
        g_variant_builder_add(&profile_builder, "s", "DIP");
    }

    profilevalue = g_variant_builder_end(&profile_builder);

    //freeing in the called function vHandleGetServicesMethod
    GVariant* profilevalueRef = g_variant_ref_sink(profilevalue);

    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING("PROFILE BUILDER changed = "));
    return profilevalueRef;
}

tVoid fc_gateway_tcl_bluetoothclienthandler::vHandleGetServicesMethod(
        const most_btsetfi_tclMsgDeviceListExtendedStatus& oDeviceListExtendedStatus)
{
    tU8 u8DeviceListExtendedSize =
            (tU8)oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems.size();

    if (0 != u8DeviceListExtendedSize)
    {
        GVariant* result = NULL;
        GVariantBuilder get_services_builder;
        g_variant_builder_init (&get_services_builder, G_VARIANT_TYPE ("a(sa{sv})"));

        for ( tU8 u8DeviceListExtendedIndex = 0;
              u8DeviceListExtendedIndex < u8DeviceListExtendedSize;
              u8DeviceListExtendedIndex++ )
        {
            most_fi_tcl_BTSetDeviceListExtendedResultItem DeviceListExtendedItem =
                    oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems[u8DeviceListExtendedIndex];

            //copy device handle
            tU8 l_u8DeviceHandle = DeviceListExtendedItem.u8DeviceHandle;
			
            //copy the device name
            std::string l_strName = DeviceListExtendedItem.sDeviceName.szValue;
			
            //copy the connected status
            bool l_bConnectedStatus =
                    DeviceListExtendedItem.bDeviceConnectedStatus;

            //get the supported profiles
            GVariant* supportedProfiles =
                    vGetSupportedProfileNames(
                        DeviceListExtendedItem.oDeviceSupportedProfiles);

            //fill the device address
            tChar acDeviceAddress[BT_MAXDEVICENAME_LENGTH];
            tChar acTempActiveDeviceBD_Address[BT_MAXDEVICENAME_LENGTH];
            memset(acDeviceAddress, 0, BT_MAXDEVICENAME_LENGTH);

            sscanf(
                        DeviceListExtendedItem.sDeviceAddress.szValue,
                        "%02x%02x%02x%02x%02x%02x",
                        (tUInt *)&acTempActiveDeviceBD_Address[0],
                        (tUInt *)&acTempActiveDeviceBD_Address[1],
                        (tUInt *)&acTempActiveDeviceBD_Address[2],
                        (tUInt *)&acTempActiveDeviceBD_Address[3],
                        (tUInt *)&acTempActiveDeviceBD_Address[4],
                        (tUInt *)&acTempActiveDeviceBD_Address[5]
                        );

            //for string print safety , making 7th char NULL
            acTempActiveDeviceBD_Address[6] = 0;

            snprintf( acDeviceAddress,
                      BT_MAXDEVICENAME_LENGTH,
                      "dev_%02x_%02x_%02x_%02x_%02x_%02x",
                      (unsigned char)acTempActiveDeviceBD_Address[0],
                      (unsigned char)acTempActiveDeviceBD_Address[1],
                      (unsigned char)acTempActiveDeviceBD_Address[2],
                      (unsigned char)acTempActiveDeviceBD_Address[3],
                      (unsigned char)acTempActiveDeviceBD_Address[4],
                      (unsigned char)acTempActiveDeviceBD_Address[5] );


            std::string strDeviceId(acDeviceAddress);
            strDeviceId = "dev_" + strDeviceId;

            // Now we have all information needed to construct gvariant.
            GVariantBuilder sub_builder;
            g_variant_builder_init(&sub_builder, G_VARIANT_TYPE("a{sv}"));

            g_variant_builder_add(&sub_builder, "{sv}", "DeviceHandle",
                                  g_variant_new_uint16(l_u8DeviceHandle));
            g_variant_builder_add(&sub_builder, "{sv}", "Address",
                                  g_variant_new_string(acDeviceAddress));
            g_variant_builder_add(&sub_builder, "{sv}", "Name",
                                  g_variant_new_string(l_strName.c_str()));
            g_variant_builder_add(&sub_builder, "{sv}", "Connected",
                                  g_variant_new_boolean(l_bConnectedStatus));
            g_variant_builder_add(&sub_builder, "{sv}", "UUIDs",
                                  supportedProfiles);
            g_variant_builder_add(&get_services_builder, "(sa{sv})",
                                  strDeviceId.c_str(), &sub_builder);
            g_variant_unref(supportedProfiles);
        }

        result = g_variant_builder_end(&get_services_builder);
        GVariant* result_ref = g_variant_ref_sink(result);
        DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING("Get Services result"));
        vGetServicesMethodResult(result_ref);
        g_variant_unref(result_ref);
    }
    else
    {
        DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                    "size of extended list is zero"));
        vGetServiceMethodError(AP_BLUETOOTH_ERROR_NO_DEVICE_CONNECTED);
    }
}

/******************************************************************************
* Function:    vHandleGetSourcesMethod
* Description: Handler for get sources method result.
* Parameters:  most_btsetfi_tclMsgDeviceListExtendedStatus
* Return:      VOID
******************************************************************************/
tVoid 
fc_gateway_tcl_bluetoothclienthandler::vHandleGetSourcesMethod(
        const most_btsetfi_tclMsgDeviceListExtendedStatus&
        oDeviceListExtendedStatus)
{
    GVariant* result = NULL;
    GVariantBuilder get_sources_builder;
    g_variant_builder_init(&get_sources_builder, G_VARIANT_TYPE("a(sa{sv})"));

    tU8 u8DeviceListExtendedSize =
            (tU8)oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems.size();

    /* Device handle - 1 car inbuilt contact manager*/
    GVariantBuilder sub_builder;
    g_variant_builder_init(&sub_builder, G_VARIANT_TYPE("a{sv}"));

    g_variant_builder_add(&sub_builder, "{sv}", "DeviceHandle",
                          g_variant_new_uint16(1));
    g_variant_builder_add(&sub_builder, "{sv}", "Connected",
                          g_variant_new_boolean(TRUE));
    g_variant_builder_add(&get_sources_builder, "(sa{sv})",
                          "car_vehicle_phonebook", &sub_builder);

    if (0 != u8DeviceListExtendedSize)
    {
        for (tU8 u8DeviceListExtendedIndex = 0;
             u8DeviceListExtendedIndex < u8DeviceListExtendedSize;
             u8DeviceListExtendedIndex++)
        {
            most_fi_tcl_BTSetDeviceListExtendedResultItem DeviceListExtendedItem =
                    oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems[u8DeviceListExtendedIndex];

            //copy device handle
            tU8 l_u8DeviceHandle = DeviceListExtendedItem.u8DeviceHandle;


            //copy the device name
            std::string l_strName = DeviceListExtendedItem.sDeviceName.szValue;


            //copy the connected status
            bool l_bConnectedStatus = DeviceListExtendedItem.bDeviceConnectedStatus;

            tChar acDeviceAddress[BT_MAXDEVICENAME_LENGTH];
            tChar acTempActiveDeviceBD_Address[BT_MAXDEVICENAME_LENGTH];
            memset(acDeviceAddress, 0, BT_MAXDEVICENAME_LENGTH);

            sscanf( DeviceListExtendedItem.sDeviceAddress.szValue,
                    "%02x%02x%02x%02x%02x%02x",
                    (tUInt *)&acTempActiveDeviceBD_Address[0],
                    (tUInt *)&acTempActiveDeviceBD_Address[1],
                    (tUInt *)&acTempActiveDeviceBD_Address[2],
                    (tUInt *)&acTempActiveDeviceBD_Address[3],
                    (tUInt *)&acTempActiveDeviceBD_Address[4],
                    (tUInt *)&acTempActiveDeviceBD_Address[5]);

            //for string print safety , making 7th char NULL
            acTempActiveDeviceBD_Address[6] = 0;

            snprintf( acDeviceAddress,
                      BT_MAXDEVICENAME_LENGTH,
                      "%02x_%02x_%02x_%02x_%02x_%02x",
                      (unsigned char)acTempActiveDeviceBD_Address[0],
                      (unsigned char)acTempActiveDeviceBD_Address[1],
                      (unsigned char)acTempActiveDeviceBD_Address[2],
                      (unsigned char)acTempActiveDeviceBD_Address[3],
                      (unsigned char)acTempActiveDeviceBD_Address[4],
                      (unsigned char)acTempActiveDeviceBD_Address[5]
                      );

            std::string strSourceID(acDeviceAddress);
            strSourceID = "Bluetooth_phonebook_"+strSourceID;

            snprintf( acDeviceAddress,
                      BT_MAXDEVICENAME_LENGTH,
                      "%02x:%02x:%02x:%02x:%02x:%02x",
                      (unsigned char)acTempActiveDeviceBD_Address[0],
                      (unsigned char)acTempActiveDeviceBD_Address[1],
                      (unsigned char)acTempActiveDeviceBD_Address[2],
                      (unsigned char)acTempActiveDeviceBD_Address[3],
                      (unsigned char)acTempActiveDeviceBD_Address[4],
                      (unsigned char)acTempActiveDeviceBD_Address[5]
                      );

            // Now we have all information needed to construct gvariant.
            GVariantBuilder sub_builder;
            g_variant_builder_init(&sub_builder, G_VARIANT_TYPE("a{sv}"));

            g_variant_builder_add(&sub_builder, "{sv}", "DeviceHandle",
                                  g_variant_new_uint16(l_u8DeviceHandle));
            g_variant_builder_add(&sub_builder, "{sv}", "Address",
                                  g_variant_new_string(acDeviceAddress));
            g_variant_builder_add(&sub_builder, "{sv}", "Connected",
                                  g_variant_new_boolean(l_bConnectedStatus));
            g_variant_builder_add(&sub_builder, "{sv}", "Name",
                                  g_variant_new_string(l_strName.c_str()));
            g_variant_builder_add(&get_sources_builder, "(sa{sv})",
                                  strSourceID.c_str(), &sub_builder);
        }
    }
    else
    {
        DLT_LOG(AGW_Bluetooth,DLT_LOG_DEBUG,
                DLT_STRING("size of extended list is zero"));
    }
    result = g_variant_builder_end(&get_sources_builder);
	GVariant* result_ref = g_variant_ref_sink(result);
    vGetSourcesMethodResult(result);
	g_variant_unref(result_ref);

}

/******************************************************************************
* Function:    vHandleDeviceListStatusChange
* Description: Handler for device list change.
* Parameters:  most_btsetfi_tclMsgDeviceListExtendedStatus
* Return:      VOID
******************************************************************************/
tVoid 
fc_gateway_tcl_bluetoothclienthandler::vHandleDeviceListStatusChange(
        const most_btsetfi_tclMsgDeviceListExtendedStatus& oDeviceListExtendedStatus)
{
    most_fi_tcl_BTSetDeviceListChangeExtended oDeviceListChange =
            oDeviceListExtendedStatus.oDeviceListChange;
    most_fi_tcl_BTSetDeviceListExtendedResult l_oDeviceListExtendedResultStored =
            m_oDeviceListExtendedResult;
    m_oDeviceListExtendedResult.oItems.clear();
    m_oDeviceListExtendedResult =
            oDeviceListExtendedStatus.oDeviceListExtendedResult;
   
//    tU8  l_u8DeviceHandle = oDeviceListChange.u8DeviceHandle;

    // its neither connected not disconnected, hence just leave it.
    /*if (oDeviceListChange.e8DeviceStatus.enType ==
            most_fi_tcl_e8_BTSetDeviceStatus::FI_EN_E8DEVICE_NO_CHANGE)
    {
        DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                    "vHandleDeviceListStatusChange = "),
                DLT_STRING( "FI_EN_E8DEVICE_NO_CHANGE"));
        return;
    }
    else
    {*/
        bool l_IsPaired = FALSE;
        char deviceHandle[5];
        tU8 l_deviceHandle = 0;

        std::vector<most_fi_tcl_BTSetDeviceListExtendedResultItem,
                std::allocator<most_fi_tcl_BTSetDeviceListExtendedResultItem> >::const_iterator
                itIteratorStored = l_oDeviceListExtendedResultStored.oItems.begin();

        std::vector<most_fi_tcl_BTSetDeviceListExtendedResultItem,
                std::allocator<most_fi_tcl_BTSetDeviceListExtendedResultItem> >::const_iterator
                itIteratorUpdated = m_oDeviceListExtendedResult.oItems.begin();


        if(l_oDeviceListExtendedResultStored.oItems.size() >
                m_oDeviceListExtendedResult.oItems.size())
        {
            //Device unpaired
            bool bIsDeviceHandleEqual = FALSE;

            for( ;
                 itIteratorStored != l_oDeviceListExtendedResultStored.oItems.end();
                 itIteratorStored++
                 )
            {
                for( ;
                     itIteratorUpdated != m_oDeviceListExtendedResult.oItems.end();
                     itIteratorUpdated++
                     )
                {
                    if(itIteratorStored->u8DeviceHandle ==
                            itIteratorUpdated->u8DeviceHandle)
                    {
                        bIsDeviceHandleEqual = TRUE;
                        break;
                    }
                    else
                    {
                        bIsDeviceHandleEqual = FALSE;
                    }
                }
                if(bIsDeviceHandleEqual == FALSE)
                {
                    DLT_LOG(AGW_Bluetooth, DLT_LOG_INFO
                            , DLT_STRING("Device handle")
                            , DLT_INT(itIteratorStored->u8DeviceHandle));
                    l_deviceHandle = itIteratorStored->u8DeviceHandle;
                    l_IsPaired = FALSE;
                    break;
                }
            }
			GVariantBuilder builder;
            GVariant *result = NULL;
            memset(deviceHandle, 0, 5);
            sprintf(deviceHandle, "%d", l_deviceHandle);
            g_variant_builder_init(&builder, G_VARIANT_TYPE("a{sv}"));
            g_variant_builder_add(&builder, "{sv}", "Paired",
                              g_variant_new_boolean(l_IsPaired));
            result = g_variant_builder_end(&builder);
		    GVariant* result_ref = g_variant_ref_sink(result);
            DLT_LOG(AGW_Bluetooth, DLT_LOG_INFO
                , DLT_STRING("Properties changed, device handle ")
                , DLT_STRING(deviceHandle));
            emit_properties_changed_signal((char*)deviceHandle, result_ref);
		    g_variant_unref(result_ref);
        }
        else if(l_oDeviceListExtendedResultStored.oItems.size() <
                m_oDeviceListExtendedResult.oItems.size())
        {
            //Device Paired
            bool bIsDeviceHandleEqual = FALSE;

            for( ;
                 itIteratorUpdated != m_oDeviceListExtendedResult.oItems.end();
                 itIteratorUpdated++
                 )
            {
                for( ;
                     itIteratorStored != l_oDeviceListExtendedResultStored.oItems.end();
                     itIteratorStored++
                     )
                {
                    if(itIteratorStored->u8DeviceHandle ==
                            itIteratorUpdated->u8DeviceHandle)
                    {
                        bIsDeviceHandleEqual = TRUE;
                        break;
                    }
                    else
                    {
                        bIsDeviceHandleEqual = FALSE;
                    }
                }
                if(bIsDeviceHandleEqual == FALSE)
                {
                    DLT_LOG(AGW_Bluetooth, DLT_LOG_INFO
                            , DLT_STRING("Device handle")
                            , DLT_INT(itIteratorUpdated->u8DeviceHandle));
                    l_deviceHandle = itIteratorUpdated->u8DeviceHandle;
                    l_IsPaired = TRUE;
                    break;
                }
            }
			
			GVariantBuilder builder;
            GVariant *result = NULL;
            memset(deviceHandle, 0, 5);
            sprintf(deviceHandle, "%d", l_deviceHandle);
            g_variant_builder_init(&builder, G_VARIANT_TYPE("a{sv}"));
            g_variant_builder_add(&builder, "{sv}", "Paired",
                              g_variant_new_boolean(l_IsPaired));
            result = g_variant_builder_end(&builder);
		    GVariant* result_ref = g_variant_ref_sink(result);
            DLT_LOG(AGW_Bluetooth, DLT_LOG_INFO
                , DLT_STRING("Properties changed, device handle ")
                , DLT_STRING(deviceHandle));
            emit_properties_changed_signal((char*)deviceHandle, result_ref);
		    g_variant_unref(result_ref);
        }
        else
        {
            //no device added or removed
			// Check if connection status changed
			for( ;
                 itIteratorUpdated != m_oDeviceListExtendedResult.oItems.end();
                 itIteratorUpdated++
                 )
            {
                for( ;
                     itIteratorStored != l_oDeviceListExtendedResultStored.oItems.end();
                     itIteratorStored++
                     )
                {
                    if(itIteratorStored->u8DeviceHandle ==
                            itIteratorUpdated->u8DeviceHandle)
                    {
						if(itIteratorStored->bDeviceConnectedStatus !=
                            itIteratorUpdated->bDeviceConnectedStatus)
                    {
                        bool l_bIsConnected = FALSE;
                        //Connected status had changed.

                        if(itIteratorUpdated->bDeviceConnectedStatus == TRUE)
                        {
                            l_bIsConnected = TRUE;
                        }
                        else
                        {
                            l_bIsConnected = FALSE;
                        }
						
						char deviceHandle[5];
                        memset(deviceHandle, 0, 5);
                        sprintf(deviceHandle, "%d",
                                itIteratorStored->u8DeviceHandle);

                        GVariantBuilder builder;
                        GVariant *result = NULL;
                        g_variant_builder_init( &builder
                                                , G_VARIANT_TYPE("a{sv}"));
                        g_variant_builder_add(&builder, "{sv}", "Connected",
                                              g_variant_new_boolean(l_bIsConnected));
                        result = g_variant_builder_end(&builder);
                        
                        GVariant* result_ref = g_variant_ref_sink(result);
                        DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                                    "vHandleDeviceListStatusChange: device handle"),
                                DLT_STRING(deviceHandle));
                        emit_properties_changed_signal((char*)deviceHandle, result_ref);
                        vHandleDeviceListChange(
                                    itIteratorUpdated->u8DeviceHandle,
                                    l_oDeviceListExtendedResultStored,
                                    l_bIsConnected);
                        g_variant_unref(result_ref);
					}
					}
				}
			}
            
        }
    //}
}

/******************************************************************************
* Function:    vHandleDeviceListChange
* Description: Sends source/device added/removed signal.
* Parameters:  tU8 u8DeviceHandle, most_fi_tcl_BTSetDeviceListExtendedResult, 
               bool
* Return:      VOID
******************************************************************************/
void fc_gateway_tcl_bluetoothclienthandler::vHandleDeviceListChange(
        tU8 u8DeviceHandle,
        const most_fi_tcl_BTSetDeviceListExtendedResult
        oDeviceListExtendedResult,
        bool bIsAdded)
{
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,
            DLT_STRING(" vSendSourceChangeSignal in Phone entered "));
    std::vector<most_fi_tcl_BTSetDeviceListExtendedResultItem,
            std::allocator<most_fi_tcl_BTSetDeviceListExtendedResultItem> >::const_iterator
            itIterator = oDeviceListExtendedResult.oItems.begin();

    for(;itIterator != oDeviceListExtendedResult.oItems.end(); itIterator++)
    {
        if(itIterator->u8DeviceHandle == u8DeviceHandle)
        {
            GVariantBuilder sub_builder;
            g_variant_builder_init(&sub_builder, G_VARIANT_TYPE("a{sv}"));

            tChar acDeviceAddress[BT_MAXDEVICENAME_LENGTH];
            tChar acTempActiveDeviceBD_Address[BT_MAXDEVICENAME_LENGTH];
            memset(acDeviceAddress, 0, BT_MAXDEVICENAME_LENGTH);

            sscanf( itIterator->sDeviceAddress.szValue,
                    "%02x%02x%02x%02x%02x%02x",
                    (tUInt *)&acTempActiveDeviceBD_Address[0],
                    (tUInt *)&acTempActiveDeviceBD_Address[1],
                    (tUInt *)&acTempActiveDeviceBD_Address[2],
                    (tUInt *)&acTempActiveDeviceBD_Address[3],
                    (tUInt *)&acTempActiveDeviceBD_Address[4],
                    (tUInt *)&acTempActiveDeviceBD_Address[5]
                    );

            //for string print safety , making 7th char NULL
            acTempActiveDeviceBD_Address[6] = 0;

            snprintf( acDeviceAddress,
                      BT_MAXDEVICENAME_LENGTH,
                      "%02x_%02x_%02x_%02x_%02x_%02x",
                      (unsigned char)acTempActiveDeviceBD_Address[0],
                      (unsigned char)acTempActiveDeviceBD_Address[1],
                      (unsigned char)acTempActiveDeviceBD_Address[2],
                      (unsigned char)acTempActiveDeviceBD_Address[3],
                      (unsigned char)acTempActiveDeviceBD_Address[4],
                      (unsigned char)acTempActiveDeviceBD_Address[5]
                      );

            DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                        "Formatted device address = "),DLT_STRING( acDeviceAddress));

            std::string sSourceID(acDeviceAddress);
            sSourceID = "Bluetooth_phonebook_"+sSourceID;
            DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                        "Formatted source ID = "),DLT_STRING( sSourceID.c_str()));

            std::string sDeviceID(acDeviceAddress);
            sDeviceID = "dev_"+sDeviceID;
            DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                        "Formatted device ID = "),DLT_STRING( sDeviceID.c_str()));

            snprintf( acDeviceAddress,
                      BT_MAXDEVICENAME_LENGTH,
                      "%02x:%02x:%02x:%02x:%02x:%02x",
                      (unsigned char)acTempActiveDeviceBD_Address[0],
                      (unsigned char)acTempActiveDeviceBD_Address[1],
                      (unsigned char)acTempActiveDeviceBD_Address[2],
                      (unsigned char)acTempActiveDeviceBD_Address[3],
                      (unsigned char)acTempActiveDeviceBD_Address[4],
                      (unsigned char)acTempActiveDeviceBD_Address[5]
                      );


            g_variant_builder_add(&sub_builder, "{sv}", "DeviceHandle",
                                  g_variant_new_uint16(u8DeviceHandle));
            g_variant_builder_add(&sub_builder, "{sv}", "Address",
                                  g_variant_new_string(acDeviceAddress));
            g_variant_builder_add(&sub_builder, "{sv}", "Connected",
                                  g_variant_new_boolean(bIsAdded));
            g_variant_builder_add(&sub_builder, "{sv}", "Name",
                                  g_variant_new_string(itIterator->sDeviceName.szValue));

            GVariant* varResult = g_variant_builder_end(&sub_builder);
            GVariant* varResultRef = g_variant_ref_sink(varResult);
            if(bIsAdded)
            {
                emit_device_added_signal(sDeviceID.c_str(),varResultRef);
                emit_source_added_signal(sSourceID.c_str(),varResultRef);
            }
            else
            {
                emit_device_removed_signal(sDeviceID.c_str());
                emit_source_removed_signal(sSourceID.c_str());
            }
            g_variant_unref(varResultRef);
        }
    }
}

/******************************************************************************
* Function:    vHandle_DeviceListExtended_Status
* Description: CCA callback function for device list changed.
* Parameters:  void
* Return:      void
******************************************************************************/
tVoid fc_gateway_tcl_bluetoothclienthandler::vHandle_DeviceListExtended_Status(
        amt_tclServiceData* poMessage)
{

    DLT_LOG(AGW_Bluetooth,DLT_LOG_DEBUG,DLT_STRING(
                " Property Status:DeviceListExtendedStatus in Phone entered "));
    // Extract required info from poMessage
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_btsetfi_tclMsgDeviceListExtendedStatus oDeviceListExtendedStatus;

    // Unmarshal the FI visitor message to the given FI data object.
    // The parameters are copied individually from shared memory to the FI data
    // object.
    if ( oVisitorMsg.s32GetData(oDeviceListExtendedStatus,
                                MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION) !=  OSAL_ERROR)
    {
        std::vector<most_fi_tcl_BTSetDeviceListExtendedResultItem,
                std::allocator<most_fi_tcl_BTSetDeviceListExtendedResultItem> >::const_iterator
                itIteratorDeviceList = oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems.begin();

        for( ;
             itIteratorDeviceList != oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems.end();
             itIteratorDeviceList++
             )
        {
            if ((itIteratorDeviceList->bDeviceConnectedStatus) &&
                    (itIteratorDeviceList->bOutgoingSourceDeviceStatus))
            {
                m_u8ChangedDeviceHandle = itIteratorDeviceList->u8DeviceHandle;
                DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                            "Device Handle = "),DLT_INT( m_u8ChangedDeviceHandle));
            }

        }
        if (m_bIsLastRequestIsMethodCall)
        {
            vHandleGetServicesMethod(oDeviceListExtendedStatus);
            m_bIsLastRequestIsMethodCall = FALSE;
            //burp!! we done with method call gracefully.
        }
        else if(m_bIsGetSourcesMethodCall)
        {
            vHandleGetSourcesMethod(oDeviceListExtendedStatus);
            m_bIsGetSourcesMethodCall = FALSE;
        }
        else
        {
            vHandleDeviceListStatusChange(oDeviceListExtendedStatus);
        }

    }
    else
    {
        DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,
                DLT_STRING(" unable to extract CCA message pointer"));

        if(m_bIsLastRequestIsMethodCall)
        {
            vGetServiceMethodError(AP_BLUETOOTH_ERROR_INTERNAL_ERROR);
            m_bIsLastRequestIsMethodCall = FALSE;
            //DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,
            //        DLT_STRING("GET services method request is consumed"));
        }
    }

    oDeviceListExtendedStatus.vDestroy();
}


#ifdef __cplusplus
extern "C"
{
#endif
/******************************************************************************
* Function:    bGetSources
* Description: Wrapper function for get sources 
* Parameters:  void
* Return:      bool
******************************************************************************/
bool bGetSources()
{
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING("+vGetSources"));
    fc_gateway_tcl_bluetoothclienthandler* btClientObj =
            fc_gateway_tcl_bluetoothclienthandler::getInstance(
                procgenericgateway_tclMainApp::poGetInstance());

    btClientObj->m_bIsGetSourcesMethodCall = TRUE;
    bool bRet = btClientObj->bGetServices_MethodStart();

    if (!bRet)
    {
        DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                    "+unable to message to bt component"));
        btClientObj->m_bIsGetSourcesMethodCall = FALSE;
        return FALSE;
    }
    return TRUE;
}

#ifdef __cplusplus
}
#endif //__cplusplus
#endif//VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC