/*!
*******************************************************************************
* @file             : AudioDBusHandler.c
*******************************************************************************
*  - PROJECT:       : Automotive Gateway
*  - SW-COMPONENT   : Gateway - Audio
*  - DESCRIPTION    : D-Bus server for audio automotive API's
*  - COPYRIGHT      : &copy; 2015 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|------------------------
*10.03.2015 | DJS5KOR(RBEI/ECO2) | 1.0.0 | methods for Automotive D-Bus Server
*10.03.2015 | RHK6KOR(RBEI/ECO2) | 1.1.0 | code refactor for common DBus server
******************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#include "AudioDBusHandler.h"
#include "AudioManager_ClientInterface.h"

//Dlt
#include "dlt/dlt.h"
//Dlt Context
DLT_IMPORT_CONTEXT(AGW_Audio);

/******************************************************************************
* Function:    handle_source_availability_method
* Description: Handler for source availability
* Parameters:  proxy object, invocation pointer, source noumber,
               availability status, user data
* Return:      boolean
*****************************************************************************/

gboolean
handle_source_availability_method (AudioManager *object,
                                   GDBusMethodInvocation *invocation,
                                   guchar arg_Source,
                                   guchar arg_Availability,
                                   gpointer user_data)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING(" Signal Info Emitted by the"\
                                              "source_availability_method"));

    if(bRequestSetSourceAvailability(arg_Source,arg_Availability))
    {
        audio_manager_complete_source_availability(object,invocation);
        return TRUE;
    }
    else
    {
        g_dbus_method_invocation_return_error (invocation,G_DBUS_ERROR,
                                               AP_AUDIO_KFAIL,
                                               "Source availability failed");
        return FALSE;
    }
}

/******************************************************************************
* Function:    handle_audio_route_request_method
* Description: handler for audio route request
* Parameters:  proxy object, invocation pointer, source number,
               status required, user data
* Return:      boolean
*****************************************************************************/
gboolean
handle_audio_route_request_method (AudioManager *object,
                                   GDBusMethodInvocation *invocation,
                                   guchar arg_Source,
                                   guchar arg_StateRequired,
                                   gpointer user_data)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

    gboolean bStatus = FALSE;

    if(AP_AUDIO_SRC_ACT_ON == arg_StateRequired) // source_activity on
    {
        bRequestAudioRoute(arg_Source, (bool *)&bStatus);
    }
    else if(AP_AUDIO_SRC_ACT_OFF == arg_StateRequired)  // source_activity off
    {
        bStatus = bReleaseAudioRoute(arg_Source);
    }
    else
    {
        g_dbus_method_invocation_return_error (invocation,G_DBUS_ERROR,
                                               AP_AUDIO_KFAIL,
                                               "The state required is invalid");
	    return TRUE;
    }
    g_dbus_method_invocation_return_value (invocation,
                                           g_variant_new ("(b)", bStatus));

    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("Handler function for AudioRoute"\
                                              "Request  Ends"));
    return TRUE;
}

/******************************************************************************
* Function:    handle_mute_state_method
* Description: handler for mute state method
* Parameters:  proxy object, invocation pointer, state required, user data
* Return:      boolean
******************************************************************************/
gboolean
handle_mute_state_method (AudioManager *object,
                          GDBusMethodInvocation *invocation,
                          guchar arg_Source,
                          gboolean arg_StateRequired,
                          gpointer user_data)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

    DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("handle_mute_state_method State Required is : "),
            DLT_INT( arg_StateRequired));

    if(bRequestMuteState(arg_Source,arg_StateRequired))
    {
        audio_manager_complete_mute_state(object,invocation);
        return TRUE;
    }
    else
    {
        g_dbus_method_invocation_return_error (invocation,G_DBUS_ERROR,
                                               AP_AUDIO_KFAIL,
                                               "Mute state failed");
        return FALSE;
    }
}

/******************************************************************************
* Function:    bSendSourceActivity
* Description: Function to send or emit source activity signal
* Parameters:  source number, source activity status
* Return:      boolean
******************************************************************************/
gboolean bSendSourceActivity(int iSrcNum, ApAudioSourceStatus eStatus)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("Source is : "),DLT_INT(iSrcNum),
            DLT_STRING("Status is"),DLT_INT(eStatus));

    AudioManager *l_poAudioProxyObj = poGetAudioProxyObj();
    if(l_poAudioProxyObj != NULL)
    {
        g_signal_emit_by_name (l_poAudioProxyObj, "source-activity", iSrcNum,
                               eStatus);
        DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("-bSendSourceActivity"));
        return TRUE;
    }
    else
    {
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("-bSendSourceActivity poGetAudioProxyObj == NULL"));
        return FALSE;
    }
}

/******************************************************************************
* Function:    bSendOnAllocate
* Description: Function to send or emit on-allocate signal
* Parameters:  source number, sink
* Return:      boolean
******************************************************************************/
gboolean bSendOnAllocate(guchar iSrcNum, GVariant* varAlsa)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("bSendOnAllocate : iSrcNum ="),
            DLT_INT( iSrcNum));
    if(varAlsa == NULL)
    {
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("-bSendOnAllocate, ALSA == NULL"));
        return FALSE;
    }
    AudioManager *l_poAudioProxyObj = poGetAudioProxyObj();
    if(l_poAudioProxyObj != NULL)
    {
        audio_manager_emit_on_allocate (l_poAudioProxyObj,iSrcNum,varAlsa);
        return TRUE;
    }
    else
    {
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("-bSendOnAllocate poGetAudioProxyObj() == NULL"));
        return FALSE;
    }
}

/******************************************************************************
* Function:    bSendDeallocate
* Description: Function to send or emit on-de-allocate signal
* Parameters:  source number
* Return:      boolean
******************************************************************************/
gboolean bSendDeallocate(int iSrcNum)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("bSendDeallocate : iSrcNum = "),
            DLT_INT( iSrcNum));
    AudioManager *l_poAudioProxyObj = poGetAudioProxyObj();
    if(l_poAudioProxyObj != NULL)
    {
        g_signal_emit_by_name (poGetAudioProxyObj(), "on-de-allocate", iSrcNum);
        return TRUE;
    }
    else
    {
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("-bSendDeallocate poGetAudioProxyObj() == NULL"));
        return FALSE;
    }
}

/******************************************************************************
* Function:    bSendMuteState
* Description: Function to send or emit on mute state signal
* Parameters:  source number
* Return:      boolean
******************************************************************************/
gboolean bSendMuteState(int iSrcNum, ApAudioMuteStatus enMuteState)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("bSendMuteState : iSrcNum ="),
            DLT_INT( iSrcNum));
    AudioManager *l_poAudioProxyObj = poGetAudioProxyObj();
    if(l_poAudioProxyObj != NULL)
    {
        g_signal_emit_by_name (poGetAudioProxyObj(), "on-mute-state",
                               iSrcNum, enMuteState);
        return TRUE;
    }
    else
    {
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("-bSendMuteState poGetAudioProxyObj() == NULL"));
        return FALSE;
    }
}

/******************************************************************************
* Function:    handle_source_activity_result_method
* Description: Function to send source activity result
* Parameters:  source number and source status
* Return:      boolean
******************************************************************************/
gboolean
handle_source_activity_result_method(AudioManager *object,
                                     GDBusMethodInvocation *invocation,
                                     guchar arg_Source,
                                     guint arg_Status,
                                     gpointer user_data)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("Source is :"),
            DLT_UINT8(arg_Source),DLT_STRING("Status is"),DLT_UINT(arg_Status));

    if(bRequestSourceActivityResult(arg_Source,arg_Status))
    {
        audio_manager_complete_source_activity_result(object,invocation);
        return TRUE;
    }
    else
    {
        g_dbus_method_invocation_return_error (invocation,G_DBUS_ERROR,
                                               AP_AUDIO_KFAIL,
                                               "Source activity result failed");
        return FALSE;
    }
}
#endif//VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
