/*!
*******************************************************************************
* @file             : FC_Gateway_AudioRouteHandler.cpp
*******************************************************************************
*  - PROJECT:       : ARL handler inside Gateway
*  - SW-COMPONENT   : Gateway - Audio
*  - DESCRIPTION    : Handler for ARL.
*  - COPYRIGHT      : &copy; 2015 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name               |  Version | Modification
* ----------|--------------------|--------------------------|-------------------
*18.11.2015 | DJS5KOR(RBEI/ECO2) |1.0.0 | Handler for ARL
*31.12.2015 | DJS5KOR(RBEI/ECO2) |1.0.1 | Added functions to stop playing music
*26.10.2016 | RHK6KOR            |1.1.0 | Compiler switch for player engine call
*08.02.2017 | BCP9KOR            |1.2.0 | Removed datapool changes, source mapping 
            |                    |      | for new sources.
*******************************************************************************/

/******************************************************************************
| includes:
| 1) system- and project-includes
| 2) needed interfaces from external components
| 3) internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#include "AudioManager_ClientInterface.h"
#include "FC_Gateway_TTSClientHandler.h"
#include "ap-audio-manager-enums.h"
#include "Utility.h"
#include "ECNR_ClientHandler.h"
//Dlt
#include "dlt/dlt.h"
//Dlt Context
DLT_DECLARE_CONTEXT(AGW_Audio); 

fc_gateway_tclAudioRouteHandler* 
fc_gateway_tclAudioRouteHandler::m_pofc_gateway_tclAudioRouteHandler = NULL; 

/******************************************************************************
*
* FUNCTION:    getInstance
*
* DESCRIPTION: Gives the Audio Route handler instance 
*
* PARAMETER:   main app pointer
*
* RETURNVALUE: pointer to Audio Route handler instance
*
******************************************************************************/
fc_gateway_tclAudioRouteHandler* 
fc_gateway_tclAudioRouteHandler::getInstance(procgenericgateway_tclMainApp* 
                                             poMainAppl)
{

    if(m_pofc_gateway_tclAudioRouteHandler == NULL)
    {
        m_pofc_gateway_tclAudioRouteHandler =
                new fc_gateway_tclAudioRouteHandler(poMainAppl);

        DLT_REGISTER_CONTEXT(AGW_Audio,"GWAD","Audio"
                             "context for DLT Logging");
    }

    return m_pofc_gateway_tclAudioRouteHandler;
}

/******************************************************************************
*
* FUNCTION:    fc_gateway_tclAudioRouteHandler
*
* DESCRIPTION: constructor, creates object fc_gateway_tclAudioRouteHandler 
*              and intializes member variables.
*
* PARAMETER:   main app pointer
*
* RETURNVALUE: none
*
******************************************************************************/
fc_gateway_tclAudioRouteHandler::fc_gateway_tclAudioRouteHandler(
        procgenericgateway_tclMainApp* poMainAppl) :
    arl_tclISource(poMainAppl)
	
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("fc_gateway_tclAudioRouteHandler() entered"));

    m_poMainApp = poMainAppl;
}

/******************************************************************************
*
* FUNCTION:    ~fc_gateway_tclAudioRouteHandler
*
* DESCRIPTION: destructor, destroys the object
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
******************************************************************************/
fc_gateway_tclAudioRouteHandler::~fc_gateway_tclAudioRouteHandler()
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("~fc_gateway_tclAudioRouteHandler() entered"));

    DLT_UNREGISTER_CONTEXT(AGW_Audio);
    if(m_pofc_gateway_tclAudioRouteHandler!=NULL)
    {
        delete m_pofc_gateway_tclAudioRouteHandler;
        m_pofc_gateway_tclAudioRouteHandler = NULL;
    }

    m_poMainApp = OSAL_NULL;
}

/******************************************************************************
*
* FUNCTION:    vSetSourceAvailability
*
* DESCRIPTION: Forwards source availability request to ARL.
*
* PARAMETER:   bSourceAvailable -TRUE if source is available otherwise FALSE
*
* RETURNVALUE: none
*
******************************************************************************/
tBool fc_gateway_tclAudioRouteHandler::bSetSourceAvailabilityReq(
        arl_tenSrcAvailability enSrcAvailability,
        arl_tenAvailabilityReason enAvailabilityReason,
        arl_tenSource enSource,tU16 u16SubSource)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("fc_gateway_tclAudioRouteHandler::"\
                       "bSetSourceAvailabilityReq"));

    // Sub source 0 by default
    if (!bSetSourceAvailability(enSrcAvailability, enAvailabilityReason,
                                enSource,u16SubSource))
    {
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("bSetSourceAvailability() failed"));

        return false;
    }

    return true;
}

/******************************************************************************
*
* FUNCTION:    bOnAllocate
*
* DESCRIPTION: Channel allocation callback from ARL.
*
* PARAMETER:   enSrcNum:               (I)   source number
*              rfcoAllocRoute:         (I)   reference to Allocate route result
*
* RETURNVALUE: TRUE
*
******************************************************************************/
tBool
fc_gateway_tclAudioRouteHandler::bOnAllocate(arl_tenSource enSrcNum,
                                             const arl_tAllocRouteResult&
                                             rfcoAllocRoute)
{
	unsigned int iInputDevIndexSize = static_cast<unsigned int>(rfcoAllocRoute.listInputDev.strALSADev.size());
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("fc_gateway_tclAudioRouteHandler::bOnAllocate entered"));

    (tVoid) enSrcNum; //lint fix

    if(enSrcNum == TTS_AUDIO_SOURCE)
    {
        return TRUE;
    }
    // get the alsa device in case of audiostreaming
	GVariantBuilder alsa_builder;
    g_variant_builder_init (&alsa_builder, G_VARIANT_TYPE ("a(ys)"));
	
	for(unsigned int iInputDevIndex = 0; 
	    iInputDevIndex < iInputDevIndexSize; 
		iInputDevIndex++)
	{
		g_variant_builder_add (
		&alsa_builder,"(ys)",AP_AUDIO_INPUT,
		rfcoAllocRoute.listInputDev.strALSADev[iInputDevIndex].szValue);
		                        
	}
	
	
	for(unsigned int iOutputDevIndex = 0; 
	    iOutputDevIndex < static_cast<unsigned int>(rfcoAllocRoute.listOutputDev.strALSADev.size()); iOutputDevIndex++)
	{
		g_variant_builder_add (
		&alsa_builder,"(ys)",AP_AUDIO_OUTPUT,
		rfcoAllocRoute.listOutputDev.strALSADev[iOutputDevIndex].szValue);
	}
	
	GVariant *varALSAdev = g_variant_builder_end(&alsa_builder);
    GVariant *varALSAdevRef = g_variant_ref_sink(varALSAdev);


    ApAudioSource enApSrcNum = AP_AUDIO_SRC_NONE;
    if(bGetGatewayAudioSrcNumber(enSrcNum,enApSrcNum))
    {
        bSendOnAllocate(enApSrcNum,varALSAdevRef);
    }
    else
    {
        return FALSE;
    }
	
	if(enApSrcNum == AP_AUDIO_SRC_SDS)
	{
		DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("Calling ECNR init"));
		vInitializeECNR();
	}
	
	g_variant_unref(varALSAdevRef);
	varALSAdevRef = NULL;
    return TRUE;
}

/******************************************************************************
*
* FUNCTION:    fc_gateway_tclAudioRouteHandler::bOnDeAllocate
*
* DESCRIPTION: Deallocate notification from ARL.
*
* PARAMETER:   enSrcNum:               (I)   source number
*
* RETURNVALUE: TRUE
******************************************************************************/
tBool fc_gateway_tclAudioRouteHandler::bOnDeAllocate(arl_tenSource enSrcNum)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("+fc_gateway_tclAudioRouteHandler::bOnDeAllocate"));

    (tVoid) enSrcNum; //lint fix

    if(enSrcNum == TTS_AUDIO_SOURCE)
    {
        return TRUE;
    }

    ApAudioSource enApSrcNum = AP_AUDIO_SRC_NONE;
    if(bGetGatewayAudioSrcNumber(enSrcNum,enApSrcNum))
    {
        bSendDeallocate(enApSrcNum);
    }
    else
    {
        return FALSE;
    }
	
	if(enApSrcNum == AP_AUDIO_SRC_SDS)
	{
		DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("Calling ECNR uninitialze"));
		vUnitializeECNR();
	}
	
    return TRUE;
}

/******************************************************************************
*
* FUNCTION:    fc_gateway_tclAudioRouteHandler::bOnSrcActivity
*
* DESCRIPTION: Source activity notification from audio routing library.
*
* PARAMETER:   enSrcNum:               (I)   source number
*              rfcoSrcActivity:        (I)   source activity
*
* RETURNVALUE: TRUE
*
******************************************************************************/
tBool
fc_gateway_tclAudioRouteHandler::bOnSrcActivity(arl_tenSource enSrcNum,
                                                const arl_tSrcActivity&
                                                rfcoSrcActivity)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("fc_gateway_tclAudioRouteHandler::bOnSrcActivity"\
                       "entered Src No:"),DLT_UINT(enSrcNum),
            DLT_STRING("Activity"),DLT_UINT(rfcoSrcActivity.enType));

    ApAudioSource enApSrcNum = AP_AUDIO_SRC_NONE;
    switch (rfcoSrcActivity.enType)
    {
    case ARL_EN_ISRC_ACT_ON:
    {
        DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("ARL_EN_ISRC_ACT_ON"));

        if(enSrcNum == TTS_AUDIO_SOURCE)
        {
            FC_Gateway_TTSClientHandler *poTTSClientHandlerObj =
                    FC_Gateway_TTSClientHandler::getInstance(m_poMainApp);
            poTTSClientHandlerObj->bPrepareVoiceOutput_MethodStart();
            bAudioSourceActivityResult(enSrcNum,AP_AUDIO_SRC_ACT_ON);
        }
        else
        {
            if(bGetGatewayAudioSrcNumber(enSrcNum,enApSrcNum))
            {
                bSendSourceActivity(enApSrcNum, AP_AUDIO_SRC_ACT_ON);
            }
            else
            {
                return FALSE;
            }
			
	        if(enApSrcNum == AP_AUDIO_SRC_SDS)
	        {
				DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                        DLT_STRING("Calling Start ECNR"));
		        vStartECNR();
	        }			
			
        }
    }

        break;

    case ARL_EN_ISRC_ACT_OFF:
    {
        DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("ARL_EN_ISRC_ACT_OFF"));

        if(enSrcNum == TTS_AUDIO_SOURCE)
        {
            DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                    DLT_STRING("TTS: ARL_EN_ISRC_ACT_OFF"));
            bAudioSourceActivityResult(enSrcNum,AP_AUDIO_SRC_ACT_OFF);
        }
        else
        {

            if(bGetGatewayAudioSrcNumber(enSrcNum,enApSrcNum))
            {
                bSendSourceActivity(enApSrcNum, AP_AUDIO_SRC_ACT_OFF);
            }
            else
            {
                return FALSE;
            }
			
			if(enApSrcNum == AP_AUDIO_SRC_SDS)
	        {
				DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                        DLT_STRING("Calling Stop ECNR"));
		        vStopECNR();
	        }	
			
        }
    }
        break;

    case ARL_EN_ISRC_ACT_PAUSE:
    {
        DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("ARL_EN_ISRC_ACT_PAUSE"));
        if(enSrcNum == TTS_AUDIO_SOURCE)
        {
            DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                    DLT_STRING("TTS :ARL_EN_ISRC_ACT_PAUSE"));

            bAudioSourceActivityResult(enSrcNum,AP_AUDIO_SRC_ACT_PAUSE);
        }
        else
        {
            if(bGetGatewayAudioSrcNumber(enSrcNum,enApSrcNum))
            {
                bSendSourceActivity(enApSrcNum, AP_AUDIO_SRC_ACT_PAUSE);
            }
            else
            {
                return FALSE;
            }
        }

    }
        break;
    }
    return TRUE;
}

/******************************************************************************
*
* FUNCTION:    fc_gateway_tclAudioRouteHandler::bOnMuteState
*
* DESCRIPTION: Callback for mute / demute request.
*
* PARAMETER:   enSrcNum:               (I)   source number
*              rfcoSrcActivity:        (I)   source activity
*
* RETURNVALUE: TRUE
*
******************************************************************************/
tBool
fc_gateway_tclAudioRouteHandler::bOnMuteState(arl_tenSource enSrcNum,
                                              arl_tenMuteState enMuteState)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("fc_gateway_tclAudioRouteHandler::bOnMuteState"\
                       "entered source num:"),
            DLT_INT(enSrcNum),DLT_STRING("MuteState"),DLT_INT(enMuteState));

    (tVoid) enSrcNum; //lint fix
    (tVoid) enMuteState; //lint fix

    ApAudioSource enApSrcNum = AP_AUDIO_SRC_NONE;

    if(bGetGatewayAudioSrcNumber(enSrcNum,enApSrcNum))
    {
        if(enMuteState >= 0 && enMuteState <= 2)
        {
            bSendMuteState(enApSrcNum, (ApAudioMuteStatus)enMuteState);
        }
        else
        {
            return FALSE;
        }
        
    }
    else
    {
        return FALSE;
    }
    return TRUE;
}

/******************************************************************************
*
* FUNCTION:    fc_gateway_tclAudioRouteHandler::bGetArlSrcNumber
*
* DESCRIPTION: Gets arl source number from gateway source number
*
* PARAMETER:   enApSrcNum:               (I)   source number of gateway
*              enArlSrcNum:              (I)   source number of arl
*
* RETURNVALUE: TRUE/FALSE
*
******************************************************************************/
bool
fc_gateway_tclAudioRouteHandler::bGetArlSrcNumber(ApAudioSource enApSrcNum,
                                                  arl_tenSource& enArlSrcNum)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("+fc_gateway_tclAudioRouteHandler::GetArlSrcNumber"),
            DLT_INT(enApSrcNum));

    bool bRet = FALSE;
    switch(enApSrcNum)
    {
    case AP_AUDIO_SRC_MAIN_1:
        enArlSrcNum = ARL_SRC_WEB_ENT;
        bRet = TRUE;
        break;

    case AP_AUDIO_SRC_MIXED_1:
        enArlSrcNum = ARL_SRC_WEB_MIX;
        bRet = TRUE;
        break;

    case AP_AUDIO_SRC_INTR_1:
        enArlSrcNum = ARL_SRC_WEB_INFO_USR;
        bRet = TRUE;
        break;

    case AP_AUDIO_SRC_INTR_2:
        enArlSrcNum = ARL_SRC_WEB_INFO_SYSTEM;
        bRet = TRUE;
        break;

    case AP_AUDIO_SRC_SDS:
        enArlSrcNum =  ARL_SRC_WEB_SDS;
        bRet = TRUE;
        break;
#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
	case AP_AUDIO_SRC_EXT_NAV:
            enArlSrcNum = ARL_SRC_EXT_NAV;
            bRet = TRUE;
        break;
#endif
    default:
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("The srcnumber is not supported from ARL :"),
                DLT_INT(enApSrcNum));

        bRet = FALSE;
    }

    DLT_LOG(AGW_Audio,DLT_LOG_INFO,
            DLT_STRING("fc_gateway_tclAudioRouteHandler::GetArlSrcNumber"\
                       "left with enArlSrcNum "),DLT_INT(enArlSrcNum));
    return bRet;
}

/******************************************************************************
*
* FUNCTION:    fc_gateway_tclAudioRouteHandler::bGetGatewayAudioSrcNumber
*
* DESCRIPTION: Gets gateway source number from arl source number
*
* PARAMETER:   enArlSrcNum:              (I)   source number of arl
               enApSrcNum:               (I)   source number of gateway
*              
*
* RETURNVALUE: TRUE/FALSE
*
******************************************************************************/
bool
fc_gateway_tclAudioRouteHandler::bGetGatewayAudioSrcNumber(
        arl_tenSource enArlSrcNUm,
        ApAudioSource& enApSrcNum)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("Arl_tenSource "),
            DLT_INT(enArlSrcNUm));

    bool bRet = FALSE;
    switch(enArlSrcNUm)
    {
    case ARL_SRC_WEB_ENT:
        enApSrcNum = AP_AUDIO_SRC_MAIN_1;
        bRet = TRUE;
        break;

    case ARL_SRC_WEB_MIX:
        enApSrcNum = AP_AUDIO_SRC_MIXED_1;
        bRet = TRUE;
        break;

    case ARL_SRC_WEB_INFO_USR:
        enApSrcNum = AP_AUDIO_SRC_INTR_1;
        bRet = TRUE;
        break;

    case ARL_SRC_WEB_INFO_SYSTEM:
        enApSrcNum = AP_AUDIO_SRC_INTR_2;
        bRet = TRUE;
        break;

    case  ARL_SRC_WEB_SDS:
        enApSrcNum = AP_AUDIO_SRC_SDS;
		bRet = TRUE;
        break;
		
#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
	case ARL_SRC_EXT_NAV:
        enApSrcNum = AP_AUDIO_SRC_EXT_NAV;
        bRet = TRUE;
        break;
#endif
    default:
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("The ARL source number doesnot belong to Gateway :"),
                DLT_INT(enApSrcNum));

        bRet = FALSE;
    }

    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING(" enApSrcNum: "),
            DLT_INT(enApSrcNum));

    return bRet;
}

/******************************************************************************
*
* FUNCTION:    bAudioRouteRequest
*
* DESCRIPTION: function for triggering bSetAudioRouteRequest call to ARL.
*
* PARAMETER:   enSrcNum:               (I)   source number
*
* RETURNVALUE: TRUE/FALSE
*
******************************************************************************/
tBool 
fc_gateway_tclAudioRouteHandler::bAudioRouteRequest(arl_tenSource enSrcNum)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("bAudioRouteRequest source "),
            DLT_INT( enSrcNum));

    (tVoid) enSrcNum;

    /*Sub Source ID(Param 4) is 0 which is the default value.*/

    if(enSrcNum == TTS_AUDIO_SOURCE)
    {
        bSetSourceAvailability(ARL_EN_SRC_PLAYABLE, ARL_EN_REASON_SAMEMEDIA,
                               enSrcNum);
    }

    if (FALSE == bSetAudioRouteRequest(enSrcNum, ARL_EN_ISRC_ACT_ON))
    {
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("Audio Route(ON) request posting failed!!"));
        return FALSE;
    }
    else
    {
        DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                DLT_STRING("Audio Route(ON) request posting SUCCESS !!"));
    }

    return TRUE;
}

/******************************************************************************
*
* FUNCTION:    bAudioReleaseRequest
*
* DESCRIPTION: function for triggering bSetAudioRouteRequest call to ARL.
*
* PARAMETER:   enSrcNum:               (I)   source number
*
* RETURNVALUE: TRUE/FALSE
*
******************************************************************************/
tBool
fc_gateway_tclAudioRouteHandler::bAudioReleaseRequest(arl_tenSource enSrcNum)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    (tVoid) enSrcNum;

    if (FALSE == bSetAudioRouteRequest(enSrcNum, ARL_EN_ISRC_ACT_OFF))
    {
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("Audio Route(OFF) request posting failed!!"));
        return FALSE;
    }
    else
    {
        DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                DLT_STRING("Audio Route(OFF) request posting SUCCESS !!"));
    }
    return TRUE;
}

/******************************************************************************
*
* FUNCTION:    bMuteStateRequest
*
* DESCRIPTION: function for triggering bSetMuteState call to ARL.
*
* PARAMETER:   enSrcNum:               (I)   source number
*
* RETURNVALUE: TRUE/FALSE
*
******************************************************************************/
tBool 
fc_gateway_tclAudioRouteHandler::bMuteStateRequest(arl_tenSource enSrcNum, 
                                                   arl_tenMuteAction
                                                   enMuteAction)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("mute action"),
            DLT_INT( enMuteAction));

    (tVoid) enSrcNum;

    if(FALSE == bSetMuteState(enSrcNum,enMuteAction,ARL_EN_MUTE_REASON_USR_ACT))
    {
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("Mute State(ON) request posting failed!!"));
        return FALSE;
    }
    else
    {
        DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                DLT_STRING("Mute State (ON) request posting SUCCESS !!"));
    }
    return TRUE;
}

/******************************************************************************
*
* FUNCTION:    bAudioSourceActivityResult
*
* DESCRIPTION: function for triggering vSourceActivityResult call to ARL.
*
* PARAMETER:   arg_Source:               (I)   source number
*            arg_Status:               (I)   status
*
* RETURNVALUE: TRUE/FALSE
*
******************************************************************************/
tBool
fc_gateway_tclAudioRouteHandler::bAudioSourceActivityResult(arl_tenSource
                                                            enSrcNum,
                                                            ApAudioSourceStatus
                                                            enStatus)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    switch (enStatus)
    {
    case AP_AUDIO_SRC_ACT_ON:
    {
        DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("ARL_EN_ISRC_ACT_ON"));
        vSourceActivityResult(enSrcNum, ARL_EN_ISRC_ACT_ON);
    }

        break;

    case AP_AUDIO_SRC_ACT_OFF:
    {
        DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("ARL_EN_ISRC_ACT_OFF"));
        vSourceActivityResult(enSrcNum,ARL_EN_ISRC_ACT_OFF);

    }
        break;

    case AP_AUDIO_SRC_ACT_PAUSE:
    {
        DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("ARL_EN_ISRC_ACT_PAUSE"));
        vSourceActivityResult(enSrcNum, ARL_EN_ISRC_ACT_PAUSE);
    }
        break;

    default:
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("Default case: return false"));
        return FALSE;

    }
    return TRUE;
}

//Functions declared in the AudioManager_ClientInterface.h

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************
*
* FUNCTION:    bRequest_MuteState
*
* DESCRIPTION: function for triggering bSetMuteState & to be called
*              from D-Bus Server.
*
* PARAMETER:   muteState: (B)   DE-MUTE (1 - Audible) / MUTE (0 - NOT audible)
*
* RETURNVALUE: TRUE/FALSE
*
******************************************************************************/
bool bRequestMuteState(int arg_Source,bool muteState)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    bool bRet = false;
    arl_tenSource enSrcNum = ARL_SRC_NONE;

    fc_gateway_tclAudioRouteHandler* audioObj  =
            procgenericgateway_tclMainApp::
            poGetInstance()->poGetAudioRouteHandler();

    if (NULL != audioObj)
    {
        if(!audioObj->bGetArlSrcNumber((ApAudioSource)arg_Source,enSrcNum))
        {
            return FALSE;
        }

        if(true == muteState) // Mute It
        {
            bRet = audioObj->bMuteStateRequest((arl_tenSource)enSrcNum,
                                               ARL_EN_MUTEACTION_MUTE);
        }
        else if(false == muteState) // Un-Mute It
        {
            bRet = audioObj->bMuteStateRequest((arl_tenSource)enSrcNum,
                                               ARL_EN_MUTEACTION_DEMUTE);
        }

        if(bRet)
        {
            DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                    DLT_STRING("bRequest_MuteState::"\
                               "Audio Route Request SUCCESS!"));
        }
        else
        {
            DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                    DLT_STRING("bRequest_MuteState:: Audio Route Request"\
                               "FAILED!!"));
        }
    }
    return bRet;
}


/***************************************************************************
*
* FUNCTION:    bRequestAudioRoute
*
* DESCRIPTION: function for triggering bSetAudioRouteRequest & to be called 
*              from D-Bus Server.
*
* PARAMETER:   enSrcNum:               (I)   source number
*
* RETURNVALUE: TRUE/FALSE
*
******************************************************************************/
bool bRequestAudioRoute(int arg_Source, bool *srcStatus)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    bool bRet = false;
    arl_tenSource enSrcNum = ARL_SRC_NONE;

    fc_gateway_tclAudioRouteHandler *audioObj =
            procgenericgateway_tclMainApp::
            poGetInstance()->poGetAudioRouteHandler();

    if (NULL != audioObj)
    {

        if(!audioObj->bGetArlSrcNumber((ApAudioSource)arg_Source,enSrcNum))
        {
            *srcStatus = false;
            return FALSE;
        }

        bRet = audioObj->bAudioRouteRequest((arl_tenSource)enSrcNum);

        if(bRet)
        {
            DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                    DLT_STRING("bRequest_RequestAudioRoute:: "\
                               "Audio Route Request SUCCESS!!"));
            *srcStatus = true;
        }
        else
        {
            DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                    DLT_STRING("bRequest_RequestAudioRoute::"\
                               "Audio Route Request FAILED!!"));
            *srcStatus = false;
        }
    }
    else
    {
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("bRequest_RequestAudioRoute:: OBJ NULL!!"));
    }

    return bRet;
}

/******************************************************************************
*
* FUNCTION:    bReleaseAudioRoute
*
* DESCRIPTION: function for triggering releasing audio route & 
*              to be called from D-Bus Server.
*
* PARAMETER:   enSrcNum:               (I)   source number
*
* RETURNVALUE: TRUE/FALSE
*
******************************************************************************/
bool bReleaseAudioRoute(int arg_srcNum)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    bool bRet = false;
    arl_tenSource enSrcNum = ARL_SRC_NONE;

    fc_gateway_tclAudioRouteHandler *audioObj  =
            procgenericgateway_tclMainApp::
            poGetInstance()->poGetAudioRouteHandler();

    if (NULL != audioObj)
    {
        if(!audioObj->bGetArlSrcNumber((ApAudioSource)arg_srcNum,enSrcNum))
        {
            return FALSE;
        }

        bRet = audioObj->bAudioReleaseRequest((arl_tenSource)enSrcNum);

        if(bRet)
        {
            DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                    DLT_STRING("Audio Route Release SUCCESS!!"));
        }
        else
        {
            DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                    DLT_STRING("Audio Route Release FAILED!!"));
        }
    }

    return bRet;
}
/******************************************************************************
*
* FUNCTION:    bRequestSetSourceAvailability
*
* DESCRIPTION: function for triggering set source availablity & 
*              to be called from D-Bus Server.
*
* PARAMETER:   enSrcNum:               (I)   source number
*
* RETURNVALUE: TRUE/FALSE
*
******************************************************************************/
bool bRequestSetSourceAvailability(int arg_Source,bool arg_Availability)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    bool bRet = false;
    arl_tenSource enSrcNum = ARL_SRC_NONE;

    fc_gateway_tclAudioRouteHandler *audioObj =
            procgenericgateway_tclMainApp::
            poGetInstance()->poGetAudioRouteHandler();

    arl_tenSrcAvailability enSrcAvailability;
    arl_tenAvailabilityReason enAvailabilityReason;

    if (NULL != audioObj)
    {
        if(!audioObj->bGetArlSrcNumber((ApAudioSource)arg_Source,enSrcNum))
        {
            return FALSE;
        }

        if(arg_Availability)
        {
            enSrcAvailability = ARL_EN_SRC_PLAYABLE;
            enAvailabilityReason = ARL_EN_REASON_SAMEMEDIA;
        }
        else
        {
            enSrcAvailability = ARL_EN_SRC_NOT_PLAYABLE;
            enAvailabilityReason = ARL_EN_REASON_NOMEDIA;
        }
        bRet = audioObj->bSetSourceAvailabilityReq(enSrcAvailability,
                                                   enAvailabilityReason,
                                                   (arl_tenSource)enSrcNum);

        if(bRet)
        {
            DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                    DLT_STRING("bRequest_SetSourceAvailability::"\
                               "Set source availablity SUCCESS!!"));
        }
        else
        {
            DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                    DLT_STRING("bRequest_SetSourceAvailability::"\
                               "Set source availablity FAILED!!"));
        }
    }
    else
    {
        DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                DLT_STRING("bRequest_SetSourceAvailability:: OBJ NULL!!"));
    }

    return bRet;

}

/******************************************************************************
*
* FUNCTION:    bRequestSourceActivityResult
*
* DESCRIPTION: function for triggering set source activity & 
*              to be called from D-Bus Server.
*
* PARAMETER:   enSrcNum:               (I)   source number
*
* RETURNVALUE: TRUE/FALSE
*
******************************************************************************/
bool bRequestSourceActivityResult(guchar arg_Source,unsigned int arg_Status)
{
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_Audio,DLT_LOG_INFO,DLT_STRING("Source"),
            DLT_UINT8(arg_Source),DLT_STRING("status"),DLT_INT(arg_Status));

    bool bRet = false;
    arl_tenSource enSrcNum = ARL_SRC_NONE;

    fc_gateway_tclAudioRouteHandler *audioObj  =
            procgenericgateway_tclMainApp::
            poGetInstance()->poGetAudioRouteHandler();


    if (NULL != audioObj)
    {
        if(!audioObj->bGetArlSrcNumber((ApAudioSource)arg_Source,enSrcNum))
        {
            return FALSE;
        }

        bRet = audioObj->bAudioSourceActivityResult(
                    enSrcNum,
                    (ApAudioSourceStatus)arg_Status);

        if(bRet)
        {
            DLT_LOG(AGW_Audio,DLT_LOG_INFO,
                    DLT_STRING("Audio SourceActivityResult SUCCESS!!"));
        }
        else
        {
            DLT_LOG(AGW_Audio,DLT_LOG_ERROR,
                    DLT_STRING("Audio SourceActivityResult FAILED!!"));
        }
    }

    return bRet;
}


#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
/******************************************************************************
*
* FUNCTION:    poGetAudioManagerInstance
*
* DESCRIPTION: function to get audio client handler pointer
*
* PARAMETER:   procgenericgateway_tclMainApp     (I)   main app pointer
*
* RETURNVALUE: fc_gateway_tclAudioRouteHandler pointer
*
******************************************************************************/
fc_gateway_tclAudioRouteHandler*
poGetAudioManagerInstance(procgenericgateway_tclMainApp* poMainAppl)
{
    fc_gateway_tclAudioRouteHandler* audioObj  =
            fc_gateway_tclAudioRouteHandler::getInstance(poMainAppl);

    return audioObj;
}
#endif
#endif //#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC

