/*!
*************************************************************************************
* @file             : Timer_Gateway.c
*************************************************************************************
*  - PROJECT:       : Automotive D-Bus server
*  - SW-COMPONENT   : Gateway
*  - DESCRIPTION    : Single Timer loop framework of gateway
*  - COPYRIGHT      : &copy; 2017 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|-----------------------------
* 14.07.2017 | IPD5KOR(RBEI/ECO2) | 1.0.0 | methods for periodic call back handling
*
**************************************************************************************/

#include "Timer_Gateway.h"
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <stdlib.h>
#include "dlt/dlt.h"

//Macro definitions
#define TIMER_POLL_FREQUENCY   500
DLT_IMPORT_CONTEXT(AGW_framework);

//Global variable declarations
static gint           numberOfEvents = 0;
static gboolean       timerRunning = FALSE;
static long int       lastCycleTime = 0;
static Gateway_Event  gatewayEvent[MAX_NUMBER_OF_EVENTS];
static guint          timerID = 0;

/******************************************************************************
 * Function:       vStartTimer
 * Description:    Starts the Timer loop
 * Parameters:     void
 * Return:         void
 *****************************************************************************/
void vStartTimer()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("+"),
            DLT_STRING(__func__));
    if(timerRunning)
    {
        return;
    }
    else
    {
        lastCycleTime = getCurrentTime();
        timerRunning  = TRUE;
        timerID = g_timeout_add(TIMER_POLL_FREQUENCY, runTimer, NULL);
    }
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("-"),
            DLT_STRING(__func__));
}

/******************************************************************************
 * Function:       vStopTimer
 * Description:    Detaches the timer loop from the proccess
 * Parameters:     void
 * Return:         void
 *****************************************************************************/
void vStopTimer()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("+"),
            DLT_STRING(__func__));

    if(timerRunning)
    {
        if(timerID > 0)
        {
            if(g_source_remove(timerID))
            {
                timerRunning = FALSE;
                timerID = 0;
            }
        }
    }
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("-"),
            DLT_STRING(__func__));
}

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
 * Function:       registerPeriodicEvent
 * Description:    Registers periodic call back events
 * Parameters:     event name, interval, call back function
 * Return:         TRUE - if registration success
                   FALSE - if registration fails
 *****************************************************************************/
gboolean registerPeriodicEvent(const gchar* name,
                               long int interval,
                               EventCB callback)
{
    if((name == NULL))
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(__func__),
                DLT_STRING("(): Null callback name"));
        return FALSE;
    }
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Request to Register periodic call back:"),DLT_STRING(name));

    if(strlen(name) > MAX_EVENT_NAME_LENGTH)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(__func__),
                DLT_STRING("():Event Name Max length exceeded"));

        return FALSE;
    }
    if(callback == NULL)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(__func__),
                DLT_STRING("():Null callback function"));
        return FALSE;
    }
    if(interval < MIN_INTERVAL)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(__func__),
                DLT_STRING("():Event interval less than min allowed"));
        return FALSE;
    }
    /* Register */
    gatewayEvent[numberOfEvents].lastExecutedTime = getCurrentTime();
    gatewayEvent[numberOfEvents].interval = interval;
    gatewayEvent[numberOfEvents].call_back= callback;
    strncpy(gatewayEvent[numberOfEvents].event_name, name, strlen(name));
    gatewayEvent[numberOfEvents].event_name[strlen(name)] = '\0';
    numberOfEvents++;

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(__func__),
            DLT_STRING("():Successfuly register event:"),DLT_STRING(name),
            DLT_STRING(" for interval(in ms): "), DLT_INT64(interval));

    if(!timerRunning)
        vStartTimer();
    return TRUE;
}

/******************************************************************************
 * Function:       getCurrentTime
 * Description:    Gives the current time of the day in milli seconds
 * Parameters:     event name, interval, call back function
 * Return:         TRUE - if registration success
                   FALSE - if registration fails
 *****************************************************************************/
long int getCurrentTime()
{
    long int current_time = 0;
    struct timespec st_time;

    clock_gettime(CLOCK_MONOTONIC, &st_time);
    current_time = (st_time.tv_sec*1000)+(st_time.tv_nsec/1000000);
    return current_time;
}

/******************************************************************************
 * Function:       unregisterPeriodicEvent
 * Description:    unregisters a peroidic callback
 * Parameters:     event name
 * Return:         TRUE -  if unregistration success
                   FALSE - if unregistration fails
 *****************************************************************************/
gboolean unregisterPeriodicEvent(const gchar* event_name)
{
    int i = 0;
    if ((event_name == NULL) || (strlen(event_name) <= 0) )
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(__func__),
                DLT_STRING("():Null or empty event name"));
        return FALSE;
    }
    DLT_LOG(AGW_framework, DLT_LOG_INFO, DLT_STRING(
                "Request to unregister periodic event:"), DLT_STRING(event_name));

    for (i = 0; i < numberOfEvents; i++)
    {
        if (strcmp(gatewayEvent[i].event_name, event_name) == 0)
        {
            deleteEvent(i);
            if(numberOfEvents == 0)
            {
                vStopTimer();
            }
            return TRUE;
        }
    }
    return FALSE;
}

/******************************************************************************
 * Function:       deleteEvent
 * Description:    deletes the periodic event from events array list
 * Parameters:     eventId
 * Return:         void
 *****************************************************************************/
void deleteEvent(int eventId)
{
    DLT_LOG(AGW_framework, DLT_LOG_INFO, DLT_STRING("+"),
            DLT_STRING(__func__));
    int i = 0;
    for(i = eventId; i < (numberOfEvents-1); i++)
    {
        gatewayEvent[i] = gatewayEvent[i + 1];
    }
    numberOfEvents--;
    DLT_LOG(AGW_framework, DLT_LOG_INFO, DLT_STRING("-"),
            DLT_STRING(__func__));
}
#ifdef __cplusplus
}
#endif

/******************************************************************************
 * Function:       resetLET
 * Description:    Resets the LastExecutedTimes of all events
 * Parameters:     void
 * Return:         void
 *****************************************************************************/
void resetLET()
{
    long int currentTime = getCurrentTime();
    int i = 0;
    for (i = 0; i < numberOfEvents; i++)
    {
        gatewayEvent[i].lastExecutedTime = currentTime;
    }
}

/******************************************************************************
 * Function:       runTimer
 * Description:    Parses all the registered events and executes the
                   callback of events whose time elapsed
 * Parameters:     gpointer
 * Return:         gboolean
 *****************************************************************************/
gboolean runTimer(gpointer data)
{
    long int currentTime = getCurrentTime();
    int i = 0;

    if ((lastCycleTime != 0) && (fabs(currentTime - lastCycleTime) > 10000))
    {
        DLT_LOG(AGW_framework, DLT_LOG_INFO, DLT_STRING(__func__),
                DLT_STRING("(): Unexpected change in clock time"));
        resetLET();
    }
    lastCycleTime = currentTime;


    for(i=0; i < numberOfEvents; i++)
    {
        if( (gatewayEvent[i].lastExecutedTime == 0) ||
                ((currentTime - gatewayEvent[i].lastExecutedTime)
                 >=  gatewayEvent[i].interval ) )
        {
            gatewayEvent[i].call_back(data);
            gatewayEvent[i].lastExecutedTime = currentTime;
        }
    }
    return TRUE;
}

