/*!
*******************************************************************************
* @file             : LCMDBusHandler.c
*******************************************************************************
*  - PROJECT:       : LCM DBus handler methods implementation
*  - SW-COMPONENT   : Gateway
*  - DESCRIPTION    : LCM handler methods for automotive API's
*  - COPYRIGHT      : &copy; 2017 Robert Bosch Engineering & Business Solutions
*  - Documents      :
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|-----------------------
* 09.01.2017 | RHK6KOR(RBEI/ECO2) | 0.1.0 | Impl. for LCM D-Bus Server handler
* 31.10.2017 | SXN5KOR(RBEI/ECO2) | 0.2.0 | Change in LCM implemenatation
* 28.12.2017 | SXN5KOR(RBEI/ECO2) | 0.2.1 | Code clean up
******************************************************************************/
#include "LCMDBusHandler.h"
#include "Utility.h"
//Dlt
#include "dlt/dlt.h"
DLT_IMPORT_CONTEXT(AGW_LCM);

bool bCheckForClientsResponse();

bool bCheckIfClientAlreadyRegd(gchar* processName);

static guint uRegistration_id = 0;
static bool bCallbackReg = false;
static guint8 u8gatewayClientRegisterID = 0;
static guint iAgwRequestId = 0;
static guint uLifeCycleRequest_timeout = 0;
static guint32 g_Request = NSM_SHUTDOWNTYPE_NOT;

static const gchar lifecycle_introspection_xml[] =
        "<node>"
        "  <interface name='org.genivi.NodeStateManager.LifeCycleConsumer'>"
        "    <method name='LifecycleRequest'>"
        "      <arg type='u' name='Request' direction='in'/>"
        "      <arg type='u' name='RequestId' direction='in'/>"
        "      <arg type='i' name='ErrorCode' direction='out'/>"
        "    <annotation name='org.freedesktop.DBus.GLib.Async' value='true'/>"
        "    </method>"
        "  </interface>"
        "</node>";
static GHashTable*            hShutDownClietHashTable = NULL;
static GDBusProxy*            poLCMProxy = NULL;

static void
LCM_callback_handler (GDBusConnection       *connection,
                      const gchar           *sender,
                      const gchar           *object_path,
                      const gchar           *interface_name,
                      const gchar           *method_name,
                      GVariant              *parameters,
                      GDBusMethodInvocation *invocation,
                      gpointer               user_data);

#if ENABLE_SYSTEMD_NOTIFY_FOR_CONTAINER									   
static bool do_systemd_notify(gchar* l_ProcessName, 
                              watchdogType l_watchdogType);
static gchar*  get_process_name_from_request_id(guint arg_request_id);
static int get_pid_from_process_name(gchar* l_ProcessName);
#endif

static GDBusProxy* get_lcm_proxy();
static gchar*  get_process_name(GVariant *arg_credentials);

static bool bEmitSignalOnRegistration(ShutdownClient* ShutdownClient);


static const GDBusInterfaceVTable life_cycle_interface_vtable =
{
    LCM_callback_handler,
    NULL,
    NULL,
    {0}
};

/******************************************************************************
 * Function:    get_lcm_proxy
 * Description: Gets the LCM proxy pointer object
 * Parameters:  void
 * Return:      GDBusProxy*
 *****************************************************************************/
static GDBusProxy* get_lcm_proxy()
{
    GDBusConnection* poSystemBusConn = poGetSystemBusConnection();

    if(NULL == poSystemBusConn)
        return NULL;

    if(NULL == poLCMProxy)
    {
        GError *l_poError = NULL;
        poLCMProxy = g_dbus_proxy_new_sync (poSystemBusConn,
                                            G_DBUS_PROXY_FLAGS_NONE,
                                            NULL,
                                            LCM_BUS_NAME,
                                            LCM_OBJECT_PATH,
                                            LCM_IFACE_NAME,
                                            NULL,
                                            &l_poError);
        if(!poLCMProxy)
        {
            if(l_poError != NULL)
            {
                DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("failed "
                 "to create LCM proxy object"),DLT_STRING(l_poError->message));

                g_clear_error(&l_poError);
            }
            return NULL;
        }
        else
        {
            DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("created "
                                        "proxy object for LCM"));
        }
    }

    return poLCMProxy;
}

/******************************************************************************
 * Function:    LCM_callback_handler
 * Description: Handles callback method from the LCM
 * Parameters:  connection, sender, object_path, interface_name,
 *              method_name, parameters, invocation, user_data
 * Return:      void
 *****************************************************************************/
static void
LCM_callback_handler (GDBusConnection       *connection,
                      const gchar           *sender,
                      const gchar           *object_path,
                      const gchar           *interface_name,
                      const gchar           *method_name,
                      GVariant              *parameters,
                      GDBusMethodInvocation *invocation,
                      gpointer               user_data)
{
	gint nsm_error_status = 1;
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("In "),DLT_STRING( __FUNCTION__));
    UNUSED(connection);
    UNUSED(sender);
    UNUSED(interface_name);
    UNUSED(user_data);

    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING(
                "LCM_callback_handler, sent by: "),DLT_STRING(object_path));
    if (g_strcmp0 (method_name, "LifecycleRequest") == 0)
    {

        /*parse paramenters and inform to gateway LCM clients */
        guint32 l_Request;
        guint32 l_RequestId;

        g_variant_get (parameters, "(uu)", &l_Request, &l_RequestId);
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING("Request and ID are: "),
                DLT_UINT(l_Request),DLT_STRING("and"),DLT_UINT(l_RequestId));
        iAgwRequestId = l_RequestId;

        /*uLifeCycleRequest_timeout = g_timeout_add (LCM_CLIENT_RESPONSE_TIME_OUT,
                lifeCycleRequestTimeoutCallback, (void*) iAgwRequestId);*/

        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING( "hash table size: "),
                DLT_UINT(g_hash_table_size(hShutDownClietHashTable)));

        GHashTableIter iter;
        gchar* key;
        ShutdownClient* value;

        g_hash_table_iter_init (&iter, hShutDownClietHashTable);
        while (g_hash_table_iter_next (&iter,(gpointer)&key, (gpointer)&value))
        {
            DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                    DLT_STRING("object path: "),DLT_STRING(object_path));
            value->is_registered = TRUE; //for future use

            DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                    DLT_STRING("sender name "),DLT_STRING( value->sender_name));

            gboolean bRet = FALSE;
            GError* l_poError = NULL;
            GDBusConnection* poConn = poGetGDBusConnection();

            g_Request = l_Request;

            DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                    DLT_STRING("sender name "),DLT_STRING( value->sender_name));
            bRet = g_dbus_connection_emit_signal (poConn,
                            value->sender_name,
                            "/com/bosch/AutomotiveProxy/LifeCycleManager",
                            "com.bosch.AutomotiveProxy.LifeCycleManager",
                            "LifecycleRequest",
                            g_variant_new ("(uu)",
                            l_Request,
                            value->request_id),
                            &l_poError);
            if(!bRet)
            {

                if(l_poError != NULL)
                {
                    DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                            DLT_STRING("unable to send signal: "),
                            DLT_STRING( l_poError->message));
                    g_error_free (l_poError);
                    l_poError = NULL;
                }
                else
                {
                    DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                       DLT_STRING("unable to send signal: Error unknown"));
                }
            }
            else
            {
                //sucess case.
                DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                        DLT_STRING( "-LifecycleRequest, emited signal"));
            }


        }
		
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("calling lifeCycleRequestTimeoutCallback withour any delay "));
        lifeCycleRequestTimeoutCallback((void*) iAgwRequestId);

    }
    else
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("Internal error: "
                     "can't handle other than LifecycleRequest"));
    }

    g_dbus_method_invocation_return_value (invocation, g_variant_new ("(i)", nsm_error_status));
	        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING("-LCM_callback_handler returned with NsmErrorStatus_Ok"));
}

/******************************************************************************
 * Function:    handle_register
 * Description: Handles LCM register method
 * Parameters:  object ptr,invocation ptr, arg_credentials and request_mode
 * Return:      gboolean
 *****************************************************************************/
gboolean handle_register(LifeCycleManager* object,
                         GDBusMethodInvocation *invocation,
                         GVariant *arg_credentials,
                         guint arg_request_mode)
{
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("+handle_register mode:"),
            DLT_UINT(arg_request_mode));

    bool bEmitSignal = false;
    gchar* l_ProcessName = get_process_name(arg_credentials);
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("process name: "),
            DLT_STRING( l_ProcessName));

    if(!l_ProcessName)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING("failed to retrive a process name from arg list"));
        g_dbus_method_invocation_return_error (invocation,
                                             G_DBUS_ERROR,
                                                     NULL,
             "failed to retrive a container/domain name");

        return TRUE;
    }
    else{

         //validate if the request came from the same process again.
        if(bCheckIfClientAlreadyRegd(l_ProcessName))
        {
            DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                  DLT_STRING("Remove old hash table entry"));
            g_hash_table_remove(hShutDownClietHashTable,
                                l_ProcessName);
        }

    }


    if(false == bCallbackReg)
    {

        if(bRegisterCallbackForLCMInterface())
        {
            DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                    DLT_STRING("callback successfully registered"));

            bool l_bRet = bRegisterForSystemStatesWithLcm();
            if(!l_bRet)
            {

                DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                        DLT_STRING("failed to register"));
                GDBusConnection* poConnection = poGetSystemBusConnection();
                if(!poConnection)
                {
                    DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                            DLT_STRING("system bus connection pointer is NULL"));
                    g_dbus_connection_unregister_object(poConnection,
                                                        uRegistration_id);
                }
                g_dbus_method_invocation_return_error (invocation,
                                                       G_DBUS_ERROR,
                                                       NULL,
                                                       "failed to register with LCM");
                return TRUE;

            }
            else{
                bCallbackReg = true;
                DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                        DLT_STRING("successfully registered with LCM"));
            }
        }
        else
        {

            DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                    DLT_STRING("failed to register callback on remote obj"));
            bCallbackReg = false;
            g_dbus_method_invocation_return_error (invocation,
                                                   G_DBUS_ERROR,
                                                   NULL,
                                                   "failed to register");
            return TRUE;

        }
    }
    else{
        if(iAgwRequestId)
        {
            bEmitSignal = true;
        }
    }

    //Need to maintain a hash map of all the members.

    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("lcm object pointer: "),
            DLT_UINT32((unsigned int)object));
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("lcm invo pointer: "),
            DLT_UINT32((unsigned int)invocation));

    const gchar *sender_name = g_dbus_method_invocation_get_sender (invocation);

    ShutdownClient* l_ShutdownClient =  new_shutdown_client((char*)sender_name,
                                            arg_request_mode, bEmitSignal);

    g_hash_table_insert( hShutDownClietHashTable,
                         l_ProcessName,
                         l_ShutdownClient);
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("hash table size: "),
            DLT_UINT32(g_hash_table_size(hShutDownClietHashTable)));

    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("-handle_request"));
    life_cycle_manager_complete_register(object,
                                         invocation,
                                         u8gatewayClientRegisterID);
    return TRUE;
}

/******************************************************************************
 * Function:    get_process_name
 * Description: Gets the process name from the handle request credentials
 * Parameters:  GVariant
 * Return:      string pointer gchar* or NULL
 *****************************************************************************/
static gchar*  get_process_name(GVariant *arg_credentials)
{
    //Get the container name from the arguments.
    GVariantIter itr;
    g_variant_iter_init (&itr, arg_credentials);
    GVariant *value;
    gchar *key;
    gchar* process_name;

    while (g_variant_iter_loop (&itr, "{sv}", &key, &value))
    {

        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING(key),
                DLT_STRING("Value:->"));
        if(!g_strcmp0(key, "Label"))
        {
            process_name  =  (gchar*)g_variant_get_string (value, NULL);
            process_name = g_strdup(process_name);
            DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING( "process name: "),
                    DLT_STRING(process_name));

            g_free(key);
            g_variant_unref (value);
            return process_name;
        }
        else {
            DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("Unrecognized Field"));
            return NULL;
        }
    }

    //should not come here.
    return NULL;
}

#if ENABLE_SYSTEMD_NOTIFY_FOR_CONTAINER			
/******************************************************************************
 * Function:    get_process_name_from_request_id
 * Description: Gets the process name from the request id.
 * Parameters:  arg_request_id
 * Return:      bool
 *****************************************************************************/

static gchar* get_process_name_from_request_id(guint arg_request_id)
{

    DLT_LOG(AGW_LCM,DLT_LOG_INFO,
            DLT_STRING( "+get_process_name_from_request_id"));

    GHashTableIter iter;
    gchar* key;
    ShutdownClient* value;
    g_hash_table_iter_init (&iter, hShutDownClietHashTable);
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING( "starting the iterator"));
    while (g_hash_table_iter_next (&iter,(gpointer)&key, (gpointer)&value))
    {
        if(arg_request_id == value->request_id)
        {
            DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                    DLT_STRING( "match found for the request ID"));
            return value->process_name;
        }
    }
    return NULL;
}
#endif

#if ENABLE_SYSTEMD_NOTIFY_FOR_CONTAINER									   
/******************************************************************************
 * Function:     get_pid_from_process_name
 * Description:  Gets the pid from the process name
 * Parameters:   process name
 * Return:       void
 *****************************************************************************/

static int get_pid_from_process_name(gchar* l_ProcessName)
{
    size_t taskNameSize = 1024;
    char*  taskName = calloc(1, taskNameSize);

    DIR* fDirectory = opendir(PROC_DIR_PATH);
    if(!fDirectory)
        return -1;

    struct dirent* entry = NULL;
    int pid = -1;
    while ((entry = readdir(fDirectory)) != 0)
    {
        if (strcmp(entry->d_name, ".") == 0 ||
                strcmp(entry->d_name, "..") == 0)
            continue;

        int res = sscanf(entry->d_name, "%d", &pid);
        if (res == 1)
        {
            char cmdline_file[1024] = {0};
            sprintf(cmdline_file, "%s/%d/cmdline", PROC_DIR_PATH, pid);
            FILE* fCmdLine = fopen(cmdline_file, "r");
            if (getline(&taskName, &taskNameSize, fCmdLine) > 0)
            {
                if (strstr(taskName, l_ProcessName) != 0)
                {
                    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("Process found"),
                                            DLT_STRING( l_ProcessName),
                                            DLT_STRING("pid"),DLT_INT(pid));
                }
            }
            fclose(fCmdLine);
        }
    }

    closedir(fDirectory);
    //free(taskName);
    return pid;
}

/******************************************************************************
 * Function:    do_systemd_notify
 * Description: sends systemd notify on behalf of the container
 * Parameters:  process name, l_watchdogType
 * Return:      void
 *****************************************************************************/

static bool do_systemd_notify(gchar* l_ProcessName,
                              watchdogType l_watchdogType)
{
    int pid = get_pid_from_process_name(l_ProcessName);
    if(!pid)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING(
                    "unable to lookup the pid from the process name"));
        return FALSE;
    }

    int iRet = 0;
    //reset the watch dog of the service file.
    if(l_watchdogType == kWatchDog)
        iRet = sd_pid_notify(pid, 0, "READY=1");
    else if (l_watchdogType == kReady)
        iRet = sd_pid_notify(pid, 0, "WATCHDOG=1");
    else
        return FALSE;

    if(!iRet)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("failed "
            "to reset the watch dog, reason:"),DLT_STRING(strerror(errno)));

        return FALSE;
    }

    return TRUE;
}
#endif

/******************************************************************************
 * Function:    handle_lifecycle_request_complete
 * Description: Handle life cycle complete reponse from client
 * Parameters:  object, invocation, arg_request_id
 * Return:      gboolean
 *****************************************************************************/
gboolean handle_lifecycle_request_complete(LifeCycleManager* object,
                                           GDBusMethodInvocation *invocation,
                                           guint arg_request_id)
{
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("+handle_lifecycle_request_complete"
                              " response from: "),DLT_UINT32(arg_request_id));


    GHashTableIter iter;
    gchar* key;
    ShutdownClient* value;

    g_hash_table_iter_init (&iter, hShutDownClietHashTable);
    while (g_hash_table_iter_next (&iter,(gpointer)&key, (gpointer)&value))
    {
        if(value->request_id ==  arg_request_id)
        {
           value->bReponseRcvd = TRUE;
        }
    }

    //check if all the client have sent response, then update NSM
   /* if(bCheckForClientsResponse())
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("calling lifeCycleRequestTimeoutCallback from"
                                                "handle_lifecycle_request_complete "));
        lifeCycleRequestTimeoutCallback((void*) iAgwRequestId);
    }*/
#if ENABLE_SYSTEMD_NOTIFY_FOR_CONTAINER									   
    //we have to get the process based on the request ID.
    gchar* processName = NULL;
    processName = get_process_name_from_request_id(arg_request_id);
    if(NULL == processName)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING(
                    "Unable to find process for the request id"));
        g_dbus_method_invocation_return_error (invocation,
                                             G_DBUS_ERROR,
                                                     NULL,
                 "no process found for the request id, Internal error");
        return TRUE;
    }

    //use the process name and notify the systemd about its presence.
    bool bRet = do_systemd_notify(processName, kReady);
    if(!bRet)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING(
                    "Unable to notify systemd about readyness"));
        g_dbus_method_invocation_return_error (invocation,
                                             G_DBUS_ERROR,
                                                     NULL,
               "unable to notify systemd, Internal error");
        return TRUE;
    }
#endif
    life_cycle_manager_complete_lifecycle_request_complete(object, invocation);
    return TRUE;
}

/******************************************************************************
 * Function:    new_shutdown_client
 * Description: creates a new shutdown client
 * Parameters:  sender name, ApLifeCycleManagerMode,bEmitSignal
 * Return:      ShutdownClient*
 *****************************************************************************/
ShutdownClient* new_shutdown_client(gchar* sender_name,
                                    ApLifeCycleManagerMode shutdown_mode,
                                    bool bEmitSignal)
{
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING( "+new_shutdown_client: "),
            DLT_UINT64(sizeof(ShutdownClient)));
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING( "new_shutdown_client: sender_name = "),
            DLT_STRING(sender_name));
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING( "new_shutdown_client: shutdown_mode = "),
            DLT_UINT(shutdown_mode));
    ShutdownClient* l_ShutdownClient = malloc(sizeof(ShutdownClient));
    memset((void*)l_ShutdownClient, 0 , sizeof(ShutdownClient));
    l_ShutdownClient->sender_name = g_strdup(sender_name);
    l_ShutdownClient->shutdown_mode = shutdown_mode;
    l_ShutdownClient->request_id = ++u8gatewayClientRegisterID;
    l_ShutdownClient->is_registered = FALSE;
    l_ShutdownClient->bReponseRcvd = FALSE;
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING( "new_shutdown_client: u8gatewayClientRegisterID = "),
            DLT_UINT32(u8gatewayClientRegisterID));

    if(bEmitSignal)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("emit LifeCycleRequest signal"));
        if(bEmitSignalOnRegistration(l_ShutdownClient))
            l_ShutdownClient->is_registered = TRUE;
    }
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("-new_shutdown_client"));
    return l_ShutdownClient;
}



void func_free_hashKey (gpointer key)
{
  DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("+ "),
            DLT_STRING( __FUNCTION__));
  if(key)
  {
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("freeing key: "),DLT_STRING(key));
    g_free(key);
  }
}

void func_free_hashValue (gpointer value)
{
  ShutdownClient* l_client = (ShutdownClient*)value;
  if(l_client)
  {
    if(l_client->sender_name)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("freeing sender name"),DLT_STRING(l_client->sender_name));
        g_free(l_client->sender_name);
    }
    g_free(l_client);
  }
}

/******************************************************************************
 * Function:    do_init_shutdown_client_hash_table
 * Description: Inits the hash table for shutdown clients/
 * Parameters:  void
 * Return:      void
 *****************************************************************************/
void do_init_shutdown_client_hash_table(void)
{
    if(!hShutDownClietHashTable)
    {
        hShutDownClietHashTable =  g_hash_table_new_full(g_str_hash,
                                                         g_str_equal,
                                                         func_free_hashKey,
                                                         func_free_hashValue);
    }
}

/******************************************************************************
 * Function:    destroy_shutdown_client
 * Description: Destroys the shutdown client
 * Parameters:  ShutdownClient *
 * Return:      void
 *****************************************************************************/

void destroy_shutdown_client(ShutdownClient* l_ShutdownClient)
{
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("+destroy_shutdown_client"));
    if(l_ShutdownClient){
        if(l_ShutdownClient->sender_name)
            g_free(l_ShutdownClient->sender_name);
        free(l_ShutdownClient);
    }
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("-destroy_shutdown_client"));
}

/******************************************************************************
 * Function:    destroy_shutdown_client_hash_table
 * Description: Destroys the shutdown clients table.
 * Parameters:  void
 * Return:      void
 *****************************************************************************/
void destroy_shutdown_client_hash_table(void)
{	
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("+ "),
            DLT_STRING( __FUNCTION__));

   if(NULL != hShutDownClietHashTable)
   {
     g_hash_table_destroy(hShutDownClietHashTable);
     hShutDownClietHashTable = NULL;
   }

    GDBusConnection* poConnection = poGetSystemBusConnection();
    if((poConnection!=NULL) && (true== bCallbackReg ))
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING("system bus connection pointer is NULL"));
        g_dbus_connection_unregister_object(poConnection,
                                            uRegistration_id);
    }
    //clear connection pointer
    if(poLCMProxy)
    {
        g_object_unref(poLCMProxy);
        poLCMProxy = NULL;
    }
    u8gatewayClientRegisterID = 0;
    iAgwRequestId = 0;
    uRegistration_id = 0;
    bCallbackReg = false;
    uLifeCycleRequest_timeout = 0;
    g_Request = NSM_SHUTDOWNTYPE_NOT;
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("- "),
            DLT_STRING( __FUNCTION__));
}

/******************************************************************************
 * Function:    lifeCycleRequestTimeoutCallback
 * Description: Callback to send reponse to LCM as acknowlegement that
 *              system system is recieved and is consumed
 * Parameters:  gpointer data
 * Return:      gboolean
 *****************************************************************************/
gboolean lifeCycleRequestTimeoutCallback(gpointer data)
{
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("+ "),
            DLT_STRING( __FUNCTION__));
    /*g_source_remove(uLifeCycleRequest_timeout);
    uLifeCycleRequest_timeout = 0;*/

    gint arg_Status = 1; //NO API support, always --> NsmErrorStatus_Ok
    guint l_request_id = 0;
    l_request_id = (guint)data;
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING(
                "l_request_id is : "),DLT_UINT(l_request_id ));

    if(poLCMProxy)
    {
        GError   *l_poError = NULL;
        GVariant *l_poResult = g_dbus_proxy_call_sync(
                                G_DBUS_PROXY (poLCMProxy),
                               "LifecycleRequestComplete",
                                    g_variant_new ("(uu)",
                                             l_request_id,
                                              arg_Status),
                                   G_DBUS_CALL_FLAGS_NONE,
                                     DEFAULT_DBUS_TIMEOUT,
                                                     NULL,
                                              &l_poError);

        if(l_poError == NULL)
        {
            DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                    DLT_STRING( "success: call to LifecycleRequestComplete"));
        }
        else
        {
            DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                    DLT_STRING("failed to call LCM Method "),
                    DLT_STRING(l_poError->message));

            g_error_free (l_poError);
        }

        if(l_poResult)
            g_variant_unref(l_poResult);

    }
    else
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("poLCMProxy is NULL"));
	
	    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("- "), 
            DLT_STRING( __FUNCTION__));

    return true;
}

/******************************************************************************
 * Function:    bRegisterCallbackForLCMInterface
 * Description: Regsiter callback method for
 *              AGW object and interface exported to LCM
 * Parameters:  void
 * Return:      bool
 *****************************************************************************/
bool bRegisterCallbackForLCMInterface(void)
{
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("+ "),DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("Get system bus connection"));
    //get the connection pointer here.
    GDBusConnection* poConnection = poGetSystemBusConnection();
    if(!poConnection)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING("Bad connection pointer to export object"));
        return FALSE;
    }
    //construct an interface node for the LifeCycleRequest handling
    GDBusNodeInfo *lifecycle_introspection_data = NULL;
    lifecycle_introspection_data = g_dbus_node_info_new_for_xml (
                lifecycle_introspection_xml, NULL);

    if(!lifecycle_introspection_data)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING("Interface node is NULL"));
        return FALSE;
    }

    GError   *l_poError = NULL;
    uRegistration_id = g_dbus_connection_register_object(poConnection,
                                           GATEWAY_SYSTEMBUS_OBJ_PATH,
                          lifecycle_introspection_data->interfaces[0],
                                         &life_cycle_interface_vtable,
                                                NULL,  /* user data */
                                      NULL,  /* user_data_free_func */
                                                          &l_poError);
    if(l_poError == NULL)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING("callback regsitered successfully"));
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING("Registration id"),DLT_UINT(uRegistration_id));
    }
    else
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING(
            "Failed to register callback"),DLT_STRING(l_poError->message));
        g_clear_error (&l_poError);
        return FALSE;
    }

    if(lifecycle_introspection_data)
    {
        g_dbus_node_info_unref(lifecycle_introspection_data);
        lifecycle_introspection_data = NULL;
    }
    return TRUE;
}

/******************************************************************************
 * Function:    bRegisterForSystemStatesWithLcm
 * Description: Register with LCM to get system states update
 * Parameters:  void
 * Return:      bool
 *****************************************************************************/
bool bRegisterForSystemStatesWithLcm(void)
{
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("+ "),
            DLT_STRING( __FUNCTION__));
    GDBusProxy* poLCMProxy = get_lcm_proxy();
    if(!poLCMProxy)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING( "failed to create LCM proxy object"));
        return FALSE;
    }

    GError *l_poError = NULL;
    //Now, we have a proxy object. So, call the LCM register method.
    GVariant *l_poResult =   g_dbus_proxy_call_sync (G_DBUS_PROXY (poLCMProxy),
                                                      "RegisterShutdownClient",
                                                       g_variant_new ("(ssuu)",
                                                       GATEWAY_SYSTEM_BUS_NAME,
                                                    GATEWAY_SYSTEMBUS_OBJ_PATH,
                                   AP_LIFECYCLE_MANAGER_MODE_SHUTDOWNTYPE_FAST,
                                                          DEFAULT_LCM_TIMEOUT),
                                                        G_DBUS_CALL_FLAGS_NONE,
                                                          DEFAULT_DBUS_TIMEOUT,
                                                                          NULL,
                                                                   &l_poError);

    if(l_poError == NULL)
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING( "success: call to RegisterShutdownClient"));
        guint32 l_errCode;
        g_variant_get (l_poResult,
                       "(u)", &l_errCode);
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING("error code received: "),DLT_UINT(l_errCode));
    }
    else
    {
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING("failed to call LCM Method "),
                DLT_STRING(l_poError->message));
        g_clear_error (&l_poError);
        return FALSE;
    }
    if(l_poResult)
        g_variant_unref(l_poResult);

    return TRUE;

}


/******************************************************************************
 * Function:    vEmitSignalOnRegistration
 * Description: Emit LifecycleRequest signal on registartion,
 *              as gateway as a whole is already registered and so callback
 *              will not be triggered again, unless change in system state
 * Parameters:  ShutdownClient*
 * Return:      bool
 *****************************************************************************/
bool bEmitSignalOnRegistration(ShutdownClient* ShutdownClient)
{
    gboolean bRet = FALSE;
    GError* l_poError = NULL;
    GDBusConnection* poConn = poGetGDBusConnection();
    //ShutdownClient* l_client = Null;
    //l_client        ShutdownClient;

    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("Emit signal for client= "),
            DLT_UINT32(ShutdownClient->request_id));


    bRet = g_dbus_connection_emit_signal (poConn,
                    ShutdownClient->sender_name,
                    "/com/bosch/AutomotiveProxy/LifeCycleManager",
                    "com.bosch.AutomotiveProxy.LifeCycleManager",
                    "LifecycleRequest",
                    g_variant_new ("(uu)",
                    NSM_SHUTDOWNTYPE_RUNUP,
                    ShutdownClient->request_id),
                    &l_poError);
    if(!bRet)
    {

        if(l_poError != NULL)
        {
            DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                    DLT_STRING("unable to send signal: "),
                    DLT_STRING( l_poError->message));
            g_error_free (l_poError);
            l_poError = NULL;
        }
        else
        {
            DLT_LOG(AGW_LCM,DLT_LOG_INFO,
               DLT_STRING("unable to send signal: Error unknown"));
        }
        return false;
    }
    else
    {
        //success case.
        DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                DLT_STRING( "-LifecycleRequest, emited signal"));
    }
    return true;
}

/******************************************************************************
 * Function:    bCheckForClientsResponse
 * Description: Function to check if reponses are received from all register
 *              client
 * Parameters:  void
 * Return:      bool
 *****************************************************************************/
bool bCheckForClientsResponse()
{
    GHashTableIter iter;
    gchar* key;
    ShutdownClient* value;

    g_hash_table_iter_init (&iter, hShutDownClietHashTable);
    while (g_hash_table_iter_next (&iter,(gpointer)&key, (gpointer)&value))
    {
        if(FALSE == value->bReponseRcvd)
        {
           return FALSE;
        }
     }
    return TRUE;
}

/******************************************************************************
 * Function:    bCheckIfClientAlreadyRegd
 * Description: Function to validate if client is
 *            	not already registered
 * Parameters:  gchar* processName
 * Return:      bool
 *****************************************************************************/
bool bCheckIfClientAlreadyRegd(gchar* processName)
{
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("+ "),DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING( "hash table size: "),
            DLT_UINT(g_hash_table_size(hShutDownClietHashTable)));
    if(g_hash_table_size(hShutDownClietHashTable))
    {
        GHashTableIter iter;
        gchar* key;
        ShutdownClient* value;
        g_hash_table_iter_init (&iter, hShutDownClietHashTable);
        while (g_hash_table_iter_next (&iter,(gpointer)&key, (gpointer)&value))
        {   
            if (strcmp(key, processName) == 0)
            {
                DLT_LOG(AGW_LCM,DLT_LOG_INFO,
                      DLT_STRING("Process has already registered,"
                                 " so rejecting this request"));
                return true;
            }
        }
    }
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("- "),DLT_STRING( __FUNCTION__));
    return false;

}

/******************************************************************************
 * Function:    iGetLCMState
 * Description: function to get LCm state
 * Parameters:  void
 * Return:      guint32
 *****************************************************************************/
guint32 iGetLCMState()
{
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("+ "),DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_LCM,DLT_LOG_INFO,DLT_STRING("LCM state = "),DLT_UINT32(g_Request));
    return g_Request;
}
