/*!
*******************************************************************************
* @file             : AutomDBusServer.h
*******************************************************************************
*  - PROJECT:       : Automotive D-Bus server
*  - SW-COMPONENT   : Gateway
*  - DESCRIPTION    : D-Bus server for automotive API's
*  - COPYRIGHT      : &copy; 2015 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|-----------------------
* 11.30.2015 | DJS5KOR(RBEI/ECO2) | 1.0.0 | methods for Automotive D-Bus Server
* 11.30.2015 | RHK6KOR(RBEI/ECO2) | 2.0.0 | methods for Automotive D-Bus Server
* 1.31.2017  | SXN5KOR(RBEI/ECO2) | 2.1.0 | Added method to get audioSource proxy
******************************************************************************/
#ifndef _AUTOMOTIVE_DBUS_SERVER_H
#define _AUTOMOTIVE_DBUS_SERVER_H

#include <glib.h>
#include <stdio.h>
#include <stdbool.h>
#include <dlfcn.h>

#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#include "URI_generated.h"
#include "ContactManager_generated.h"
#include "TelephonyManager_generated.h"
#include "PhoneDBusHandler.h"
#include "URI_DBusHandler.h"
#include "bluetooth-manager.h"
#include "TTSDBusHandler.h"
#include "MediaPlayerDbusHandler.h"
#include "AudioManager_generated.h"
#include "AudioDBusHandler.h"


#include "TunerControlDBusHandler.h"
#include "TunerInfoDBusHandler.h"
#endif

#include "GatewayLCMClient.h"
#include "FactoryResetDBusHandler.h"


#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
#include "ConfigurationDbusHandler.h"
#include "Navigation_Route_generated.h"
#include "Geocode_generated.h"
#include "SettingsProvider_Multimedia_generated.h"
#include "PersonalizationClientHandler.h"
#include "SettingsProviderDBusHandler.h"
#include "SpmDBusHandler.h"
#include "VehicleInfoManager_generated.h"
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
#if defined(VARIANT_GEN3ARM_RNAIVI) || defined(VARIANT_COMPILER_ENV_GEN4RCAR)
#include "FcOtaScomoCmpClientHandler.h"
#include "PackageUpdaterDBusHandler.h"
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#include "AudioSource_ClientInterface.h"
#include "DataNotifier_generated.h"
#include "DataNotifierDBusHandler.h"
#include "cea_DiagnosticDbusHandler.h"
#if ENABLE_CSM_IN_GATEWAY
#include "VehicleInfoManager_generated.h"
#include "cea_VIMDBusHandler.h"
#endif //ENABLE_CSM_IN_GATEWAY

#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC 


#ifdef ENABLE_NON_CMD_IN_GATEWAY
#if defined(VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC)
#include "FactoryResetDBusHandler.h"
#endif
#include "ECMDbusHandler.h"
#endif //ENABLE_NON_CMD_IN_GATEWAY


//Gateway Interface and object path names
#define AUTOMOTIVE_PROXY_BUS_NAME "com.bosch.AutomotiveProxy"
#define AGW_NAVIGATION_OBJ_PATH "/com/bosch/AutomotiveProxy/Navigation"
#define AGW_NAVIGATION_INTERFACE_NAME "com.bosch.AutomotiveProxy.Navigation"

/******************************************************************************
 * Function:    vStartAutomServer
 * Description: Starts the Automotive D-Bus server
 * Parameters:  void
 * Return:      void
 *****************************************************************************/

void       vStartAutomServer();

/******************************************************************************
 * Function:    vStopAutomServer
 * Description: Stops the Automotive D-Bus server
 * Parameters:  void
 * Return:      void
 *****************************************************************************/

void       vStopAutomServer();

/******************************************************************************
 * Function:    poGetSystemBusConnection
 * Description: Gets the GDBus connection pointer
 * Parameters:  void
 * Return:      GDBusConnection*
 *****************************************************************************/

GDBusConnection* poGetSystemBusConnection();

/******************************************************************************
 * Function:    poGetGDBusConnection
 * Description: Gets the GDBus session connection pointer
 * Parameters:  void
 * Return:      GDBusConnection*
 *****************************************************************************/

GDBusConnection* poGetGDBusConnection();

/******************************************************************************
 * Function:    poGetSessionBusConnection
 * Description: Gets the Session bus connection pointer
 * Parameters:  void
 * Return:      GDBusConnection*
 *****************************************************************************/
GDBusConnection* poGetSessionBusConnection();

/******************************************************************************
 * Function:    vCleanUpProxyOnShutDown
 * Description: Cleans proxy object on shutdown
 * Parameters:  void
 * Return:      void
 *****************************************************************************/

void vCleanUpProxyOnShutDown();

/******************************************************************************
 * Function:    	bDoComponentsSetup
 * Description:	    Performs all components setup.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/
bool bDoComponentsSetup(GDBusConnection *pConnection);

#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
/******************************************************************************
 * Function:    poGetAudioProxyObj
 * Description: Gets the audio proxy object
 * Parameters:  void
 * Return:      AudioManager* (Instance of proxy object)
 *****************************************************************************/
AudioManager* poGetAudioProxyObj();

/******************************************************************************
 * Function:    	bDoAudioSetup
 * Description:	    Performs the audio skeleton export and registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/

bool bDoAudioSetup(GDBusConnection *pConnection);

/******************************************************************************
 * Function:    poGetURIProxyObj
 * Description: Gets the URI proxy object
 * Parameters:  void
 * Return:      URI*
 *****************************************************************************/
URI* poGetURIProxyObj();

/******************************************************************************
* Function:    poGetMediaPlayerProxyObj
* Description: Gets the MediaPlayer proxy object
* Parameters:  void
* Return:      MediaPlayerPlayer*
*****************************************************************************/
MediaPlayerPlayer* poGetMediaPlayerProxyObj();

/******************************************************************************
* Function:    	poGetContactManager
* Description:	    Gets the contact manager pointer.
* Parameters:		void
* Return:			GDBusProxy
*****************************************************************************/
ContactManager* poGetContactManager();

/******************************************************************************
* Function:    	poGetSourceManager
* Description:	    Gets the source manager pointer.
* Parameters:		void
* Return:			GDBusProxy
*****************************************************************************/
SourceManager* poGetSourceManager();

/******************************************************************************
* Function:    	    poGetSMSManager
* Description:	    Gets the sms manager
* Parameters:		void
* Return:			SMSManager
*****************************************************************************/
SmsManager*  poGetSMSManager();

/******************************************************************************
* Function:    	poGetCallManager
* Description:	    Gets the source manager pointer.
* Parameters:		void
* Return:			CallManager
*****************************************************************************/
CallManager*  poGetCallManager();

/******************************************************************************
 * Function:    	bDoSrcMngrDBusSetup
 * Description:	Performs the source manager skeleton export and
                  registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/
bool bDoSrcMngrDBusSetup();

/******************************************************************************
 * Function:    	bDoUriSetup
 * Description:	    Performs the uri skeleton export and registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/

bool bDoUriSetup(GDBusConnection *pConnection);

/******************************************************************************
 * Function:    poGetTunerControlProxyObj
 * Description: Gets the TunerControl proxy object
 * Parameters:  void
 * Return:      TunerControl*
 *****************************************************************************/
TunerControl* poGetTunerControlProxyObj();

/******************************************************************************
 * Function:    poGetTunerInfoProxyObj
 * Description: Gets the TunerInfo proxy object
 * Parameters:  void
 * Return:      TextToSpeech*
 *****************************************************************************/
TunerInfo* poGetTunerInfoProxyObj();

/******************************************************************************
 * Function:    	bDoTunerSetup
 * Description:	Performs the Tuner skeleton export and registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/

bool bDoTunerSetup(GDBusConnection *pConnection);
#endif


#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
/******************************************************************************
* Function:    poConnectToNavBus
* Description: connect to Nav bus
* Parameters:  void
* Return:      void
*****************************************************************************/
void poConnectToNavBus();

/******************************************************************************
* Function:    get_dbus_connection_cb
* Description: call back that creates a proxy for accessing interface_name
*				    on remote object owned by unique name
* Parameters:  source_object, result pointer, user_data
* Return:      void
*****************************************************************************/
void get_dbus_connection_cb(GObject *source_object,
                            GAsyncResult *res,
                            gpointer user_data);


/******************************************************************************
* Function:    connect_to_nav_bus_cb
* Description: call back for connection to nav dbus
* Parameters:  source_object, result pointer, user_data
* Return:      void
*****************************************************************************/
void connect_to_nav_bus_cb (GObject    *source_object,
                            GAsyncResult *res,
                            gpointer user_data);

/******************************************************************************
 * Function:    	poGetSettingsProviderProxy
 * Description:	Gets the settings provider proxy pointer.
 * Parameters:		void
 * Return:			SettingsProvider
 *****************************************************************************/
SettingsProvider*  poGetSettingsProviderProxy();

/******************************************************************************
 * Function:    	poGetSettingsProviderProxy
 * Description:	Gets the settings provider proxy pointer.
 * Parameters:		void
 * Return:			SettingsProvider
 *****************************************************************************/
SettingsProviderMultimediaState*  poGetSettingsProviderMultimediaStateProxy();

/******************************************************************************
 * Function:    	poGetNaviRouteObjManager
 * Description:	Get object manager pointer.
 * Parameters:		void
 * Return:			GDBusObjectManagerServer
 *****************************************************************************/
GDBusObjectManagerServer* poGetNaviRouteObjManager();

/******************************************************************************
 * Function:    	poGetGeoCodeObjManager
 * Description:	Get object manager pointer.
 * Parameters:		void
 * Return:			GDBusObjectManagerServer
 *****************************************************************************/
GDBusObjectManagerServer* poGetGeoCodeObjManager();

/******************************************************************************
 * Function:    	poGetGeoCodeLocationObjManager
 * Description:	Get object manager pointer.
 * Parameters:		void
 * Return:			GDBusObjectManagerServer
 *****************************************************************************/
GDBusObjectManagerServer* poGetGeoCodeLocationObjManager();

/******************************************************************************
 * Function:    	bDoSettingsProviderDBusSetup
 * Description:	    Performs the settings provider skeleton export
                    and registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/

bool bDoSettingsProviderDBusSetup(GDBusConnection *pConnection);

/******************************************************************************
 * Function:    	poGetCarConfigurationProxy
 * Description:	Gets the CarConfiguration provider proxy pointer.
 * Parameters:		void
 * Return:			CarConfiguration
*****************************************************************************/
CarConfiguration*  poGetCarConfigurationProxy();

/******************************************************************************
 * Function:    	bDoCarConfigurationDBusSetup
 * Description:	    Performs the car configuration skeleton export and
                    registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/
bool bDoCarConfigurationDBusSetup();

#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

#ifdef ENABLE_NON_CMD_IN_GATEWAY
#if defined(VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC)
/******************************************************************************
 * Function:       poGetFactoryResetProxy
 * Description:   Gets the factory reset proxy pointer.
 * Parameters:      void
 * Return:         SettingsProviderFactoryReset
 *****************************************************************************/
SettingsProviderFactoryReset*  poGetFactoryResetProxy();

/******************************************************************************
 * Function:    	bDoFactoryResetDBusSetup
 * Description:	Performs the factory reset skeleton export and registrations.
 * Parameters:		GDBusConnection*
 * Return:			bool
 *****************************************************************************/
bool bDoFactoryResetDBusSetup(GDBusConnection *pConnection);
#endif
#endif //ENABLE_NON_CMD_IN_GATEWAY

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
/******************************************************************************
 * Function:    	vExportNaviInterface
 * Description:	    Performs the navi skeleton export and registrations.
 * Parameters:		void
 * Return:			void
 *****************************************************************************/
void vExportNaviInterface();

/******************************************************************************
 * Function:    	vUnexportNaviInterface
 * Description:	    Performs the navi skeleton unexport
 * Parameters:		void
 * Return:			void
 *****************************************************************************/
void vUnexportNaviInterface();

/******************************************************************************
 * Function:    	vExportNaviInterface
 * Description:	    Performs the navi skeleton export and registrations.
 * Parameters:		void
 * Return:			void
 *****************************************************************************/
void vUnexportE2ENaviInterface();

/******************************************************************************
* Function:    connect_to_e2e_nav_bus_cb
* Description: call back for connection to nav dbus
* Parameters:  source_object, result pointer, user_data
* Return:      void
*****************************************************************************/
void connect_to_e2e_nav_bus_cb (GObject    *source_object,
                                   GAsyncResult *res,
                                   gpointer user_data);

#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

#if ENABLE_CSM_IN_GATEWAY
/******************************************************************************
 * Function:    poGetVIMProxyObj
 * Description: Gets the VIM proxy object
 * Parameters:  void
 * Return:      VIM*
 *****************************************************************************/
VehicleInfoManager* poGetVehicleInfoManagerProxyObj();

#endif //ENABLE_CSM_IN_GATEWAY


#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
/*******************************************************************************
 * Function:    poGetDataNotifierProxyObj
 * Description: Gets the Data notifier proxy object
 * Parameters:  void
 * Return:      DataNotifier*
 ******************************************************************************/
DataNotifier* poGetDataNotifierProxyObj();

/******************************************************************************
 * Function:    	bDoDataNotifierSetup
 * Description:	    Performs the Data Notifier skeleton export and registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/

bool bDoDataNotifierSetup(GDBusConnection *pConnection);

#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC

#ifdef ENABLE_CGROUP_IMPL
void unexportAllInterfaces();
#endif //ENABLE_CGROUP_IMPL

/******************************************************************************
 * Function:    	vCloseDbusConnection
 * Description:	    close dbus connection
 * Parameters:		void
 * Return:			void
 *****************************************************************************/
void vCloseDbusConnection();

#endif //_AUTOMOTIVE_DBUS_SERVER_H
