
// system headers
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>   // implicitly links generic <osal_if.h>

//#define SYSTEM_S_IMPORT_INTERFACE_VECTOR
//#include <stl_pif.h>





#include <vector>

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

// checksum calculation
#define AHL_S_IMPORT_INTERFACE_CRC
#include <ahl_if.h>


// project-specific headers
#include "vd_adr3_Config_Types.h"
#include "vd_adr3_Config_Streamers.h"
#include "vd_adr3_Config_File.h"

#include "vd_adr3Msg_If.h"  // implicitly links "vd_adr3Message_Data.h"

// audio persistent data pool
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_audio_if.h"


// ETG trace
#include "fc_audiomanager_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_ADR3_MAIN
   #include "trcGenProj/Header/vd_adr3_Config_Streamers.cpp.trc.h"
#endif


// =============================================================================
//
//   D a t a p o o l  support (scope = file local)
//

tU16 u16DpAccessId DP_DATAPOOL_ID;


tS32 s32CalcHash(tCString sDpElementName)
{
   // redo the calculation of private dp_tclBaseElement::s32CalcHash function,
   // which complains with a fatal trace if directly used

   tS32 hash = 0;
   for (unsigned int i=0; i < strlen(sDpElementName); i++)
      hash = 5*hash + sDpElementName[i];

   return hash;
}

// =============================================================================
//
//                 F o r m a t t e d  B u f f e r  O u t p u t

/* static */ tCString adr3_tclStreamerBase :: sGetDpElementName(adr3_tenDataFunctionID enDataFID)
{
   // map data pool element name to function-ID

   switch (enDataFID)
   {
      case BAL_DATA_ID:
         return "ADR3_ConfBalance";

      case BASS_DATA_ID:
         return "ADR3_ConfBass";

      case MID_DATA_ID:
         return "ADR3_ConfMidRange";

      case FAD_DATA_ID:
         return "ADR3_ConfFader";

      case LOUD_LOW_DATA_ID:
         return "ADR3_ConfLoudness";

      case TREB_DATA_ID:
         return "ADR3_ConfTreble";

      case SRCOFF_DATA_ID:
         return "ADR3_ConfSrcOffset";

      case LOUD_HIGH_DATA_ID:
         return "ADR3_ConfLoudnessHigh";

      case DELAY_DATA_ID:
         return "ADR3_ConfDelay";

      case CAR_EQU_DATA_ID:
         return "ADR3_ConfEqu";

      case SDDC_DATA_ID:
         return "ADR3_ConfSddc";

      case SDVC_DATA_ID:
         return "ADR3_ConfSdvc";

      case INPUT_DATA_ID:
         return "ADR3_ConfSrcOffsetIn";

      case OUTPUT_DATA_ID:
         return "ADR3_ConfSrcOffsetOut";

      case PREMIX_DATA_ID:
         return "ADR3_ConfPreMix";

      case SDVC_FILTER_DATA_ID:
         return "ADR3_ConfSDFC";

      case OUTPUTPHASE_DATA_ID:
         return "ADR3_ConfOutputPhase";

      case SD_BASS_DATA_ID:
         return "ADR3_Conf_SD_Bass";

      case SD_TREBLE_DATA_ID:
         return "ADR3_Conf_SD_Treble";

      default:  // should never happen
        return "unknown_DpElement";
   }
}

// -----------------------------------------------------------------------------

/* constructor */ adr3_tclStreamerBase :: adr3_tclStreamerBase(tCString _sName, tU8* _pBuffer, tU32 _u32BufSize)
   : sName(_sName)
   , u32BufSize(_u32BufSize)
   , pLocalBuffer(NULL)
   , pBuffer(_pBuffer)
   , pu8Position(4 + pBuffer)  // start behind version
{
   // nothing to do except above initializers
}

// -----------------------------------------------------------------------------

/* constructor */ adr3_tclStreamerBase :: adr3_tclStreamerBase(tCString _sName, tU32 _u32BufSize)
   : sName(_sName)
   , u32BufSize(_u32BufSize)
   , pLocalBuffer(new tU8[u32BufSize])
   , pBuffer(pLocalBuffer)
   , pu8Position(NULL)
{
   if (pBuffer)
      pu8Position = 4U + pBuffer;  // start behind version
}

// -----------------------------------------------------------------------------

/* destructor */ adr3_tclStreamerBase :: ~adr3_tclStreamerBase()
{
   delete[] pLocalBuffer;
   pLocalBuffer = NULL;
   sName = NULL;
   pu8Position = NULL;
}

// -----------------------------------------------------------------------------

tBool adr3_tclStreamerBase :: bSendToADR(tU8 u8FileID)
{
   // send entire file to ADR3 co-processor
   if (NULL == pBuffer)
   {
      ETG_TRACE_ERR(("cannot send file to ADR; invalid buffer: %08x", pBuffer))
      return FALSE;
   }

   trMsgOut oMsgOut;
   //Deliver message to vd_adr3Msg_If::vSendAdr3Msg
   oMsgOut.u8InstId  = VD_ADR3_INST_ID_LS_1;
   oMsgOut.u16FktId  = VD_ADR3_FKT_ID_SOUND_CONFIG_SET;
   oMsgOut.enOpType  = VD_ADR3_OPTYPE_SET;
   oMsgOut.u16MsgLen = (tU16)(1 + u32BufSize);
   oMsgOut.au8MsgData[0] = u8FileID;
   OSAL_pvMemoryCopy(&oMsgOut.au8MsgData[1], pBuffer, u32BufSize);

   switch (u8FileID)
   {
      case BASS_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_BASS;
         break;

      case MID_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_MIDRANGE;
         break;

      case TREB_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_TREBLE;
         break;

      case SD_BASS_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_SD_BASS;
         break;

      case SD_TREBLE_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_SD_TREBLE;
         break;

      case LOUD_LOW_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_LOUDNESS;
         break;

      case BAL_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_BALANCE;
         break;

      case FAD_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_FADER;
         break;

      case CAR_EQU_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_CAREQU;
         break;

      case SRCOFF_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_SOURCEOFFSET;
         break;

      case LOUD_HIGH_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_LOUDNESSHIGH;
         break;

      case DELAY_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_DELAY;
         break;

      case SDVC_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_SDVC;
         break;

      case SDDC_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_SDDC;
         break;

      case SDVC_FILTER_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_SDVC_FILTER;
         break;

      case INPUT_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_SOURCEOFFSETIN;
         break;

      case OUTPUT_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_SOURCEOFFSETOUT;
         break;

      case PREMIX_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_PREMIX;
         break;

      case OUTPUTPHASE_DATA_ID:
         oMsgOut.u8ThreadMessageType = (tU16)SOUNDCONFIG_SET_OUTPUTPHASE;
         break;

      default:
         ETG_TRACE_ERR(("!!! vSendConfigFile for undefined fileID  : 0x%4x; aborting ...", u8FileID));
         return FALSE;
   }

   vd_adr3Msg_If::vSendAdr3Msg(&oMsgOut);

   return TRUE;
}

// -----------------------------------------------------------------------------

tVoid adr3_tclStreamerBase :: vPrintFormatted (tCString sFilePath)
{
   FILE* formatted = fopen(sFilePath, "w");
   if (formatted)
   {
      vPrintFormatted (formatted);
      fclose(formatted);
   }
}

// -----------------------------------------------------------------------------

tVoid  adr3_tclStreamerBase :: vDumpHex (FILE* poDumpFile) const
{
   if ((0 == u32BufSize) || (NULL == pBuffer))
   {
      if (poDumpFile)
         fprintf(poDumpFile, "Buffer empty for %s\n", sName);
      #if defined VARIANT_S_FTR_ENABLE_TRC_GEN
         ETG_TRACE_FATAL(("Buffer empty for %s", sName))
      #endif

      return;
   }

   #if defined VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_FATAL(("--------------------------------------------------------------------------------------------"))
      ETG_TRACE_FATAL(("\tHex-Dump of %d bytes from file %s\t", u32BufSize, sName))
   #endif

   char u8Line[239] = "\t";  // maximum length of ETG trace
   for (tU32 pOffset = 0; pOffset < u32BufSize; ++pOffset)
   {
      if (pOffset + 1 == u32BufSize)
         sprintf(u8Line + strlen(u8Line), " 0x%02x", *(pBuffer + pOffset) );  // last element without comma
      else if (9 == (pOffset % 10))
         sprintf(u8Line + strlen(u8Line), " 0x%02x,\\", *(pBuffer + pOffset) );  // new line after every 10 elements
      else
      {
         sprintf(u8Line + strlen(u8Line), " 0x%02x,", *(pBuffer + pOffset) );
         continue;
      }

      if (poDumpFile)
         fprintf(poDumpFile, "%s\n", u8Line);
      #if defined VARIANT_S_FTR_ENABLE_TRC_GEN
         ETG_TRACE_FATAL(("%s\t", u8Line))
      #endif
      u8Line[1] = 0; // reset line to "\t";
   }
   #if defined VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_FATAL(("--------------------------------------------------------------------------------------------"))
   #endif
}

// -----------------------------------------------------------------------------

tCRCCheckSum adr3_tclStreamerBase :: oReadCRC() const
{
   if ((NULL == pBuffer) || (8 > u32BufSize))
      return 0;

   tU8* pu8CRCPosition = pBuffer + u32BufSize - 4;
   tCRCCheckSum oCRC
      =  *(pu8CRCPosition + 0U)
      + (*(pu8CRCPosition + 1U) << 8U)
      + (*(pu8CRCPosition + 2U) << 16U)
      + (*(pu8CRCPosition + 3U) << 24U);

   return oCRC;
}

// -----------------------------------------------------------------------------

tVoid adr3_tclStreamerBase :: vViewData(const tU8*& rpu8Data, tU32& ru32DataSize) const
{
   rpu8Data = pBuffer;
   ru32DataSize = u32BufSize;
}

// -----------------------------------------------------------------------------

tFileVersion adr3_tclStreamerBase :: oReadVersion() const
{
   // reads first 4 bytes from buffer, but consumes nothing

   tFileVersion v;
   if (pBuffer && (pBuffer + 4 <= pBuffer + u32BufSize))
   {
      v.majorVersion   = *(pBuffer + 2);
      v.year           = *(pBuffer + 1);
      v.month          = *(pBuffer + 0);
      v.day            = *(pBuffer + 3);
   }

   return v;
}

// -----------------------------------------------------------------------------

/* static */ tVoid adr3_tclStreamerBase :: fprintValue(FILE* f, tS16 s16Value)
{
   // prints a 2-byte hex-representation of given value into file

   fprintf(f, " 0x%02x,0x%02x,", (tU8)(s16Value) & 0x00FF, (tU8)((s16Value & 0xFF00) >> 8));
};

// -----------------------------------------------------------------------------

tVoid  adr3_tclStreamerBase :: vPrintFormatted (FILE* f)
{

   adr3_tclParser P(sName, pBuffer, u32BufSize);

   fprintf (f, "/* sound configuration file for %s settings */\\\n", sName);

   // ---   Version   ---
   {
      tFileVersion v = P.oReadVersion();

      fprintf (f, " 0x%02x, 0x%02x, 0x%02x, 0x%02x,  /* version %d, %d-%d-%d */\\\n"
         , v.month, v.year, v.majorVersion, v.day
         , v.majorVersion, 2000 + v.year, v.month, v.day);
   }


   // ---   Common header   ---
   tS16 s16Value;
   for (tU32 i = 0; i < 2; ++i)
   {
      s16Value = P.s16Read();
      fprintValue(f, s16Value);
   }
   fprintf(f, "  /* contentID, functionID=%d */\\\n\\\n",  s16Value);

   // ---   Mapping Section   ---
   fprintf(f, " /* mapping section */ \\\n");

   tS16 s16MappedDataSetCount = 0;
   tS16 s16NodeCount = P.s16Read();
   tS16 s16MappingLineCount = P.s16Read();
   fprintValue(f, s16NodeCount);
   fprintValue(f, s16MappingLineCount);
   fprintf(f, "  /* %d nodes, %d mapping lines */ \\\n\\\n", s16NodeCount, s16MappingLineCount);

   // Mapping Lines
   fprintf(f, " /* data set per mapping line */ \\\n");
   for (tS16 i = 0; i < s16MappingLineCount; ++i)
   {
      s16Value = P.s16Read();
      fprintValue(f, s16Value);
      if (s16Value > s16MappedDataSetCount)
         s16MappedDataSetCount = s16Value;
   }
   fprintf(f, "\\\n\\\n");

   // Node Map
   fprintf(f, "/* node information: */ \\\n/* mapping line, config item, value */ \\\n");
   for (tS16 i = 0; i < s16NodeCount; ++i)
   {
      tS16 line = P.s16Read();
      tS16 column = P.s16Read();
      tS16 item = P.s16Read();

      fprintValue(f, line);
      fprintValue(f, column);
      fprintValue(f, item);
      fprintf(f, "\\\n");
   }
   fprintf(f, "\\\n");

   // mapping line names
   for (tS16 i = 0; i < s16MappedDataSetCount; ++i)
   {
      tS16 s16SetID = P.s16Read();
      tString sMappingLineName(P.sRead());  // creates a new heap object
      if (NULL == sMappingLineName)
         continue;

      tU16 length = (tU16)strlen(sMappingLineName);

      fprintValue(f, s16SetID);
      fprintValue(f, length);
      for (tU16 j = 0; j < length; ++j)
         fprintValue(f, sMappingLineName[j]);
      fprintf(f, "  /* Set %d: %s */ \\\n", s16SetID, sMappingLineName);

      delete sMappingLineName;
   }

   // ---   Data Section   ---
   {
      fprintf(f, "\\\n /* data section */  \\\n");
      tS16 s16ContentID = P.s16Read();
      tS16 s16ConfigFunctionID = P.s16Read();
      tS16 s16Length = P.s16Read();

      fprintValue(f, s16ContentID);
      fprintValue(f, s16ConfigFunctionID);
      fprintValue(f, s16Length);

      fprintf(f, " /* CONF_ID, FunctionID = %d, Length = constant %d */ \\\n"
         , s16ConfigFunctionID
         , s16Length);
   }

   // data set dimensions
   adr3_tDataDimensions oDimensions;
   {
      s16Value = P.s16Read();
      oDimensions.s16Min = P.s16Read();
      oDimensions.s16Max = P.s16Read();
      oDimensions.u16Columns = P.u16Read();
      fprintValue(f, s16Value);
      fprintValue(f, oDimensions.s16Min);
      fprintValue(f, oDimensions.s16Max);
      fprintValue(f, oDimensions.u16Columns);
      fprintf(f, " /* contentID, min step = %d, max step = %d,  %d parameters per entry */\\\n\\\n"
         , oDimensions.s16Min, oDimensions.s16Max, oDimensions.u16Columns);
   }

   // Data Set Header
   s16Value = P.s16Read();
   tS16 s16DataFunctionID = P.s16Read();
   tS16 s16ValuesPerDataSet = P.s16Read();
   tS16 s16DataSetCount = P.s16Read();
   fprintValue(f, s16Value); // contentID
   fprintValue(f, s16DataFunctionID);
   fprintf(f, " /* contentID, functionID = %d */\\\n", s16DataFunctionID);
   fprintValue(f, s16ValuesPerDataSet);
   fprintf(f, " /* total length of set = %d values */ \\\n", s16ValuesPerDataSet);
   fprintValue(f, s16DataSetCount);
   fprintf(f, " /* %d sets */ \\\n", s16DataSetCount);

   // Data Set Values
   for (tS16 k = 0; k < s16DataSetCount; ++k)
   {
      fprintf(f, "\\\n /* data set %u */ \\\n", k + 1U);
      for (tU16 i = 0; i < oDimensions.u16Rows(); ++i)
      {
         fprintf(f, " /* %4d: */ ", oDimensions.s16Min + i);
         for (tU16 j = 0; j < oDimensions.u16Columns; ++j)
         {
            s16Value = P.s16Read();
            fprintValue(f, s16Value);
         }
         fprintf(f, "\\\n");
      }
   }

   // ---   CRC Checksum   ---
   tCRCCheckSum oCRC = P.oReadCRC();
   fprintf(f, "\\\n  /* CRC */ \\\n 0x%02x, 0x%02x, 0x%02x, 0x%02x \\\n"
      , (tU8)((oCRC & 0x000000FFUL) >> 0)
      , (tU8)((oCRC & 0x0000FF00UL) >> 8)
      , (tU8)((oCRC & 0x00FF0000UL) >> 16)
      , (tU8)((oCRC & 0xFF000000UL) >> 24));
}

// -----------------------------------------------------------------------------

/* static */ tU32 adr3_tclStreamerBase :: u32GetDatapoolElementSize (tCString sDpElementName)
{
   // extract size of datapool element

   dp_tclBaseElement myDpElem(sDpElementName, s32CalcHash(sDpElementName));
   tS32 s32Status = DP_s32GetElement(DP_U32_POOL_ID_AUDIOMANAGERDP, &myDpElem, u16DpAccessId);
   if (DP_S32_NO_ERR == s32Status)
      return myDpElem.u32GetDataLength();
   else
      return 0;
}

// -----------------------------------------------------------------------------

/* static */ tVoid adr3_tclStreamerBase :: vCopyFromDatapool (tCString sDpElementName
      , tU8 pau8DestinationBuffer[], tU32 u32DestBufSize, tU32& ru32BytesCopied)
{
   ru32BytesCopied = 0U;
   tU32 u32DpElementSize = u32GetDatapoolElementSize(sDpElementName);
   if (0U == u32DpElementSize)
   {
      ETG_TRACE_ERR(("adr3_tclParser::vCopyFromDatapool() Error - %s", sDpElementName))
      return;
   }
   else if (u32DpElementSize > u32DestBufSize)
   {
      ETG_TRACE_ERR(("adr3_tclParser::vCopyFromDatapool() Error - insufficient buffer provided: is %u; should be %u for file %s"
         , u32DestBufSize, u32DpElementSize, sDpElementName))
      return;
   }

   dp_tclBaseElement myDpElem(sDpElementName, s32CalcHash(sDpElementName));
   tS32 s32Status = DP_s32GetElement(DP_U32_POOL_ID_AUDIOMANAGERDP, &myDpElem, u16DpAccessId);

   if (DP_S32_NO_ERR == s32Status)
   {
      tVoid* pvDpData = myDpElem.pvGetData();
      if (pvDpData)
      {
         (tVoid)memcpy (pau8DestinationBuffer, pvDpData, u32DpElementSize);
         ru32BytesCopied = u32DpElementSize;
      }
   }
   else
      ETG_TRACE_ERR(("Failed loading datapool element %s", sDpElementName))
}

// -----------------------------------------------------------------------------

tVoid adr3_tclStreamerBase :: vUpdateDatapoolElement()
{
   dp_tclBaseElement myDpElem(sName, s32CalcHash(sName));
   (tVoid)myDpElem.bFillData((tVoid*)pBuffer, u32BufSize);
   (tVoid)DP_s32SetElement(DP_U32_POOL_ID_AUDIOMANAGERDP, &myDpElem, u16DpAccessId);
}

// =============================================================================
//
//                                   P a r s e r
//
//                     Stream for Reading from ADR3-formatted Buffer
//

/* constructor */ adr3_tclParser :: adr3_tclParser (tCString _sName, const tU8* _pu8Start, size_t _u32Size)
: adr3_tclStreamerBase (_sName, (tU32)_u32Size)
{
   // copy to get rid of "const" constraint
   if (pBuffer && _pu8Start)
      (tVoid)memcpy(pBuffer, _pu8Start, _u32Size);
}

// -----------------------------------------------------------------------------

/* constructor */ adr3_tclParser :: adr3_tclParser (adr3_tenDataFunctionID enDataFID)
   : adr3_tclStreamerBase (sGetDpElementName(enDataFID), u32GetDatapoolElementSize(sGetDpElementName(enDataFID)))
{
   // reads from datapool element

   dp_tclBaseElement myDpElem(sName, s32CalcHash(sName));
   tS32 s32Status = DP_s32GetElement(DP_U32_POOL_ID_AUDIOMANAGERDP, &myDpElem, u16DpAccessId);

   if (DP_S32_NO_ERR == s32Status)
   {
      tVoid* pvDpData = myDpElem.pvGetData();
      if (pBuffer && pvDpData)
         (tVoid)memcpy(pBuffer, pvDpData, u32BufSize);
   }
   #if defined VARIANT_S_FTR_ENABLE_TRC_GEN
   else
      ETG_TRACE_ERR(("Failed loading datapool element %s", sName))
   #endif
}

// -----------------------------------------------------------------------------

tU8  adr3_tclParser :: u8Read ()
{
   // consumes 2 bytes anyway, but assigns only LSB to given variable
   tU8 u8Value = 0;
   if (pu8Position && bCheckSpace(2))
   {
      u8Value = *pu8Position;
      pu8Position += 2;
   }

   return u8Value;
}

// -----------------------------------------------------------------------------

tS16  adr3_tclParser :: s16Read ()
{
   // consumes 2 bytes, LSB first
   tS16 s16Value = (tS16)0;
   if (pu8Position && bCheckSpace(2))
   {
      s16Value = (tS16)*pu8Position;                        // LSB
      s16Value = (tS16)(s16Value + (*(pu8Position + 1) << 8));  // MSB
      pu8Position += 2;
   }
   return s16Value;
}

// -----------------------------------------------------------------------------

tString  adr3_tclParser :: sRead ()
{
   // determine length and allocate new memory
   tU16 length = u16Read();
   tString sRetVal(new tChar[1 + length]);

   // parse characters and terminate string
   if (sRetVal)
   {
      for (size_t j = 0; j < length; j++)
         sRetVal[j] = u8Read();
      sRetVal[length] = '\0';
   }

   return sRetVal;
}

// =============================================================================
//
//                                 S e r i a l i z e r
//
//                     Stream for Writing into ADR3-formatted Buffer
//

// -----------------------------------------------------------------------------

/* constructor */ adr3_tclSerializer :: adr3_tclSerializer(const adr3_tclFile& roAudioFile)
   : adr3_tclStreamerBase(roAudioFile.sName, roAudioFile.u32SerializedSize())
{
   roAudioFile.vSerialize(*this);
}

// -----------------------------------------------------------------------------

/* constructor */ adr3_tclSerializer :: adr3_tclSerializer (adr3_tenDataFunctionID enDataFID, tU8* pu8Data, tU32 u32Size)
   : adr3_tclStreamerBase(sGetDpElementName(enDataFID), pu8Data, u32Size)
{
   // inherit existing buffer (e. g. from SoundTool)
}

// -----------------------------------------------------------------------------

tVoid adr3_tclSerializer :: vWriteS16 (const tS16 s16Value)
{
   // consumes 2 bytes, LSB first
   if (bCheckSpace(2))
   {
      *pu8Position       = (tU8)s16Value;                // LSB
      *(pu8Position + 1) = (tU8)((tU16)s16Value >> 8u);  // MSB
      pu8Position += 2;
   }
}

// -----------------------------------------------------------------------------

tVoid adr3_tclSerializer :: vWriteString(tCString sString)
{
   // writing operator for strings in files

   tU16 length = 0;
   if (sString)
   {
      length = (tU16)strlen(sString);

      vWriteU16(length);
      for (tU16 j = 0; j < length; j++)
         vWriteU8(sString[j]);
   }
}

// -----------------------------------------------------------------------------

tVoid  adr3_tclSerializer :: vUpdateVersion (const tFileVersion& roVersion)
{
   // reads first 4 bytes from buffer, but consumes nothing

   if (u32BufSize >= 4)
   {
      *(pBuffer + 2) = roVersion.majorVersion;
      *(pBuffer + 1) = roVersion.year;
      *(pBuffer + 0) = roVersion.month;
      *(pBuffer + 3) = roVersion.day;
   }
}

// -----------------------------------------------------------------------------

tVoid adr3_tclSerializer :: vUpdateCRC()
{
   if (8 > u32BufSize)
   {
      #if defined VARIANT_S_FTR_ENABLE_TRC_GEN
         ETG_TRACE_FATAL(("Buffer too small (%d bytes) for %s", u32BufSize, sName))
      #endif

      return;
   }

   #ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
   ETG_TRACE_ERR(("adr3_tclSerializer::vUpdateCRC()-> ahl_calc_crc_32 function call is avoided for 64 bit platform -> temporary solution"))
   #else
   tCRCCheckSum      oCRC;
   ahl_calc_crc_32(pBuffer, u32BufSize - 4, &oCRC);
   tU8* pu8CRCPosition = pBuffer + u32BufSize - 4;
   *(pu8CRCPosition + 0) = (tU8)(oCRC & 0x000000FF);
   *(pu8CRCPosition + 1) = (tU8)((oCRC & 0x0000FF00) >> 8);
   *(pu8CRCPosition + 2) = (tU8)((oCRC & 0x00FF0000) >> 16);
   *(pu8CRCPosition + 3) = (tU8)((oCRC & 0xFF000000) >> 24);
   #endif
}

