//
// Main.cpp   for sub-component VolumeManager
//
//  Created on: Jun 16, 2014
//      Author: Martin Koch, Fa. ESE
//

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h> // implicitly links <osal_if.h>


//#define SYSTEM_S_IMPORT_INTERFACE_VECTOR
//#include <stl_pif.h>


#include <vector>
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "./Types.h"
#include "./VolumeManager.h"
// - - - - - - - - - - - -

//Include public FI interface of this service.
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE   // import fi_tclVisitorMessage
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_SERVICEINFO
#include <midw_fi_if.h>

#include "Configuration/Configuration.h"
#include "./PropertyStore.h"
#include "vd_adr3Msg_If.h"
#include "./CCAListener.h"
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "InternalComponentCommunication/MessageConfig.h"
#include "Volume/Engine/Engine.h"

#include "fc_audiomanager_service_Audio_Function.h"

// PO framework for amplifier command
#include "InternalComponentCommunication/InternalCommunicationAdapter.h"
#include "InternalComponentCommunication/Messages/Volume/ID_Amp_SetVolume.h"


#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/VolumeManager.cpp.trc.h"

#include "Configuration/dBCalculator.h"


namespace VolumeManager
{

   // --------------------------------------------------------------------------

   /* constructor */ VolumeManager:: VolumeManager (fc_audiomanager_tclService_Audio_Function& parent)
      : _parent(parent)
      , _pConfiguration(new Configuration())
      , _pProperties(new PropertyStore(parent))
      , _pEngine(NULL)
      , _pCCAListener(new CCAListener)
   {
      if (NULL == _pConfiguration)
         ETG_TRACE_FATAL(("VolumeManager - Main: E R R O R :  failed to create Configuration database"))

      if (NULL == _pProperties)
         ETG_TRACE_FATAL(("VolumeManager - Main: E R R O R :  failed to create PropertyStore"))

      if (_pConfiguration && _pProperties)
      {
         _pProperties->vInitAll(*_pConfiguration);
         _pEngine = new Engine(*_pProperties, *_pConfiguration, parent.m_pFading);  // implicitly instantiates InternalCommunicationAdapter
      }
      if (NULL == _pEngine)
         ETG_TRACE_FATAL(("VolumeManager - Main: E R R O R :  failed to create Engine"))

      ETG_TRACE_USR1(("VolumeManager - Main: established"))
   }

   // --------------------------------------------------------------------------

   /* destructor */ VolumeManager:: ~VolumeManager ()
   {
      ETG_TRACE_USR1(("VolumeManager - Main: terminating ..."))

      // cleanup, but prevent throwing exceptions on failure
      delete _pCCAListener;
      delete _pEngine;
      delete _pProperties;
      delete _pConfiguration;
   }

   // --------------------------------------------------------------------------

   // support delegating CCA message processing
   tVoid  VolumeManager:: vOnMethodStart (tU16 u16FunctionId, amt_tclServiceData& roMessage)
   {
      switch (u16FunctionId)
      {
         case MIDW_MASCFFI_C_U16_PERFORMONVOLUMELIMIT:
            {
               // empty message; don't need to extract data
               if (_pEngine)
                  _pEngine->vApplyOnVolumeLimits();
            }
            break;

         default:
            if (_pCCAListener)
               _pCCAListener->vOnMethodStart(u16FunctionId, roMessage);
            break;
      }
   }

   // --------------------------------------------------------------------------

//      tVoid  Main:: vOnMethodAbort(tU16 u16FunctionId, amt_tclServiceData* poMessage);

   // --------------------------------------------------------------------------

   tBool  VolumeManager:: bProcessSet (amt_tclServiceData& roMessage, tBool& bPropertyChanged, tU16& u16Error)
   {
      // we manage all related client-notifications ourself, so
      bPropertyChanged = false;

      if ((NULL == _pEngine) || (NULL == _pCCAListener))
      {
         u16Error = MIDW_MASCFFI_C_U16_ERROR_INTERNALFAILURE;
         ETG_TRACE_ERR(("bProcessSet() - E R R O R  : invalid pointer: _pEngine = %p, _pCCAListener = %p"
               , _pEngine, _pCCAListener))
         return false;
      }

      bool retVal = false;
      switch (roMessage.u16GetFunctionID())
      {
         // hijack prior responsibilities of Volume...Handler classes
         case MIDW_MASCFFI_C_U16_VOLUMEMODESTATUS:
         case MIDW_MASCFFI_C_U16_VOLUME:
         case MIDW_MASCFFI_C_U16_VOLUMELOCK:
         case MIDW_MASCFFI_C_U16_ENTONVOLUMESETUP:
         case MIDW_MASCFFI_C_U16_MINIMUMONVOLUMESETUP:
         case MIDW_MASCFFI_C_U16_OVERTEMPAMPLIFIERSTATUS:
            retVal = _pCCAListener->bProcessSet(roMessage, bPropertyChanged, u16Error);
            break;

         case MIDW_MASCFFI_C_U16_VOLUMELIST:
            {
               // get local data structure from incoming message
               midw_mascffi_tclMsgVolumeListSet data;
               fi_tclVisitorMessage(&roMessage).s32GetData(data);
               retVal = _pEngine->bProcessListChange (data.AudVolumeList, data.Resource.enType, u16Error);
               // cleanup
               data.vDestroy();
            }
            break;

         case MIDW_MASCFFI_C_U16_PDCATTENUATION:
            {
               // get local data structure from incoming message
               midw_mascffi_tclMsgPDCAttenuationSet data;
               fi_tclVisitorMessage(&roMessage).s32GetData(data);
               retVal = _pEngine->bProcessPDCCommand (data.PDCAttenuationCommand.enType, u16Error);
               // cleanup
               data.vDestroy();
            }
            break;

         case MIDW_MASCFFI_C_U16_PDCATTENUATIONLEVEL:
            {
               // get local data structure from incoming message
               midw_mascffi_tclMsgPDCAttenuationLevelSet data;
               fi_tclVisitorMessage(&roMessage).s32GetData(data);
               retVal = _pEngine->bProcessPDCLevel (data.PDCAttenutionLevel, u16Error);
               // cleanup
               data.vDestroy();
            }
            break;

         case MIDW_MASCFFI_C_U16_OVERTEMPREDUCTION:
            {
               midw_mascffi_tclMsgOverTempReductionSet data;
               fi_tclVisitorMessage(&roMessage).s32GetData(data);
               tBool downScaling_Active = data.DownScaling_Active;
               tU8 reductionInterval = data.ReductionInterval;
               retVal = _pEngine->bProcessOverTempReduction (downScaling_Active, reductionInterval, u16Error);
               // cleanup
               data.vDestroy();
            }
            break;

         default:
            // something got wrong
            u16Error = MIDW_MASCFFI_C_U16_ERROR_INTERNALFAILURE;
            break;
      }

      return retVal;
   }

   // --------------------------------------------------------------------------

   tBool VolumeManager:: bGetStatusMessage (tU16 u16FunctionId, amt_tclServiceData& roOutMsg, amt_tclServiceData* /* poInMsg */)
   {
      // provide status message for volume-related properties
      // from PropertyStore

      // prerequisites
      if (NULL == _pProperties)
      {
         ETG_TRACE_FATAL(("VolumeManager:: bGetStatusMessage() - E R R O R :  property store invalid  "))
         return FALSE;
      }
      const fi_tclMessageBase& fiMsg = _pProperties->rGetFIMessage(u16FunctionId);
      const fi_tclTypeBase& fiType = fiMsg.corfoGetTypeBase();
      if ((tS32)fi_tclToken::EN_UNKNOWN == fiType.s32GetTypeId())
      {
         ETG_TRACE_FATAL(("VolumeManager:: bGetStatusMessage() - E R R O R :  Unsupported FunctionID 0x%x "
            , u16FunctionId))
         return FALSE;
      }
      else if (u16FunctionId != fiMsg.u16GetFunctionID())
      {
         ETG_TRACE_FATAL(("VolumeManager:: bGetStatusMessage() - E R R O R :  FunctionID does not match: requested 0x%x - found 0x%x"
            , u16FunctionId, fiMsg.u16GetFunctionID()))
         return FALSE;
      }

      // stream to shared memory and hand-over to caller-provided object
      ETG_TRACE_USR4(("VolumeManager:: bGetStatusMessage() - returning %u bytes in message for FunctionID  0x%x "
         , fiType.u32GetSize(MIDW_MASCFFI_C_U16_SERVICE_MAJORVERSION)
         , fiMsg.u16GetFunctionID()))
      return fi_tclVisitorMessage(fiType) .bHandOver(&roOutMsg);
   }

   // --------------------------------------------------------------------------

   void VolumeManager:: vOnLoadSettings()
   {
      if (_pProperties && _pEngine)
         _pProperties->vLoadPersistentData();

      //Get AmpDevice status from ADR3
      vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_AMPLIFIERDEVICE, VD_ADR3_OPTYPE_GET
            , 0, OSAL_NULL, AMPLIFIERDEVICE_ID);//send data to ADR3

   }

   // --------------------------------------------------------------------------

   /* static */ void VolumeManager:: vLaunchAmplifierCommand(const AmpVolumeData& ampData)
   {
      ID_Amp_SetVolume outMsg(ampData);
      InternalCommunicationAdapter::POMessages->DeliverMsg(&outMsg);

      ETG_TRACE_USR2(("VolumeManager - Main:"
         " - launched amplifier volume command for stream = %u, Step = %d"
         " , dB = %d, Ramp (lin/log) = %d/%d, Sink : %d"
         , ampData.m_enStream, ampData.m_VolStep
         , ampData.m_VoldB, ampData.m_RampLin, ampData.m_RampdB, ampData.m_Sink ))
   }

   // --------------------------------------------------------------------------

   void VolumeManager:: vHandleTTFisCommand (tU32 size, const tU8* pcu8Data)
   {
      // TTFis support
      if (NULL == pcu8Data)
      {
         ETG_TRACE_FATAL(("vHandleTTFisCommand() - E R R O R  : invalid TTFis message"))
         return;
      }
      else if (size <= 1)
      {
         ETG_TRACE_FATAL(("vHandleTTFisCommand() - E R R O R  : TTFis message too short"))
         return;
      }

      switch (pcu8Data[1])
      {
         case TRC::OverTempReduction:
         case TRC::enSetVolume:
         case TRC::enSetVolumeMode:
         case TRC::VolumeLock:
         case TRC::PerformOnVolumeLimit:
         case TRC::PDCAttenuation:
         case TRC::PDCAttenuationLevel:
            if (_pEngine)
               _pEngine->vHandleTTFisCommand(size, pcu8Data);
            break;

         default:
            ETG_TRACE_ERR(("vHandleTTFisCommand() - E R R O R  : unhandled TTFis message received: ID = %u"
                  , pcu8Data[1]))
            break;
      }
   }

 // --------------------------------------------------------------------------

       tS16 VolumeManager::s16getDBValue(tU8 u8UserStep, tenStream enStream)
    {
    tS16 retVal = -120;  // muting instead of roaring in case of invalid configuration & property store
    if (NULL == _pConfiguration)
    {
      ETG_TRACE_ERR(("VolumeManager: Invalid Configuration database"))
      return retVal;
    }

    if (NULL == _pProperties)
    {
      ETG_TRACE_ERR(("VolumeManager: Invalid PropertyStore"))
      return retVal;
    }
    const dBCalculator* pCalc = _pConfiguration->pGetDBCalculator(_pProperties->enGetCurrentSource());
  if(pCalc!=NULL)
      retVal = pCalc->s16GetdBValue(u8UserStep,enStream);
    return retVal;
    }
   // --------------------------------------------------------------------------

}   // namespace VolumeManager
