//
//  VolumeManager/Engine/Functions/NavVolumeFilter.cpp
//
//  Created on: Sep 26, 2014
//      Author: Martin Koch, Fa. ESE
//



// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>

#include "Volume/Types.h"
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "./IFunction.h"
#include "./FunctionBase.h"
#include "InternalComponentCommunication/MessageConfig.h"
#include "./NavVolumeFilter.h"
// - - - - - - - - - - - - -


#include "Volume/PropertyStore.h"
#include "Volume/Engine/Stream.h"
#include "Volume/Engine/StreamSet.h"
#include "Volume/Engine/Actions/IAction.h"
#include "Volume/Engine/Actions/ActionBase.h"
#include "Volume/Engine/Actions/VolumeSetAction.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "Volume/Configuration/dBCalculator.h"
#include "Volume/Configuration/Configuration.h"
#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeData.h"

#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/NavVolumeFilter.cpp.trc.h"

#include "fc_audiomanager_service_Audio_Function.h"   // needed for fading ramp



namespace VolumeManager
{

   // -----------------------------------------------------------------------------
   //
   //       class   N a v i g a t i o n - V o l u m e - F i l t e r
   //
   //


   /* constructor */ NavVolumeFilter:: NavVolumeFilter (PropertyStore& properties, const Configuration& config
            , const FunctionConfig& functionCfg, StreamSet& streamSet)
      : FunctionBase(properties, config, functionCfg, streamSet)
      , IF_MessageObserver<PO_MessageConfig::enID>("VolumeManager_NavVolumeFilter")
   {
      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      if (pPO)
         pPO->AddObserver(this, PO_MessageConfig::ID_NotifyMute);
   }

   // -----------------------------------------------------------------------------

   /* virtual destructor */ NavVolumeFilter:: ~NavVolumeFilter ()
   {
      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      if (pPO)
         pPO->RemoveObserver(this, PO_MessageConfig::ID_NotifyMute);
   }

   // -----------------------------------------------------------------------------

   /* virtual */ tenFunctionType NavVolumeFilter:: enGetType ()
   {
      return enNavVolumeFilter;
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void NavVolumeFilter:: vAdjustAmplifierVolume (tenFadingReason enReason, const VolumeData& volumeData, bool& /* useStandardAmplifierCommand */)
   {
      // check applicability
      if (EN_AUDIO_FADING_RES_USR != enReason)
         return;
      tenVolumeMode enMode = _properties.enGetMode();
      bool applicable
         =     (EN_INT_VOLMODE_NAV_ANNOUNCEMENT == enMode)
            || (EN_INT_VOLMODE_SETUP_NAV_VOL == enMode);
      if ( ! applicable)
         return;;

      // look for dependent stream and update its volume
      const dBCalculator* pCalc = _config.pGetDBCalculator(_properties.enGetCurrentSource());
      if (NULL == pCalc)
         return;

      tU8 volume = volumeData.m_Volume;
      tenActions enAction = VolumeSetAction::enGetFadingAction (enReason);
      tenStream enDependentStreamID = pCalc->enGetSecondaryStreamID();
      Ramp secondaryRamp = _streamSet.queryRamp(enAction, enReason, enDependentStreamID);
      Stream* pDependentStream = _streamSet.pGetStream(enDependentStreamID);
       if (pDependentStream)
         pDependentStream->setVolume(*pCalc, volume, secondaryRamp);
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void NavVolumeFilter:: MessageNotification (PO_MessageConfig::enID MsgId)
   {
      // implement IF_MessageObserver
      if (PO_MessageConfig::ID_NotifyMute != MsgId)
      {
         ETG_TRACE_ERR(("NavVolumeFilter::MessageNotification() - unhandled message %u received", MsgId))
         return;
      }

      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      const ID_NotifyMute* pMsg = pPO->QueryMessage<ID_NotifyMute>(MsgId);
      tenStream enStream = pMsg->enStream;
      tenMuteResult enResult = pMsg->enMuteResult;

      ETG_TRACE_USR4(("NavVolumeFilter::MessageNotification() : ID_NotifyMute received for Stream %u with result %u"
            , ETG_CENUM(tenStream, enStream), ETG_CENUM(tenMuteResult, enResult)))

      if (EN_AUDIO_SOURCE_STREAM_MIX1 != enStream)
         return;

//      tenResource enResource = Speaker;
      const dBCalculator* pCalc = _config.pGetDBCalculator(_properties.enGetCurrentSource());
      if (NULL == pCalc)
         return;

      tenStream enID = pCalc->enGetSecondaryStreamID();
      Stream* pDependentStream = _streamSet.pGetStream(enID);
      if (NULL == pDependentStream)
      {
         ETG_TRACE_FATAL(("NavVolumeFilter::SetDependentVolume() - E R R O R :  dependent stream invalid"))
         return;
      }

      ETG_TRACE_USR4(("enGetCurrentSource : %d", pDependentStream->enGetCurrentSource()));

      const SourceConfig* pSecondarySource = _config.pGetSourceConfig(pDependentStream->enGetCurrentSource());
      if (pSecondarySource && affectsVolumeGroup(pSecondarySource->u8GetGroupType()))
      {
         if (EN_AUDIO_MUTE_MUTERESULT_DEMUTE_RUNNING == enResult)
            SetDependentVolume(*pDependentStream, *pCalc);
         else if (EN_AUDIO_MUTE_MUTERESULT_MUTE == enResult)
            ResetDependentVolume(*pDependentStream);
      }
   }

   // -----------------------------------------------------------------------------

   void NavVolumeFilter:: SetDependentVolume (Stream& dependentStream, const dBCalculator& calc) const
   {
      // query current dependent volume from property store
      const SourceConfig* pSourceCfg = _config.pGetSourceConfig(dependentStream.enGetCurrentSource());
      if (pSourceCfg)
      {
         tU8 sourceType = pSourceCfg->u8GetGroupType();
         const tU8* pVolume = _properties.pGetVolume(dependentStream.enGetResource(), sourceType);
         if (NULL == pVolume)
         {
            ETG_TRACE_ERR(("NavVolumeFilter::SetDependentVolume() - E R R O R :  no recent volume found for type %u at resource %u"
            , ETG_CENUM(midw_fi_tcl_e8_Aud_VolumeType::tenType, sourceType)
            , ETG_CENUM(tenResource, _streamSet.enGetResource())))
            return;
         }

         ETG_TRACE_USR4(("NavVolumeFilter::SetDependentVolume() -  setting dependent volume to %u for source %u"
               , *pVolume, ETG_CENUM(tenInternalSource, dependentStream.enGetCurrentSource())))

         // use calculator of active source (passed as parameter
         Ramp ramp = _streamSet.queryRamp(EN_AUDIO_FADING_NOTIFY_SET_NAV_ATTENUATION, EN_AUDIO_FADING_RES_USR, dependentStream.enGetStreamID());
         dependentStream.setVolume(calc, *pVolume, ramp);
      }
      else
      {
         ETG_TRACE_ERR(("NavVolumeFilter::SetDependentVolume() -  E R R O R :  invalid configuration for dependent source %u"
               , ETG_CENUM(tenInternalSource, dependentStream.enGetCurrentSource())))
      }
   }

   // -----------------------------------------------------------------------------

   void NavVolumeFilter:: ResetDependentVolume (Stream& dependentStream) const
   {
      const SourceConfig* pSourceCfg = _config.pGetSourceConfig(dependentStream.enGetCurrentSource());
      tenInternalSource enSourceID = dependentStream.enGetCurrentSource();

      // for reset, use calculator of source in dependent stream
      const dBCalculator* pCalc = _config.pGetDBCalculator(enSourceID);

      if (pSourceCfg && pCalc)
      {
         // query current dependent volume from property store
         tU8 sourceType = _config.pGetSourceConfig(enSourceID)->u8GetGroupType();
         tenResource enResource = _streamSet.enGetResource();
         tU8 Volstep = 0;
         const tU8* pVolume = _properties.pGetVolume(enResource, sourceType);

         if(dependentStream.enGetCurrentSource() == AUD_INT_SRC_FM_TA || dependentStream.enGetCurrentSource() == AUD_INT_SRC_DAB_TA)
         {
      #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_TA_DAN
        const tU8* pEntVolume  = _properties.pGetVolume(dependentStream.enGetResource(), midw_fi_tcl_e8_Aud_VolumeType::FI_EN_VOLUME_ENTERTAINMENT);

        if(pEntVolume)
          Volstep = (tU8)(*pEntVolume + _properties.s8GetDanValue()); //gen4 prio3 warning fix
      #else
        if(pVolume)
          Volstep = *pVolume;
      #endif

         ETG_TRACE_USR4(("SetDependentVolume() -  re-setting dependent volume to %u for source %u"
               , Volstep, ETG_CENUM(tenInternalSource, dependentStream.enGetCurrentSource())))
         }
         else
         {
             if (NULL == pVolume)
             {
               ETG_TRACE_ERR(("NavVolumeFilter::ResetDependentVolume() - E R R O R :  no recent volume found for type %u at resource %u"
               , ETG_CENUM(midw_fi_tcl_e8_Aud_VolumeType::tenType, sourceType)
               , ETG_CENUM(tenResource, _streamSet.enGetResource())))
               return;
             }

            Volstep = *pVolume;
            ETG_TRACE_USR4(("SetDependentVolume() -  re-setting dependent volume to %u for source %u"
               , *pVolume, ETG_CENUM(tenInternalSource, dependentStream.enGetCurrentSource())))
          }

         Ramp ramp = _streamSet.queryRamp(EN_AUDIO_FADING_NOTIFY_CHG_VOL, EN_AUDIO_FADING_RES_SOURCESEL
                                          , dependentStream.enGetStreamID());
         dependentStream.setVolume(*pCalc, Volstep, ramp);

         _streamSet.vSetActiveStream(dependentStream.enGetStreamID());
         const SourceConfig* pSecondarySource = _config.pGetSourceConfig(dependentStream.enGetCurrentSource());
         if(NULL != pSecondarySource)
         {
         _properties.vSetCurrentSource(*pSecondarySource,dependentStream.enGetResource());
         }
         _properties.NotifyAllUpdates();
      }
      else
      {
         ETG_TRACE_ERR(("NavVolumeFilter::SetDependentVolume() -  E R R O R :  invalid configuration for dependent source %u"
               , ETG_CENUM(tenInternalSource, dependentStream.enGetCurrentSource())))
      }
   }

   // -----------------------------------------------------------------------------

}  // namespace VolumeManager


