//
//  VolumeManager/Engine/Functions/BeepVolumeFilter.cpp
//
//  Created on: Sep 26, 2014
//      Author: Martin Koch, Fa. ESE
//



// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>

#include "Volume/Types.h"
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "./IFunction.h"
#include "./FunctionBase.h"
#include "InternalComponentCommunication/MessageConfig.h"
#include "./BeepVolumeFilter.h"
// - - - - - - - - - - - - -


#include "Volume/PropertyStore.h"
#include "Volume/Engine/Stream.h"
#include "Volume/Engine/StreamSet.h"
#include "Volume/Engine/Actions/IAction.h"
#include "Volume/Engine/Actions/ActionBase.h"
#include "Volume/Engine/Actions/VolumeSetAction.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "Volume/Configuration/dBCalculator.h"
#include "Volume/Configuration/Configuration.h"
#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeData.h"

#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/BeepVolumeFilter.cpp.trc.h"

#include "fc_audiomanager_service_Audio_Function.h"   // needed for fading ramp



namespace VolumeManager
{

   // -----------------------------------------------------------------------------
   //
   //               class   B e e p - V o l u m e - F i l t e r
   //
   //


   /* constructor */ BeepVolumeFilter:: BeepVolumeFilter (PropertyStore& properties, const Configuration& config
            , const FunctionConfig& functionCfg, StreamSet& streamSet)
      : FunctionBase(properties, config, functionCfg, streamSet)
      , IF_MessageObserver<PO_MessageConfig::enID>("VolumeManager_BeepVolumeFilter")
      , _isMainAudioMuted(false)
      , _enBeepStreamID(EN_AUDIO_SOURCE_STREAM_DEFAULT)
      , _u8MinBeepVolume(0)
      , _u8MaxBeepVolume(0)
   {
      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      if (pPO)
         pPO->AddObserver(this, PO_MessageConfig::ID_NotifyMute);

      // determine stream dedicated to beeping
      {
         const GroupConfig* pGroups;
         size_t              numberOfGroups;
        _config.vGetSourceGroupConfigs(pGroups, numberOfGroups);

         if (NULL == pGroups)
            return;

         const GroupConfig* pBeepGrp = NULL;
         unsigned i = 0;
         for (   ; i < numberOfGroups; ++i)
            if (midw_fi_tcl_e8_Aud_VolumeType::FI_EN_VOLUME_BEEP == pGroups[i].typeId)
            {
               pBeepGrp = &pGroups[i];
               break;
            }

         if (pBeepGrp)
         {
            _u8MinBeepVolume = pBeepGrp->constraints.minStep;
            _u8MaxBeepVolume = pBeepGrp->constraints.maxStep;
            for (unsigned j = 0; j < pBeepGrp->sinkCount; ++j)
               if (pBeepGrp->ppSinks[j] && pBeepGrp->ppSinks[j]->pStreamSet && (streamSet.enGetResource() == pBeepGrp->ppSinks[j]->pStreamSet->enResource))
               {
                  _enBeepStreamID = pBeepGrp->ppSinks[j]->enStreamID;
                  break;
               }
         }
      }

   }

   // -----------------------------------------------------------------------------

   /* virtual destructor */ BeepVolumeFilter:: ~BeepVolumeFilter ()
   {
      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      if (pPO)
         pPO->RemoveObserver(this, PO_MessageConfig::ID_NotifyMute);
   }

   // -----------------------------------------------------------------------------

   /* virtual */ tenFunctionType BeepVolumeFilter:: enGetType ()
   {
      return enBeepVolumeFilter;
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void BeepVolumeFilter:: vAdjustAmplifierVolume (tenFadingReason enReason, const VolumeData& volumeData, bool& useStandardAmplifierCommand)
   {
     ETG_TRACE_USR4(("BeepVolumeFilter::vAdjustAmplifierVolume"));
      // check applicability
      if (EN_AUDIO_FADING_RES_USR != enReason)
         return;
      Stream* pBeepStream = _streamSet.pGetStream(_enBeepStreamID);
      if (NULL == pBeepStream)
         return;
      tenVolumeMode enMode = pBeepStream->enGetCurrentMode();
      bool applicable
         =     (EN_INT_VOLMODE_ACTUATOR_TEST_HMI == enMode)
            || (EN_INT_VOLMODE_ACTUATOR_TEST_DIAGNOSIS == enMode);
      if ( ! applicable)
         return;;

      // update volume on stream dedicated for beeps
      const dBCalculator* pCalc = _config.pGetDBCalculator(_properties.enGetCurrentSource());
      if (NULL == pCalc)
         return;

      // suppress unnecessary amplifier command for active stream
      if (volumeData.m_Volume == _properties.u8GetVolume())
         useStandardAmplifierCommand = false;

      // apply limits on beep-stream
      tU8 volume = volumeData.m_Volume;
      if (_isMainAudioMuted)
         volume = _u8MinBeepVolume;
      else if (_u8MinBeepVolume > volume)
         volume = _u8MinBeepVolume;
      else if (_u8MaxBeepVolume < volume)
         volume = _u8MaxBeepVolume;
      tenActions enAction = VolumeSetAction::enGetFadingAction (enReason);
      Ramp ramp = _streamSet.queryRamp(enAction, enReason, _enBeepStreamID);
      pBeepStream->setVolume(*pCalc, volume, ramp);
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void BeepVolumeFilter:: MessageNotification (PO_MessageConfig::enID MsgId)
   {
      // implement IF_MessageObserver
      if (PO_MessageConfig::ID_NotifyMute != MsgId)
      {
         ETG_TRACE_ERR(("BeepVolumeFilter::MessageNotification() - unhandled message %u received", MsgId))
         return;
      }

      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      const ID_NotifyMute* pMsg = pPO->QueryMessage<ID_NotifyMute>(MsgId);
      tenStream enStream = pMsg->enStream;
      tenMuteResult enResult = pMsg->enMuteResult;

      if (EN_AUDIO_SOURCE_STREAM_MAIN != enStream)
         return;

      bool mainAudioMutes = (enResult == EN_AUDIO_MUTE_MUTERESULT_MUTE);
      if (_isMainAudioMuted == mainAudioMutes)
         return;

      ETG_TRACE_USR4(("BeepVolumeFilter::MessageNotification() : ID_NotifyMute received for Stream %u with %u; changing _isMainAudioMuted from %u to %u"
            , ETG_CENUM(tenStream, enStream), ETG_CENUM(tenMuteResult, enResult), _isMainAudioMuted, mainAudioMutes))

      _isMainAudioMuted = mainAudioMutes;
   }

   // -----------------------------------------------------------------------------

}  // namespace VolumeManager


