//
// VolumeManager/Engine/Actions/SourceSwitchAction.cpp
//
//  Created on: Jul 30, 2014
//      Author: Martin Koch, Fa. ESE
//


// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>

#include "Volume/Types.h"  // implicitly links <midw_fi_if.h> for alltypes.h and <stl_pif.h> for vector
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "./IAction.h"
#include "./ActionBase.h"
#include "./SourceSwitchAction.h"
// - - - - - - - - - - - - -

#include "./VolumeSetAction.h"
#include "Volume/Configuration/Configuration.h"
#include "Volume/Engine/Stream.h"
#include "Volume/Engine/StreamSet.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "Volume/Configuration/dBCalculator.h"
#include "Volume/PropertyStore.h"
#include "Volume/TypeConversions/Functions.h"

#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeData.h"
#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeSourceData.h"

#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/SourceSwitchAction.cpp.trc.h"



namespace VolumeManager
{

   // -----------------------------------------------------------------------------

   /* constructor */ SourceSwitchAction:: SourceSwitchAction (PropertyStore& properties, const Configuration& config, const VolumeSourceData& data)
      : ActionBase(properties, config)
      , _enLastSource(_properties.enGetCurrentSource())
      , _pNextSource(_config.pGetSourceConfig(data.m_Source))
   {
      ETG_TRACE_USR4(("SourceSwitchAction() invoked for switching from %u to %u: on Sink %d"
            , ETG_CENUM(tenInternalSource, _enLastSource), ETG_CENUM(tenInternalSource, data.m_Source), data.m_Resource.getResource()))

      if (NULL == _pNextSource)
         ETG_TRACE_ERR(("SourceSwitchAction() - E R R O R  :  target source configuration not found!"))
   }

   // -----------------------------------------------------------------------------

   /* virtual destructor */ SourceSwitchAction:: ~SourceSwitchAction ()
   {
      _pNextSource = NULL;
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void SourceSwitchAction:: vExecute (StreamSet& streamSet)
   {
      if (NULL == _pNextSource)
         return;

      tenStream enStream = _pNextSource->enGetStreamID(streamSet.enGetResource());

      if (enStream == EN_AUDIO_SOURCE_STREAM_MIX2 && (_properties.bIsSPIDuckingActive() == false))
      {
        vSpecialHandlingForMIX2(streamSet);
        return;
      }

      vUpdateProperties(streamSet);

      vRestoreRecentVolume(streamSet);

      return;
}
   // -----------------------------------------------------------------------------

   void SourceSwitchAction:: vUpdateProperties (StreamSet& streamSet)
   {
      if (NULL == _pNextSource)
         return;

      // apply changes
      tenResource enResource = streamSet.enGetResource();
      tenStream enStream = _pNextSource->enGetStreamID(streamSet.enGetResource());
      tenInternalSource enSourceID = _pNextSource->enGetSourceID();
      Stream* pStream = streamSet.pGetStream(enStream);

      if (pStream)
      {
         tenVolumeMode enMode = pStream->enGetCurrentMode();
         tenStream enStreamID = pStream->enGetStreamID();
         pStream->vSetCurrentSource(enSourceID);
         streamSet.vSetActiveStream(enStreamID);
         _properties.vUpdateMode(enMode, enStreamID, enResource);
         _properties.vSetCurrentSource(*_pNextSource, enResource);
      }
      else
      {
         ETG_TRACE_FATAL(("vApply - E R R O R :  could not locate stream for source %u at resource %u"
             ,  ETG_CENUM(tenInternalSource, enSourceID)
             ,  ETG_CENUM(tenResource, streamSet.enGetResource())))
      }
   }

   // -----------------------------------------------------------------------------

   void SourceSwitchAction:: vResetCurrentVolume (StreamSet& streamSet)
   {
      if ((AUD_INT_SRC_NOSOURCE == _enLastSource) || (AUD_INT_SRC_UNDEFINED == _enLastSource))
         return;  // no previous source, don't try to get its configuration

      const SourceConfig* pLastSource = _config.pGetSourceConfig(_enLastSource);
      if (NULL == pLastSource)
         return;   // can't do anything

      Stream* pStream = streamSet.pGetActiveStream();
      if (NULL == pStream)
         return;

      if (_enLastSource != streamSet.enGetCurrentSource())
         return;

      // set amplifier volume
      tenStream enStreamID = streamSet.enGetActiveStreamID();
      const dBCalculator* pCalc = _config.pGetDBCalculator(_enLastSource);
      if (pCalc)
      {
         Ramp ramp = streamSet.queryRamp(EN_AUDIO_FADING_NOTIFY_CHG_VOL, EN_AUDIO_FADING_RES_SOURCESEL, enStreamID);
         pStream->setVolume(*pCalc, pLastSource->getVolumeLimits().minStep, ramp);
      }
   }

   // -----------------------------------------------------------------------------

   void SourceSwitchAction:: vRestoreRecentVolume (StreamSet& streamSet)
   {
      if (NULL == _pNextSource)
         return;

      // query current navigation volume from property store
      tenResource enResource = streamSet.enGetResource();
      tU8 nextSourceGroup = _pNextSource->u8GetGroupType();

      VolumeData data;
      data.m_Channel = fiAudioChannel_StreamID_Map::getFirst(streamSet.enGetActiveStreamID());
      data.m_Resource.setResource(fiResource_Map::getFirst(enResource));
      data.m_VolumeType.setVolumeType(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS);

      const tU8* pVolume  = _properties.pGetVolume(enResource, nextSourceGroup);
      if (pVolume)
      {
         data.m_Volume = *pVolume;
      }
      else
      {
         data.m_Volume = _properties.u8GetVolume();
         ETG_TRACE_USR1(("vRestoreRecentVolume():  no recent volume found for type %u at resource %u, using current"
         , ETG_CENUM(midw_fi_tcl_e8_Aud_VolumeType::tenType, nextSourceGroup)
         , ETG_CENUM(tenResource, enResource)))
      }

      // delegate to embedded volume-set action
      VolumeSetAction(_properties, _config, data, EN_AUDIO_FADING_RES_SOURCESEL).vExecute(streamSet);
   }

   // -----------------------------------------------------------------------------

   void SourceSwitchAction::vSpecialHandlingForMIX2(StreamSet& streamSet)
   {
     if (NULL == _pNextSource)
        return;

     //Only volume should be set for SPI_MIX to ADR.
     //Volume Status should be blocked as it should be sent only on Ducking request
       tenResource enResource = streamSet.enGetResource();
     tenStream enStream = _pNextSource->enGetStreamID(streamSet.enGetResource());
     Stream* pStream = streamSet.pGetStream(enStream);
     tenStream  currentActiveStream = streamSet.enGetActiveStreamID();

     tU8 nextSourceGroup = _pNextSource->u8GetGroupType();

      if(pStream!=NULL)
       {
         pStream->vSetCurrentSource(_pNextSource->enGetSourceID());
         streamSet.vSetActiveStream(pStream->enGetStreamID());
       }
       VolumeData data;
       data.m_Channel = fiAudioChannel_StreamID_Map::getFirst(enStream);
       data.m_Resource.setResource(fiResource_Map::getFirst(enResource));
       data.m_VolumeType.setVolumeType(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS);

       const tU8* pVolume  = _properties.pGetVolume(enResource, nextSourceGroup);
       if (pVolume)
       {
          data.m_Volume = *pVolume;
       }
       else
       {
          data.m_Volume = _properties.u8GetVolume();
          ETG_TRACE_USR1(("vRestoreRecentVolume():  no recent volume found for type %u at resource %u, using current"
          , ETG_CENUM(midw_fi_tcl_e8_Aud_VolumeType::tenType, nextSourceGroup)
          , ETG_CENUM(tenResource, enResource)))
       }

       // delegate to embedded volume-set action
       VolumeSetAction(_properties, _config, data, EN_AUDIO_FADING_RES_SOURCESEL).vExecute(streamSet);
       streamSet.vSetActiveStream(currentActiveStream);
   }

}  // namespace VolumeManager
