//
// VolumeManager/Engine/Actions/ModeChangeAction.cpp
//
//  Created on: Sep 2, 2014
//      Author: Martin Koch, Fa. ESE
//


// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>

#include "Volume/Types.h"  // implicitly links <midw_fi_if.h> for alltypes.h and <stl_pif.h> for vector
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "./IAction.h"
#include "./ActionBase.h"
#include "./ModeChangeAction.h"
// - - - - - - - - - - - - -

#include "Volume/Configuration/Configuration.h"
#include "Volume/Engine/Stream.h"
#include "Volume/Engine/StreamSet.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "Volume/Configuration/dBCalculator.h"
#include "Volume/PropertyStore.h"
#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeModeData.h"
#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeSourceData.h"
#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeData.h"

#include "./SourceSwitchAction.h"
#include "./VolumeSetAction.h"
#include "Volume/TypeConversions/Functions.h"


#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/ModeChangeAction.cpp.trc.h"



namespace VolumeManager
{

   // -----------------------------------------------------------------------------

   /* constructor */ ModeChangeAction:: ModeChangeAction (PropertyStore& properties, const Configuration& config, const VolumeModeData& modeData)
      : ActionBase(properties, config)
      , _modeData(modeData)
   {
      ETG_TRACE_USR4(("ModeChangeAction() invoked with data:"))
      _modeData.printData();
   }

   // -----------------------------------------------------------------------------

   /* virtual destructor */ ModeChangeAction:: ~ModeChangeAction ()
   {
      ETG_TRACE_USR4(("ModeChangeAction() completed."))
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void ModeChangeAction:: vExecute (StreamSet& streamSet)
   {
      tenStream enStream = fiAudioChannel_StreamID_Map::getSecond(_modeData.m_Channel);
      Stream* pStream = streamSet.pGetStream(enStream);
      if (NULL== pStream)
      {
         ETG_TRACE_ERR(("vExecute() - E R R O R  :  no stream found for %u"
               , ETG_CENUM(tenStream, enStream)))
         return;
      }
      tenVolumeMode enPreviousMode = pStream->enGetCurrentMode();

      ETG_TRACE_USR3(("vExecute() : switching mode from %u to %u for stream %s"
            , ETG_CENUM(tenVolumeMode, enPreviousMode)
            , ETG_CENUM(tenVolumeMode, _modeData.m_VolumeModeType)
            , pStream->getName() ))

      pStream->vSetCurrentMode(_modeData.m_VolumeModeType);

      if (bImpliesSourceChange())
         vExecuteImplicitSourceChange(streamSet);

      // if source in mentioned stream is active, update mode and volume properties
      if (_properties.enGetCurrentSource() == pStream->enGetCurrentSource())
      {
         _properties.vUpdateMode(_modeData.m_VolumeModeType, pStream->enGetStreamID(), pStream->enGetResource());
      }

      if (enPreviousMode == _modeData.m_VolumeModeType)
         return;

      // allow for volume attenuation in case of PDC Attenuation Setup
      if ((enPreviousMode == EN_INT_VOLMODE_SETUP_PDC_ATTENUATION) || (_modeData.m_VolumeModeType == EN_INT_VOLMODE_SETUP_PDC_ATTENUATION))
      {
         // launch new VolumeSetAction with current data
         VolumeData data;
         data.m_Channel = _modeData.m_Channel;
         data.m_Resource.setResource(_modeData.m_Resource);
         data.m_Volume  = _properties.u8GetVolume();
         data.m_VolumeType.setVolumeType(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS);
         VolumeSetAction(_properties, _config, data, EN_AUDIO_FADING_RES_USR).vExecute(streamSet);
      }

      if (_modeData.m_VolumeModeType == EN_INT_VOLMODE_REMOTE_CONTROL_START)
      {
         // launch new VolumeSetAction with current data
         VolumeData data;
         data.m_Channel = _modeData.m_Channel;
         data.m_Resource.setResource(_modeData.m_Resource);
         data.m_Volume  = _properties.u8GetVolume();
         data.m_VolumeGain = -120;
         data.m_VolumeType.setVolumeType(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS);
         VolumeSetAction(_properties, _config, data, EN_AUDIO_FADING_RES_USR).vExecute(streamSet);
      }
      // delegate special handling of beep volume to VolumeSetAction
      if (    (EN_INT_VOLMODE_BEEP == enPreviousMode)
           || (EN_INT_VOLMODE_ACTUATOR_TEST_HMI == enPreviousMode)
           || (EN_INT_VOLMODE_ACTUATOR_TEST_DIAGNOSIS == enPreviousMode)
           || (EN_INT_VOLMODE_BEEP == _modeData.m_VolumeModeType)
           || (EN_INT_VOLMODE_ACTUATOR_TEST_HMI == _modeData.m_VolumeModeType)
           || (EN_INT_VOLMODE_ACTUATOR_TEST_DIAGNOSIS == _modeData.m_VolumeModeType)  )
      {
         VolumeData data;
         data.m_Channel = fiAudioChannel_StreamID_Map::getFirst(streamSet.enGetActiveStreamID());
         data.m_Resource.setResource(_modeData.m_Resource);
         data.m_Volume  = _properties.u8GetVolume();
         data.m_VolumeType.setVolumeType(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS);
         VolumeSetAction(_properties, _config, data, EN_AUDIO_FADING_RES_USR).vExecute(streamSet);
      }
   }

   // -----------------------------------------------------------------------------

   bool ModeChangeAction:: bImpliesSourceChange ()
   {
      tenInternalSource enCurrentSource = _properties.enGetCurrentSource();
      tenVolumeMode      enCurrentMode  = _properties.enGetMode();

      // TA-Setup with current non-FM source
      if ((_modeData.m_VolumeModeType == EN_INT_VOLMODE_SETUP_TA) && (enCurrentSource != AUD_INT_SRC_FM))
         return true;

      if(_modeData.m_VolumeModeType == EN_INT_VOLMODE_REMOTE_CONTROL_END)
       return true;

      // leave TA-Setup
      if ((_modeData.m_VolumeModeType != enCurrentMode) && (enCurrentMode == EN_INT_VOLMODE_SETUP_TA))
         return true;

      // Navigation-Setup start
      if ((_modeData.m_VolumeModeType == EN_INT_VOLMODE_SETUP_NAV_VOL) && (enCurrentSource != AUD_INT_SRC_NAVI))
         return true;

      // Leave Navigation-Setup
      if ((_modeData.m_VolumeModeType != EN_INT_VOLMODE_SETUP_NAV_VOL) && (enCurrentSource == AUD_INT_SRC_NAVI))
         return true;

      return false;
   }

   // -----------------------------------------------------------------------------

   void ModeChangeAction:: vExecuteImplicitSourceChange (StreamSet& streamSet)
   {
      // for some mode transitions we need to adjust the active source.
      // This can easily be done by executing a SourceSwitch action

      tenVolumeMode      enCurrentMode  = _properties.enGetMode();

      if (    (_modeData.m_VolumeModeType == EN_INT_VOLMODE_SETUP_NAV_VOL)
           && (AUD_INT_SRC_NAVI != streamSet.enGetCurrentSource()))
      {
         // switch to Navigation Setup

         VolumeSourceData sourceData;
         sourceData.m_Channel  = fiAudioChannel_StreamID_Map::getFirst(streamSet.enGetActiveStreamID());
         sourceData.m_Resource.setResource(fiResource_Map::getFirst(streamSet.enGetResource()));
         sourceData.m_Source   = AUD_INT_SRC_NAVI;

         SourceSwitchAction(_properties, _config, sourceData).vExecute(streamSet);
      }
      else if ( ( (_modeData.m_VolumeModeType == EN_INT_VOLMODE_NORMAL)
                 && (EN_INT_VOLMODE_NORMAL != enCurrentMode) ) ||
                 ( (_modeData.m_VolumeModeType == EN_INT_VOLMODE_REMOTE_CONTROL_END )
                  && ( EN_INT_VOLMODE_NORMAL == enCurrentMode ) ) )
       {
          // switch back from Navigation-Setup or TA-Setup

          Stream* pMainStream = streamSet.pGetStream(EN_AUDIO_SOURCE_STREAM_MAIN);
          if (pMainStream)
          {
             VolumeSourceData sourceData;
             sourceData.m_Channel = fiAudioChannel_StreamID_Map::getFirst(streamSet.enGetActiveStreamID());
             sourceData.m_Resource.setResource(fiResource_Map::getFirst(streamSet.enGetResource()));
             sourceData.m_Source   = streamSet.enGetRecentEntertainmentSource();

             SourceSwitchAction(_properties, _config, sourceData).vExecute(streamSet);
          }
          else
          {
             ETG_TRACE_FATAL(("vExecuteImplicitSourceChange() - E R R O R :  could not locate Stream 'Main'"))
          }
       }
      else if (    (_modeData.m_VolumeModeType == EN_INT_VOLMODE_SETUP_TA)
            && (AUD_INT_SRC_FM != streamSet.enGetCurrentSource()))
      {
         // switch to TA Setup

         VolumeSourceData sourceData;
         sourceData.m_Channel  = fiAudioChannel_StreamID_Map::getFirst(streamSet.enGetActiveStreamID());
         sourceData.m_Resource.setResource(fiResource_Map::getFirst(streamSet.enGetResource()));
         sourceData.m_Source   = AUD_INT_SRC_FM;

         SourceSwitchAction(_properties, _config, sourceData).vExecute(streamSet);
      }
      else
      {
         ETG_TRACE_COMP(("vExecute() : No source change done for stream %u in mode %u"
               , ETG_CENUM(tenVolumeMode, streamSet.enGetVolumeMode())
               , ETG_CENUM(tenStream, streamSet.enGetActiveStreamID()) ))
      }

   }

   // -----------------------------------------------------------------------------

}  // namespace VolumeManager
