//
// VolumeManager/Configuration/Configuration.cpp
//
// Provide configuration details for audio sources regarding
//    source grouping
//    persistence
//    min/max/default user-step constraints
//    related output stream,
//    calculation of dB-values forwarded to amplifier
// as defined in file "Configuration.xml"
//
//  Created on: Aug 1, 2014
//      Author: Martin Koch, Fa. ESE
//


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h> // implicitly links <osal_if.h>


#include "Volume/Types.h"
#include "./Configuration.h"
// - - - - - - - - - - - -


#include "./ConfigDetails.hpp"
#include "./XMLParser.h"
#include "./dBCalculator.h"
#include "Volume/Utilities/StringCopy.hpp"


#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/Configuration.cpp.trc.h"




namespace VolumeManager
{

   // --------------------------------------------------------------------------

   /* constructor */ Configuration:: Configuration ()
      : _pCalculators(NULL)
      , _pFunctions(NULL)
      , _pSets(NULL)
      , _pStreams(NULL)
      , _pStreamMap(NULL)
      , _pGroups(NULL)
      , _pSources(NULL)
      , _calculatorCount(0)
      , _functionCount(0)
      , _setCount(0)
      , _streamCount(0)
      , _sourceCount(0)
      , _sourceGroupCount(0)
   {
      XMLParser parser;
      parser.vPopulateCalculators(/* output */ _pCalculators, _calculatorCount);
      parser.vPopulateFunctions(/* output */ _pFunctions, _functionCount);
      parser.vPopulateStreamSets(_pFunctions, _functionCount
            , /* output */ _pSets, _setCount, _pStreams,  _streamCount, _pCalculators, _calculatorCount);
      parser.vPopulateSourceGroups(_pStreams, _streamCount, _pCalculators, _calculatorCount
            , /* output */ _pStreamMap, _pGroups, _sourceGroupCount);
      parser.vPopulateSources(_pGroups, _sourceGroupCount, /* output */ _pSources, _sourceCount);
   }

   // --------------------------------------------------------------------------

   /* destructor */ Configuration:: ~Configuration ()
   {
      if (_pSources)
         for (size_t i = 0; i < _sourceCount; ++i)
            delete[] _pSources[i].name;
      delete[] _pSources;

      if (_pGroups)
         for (size_t i = 0; i < _sourceGroupCount; ++i)
            delete[] _pGroups[i].name;
      delete[] _pGroups;

      delete[] _pStreamMap;

      if (_pStreams)
         for (size_t i = 0; i < _streamCount; ++i)
           delete[] _pStreams[i].name;
      delete[] _pStreams;

      if (_pSets)
         for (size_t i = 0; i < _setCount; ++i)
            delete[] _pSets[i].name;
      delete[] _pSets;

      if (_pFunctions)
         for (size_t i = 0; i < _functionCount; ++i)
         {
            delete[] _pFunctions[i].name;
            delete[] _pFunctions[i].params;
            delete[] _pFunctions[i].groupRefs;
         }
      delete[] _pFunctions;

      delete[] _pCalculators;


      ETG_TRACE_USR1(("Configuration dissolved."))
   }

   // --------------------------------------------------------------------------

   size_t Configuration:: getStreamSetCount () const
   {
      return _setCount;
   }

   // --------------------------------------------------------------------------

   size_t Configuration:: getStreamCount () const
   {
      return _streamCount;
   }

   // --------------------------------------------------------------------------

   size_t Configuration:: getSourceGroupCount () const
   {
      return _sourceGroupCount;
   }

   // --------------------------------------------------------------------------

   const VolumeConstraints* Configuration:: pGetVolumeLimitsByGroup (tU8 u8GroupTypeId, bool requestPersistenceLimits) const
   {
      for (size_t i = 0; i < _sourceGroupCount;  ++i)
      {
         if (_pGroups[i].typeId == u8GroupTypeId)
         {
            if (requestPersistenceLimits)
               return &(_pGroups[i].persistenceConstraints);
            else
               return &(_pGroups[i].constraints);
         }
      }
      // not found
      return NULL;
   }

   // --------------------------------------------------------------------------

   const StreamSetConfig* Configuration:: pGetStreamSetConfig (size_t configIndex) const
   {
      if (configIndex < _setCount)
         return &_pSets[configIndex];
      else
         return NULL;
   }

   void Configuration:: pGetStreamSet (const StreamSetConfig*& StreamSet) const
   {
      StreamSet = _pSets;
   }
   // --------------------------------------------------------------------------

   const StreamConfig* Configuration:: pGetStreamConfig (size_t configIndex) const
   {
      if (configIndex < _streamCount)
         return &_pStreams[configIndex];
      else
         return NULL;
   }

   // --------------------------------------------------------------------------

   const dBCalculator* Configuration:: pGetDBCalculator (tenInternalSource enSource) const
   {
      if (_pSources)
         for (size_t i = 0; i < _sourceCount; ++i)
            if (enSource == _pSources[i].enGetSourceID())
            {
               const dBCalculator* pCalc = _pSources[i].pGetCalculator();
               if (pCalc)
                  return pCalc;
               else
                  break;  // not valid
            }

      // not found or not valid
      ETG_TRACE_FATAL(("getDBCalculator() - E R R O R :  could not locate dBCalculator for source %u"
            , ETG_CENUM(tenInternalSource, enSource)))
      return NULL;
   }

   // --------------------------------------------------------------------------

   const SourceConfig* Configuration:: pGetSourceConfig (tenInternalSource enSource) const
   {
      if (_pSources)
         for (size_t i = 0; i < _sourceCount; ++i)
            if (enSource == _pSources[i].enGetSourceID())
               return &_pSources[i];

      // not found
      ETG_TRACE_USR2(("pGetSourceConfig() - E R R O R : could not locate config for source %u"
            , ETG_CENUM(tenInternalSource, enSource)))
      return NULL;
   }

   // --------------------------------------------------------------------------

   void Configuration:: vGetSourceGroupConfigs (const GroupConfig*& ext_pGroups, size_t& ext_NumberOfGroups) const
   {
      ext_pGroups = _pGroups;
      ext_NumberOfGroups = _sourceGroupCount;
   }

   // --------------------------------------------------------------------------

}
