/**
 * @file ESETuningHandlerRNAIVI.cpp
 * @author CM-AI wie1hi
 * @copyright (c) 2014 Robert Bosch Car Multimedia GmbH
 * @addtogroup fc_audiomanager
 * @{
 */

#include <stdio.h>
#include <netinet/in.h>

#include "aud_sinkmanager/vd_adr3/vd_adr3Msg_If.h"

#include "../../../../../ai_osal_linux/components/system/system_types.h"
#include "fc_audiomanager_main.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include <etrace_if.h>

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

#include "fc_audiomanager_trace.h"
#include "fc_audiomanager_trace_macros.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_TUNING_DATA_SERVICE
#include "trcGenProj/Header/ESETuningHandlerRNAIVI.cpp.trc.h"
#endif

#include "TuningDataService.h"
#include "ESETuningHandlerRNAIVI.h"
#include "EngineSound/RNAIVI/EngineSoundHandlerIVI.h"
#include "EngineSound/RNAIVI/EseHardCodedData.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_diagdebug_if.h" // For EngineSound EOL Datapool
#include "dp_generic_if.h"
#include "dp_audio_if.h"

ESETuningHandlerRNAIVI::ESETuningHandlerRNAIVI()
{
   m_txHandle = 0;
   m_pU8Data = OSAL_NULL;
   m_u32Size = 0;
   m_bWaitEseVolume = false;
   m_bWaitEseOnOff = false;
   m_bESEOnOffState = false;
}

ESETuningHandlerRNAIVI::~ESETuningHandlerRNAIVI()
{
   m_pU8Data = OSAL_NULL;
}

ESETuningHandlerRNAIVI* ESETuningHandlerRNAIVI::getInstance()
{
   static ESETuningHandlerRNAIVI theInstance;
   return &theInstance;
}

/**
 * @method tVoid DataRx(tU32 size, tVoid* data, tU8 txHandle)
 * @description Receives Data from the TuningDataService:
 *             Command (1 Byte)
 *             Address (4 Bytes)
 *             Length  (4 Bytes)
 *             Value   (<Length> Bytes)
 *             optional: additional [Address, Length, Value] sequences
 * @param size Length of the data that the parameter <data> points to
 * @param data Pointer to the received tuning data. The pointer gets invalid
 *             upon leaving this function. Don't store it!
 * @param txHandle A handle that is needed to send data back to the sender
 */
tVoid ESETuningHandlerRNAIVI::DataRx(tU32 size, tVoid* pData, tU8 txHandle)
{
  ETG_TRACE_USR4(("ESETuningHandlerRNAIVI: Received %i bytes of data. TxHandle: %i", size, txHandle));
  m_txHandle = txHandle;

  if (size < 2)
  {
    ETG_TRACE_ERR(("ESETuningHandlerRNAIVI: Received invalid command."));
    vSendErrToClient(ERR_INVALID_DATA_LENGTH);
    return;
  }

  m_u32Size = size;
  m_pU8Data = (tU8*)pData;

  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  // handle different commands from tuning tool
  switch(m_pU8Data[1])
  {
    case TUNING_ESE_VERSION:
      if(m_pU8Data[0] == TUNING_GetCmd)
        vPrepareEseVersionResponse();
      break;

    case TUNING_ESE_ONOFF:
      vSendESEOnOffData();
      break;

    case TUNING_ESE_VOLUME:
      vSendESEVolumeData();
      break;

    case TUNING_ESE_PARAMETER:
      vSendESEParameter();
      break;

    case TUNING_ESE_COMM_STATUS:
      vSendAckToClient(m_pU8Data[1]);
      break;

    default:
         vSendErrToClient(ERR_UNKNOWN_PARAMID);
      ETG_TRACE_ERR(("ESETuningHandlerRNAIVI::DataRx() received invalid parameter"));
      break;
  }
}

/**
 * @metohd AdrMessageRx(tU8 cmd, tU32 adr, tU32 len, tS32* data)
 * @description This method receives messages from the ADR function VD_ADR3_FKT_ID_ESE_PARAMETER (0x0FE1)
 *              and relays them via TCP/IP to the connected Tuning Tool.
 */
tVoid ESETuningHandlerRNAIVI::AdrMessageRx(tU16 u16FuncID, tU8* u8Data, tU32 u32Length )
{
   ETG_TRACE_USR4(("Got ESE Data: function:%x ,  Len:%x ", u16FuncID, u32Length));

  if(NULL == u8Data)
    return;

  switch(u16FuncID)
  {
    case VD_ADR3_FKT_ID_ENGINE_SOUND_ONOFF:
         if(m_bWaitEseOnOff)
         {
            vPrepareEseOnOffResponse(u8Data,u32Length);
            m_bWaitEseOnOff = false;
         }
      break;

    case VD_ADR3_FKT_ID_ENGINE_SOUND_VOLUME:
         if(m_bWaitEseVolume)
         {
            vPrepareEseVolumeResponse(u8Data,u32Length);
            m_bWaitEseVolume = false;
         }
      break;

    default:
      break;
  }
}
tVoid ESETuningHandlerRNAIVI::vPrepareEseVersionResponse()
{
  reg_tclRegKey reg;
  char chararray[10] = { '\0' };

   vSendAckToClient(m_pU8Data[1]); //first prepare an ACK to the tool

  if (reg.bOpen("/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS") == true)
  {
    if (reg.bQueryString("CUSTOMERVERSION", chararray, sizeof(chararray)) == true)
    {
      TuningDataService* tuningService = TuningDataService::getInstance();

      ETG_TRACE_USR4(("ESETuningHandlerRNAIVI: Customer Version read as %s", chararray ));

      //size of buffer = 10 -> 4 bytes for 'length' field + 1 byte 'Cmd' field + 1 byte 'parameter' field + 4 byte 'data' field
      tU8 u8ReplyBuf[10] = {0x00};

      u8ReplyBuf[3] = 6;  //Length -> Datalength(4) + Cmd(1) + Parameter(1)
      u8ReplyBuf[4] = TUNING_GetCmd; //Cmd
      u8ReplyBuf[5] = TUNING_ESE_VERSION; //Parameter

      memcpy(&u8ReplyBuf[6],chararray,4); //data

      if(tuningService)
        tuningService -> vDataTx(TuningDataService::ID_ESE_RNAIVI, m_txHandle, u8ReplyBuf, 10);
    }

    reg.vClose();
  }
}
tVoid ESETuningHandlerRNAIVI::vPrepareEseOnOffResponse(tU8* u8Data, tU32 u32Length)
{
  TuningDataService* tuningService = TuningDataService::getInstance();

   if(!u32Length)
      return;

  //size of buffer = 7 -> 4 bytes for 'length' field + 1 byte 'Cmd' field + 1 byte 'parameter' field + 1 byte 'data' field
  tU8 u8ReplyBuf[7] = {0x00};

  u8ReplyBuf[3] = 3;  //Length -> Datalength(1) + Cmd(1) + Parameter(1)
  u8ReplyBuf[4] = TUNING_GetCmd; //Cmd
  u8ReplyBuf[5] = TUNING_ESE_ONOFF; //Parameter
  u8ReplyBuf[6] = u8Data[0]; //Data

  if(tuningService)
    tuningService -> vDataTx(TuningDataService::ID_ESE_RNAIVI, m_txHandle, u8ReplyBuf, 7);
}

tVoid ESETuningHandlerRNAIVI::vPrepareEseVolumeResponse(tU8* u8Data, tU32 u32Length)
{
  TuningDataService* tuningService = TuningDataService::getInstance();

   if(u32Length < 2)
      return;

  //size of buffer = 7 -> 4 bytes for 'length' field + 1 byte 'Cmd' field + 1 byte 'parameter' field + 1 byte 'data' field
  tU8 u8ReplyBuf[7] = {0x00};

  u8ReplyBuf[3] = 3;  //Length -> Datalength(1) + Cmd(1) + Parameter(1)
  u8ReplyBuf[4] = TUNING_GetCmd; //Cmd
  u8ReplyBuf[5] = TUNING_ESE_VOLUME; //Parameter

  tS16 s16Offset = ((tS16)(((tU8)(u8Data[1]))|(((tS16)((tU8)(u8Data[0])))<<8)));
  u8ReplyBuf[6] = (tU8)(((s16Offset / 4) + 28) / 2);  //Data

  if(tuningService)
    tuningService -> vDataTx(TuningDataService::ID_ESE_RNAIVI, m_txHandle, u8ReplyBuf, 7);
}

tVoid ESETuningHandlerRNAIVI::vSendESEOnOffData()
{
  ETG_TRACE_USR4(("ESETuningHandlerRNAIVI: vSendESEOnOffData Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  if(m_pU8Data[0] == TUNING_SetCmd)
    vSetESEOnOff(m_pU8Data[2]);
  else if(m_pU8Data[0] == TUNING_GetCmd)
    vGetESEOnOff();
  else
   {
    ETG_TRACE_ERR(("ESETuningHandlerRNAIVI::vSendESEOnOffData() received invalid command"));
      vSendErrToClient(ERR_UNKNOWN_CMDID);
   }
}

void ESETuningHandlerRNAIVI::vSetESEOnOff(tBool bOn)
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);
  ETG_TRACE_USR4(("ESETuningHandlerRNAIVI::vSetESEOnOff() entered. bOn : %d", bOn));
  m_bESEOnOffState = bOn;
   EngineSoundHandlerIVI::getInstance()->vSendESEOnOff(bOn);

  vSendAckToClient(m_pU8Data[1]);
}

void ESETuningHandlerRNAIVI::vGetESEOnOff()
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);
  ETG_TRACE_USR4(("ESETuningHandlerRNAIVI::vGetESEOnOff() entered. "));
   m_bWaitEseOnOff = true;

  EngineSoundHandlerIVI::getInstance()->vRequestESEOnOffStatus();

  vSendAckToClient(m_pU8Data[1]);
}


tVoid ESETuningHandlerRNAIVI::vSendESEVolumeData()
{
  ETG_TRACE_USR4(("ESETuningHandlerRNAIVI: vSendESEVolumeData Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  if(m_pU8Data[0] == TUNING_SetCmd)
    vSetESEVolume(m_pU8Data[2]);
  else if(m_pU8Data[0] == TUNING_GetCmd)
    vGetESEVolume();
  else
   {
    ETG_TRACE_ERR(("ESETuningHandlerRNAIVI::vSendESEVolumeData() received invalid command"));
      vSendErrToClient(ERR_UNKNOWN_CMDID);
   }
}

void ESETuningHandlerRNAIVI::vSetESEVolume(tU8 u8VolumeLevel)
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);
  ETG_TRACE_USR4(("ESETuningHandlerRNAIVI::vSetESEVolume() entered. Volume Level : %d", u8VolumeLevel));

  EngineSoundHandlerIVI::getInstance()->vSendRSoundVolume(u8VolumeLevel);

  vSendAckToClient(m_pU8Data[1]);
}

void ESETuningHandlerRNAIVI::vGetESEVolume()
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);
  ETG_TRACE_USR4(("ESETuningHandlerRNAIVI::vGetESEVolume() entered. "));
   m_bWaitEseVolume = true;

  EngineSoundHandlerIVI::getInstance()->vRequestESEVolumeStatus();

  vSendAckToClient(m_pU8Data[1]);
}

tVoid ESETuningHandlerRNAIVI::vSendESEParameter()
{
   ETG_TRACE_USR4(("ESETuningHandlerRNAIVI: vSendESEParameter Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  if(m_pU8Data[0] == TUNING_SetCmd)
  {
    vSendAckToClient(m_pU8Data[1]);
    if(m_bESEOnOffState)
    {
      EngineSoundHandlerIVI::getInstance()->vSendESEOnOff(false);
      EngineSoundHandlerIVI::getInstance()->vLoadESEParameter(&m_pU8Data[2],(m_u32Size-2));
      EngineSoundHandlerIVI::getInstance()->vSendESEOnOff(true);
    }
    else
      EngineSoundHandlerIVI::getInstance()->vLoadESEParameter(&m_pU8Data[2],(m_u32Size-2));
  }
  else if(m_pU8Data[0] == TUNING_GetCmd)
    vPrepareEseParameterResponse(m_pU8Data[2]);
  else
   {
    ETG_TRACE_ERR(("ESETuningHandlerRNAIVI::vSendESEParameter() received invalid command"));
      vSendErrToClient(ERR_UNKNOWN_CMDID);
   }
}

tVoid ESETuningHandlerRNAIVI::vPrepareEseParameterResponse(tU8 u8FileId)
{
   vSendAckToClient(m_pU8Data[1]); //first prepare an ACK to the tool

   TuningDataService* tuningService = TuningDataService::getInstance();

   tU8 u8IntID = (tU8)(u8FileId + 1); //The tool maps FileIDs from 0 (Neutral) to 8(Hard-coded2). Audio middleware internally uses index from 1 to 9

   tU16 au16Index[] = {EOL_INDEX_NEUTRAL_DATA,EOL_INDEX_SOUND1_DATA,EOL_INDEX_SOUND2_DATA,EOL_INDEX_RSOUND1_DATA,EOL_INDEX_RSOUND2_DATA,EOL_INDEX_RSOUND3_DATA, EOL_INDEX_RSOUND4_DATA};

   //Read MEX soundtypes from EOL
   if((u8IntID >= ENGINESOUND_SOUNDTYPE_NEUTRAL) && (u8IntID <= ENGINESOUND_SOUNDTYPE_SOUND2))
   {
      tU8 au8DatapoolData[ENGINE_SOUND_DATA_SIZE_BYTES];
      dp_tclEngineSoundDPEngineSound_Data oDpEngineSound;

      tU32 u32Len = 3 + ESE_PARAM_SIZE_MEX;
      tU8 u8ReplyBuf[7 + ESE_PARAM_SIZE_MEX] = {0x00,0x00,0x00,0x00, /*Length*/
                                                 TUNING_GetCmd, /*Cmd*/
                                                 TUNING_ESE_PARAMETER, /*Parameter*/
                                                 u8FileId};

      if(ENGINE_SOUND_DATA_SIZE_BYTES == oDpEngineSound.s32GetData(au8DatapoolData, ENGINE_SOUND_DATA_SIZE_BYTES))
         memcpy(&u8ReplyBuf[7],&au8DatapoolData[au16Index[u8IntID-1]],ESE_PARAM_SIZE_MEX);
      else
         ETG_TRACE_ERR(("ESETuningHandlerRNAIVI::Failed to read the ESE data from datapool"));

      u32Len = htonl(u32Len);
      memcpy(u8ReplyBuf,&u32Len,4); //Copy the length

      if(tuningService)
         tuningService -> vDataTx(TuningDataService::ID_ESE_RNAIVI, m_txHandle, u8ReplyBuf, sizeof(u8ReplyBuf));
   }

   //RSOUND1 to 4 are in EOL
   else if((u8IntID >= ENGINESOUND_SOUNDTYPE_RSOUND1) && (u8IntID <= ENGINESOUND_SOUNDTYPE_RSOUND4))
   {
      tU8 au8DatapoolData[ENGINE_SOUND_DATA_SIZE_BYTES];
      dp_tclEngineSoundDPEngineSound_Data oDpEngineSound;

      tU32 u32Len = 3 + ESE_PARAM_SIZE_RSOUND;
      tU8 u8ReplyBuf[7 + ESE_PARAM_SIZE_RSOUND] = {0x00,0x00,0x00,0x00, /*Length*/
                                                 TUNING_GetCmd, /*Cmd*/
                                                 TUNING_ESE_PARAMETER, /*Parameter*/
                                                 u8FileId};


      if(ENGINE_SOUND_DATA_SIZE_BYTES == oDpEngineSound.s32GetData(au8DatapoolData, ENGINE_SOUND_DATA_SIZE_BYTES))
         memcpy(&u8ReplyBuf[7],&au8DatapoolData[au16Index[u8IntID-1]],ESE_PARAM_SIZE_RSOUND);
      else
         ETG_TRACE_ERR(("ESETuningHandlerRNAIVI::Failed to read the ESE data from datapool"));

      u32Len = htonl(u32Len);
      memcpy(u8ReplyBuf,&u32Len,4); //Copy the length

      if(tuningService)
         tuningService -> vDataTx(TuningDataService::ID_ESE_RNAIVI, m_txHandle, u8ReplyBuf, sizeof(u8ReplyBuf));
   }

   //RSOUND5 is Hardcoded data
   else if(ENGINESOUND_SOUNDTYPE_RSOUND5 == u8IntID)
   {
      tU32 u32Len = 3 + ESE_PARAM_SIZE_RSOUND;
      tU8 u8ReplyBuf[7 + ESE_PARAM_SIZE_RSOUND] = {0x00,0x00,0x00,0x00, /*Length*/
                                                 TUNING_GetCmd, /*Cmd*/
                                                 TUNING_ESE_PARAMETER, /*Parameter*/
                                                 u8FileId,
                                                 Hardcoded_1 /*Sound data*/};

      u32Len = htonl(u32Len);
      memcpy(u8ReplyBuf,&u32Len,4); //Copy the length

      if(tuningService)
         tuningService -> vDataTx(TuningDataService::ID_ESE_RNAIVI, m_txHandle, u8ReplyBuf, sizeof(u8ReplyBuf));
   }

   //RSOUND6 is Hardcoded data
   else if(ENGINESOUND_SOUNDTYPE_RSOUND6 == u8IntID)
   {
      tU32 u32Len = 3 + ESE_PARAM_SIZE_RSOUND;
      tU8 u8ReplyBuf[7 + ESE_PARAM_SIZE_RSOUND] = {0x00,0x00,0x00,0x00, /*Length*/
                                                 TUNING_GetCmd, /*Cmd*/
                                                 TUNING_ESE_PARAMETER, /*Parameter*/
                                                 u8FileId,
                                                 Hardcoded_2 /*Sound data*/};

      u32Len = htonl(u32Len);
      memcpy(u8ReplyBuf,&u32Len,4); //Copy the length

      if(tuningService)
         tuningService -> vDataTx(TuningDataService::ID_ESE_RNAIVI, m_txHandle, u8ReplyBuf, sizeof(u8ReplyBuf));
   }
   else
   {
      vSendErrToClient(ERR_PARAM_OUT_OF_RANGE);
   }
}

tVoid ESETuningHandlerRNAIVI::vSendAckToClient(tU8 u8ParameterID)
{
   ETG_TRACE_USR4(("Send Acknowledge to '%i' parameter", ETG_CENUM(tenESETuningParamID, u8ParameterID)));
   TuningDataService::getInstance() -> vDataTx(TuningDataService::ID_ESE_RNAIVI, m_txHandle, &u8ParameterID, 1, TuningDataService::MSG_TYPE_ACK);
}

tVoid ESETuningHandlerRNAIVI::vSendErrToClient(tU16 u16ErrorCode)
{
   tU8 u8ReplyBuf[2];
   memcpy(u8ReplyBuf,&u16ErrorCode,2);
   TuningDataService::getInstance() -> vDataTx(TuningDataService::ID_ESE_RNAIVI, m_txHandle, u8ReplyBuf, 2, TuningDataService::MSG_TYPE_ERROR);
}




