/**
 * @file ArkamysTuningHandlerRNAIVI.cpp
 * @author pau4kor
 * @copyright (c) 2016 RBEI
 * @addtogroup fc_audiomanager
 * @{
 */

#include <stdio.h>
#include <netinet/in.h>


#include "../../../../../ai_osal_linux/components/system/system_types.h"
#include "fc_audiomanager_main.h"

#include "AudioProxy/aud_gio_dbus_handler.h"
#include "AudioProxy/aud_gio_dbus_audproc_proxy.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include <etrace_if.h>

#include "fc_audiomanager_trace.h"
#include "fc_audiomanager_trace_macros.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_TUNING_DATA_SERVICE
#include "trcGenProj/Header/ArkamysTuningHandlerRNAIVI.cpp.trc.h"
#endif

#include "Arkamys/ArkamysADRInterface.h"
#include "Arkamys/RNAIVI/ArkamysFeatureHandlerRNAIVI.h"

#include "TuningDataService.h"
#include "ArkamysTuningHandlerRNAIVI.h"

#include "vd_adr3Msg_If.h"

//needed for SDEQ - HP
#include "vd_adr3_Config_Types.h"
#include "vd_adr3_Config_File.h"
#include "vd_adr3_Config_Streamers.h"
#include "vd_adr3_Config.h"

tU8 ArkamysTuningHandlerRNAIVI::u8FilterLenCount = 0;

ArkamysTuningHandlerRNAIVI::ArkamysTuningHandlerRNAIVI():IF_MessageObserver<PO_MessageConfig::enID>("AIVI-Arkamys tuning")
{
   m_txHandle = 0;
   m_bHandleValid = false;
   m_bCommExists = false;
   m_pU8Data = OSAL_NULL;
   m_u32Size = 0;

   m_bADR3SendReq = false;

   m_bVolGetReq = false;
   m_bMuteGetReq = false;
   m_bByPassGetReq = false;
   m_bFMEnhancerGetReq = false;

   m_bMuteSetReq = false;

   OSAL_pvMemorySet(m_u8FMEnhancer1Data, 0, LEN_OF_FILTER_PARAMETERS);

   OSAL_pvMemorySet(m_u8SDEQData, 0, MAX_LEN_OF_SDEQ_PARAMETERS);

   // Register for Stream Mute
   m_u8MuteRegId = StreamMute::getInstance() -> registerForStreamMute(StreamMute_IF::INTERNAL_BEH, this);

   //register for PO messages
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this , PO_MessageConfig::ID_NotifyMute);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this , PO_MessageConfig::ID_NotifyVolume);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this , PO_MessageConfig::ID_NotifyFilter);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this , PO_MessageConfig::ID_ArkamysMode);

}

ArkamysTuningHandlerRNAIVI::~ArkamysTuningHandlerRNAIVI()
{
  InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);
  m_pU8Data = OSAL_NULL;
}

ArkamysTuningHandlerRNAIVI* ArkamysTuningHandlerRNAIVI::getInstance()
{
   static ArkamysTuningHandlerRNAIVI theInstance;
   return &theInstance;
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::MessageNotification()
 *
 * DESCRIPTION:This method receives the notification for Post Office messages.
 *
 * PARAMETER: PO_MessageConfig::enID MsgId
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
void ArkamysTuningHandlerRNAIVI::MessageNotification(PO_MessageConfig::enID MsgId)
{
 switch (MsgId)
   {
      case PO_MessageConfig::ID_NotifyMute:
      {
         const ID_NotifyMute* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_NotifyMute>(MsgId);
         if(m_bMuteGetReq)//only if mutestate get is requested
         {
           PublicParamMessageRx(Tuning_Lib_Mute, (tU8)pMsg->enMuteResult);
           m_bMuteGetReq = false;
         }
      }
      break;

      case PO_MessageConfig::ID_NotifyVolume:
      {
         const ID_NotifyVolume* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage <ID_NotifyVolume> (MsgId);
         if(m_bVolGetReq)//only if volume get is requested
         {
           PublicParamMessageRx(Tuning_Lib_Volume, pMsg->u8VolumeStep);
           m_bVolGetReq = false;
         }
      }
      break;

      case PO_MessageConfig::ID_NotifyFilter:
      {
         const IDNotifyFilter* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage <IDNotifyFilter> (MsgId);
         if(m_bFMEnhancerGetReq)
            FMEnhancer1MessageRx(Tuning_Lib_FMEnhancer1, pMsg->values);
      }
      break;

      case PO_MessageConfig::ID_ArkamysMode:
      {
         const ID_ArkamysMode* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage <ID_ArkamysMode> (MsgId);
         if(m_bByPassGetReq)//only if bypass get is requested
         {
             PublicParamMessageRx(Tuning_Lib_ByPass, (tU8)pMsg->arkamysMode);
             m_bByPassGetReq = false;
         }

      }
      break;

      default:
      {
         ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI::MessageNotification - unhandled message received: %u", MsgId));
      }
      break;
   }
}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::StreamMuteNotify()
 *
 * DESCRIPTION:This method receives the notification for mute/demute.
 *
 * PARAMETER: StreamMuteNotify_IF::tenMuteNotification muteAck, tU8 stream
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
void ArkamysTuningHandlerRNAIVI::StreamMuteNotify(StreamMuteNotify_IF::tenMuteNotification muteAck, tU8 stream, tU16 sink)
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI() muteAck=%u,stream = %u",(tU8) muteAck, stream));
  if(muteAck == MUTE_ACK_MUTE || muteAck == MUTE_ACK_DEMUTE)
  {
    if(muteAck == MUTE_ACK_MUTE)
    {
         AUD_POINTER_CHECK_CALL_NO_RET_VALUE(fc_audiomanager_tclApp::theServer()->poDBus()->poGetAudioProcProxy());
         tBool bRetVal = fc_audiomanager_tclApp::theServer()->poDBus()->poGetAudioProcProxy()->set_aplArkamysFlush((tS32)muteAck);
         ETG_TRACE_USR4(("Audio Process aplArkamysFlush function call returned with (%d)", bRetVal));
    }
    if(m_bMuteSetReq)//only if set is requested
    {
      vSendAckToClient(TUNING_WriteCmd);//send ack to mute write command
      m_bMuteSetReq = false;
    }
  }

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::Mute()
 *
 * DESCRIPTION:This method sends the mute request to ADR3.
 *
 * PARAMETER: tU8 muteReqId
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::Mute(tU8 muteReqId)
{
   StreamMute::getInstance() -> muteStream(muteReqId, StreamMute_IF::MUTE_ACTION_MUTE, tenFadingReason(EN_AUDIO_FADING_RES_ARKAMYS), tenStream(EN_AUDIO_SOURCE_STREAM_MAIN));
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::Demute()
 *
 * DESCRIPTION:This method sends the demute request to ADR3.
 *
 * PARAMETER: tU8 muteReqId
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::Demute(tU8 muteReqId)
{
   StreamMute::getInstance() -> muteStream(muteReqId, StreamMute_IF::MUTE_ACTION_DEMUTE, tenFadingReason(EN_AUDIO_FADING_RES_ARKAMYS), tenStream(EN_AUDIO_SOURCE_STREAM_MAIN));
}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::AdrMessageRx()
 *
 * DESCRIPTION:Receives Data from the TuningDataService:
 *             Command (4 Byte)
 *             Address (4 Bytes)
 *             Length  (4 Bytes)
 *             Value   (<Length> Bytes)
 *             optional: additional [Address, Length, Value] sequences
 * @param size Length of the data that the parameter <data> points to
 * @param data Pointer to the received tuning data. The pointer gets invalid
 *             upon leaving this function. Don't store it!
 * @param txHandle A handle that is needed to send data back to the sender
 *
 * PARAMETER: tU32 size, tVoid* data, tU8 txHandle
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::DataRx(tU32 size, tVoid* pData, tU8 txHandle)
{
   ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: Received %i bytes of data. TxHandle: %i", size, txHandle));
   m_txHandle = txHandle;
   m_bHandleValid = true;

   if (size < 4)
   {
      ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI: Received invalid command."));
      vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
      return;
   }
   m_u32Size = size;
   m_pU8Data = (tU8*)pData;

   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

// handle different commands from tuning tool
  switch(m_pU8Data[0])
  {
  case TUNING_OpenCmd:
  {
    m_bCommExists = true;
    vSendAckToClient(m_pU8Data[0]);
    break;
  }
  case TUNING_CloseCmd:
  {
    m_bCommExists = false;
    vSendAckToClient(m_pU8Data[0]);
    break;
  }
  case TUNING_WriteCmd:
  case TUNING_ReadCmd:
  {
    vSendData();
      break;
  }
  default:
    ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI::DataRx() received invalid command"));
    break;
  }

}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::AdrMessageRx()
 *
 * DESCRIPTION:This method receives messages from the ADR function VD_ADR3_FKT_ID_ARKAMYS_PARAMETER (0x0FE0)
 *              and relays them via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: tU8 cmd, tU32 adr, tU32 len, tS32* data
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::AdrMessageRx(tU8 u8Command, tU32 u32Address, tU32 u32Length, tS32* pS32Data)
{
   if (!m_bHandleValid || !m_bCommExists)
   {
      return;
   }

   tU32 u32LengthWord_HostOrder = ntohl(u32Length);
   tU32 u32LengthByte_HostOrder = (tU32)(u32LengthWord_HostOrder * sizeof(tU32)); // Length in bytes

   ETG_TRACE_USR4(("Got Arkamys Data: Cmd:%x Adr:%x Len:%x (%i)"
         , u8Command
         , u32Address
         , u32Length, u32LengthWord_HostOrder
         ));

   TuningDataService* tuningService = TuningDataService::getInstance();

   if (u32LengthByte_HostOrder > TUNING_DATA_SERVICE_MAX_MSG_DATA_SIZE)
   {
      ETG_TRACE_ERR(("Response too long: %i", u32LengthByte_HostOrder));
      vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
      return;
   }

   if ((u8Command != ArkamysStaticMessage::ArkamysOpTypeStatus))
   {
      ETG_TRACE_ERR(("Unknown command %i", u8Command));
      if(u8Command == ArkamysStaticMessage::ArkamysOpTypeError)
      {
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
      }
      return;
   }


   tU32 dataSize =   3 +      //Ack_Cmd, Error status
                 4 +      // data length
        //   1 +      // command
                 //    4 + 4 +  // Address, Length
                     u32LengthByte_HostOrder; // Data Size
   tU8 *pReplyBuf = OSAL_NEW tU8[dataSize];
   if (pReplyBuf == OSAL_NULL) // lint
   {
      return;
   }

    pReplyBuf[0] = TUNING_AckCmd; //ack_cmd
    pReplyBuf[1] = pReplyBuf[2] = 0x00; // error status
    tU32 tempSize = dataSize - 7; // data length
    ETG_TRACE_USR4(("tempSize : %i (%i)", tempSize, dataSize));
    OSAL_pvMemoryCopy(&pReplyBuf[3], &tempSize, sizeof(tU32));

   OSAL_pvMemoryCopy(&pReplyBuf[7], pS32Data, u32LengthByte_HostOrder);


   tuningService -> vDataTx(TuningDataService::ID_ARKAMYS_RNAIVI, m_txHandle, pReplyBuf, dataSize);

   OSAL_DELETE[] pReplyBuf;
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::IMXMessageRx()
 *
 * DESCRIPTION:This method receives messages from the IMX library
 *              and relays them via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: tU32 len, tU8* pU8Data
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::IMXMessageRx(tU32 len, tU8* pU8Data)
{
   if (!m_bHandleValid || !m_bCommExists)
   {
      return;
   }


   ETG_TRACE_USR4(("Got IMX Arkamys Data: Len:%i ", len));

   len -= (LENGTH_OFFSET);//remove error code and data length field
   ETG_TRACE_USR4(("IMX Arkamys updated Data Len:%i ", len));

   if(!len)
   {
        ETG_TRACE_ERR(("Response too short: %i", len));
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        return;
   }

   TuningDataService* tuningService = TuningDataService::getInstance();

   tU32 dataSize =   3 +      //Ack_Cmd, Error status
                 4 +      // data length
           len;     // Data Size
   tU8 *pReplyBuf = OSAL_NEW tU8[dataSize];
   if (pReplyBuf == OSAL_NULL) // lint
   {
      return;
   }

    pReplyBuf[0] = TUNING_AckCmd; //ack_cmd
    pReplyBuf[1] = pReplyBuf[2] = 0x00; // error status

    OSAL_pvMemoryCopy(&pReplyBuf[3], &len, sizeof(tU32));

    OSAL_pvMemoryCopy(&pReplyBuf[7], &pU8Data[8], len);

    tuningService -> vDataTx(TuningDataService::ID_ARKAMYS_RNAIVI, m_txHandle, pReplyBuf, dataSize);

    OSAL_DELETE[] pReplyBuf;
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::PublicParamMessageRx()
 *
 * DESCRIPTION:This method receives messages for the mute, volume and bypass status
 *              and relays them via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: tU32 len, tU8* pU8Data
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::PublicParamMessageRx(tU8 u8PublicParam, tU8 u8Data)
{
   if (!m_bHandleValid || !m_bCommExists)
   {
      return;
   }

   ETG_TRACE_USR4(("status message for %i public parameter", ETG_CENUM( tenTuningProcLibID, u8PublicParam)));

   TuningDataService* tuningService = TuningDataService::getInstance();

   tU32 dataSize =   3 +      //Ack_Cmd, Error status
                 4 +      // data length
           1;       // Data Size
   tU8 *pReplyBuf = OSAL_NEW tU8[dataSize];
   if (pReplyBuf == OSAL_NULL) // lint
   {
      return;
   }

    pReplyBuf[0] = TUNING_AckCmd; //ack_cmd
    pReplyBuf[1] = pReplyBuf[2] = 0x00; // error status
    tU32 tempSize = dataSize - 7; // data length
    OSAL_pvMemoryCopy(&pReplyBuf[3], &tempSize, sizeof(tU32));
    pReplyBuf[7] = u8Data;

    tuningService -> vDataTx(TuningDataService::ID_ARKAMYS_RNAIVI, m_txHandle, pReplyBuf, dataSize);

    OSAL_DELETE[] pReplyBuf;
}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::FMEnhancerMessageRx()
 *
 * DESCRIPTION:This method receives messages for fm enhancer status
 *              and relays them via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: tU8 u8PublicParam, SetFilterData data.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::FMEnhancer1MessageRx(tU8 u8PublicParam, SetFilterData data)
{
   if (!m_bHandleValid || !m_bCommExists)
   {
      return;
   }


   ETG_TRACE_USR4(("status message for %i fm enhancer1", ETG_CENUM( tenTuningProcLibID, u8PublicParam)));

   m_u8FMEnhancer1Data[u8FilterLenCount]       = data.u8FilterCharacteristic;
   u8FilterLenCount++;
   m_u8FMEnhancer1Data[u8FilterLenCount]       = data.u8Order;
   u8FilterLenCount++;
   m_u8FMEnhancer1Data[u8FilterLenCount]       = (tU8) (((data.u16Frequency) & 0xFF00)>>8);
   u8FilterLenCount++;
   m_u8FMEnhancer1Data[u8FilterLenCount]       = (tU8) ((data.u16Frequency) & 0x00FF);
   u8FilterLenCount++;
   m_u8FMEnhancer1Data[u8FilterLenCount]       = (tU8) (((data.s16Gain) & 0xFF00)>>8);
   u8FilterLenCount++;
   m_u8FMEnhancer1Data[u8FilterLenCount]       = (tU8) ((data.s16Gain) & 0x00FF);
   u8FilterLenCount++;
   m_u8FMEnhancer1Data[u8FilterLenCount]       = data.u8QFactor;
   u8FilterLenCount++;

   ETG_TRACE_USR4(("fm enhancer1 len count :%i", u8FilterLenCount));

   if(u8FilterLenCount == LEN_OF_FILTER_PARAMETERS)
   {

   TuningDataService* tuningService = TuningDataService::getInstance();
   tU32 len = LEN_OF_FILTER_PARAMETERS; // 3 filter parameters * 7 bytes of data
   tU32 dataSize =   3 +      //Ack_Cmd, Error status
           4 +      // data length
           len; // Data Size
   tU8 *pReplyBuf = OSAL_NEW tU8[dataSize];
   if (pReplyBuf == OSAL_NULL) // lint
   {
    return;
   }

  pReplyBuf[0] = TUNING_AckCmd; //ack_cmd
  pReplyBuf[1] = pReplyBuf[2] = 0x00; // error status
   // tU32 tempSize = dataSize - 7; // data length
  OSAL_pvMemoryCopy(&pReplyBuf[3], &len, sizeof(tU32));

  OSAL_pvMemoryCopy(&pReplyBuf[7], m_u8FMEnhancer1Data, LEN_OF_FILTER_PARAMETERS);

  tuningService -> vDataTx(TuningDataService::ID_ARKAMYS_RNAIVI, m_txHandle, pReplyBuf, dataSize);

  OSAL_DELETE[] pReplyBuf;

  //reset counter and flag after reading the data
  u8FilterLenCount = 0;
  m_bFMEnhancerGetReq = false;
  ETG_TRACE_USR4(("fm enhancer1 len count after reset:%i", u8FilterLenCount));
   }

}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vStoreAndSendSDEQParams()
 *
 * DESCRIPTION:This method stores the SDEQ data from tuning tool which can be used to send the response.
 *             and forwards the same to ADR3 also
 *
 * PARAMETER: tU32 u32WriteLen, tU8* u8WriteData
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vStoreAndSendSDEQParams(tU32 u32WriteLen, tU8* u8WriteData)
{
   ETG_TRACE_USR4(("Store SDEQ Data: Len:%x ", u32WriteLen));

   if((u32WriteLen) && (u32WriteLen <= (MAX_LEN_OF_SDEQ_PARAMETERS - 1)))
   {
     ETG_TRACE_USR4(("Updating SDEQ data: Size:%i, data: %x", u32WriteLen, ETG_LIST_LEN(u32WriteLen), ETG_LIST_PTR_T8(u8WriteData)));
     //store the values in a buffer
     m_u8SDEQData[0] = 36;//hardcoded to 36 pairs as entire data has to be sent to ADR3 even from tuning tool & on start-up
     for(tU32 i = 1; i <= u32WriteLen; i++)
     {
       m_u8SDEQData[i] = u8WriteData[i - 1];
     }

     //send the stored values to adr3
     adr3_tenDataFunctionID enFileID =  SDVC_FILTER_DATA_ID;
     adr3_tclParser oParser(enFileID);               // data from pool
     adr3_tclSerializer* poSerializer = NULL;        // modified data
     adr3_tclStreamerBase* poStreamer = &oParser;    // either pool or modified data

     // modify sound configuration file, if project provides modifier function
     /*pfnADRFileModifier pfnFileModifier = pfnGetModifierFunction(enFileID);
       if (pfnFileModifier)
       {*/
     ETG_TRACE_USR1(("ArkamysTuningHandlerRNAIVI: vStoreAndSendSDEQParams now modifying File: %s", oParser.sGetName()));

     // parse into structured file object
     adr3_tclFile oFile(enFileID , oParser);

     // call modifier function provided by project
     //(*pfnFileModifier)(oFile, u32MsgLen, &m_pU8Data[8]);
     tBool bTriggerUpdateOnChange = false;
     vModify_SDVCFilter(oFile, MAX_LEN_OF_SDEQ_PARAMETERS, m_u8SDEQData, bTriggerUpdateOnChange);

     ETG_TRACE_USR1(("ArkamysTuningHandlerRNAIVI: vStoreAndSendSDEQParams data changed: %i", bTriggerUpdateOnChange));

     // Send the file if there is change in data
     if(bTriggerUpdateOnChange)
     {
       if (et_bIsTraceActive((tU16)TR_CLASS_VD_ADR3_MAIN, (tU16)TR_LEVEL_USER_3))
         oFile.vPrintLog(NULL);

       // stream back to linear buffer
       poSerializer = new adr3_tclSerializer (oFile);
       if (poSerializer)
       {
         poStreamer = poSerializer; // use modified data
       }
       else
       {
         ETG_TRACE_FATAL(("vd_adr3_config :: vCheckSoundConfigVersion():  poSerializer is NULL"));
       }

       ETG_TRACE_USR1(("Sending  %s  to ADR3 "
           , poStreamer->sGetName()));
       if (et_bIsTraceActive((tU16)TR_CLASS_VD_ADR3_MAIN, (tU16)TR_LEVEL_USER_3))
         poStreamer->vDumpHex(NULL);

       (tVoid) poStreamer->bSendToADR(enFileID);
     }
   // clean-up
     if (poSerializer)
       delete poSerializer;

   }

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::SDEQMessageRx()
 *
 * DESCRIPTION:This method receives messages for the SDEQ status
 *              and relays them via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: tU32 len, tU8* pU8Data
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::SDEQMessageRx(tU32 len, tU8* pU8Data)
{
   if (!m_bHandleValid || !m_bCommExists)
   {
      return;
   }


   ETG_TRACE_USR4(("Got SDEQ HP Data: Len:%x ", len));

   TuningDataService* tuningService = TuningDataService::getInstance();

   tU32 dataSize =   3 +      //Ack_Cmd, Error status
                 4 +      // data length
           len; // Data Size
   tU8 *pReplyBuf = OSAL_NEW tU8[dataSize];
   if (pReplyBuf == OSAL_NULL) // lint
   {
      return;
   }

    pReplyBuf[0] = TUNING_AckCmd; //ack_cmd
    pReplyBuf[1] = pReplyBuf[2] = 0x00; // error status

    OSAL_pvMemoryCopy(&pReplyBuf[3], &len, sizeof(tU32));

    OSAL_pvMemoryCopy(&pReplyBuf[7], pU8Data, len);

    tuningService -> vDataTx(TuningDataService::ID_ARKAMYS_RNAIVI, m_txHandle, pReplyBuf, dataSize);

    OSAL_DELETE[] pReplyBuf;
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vAdrOkCallback()
 *
 * DESCRIPTION:This method is called when ADR connection is established
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vAdrOkCallback()
{
   ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI::adrOkCallback()"));
   if(m_bADR3SendReq)
   {
     if (ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pADRInterface != OSAL_NULL)
     {
       if(! (ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pADRInterface-> bExistMsgToSend()) )
       {
        //send acknowledgement only for the ADR3 write command received
         ETG_TRACE_USR2(("ArkamysTuningHandlerRNAIVI::adrOkCallback() send ack for ADR write command"));
           vSendAckToClient(TUNING_WriteCmd);
         m_bADR3SendReq = false;
       }
     }
     else
     {
       ETG_TRACE_ERR(("ArkamysFeatureHandler::getInstance() -> m_pArkamysADRInterface == OSAL_NULL"));
     }
   }
}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSendData()
 *
 * DESCRIPTION:This method receives the data to be sent to ADR, iMX or public parameters and relays it respectively.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSendData()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendData Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  if(m_bCommExists)
  {
  //handle different processors (ADR3, iMX, public params)
    switch(m_pU8Data[1])
    {
    case Tuning_Proc_Public:
      vSendPublicParamData();
      break;
    case Tuning_Proc_ADR3:
      vSendADR3Data();
      break;
    case Tuning_Proc_IMX6:
      vSendIMXData();
      break;
    default:
      ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI::vSendData() received invalid proc id"));
      break;
    }
  }
  else
    ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendData communication with the system is : %i (closed)", m_bCommExists));

}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSendIMXData()
 *
 * DESCRIPTION: This method receives the data to be sent to IMX and relays it respectively.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSendIMXData()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendIMXData Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(fc_audiomanager_tclApp::theServer()->poDBus());
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(fc_audiomanager_tclApp::theServer()->poDBus()->poGetAudioProcProxy());

  tS32 s32errorCode = TUNING_BOSCH_LIB_ERROR_CODE;
  tU32 dataIndex = 4;// start point of the data to be read

  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendIMXData Received %i number of commands. ", m_pU8Data[3]));

  if(m_pU8Data[0] == TUNING_ReadCmd)
    m_pU8Data[3] = 1;// for read commands explicitly set the number of commands to 1

  while(dataIndex < m_u32Size)
  {

    if(TUNING_MSG_FOOTER_VALUE == m_pU8Data[dataIndex])
      break;// end of msg reached

    if(m_u32Size < (dataIndex + (3 * TUNING_DATA_OFFSET_IVI)) )
    {
      vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);//invalid length
      break;
    }
    tU32 u32BlockID = 0;
    tU32 u32ParamID = 0;
    tU32 u32IMXDataLen = 0;
    tU32 bufferSize = 0;

    //for multi-frame write commands only
    if((m_pU8Data[3] > 1) && (m_pU8Data[0] == TUNING_WriteCmd))
    {
      //removing 4 bytes of data - 1 byte end marker, 3 bytes header.
      //number of commands is sent to AudioProcess
      bufferSize = m_u32Size - TUNING_DATA_OFFSET_IVI;
      dataIndex -= 1;//to include the number of commands
      ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendIMXData number of multiframe commands (%d)", m_pU8Data[dataIndex]));
      vSetIMXData(bufferSize, &m_pU8Data[dataIndex], s32errorCode, AUD_PROC_WRITE_MULTI_FRAME_PARA_ID);
    }
    else if (m_pU8Data[3] == 1)
    {

      OSAL_pvMemoryCopy(&u32BlockID, &m_pU8Data[dataIndex], sizeof(tU32));
      OSAL_pvMemoryCopy(&u32ParamID, &m_pU8Data[dataIndex + (TUNING_DATA_OFFSET_IVI)], sizeof(tU32));
      OSAL_pvMemoryCopy(&u32IMXDataLen, &m_pU8Data[dataIndex + (2 * TUNING_DATA_OFFSET_IVI)], sizeof(tU32));

      ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendIMXData Received %i block id,  %i param id, %i length",
          u32BlockID, u32ParamID, u32IMXDataLen));

      if(m_pU8Data[0] == TUNING_WriteCmd)
      {
        bufferSize = (3 * TUNING_DATA_OFFSET_IVI) + u32IMXDataLen;
        vSetIMXData(bufferSize, &m_pU8Data[dataIndex], s32errorCode, AUD_PROC_WRITE_PARA_ID);
      }
      else if(m_pU8Data[0] == TUNING_ReadCmd)
      {
        bufferSize = (3 * TUNING_DATA_OFFSET_IVI);
        vGetIMXData(bufferSize,  &m_pU8Data[dataIndex], u32IMXDataLen);
      }

    }
    else
    {
      break;//break out of the loop
    }
    dataIndex += bufferSize;
  }

  //ack should be sent if there is no error in the received command
  if(m_pU8Data[0] == TUNING_WriteCmd)
  {
    if(!s32errorCode)
      vSendAckToClient(m_pU8Data[0]);
  }
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSetIMXData()
 *
 * DESCRIPTION: This method receives the data to be set to IMX6 and relays it respectively.
 *
 * PARAMETER: tU32 u32WriteLen, tU8* u8WriteData, tS32 & s32ErrorCode, tU32 u32ParamID
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSetIMXData(tU32 u32WriteLen, tU8* u8WriteData, tS32 & s32ErrorCode, tU32 u32ParamID)
{
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(fc_audiomanager_tclApp::theServer()->poDBus()->poGetAudioProcProxy());

  tU8 *u8IMXData = OSAL_NEW tU8[u32WriteLen];
  if (u8IMXData == OSAL_NULL) // Lint wants a check
  {
      ETG_TRACE_FATAL(("Could not allocate %i bytes.", u32WriteLen));
      return;
  }
  OSAL_pvMemoryCopy(u8IMXData, u8WriteData, u32WriteLen);

  tU8 *u8IMXResult = OSAL_NEW tU8[TUNING_DATA_OFFSET_IVI];

   ETG_TRACE_USR4(("Invoking AudProcWrite via DBus here"));
   AudProc_Error_e errVal = fc_audiomanager_tclApp::theServer()->poDBus()->poGetAudioProcProxy()->AudprocWriteData_sync(AUD_PROC_SRC_DEFAULT, u32ParamID, u8IMXData, u32WriteLen, u8IMXResult);
   if(errVal == AUDPROC_ERR_OK)
   {
     OSAL_pvMemoryCopy( &s32ErrorCode, u8IMXResult, sizeof(tS32));
     ETG_TRACE_USR2(("Audio Process returned with error code %i for write command", s32ErrorCode));

     //send error acknowledgement for the commands received if any
     if(s32ErrorCode)
       vSendErrToClient((tS16)s32ErrorCode);
   }
   else
   {
     ETG_TRACE_USR4(("Audio Process error in writing the data %i", ETG_CENUM(AudProc_Error_e, errVal)));
     s32ErrorCode = (tS32)errVal;
     vSendErrToClient(errVal);
   }

   OSAL_DELETE[]u8IMXData;
   OSAL_DELETE[]u8IMXResult;
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vGetIMXData()
 *
 * DESCRIPTION: This method receives the data to be read from IMX6 and relays it respectively.
 *
 * PARAMETER: tU32 u32WriteLen, tU8* u8WriteData, tU32 u32ReadLen.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vGetIMXData(tU32 u32WriteLen, tU8* u8WriteData, tU32 u32ReadLen)
{
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(fc_audiomanager_tclApp::theServer()->poDBus()->poGetAudioProcProxy());

  tU8 *u8IMXSendData = OSAL_NEW tU8[u32WriteLen];
   if (u8IMXSendData == OSAL_NULL) // Lint wants a check
      {
         ETG_TRACE_FATAL(("Could not allocate %i bytes.", u32WriteLen));
         return;
      }

  OSAL_pvMemoryCopy(u8IMXSendData, u8WriteData, u32WriteLen);

  //ToDo: reduce the length to 8 bytes once the reset is fixed in audioprocess proxy
  tU32 u32ReadBufferSize = u32ReadLen + (5 * LENGTH_OFFSET);//error code and data length added from audioprocess
  tU8 *u8IMXReceiveData = OSAL_NEW tU8[u32ReadBufferSize];
   if (u8IMXReceiveData == OSAL_NULL) // Lint wants a check
      {
         ETG_TRACE_FATAL(("Could not allocate %i bytes.", u32ReadLen));
         return;
      }
   ETG_TRACE_USR4(("Length to be read %d", u32ReadLen));
   ETG_TRACE_USR4(("Invoking AudProcRead via DBus here"));

   AudProc_Error_e errVal = fc_audiomanager_tclApp::theServer()->poDBus()->poGetAudioProcProxy()->AudprocReadData_sync(AUD_PROC_SRC_DEFAULT, AUD_PROC_READ_PARA_ID, u8IMXSendData, u32WriteLen, u8IMXReceiveData, u32ReadLen);

   if(errVal == AUDPROC_ERR_OK)
   {
       //if read length is 4 then send error code
     if(u32ReadLen == (TUNING_DATA_OFFSET_IVI))
     {
       tS32 s32errorCode = 0 ;
       OSAL_pvMemoryCopy(&s32errorCode, u8IMXReceiveData, sizeof(tS32));
       ETG_TRACE_USR4(("Audio Process returned with error code %i for read command", s32errorCode));
       vSendErrToClient((tS16)s32errorCode);
     }
     else
     {
       IMXMessageRx(u32ReadLen, u8IMXReceiveData);//4 bytes-error code, 4 bytes - length, rest actual data
     }
   }
   else
   {
     ETG_TRACE_USR4(("Audio Process error in reading the data %i", ETG_CENUM(AudProc_Error_e, errVal)));
     vSendErrToClient(errVal);
   }

   OSAL_DELETE[]u8IMXSendData;
   OSAL_DELETE[]u8IMXReceiveData;

}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSendADR3Data()
 *
 * DESCRIPTION: This method receives the data to be sent to ADR3 and relays it respectively.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSendADR3Data()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendADR3Data Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  // Discard too big messages
  // deduct 1 byte end marker - ee
  if ((m_u32Size - 1) > MIN(AUD_ADR_MAX_PAYLOAD, ARKAMYS_ADR_MAX_MESSAGE_SIZE_BYTES))
  {
    ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI: vSendADR3Data Message size too big: %i > %i.", m_u32Size, MIN(AUD_ADR_MAX_PAYLOAD, ARKAMYS_ADR_MAX_MESSAGE_SIZE_BYTES)));
    vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);//invalid length
    return;
  }

  tU8 u8NumOfWriteCmd = 0;
  tU32 tempSize;
  tU32 dataIndex = 4;// start point of the data to be read

  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendADR3Data Received %i number of commands. ", m_pU8Data[3]));

  tU8 u8Command = 0;

  if(m_pU8Data[0] == TUNING_ReadCmd)
  {
    m_pU8Data[3] = 1;// for read commands explicitly set the number of commands to 1
    u8Command = ArkamysStaticMessage::ArkamysOpTypeGet;
  }
  else if(m_pU8Data[0] == TUNING_WriteCmd)
  {
    u8Command = ArkamysStaticMessage::ArkamysOpTypeSet;
  }

  m_bADR3SendReq = true;

  while(dataIndex < m_u32Size)
  {

    if(TUNING_MSG_FOOTER_VALUE == m_pU8Data[dataIndex])
      break;// end of msg reached

    if(m_u32Size < (dataIndex + (2 * TUNING_DATA_OFFSET_IVI)) )
    {
      vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);//invalid length
      m_bADR3SendReq = false;
      break;
    }
    OSAL_pvMemoryCopy(&tempSize, &m_pU8Data[dataIndex + TUNING_DATA_OFFSET_IVI], sizeof(tU32));
    tempSize = ntohl(tempSize); //convert network to host byte order
    tempSize *= 4; //converting words to bytes
    tempSize += 8; // address + length (both 4 bytes)


    if (u8NumOfWriteCmd < m_pU8Data[3])
    {
      vValidateADRCommand(&m_pU8Data[dataIndex], tempSize);
      u8NumOfWriteCmd++;
    }
    dataIndex += tempSize;

    }

    if(m_bADR3SendReq)//send the data to ADR only if there are no errors in the commands received
    {
      ArkamysStaticMessage oMsg;
      oMsg.command = u8Command;
      oMsg.size = m_u32Size - 5; //4bytes header + 1byte endmarker
      OSAL_pvMemoryCopy(oMsg.data, &m_pU8Data[TUNING_DATA_OFFSET_IVI], oMsg.size);

      ETG_TRACE_USR4(("Command:%d Size:%i", ETG_CENUM(ArkamysStaticMessage::tenAdrOpType, oMsg.command), oMsg.size));

      if (oMsg.command == ArkamysStaticMessage::ArkamysOpTypeGet)
        m_bADR3SendReq = false;//reset m_bADR3SendReq flag to prevent sending invalid Ack to tuning tool

      if (oMsg.command == ArkamysStaticMessage::ArkamysOpTypeSet)
      {
        ETG_TRACE_USR4(("Raw Data:%02x", ETG_LIST_LEN(oMsg.size), ETG_LIST_PTR_T8(oMsg.data)));
      }


      if (ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pADRInterface != OSAL_NULL)
      {
        ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pADRInterface -> SendToADR(oMsg);
      }

      else
      {
        ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pADRInterface == OSAL_NULL"));
      }
    }

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vValidateADRCommand()
 *
 * DESCRIPTION: This method is used to validate the address and length of each ADR3 command
 *
 * PARAMETER: const tU8* pData, tU32 length
 *
 * RETURNVALUE: tVoid
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vValidateADRCommand(const tU8* pData, tU32 length)
{
   tU32 address;
   OSAL_pvMemoryCopy(&address, &pData[0], sizeof(tU32));   // Address
   address = ntohl(address);//convert network to host byte order

   ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI::vValidateADRCommand(): address after conversion Adr:%p "
         ,address));

   tU32 dataLength ;
   OSAL_pvMemoryCopy(&dataLength, &pData[4], sizeof(tU32));   // Length
   dataLength = ntohl(dataLength);//convert network to host byte order

   ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI::vValidateADRCommand(): length after conversion length:%p "
         ,dataLength));

   const tU32 headerSize  = 2 * sizeof(tU32); // Address + Length (both 4 byte)
   const tU32 messageSize = (tU32)((dataLength * sizeof(tU32)) + headerSize);
   ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI::vValidateADRCommand(): Adr:%p  Length:%i headerSize:%i msgSize:%i"
         ,address, dataLength, headerSize, messageSize));

   if (address == 0)
   {
      ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI: vValidateADRCommand (%p, %i): Target address == 0. Will not send.", pData, length));
      vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);//invalid length
      m_bADR3SendReq = false;
      return;
   }
   if ((dataLength + headerSize) > length)
   {
      ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI: vValidateADRCommand(): Data Length (%u) would cause buffer read overflow (length: %u). Will not send.", dataLength, length));
      vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);//invalid length
      m_bADR3SendReq = false;
      return;
   }
}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSendPublicParamData()
 *
 * DESCRIPTION: This method receives the data to be sent to public parameters and relays it respectively.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSendPublicParamData()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendPublicParamData Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  //handle different parameters (volume, mute, SDEQ, FM Enhancer, ByPass ADR3)
  switch(m_pU8Data[2])
  {
  case Tuning_Lib_Volume:
    vSendVolume();
    break;
  case Tuning_Lib_Mute:
    vSendMute();
    break;
  case Tuning_Lib_SDEQ:
    vSendSDEQ();
    break;
  case Tuning_Lib_FMEnhancer1:
    vSendFMEnhancer1();
    break;
  case Tuning_Lib_ByPass:
    vSendByPass();
    break;
  default:
    ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI::vSendPublicParamData() received invalid Lib id"));
    break;
  }

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSendVolume()
 *
 * DESCRIPTION: This method sends the volume message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSendVolume()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendVolume Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);
  if(m_pU8Data[0] == TUNING_WriteCmd)
    vSetVolume();
  else
    vGetVolume();

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSetVolume()
 *
 * DESCRIPTION: This method sends the set volume message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSetVolume()
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);
  tU32 u32MsgLen;

  tU8 dataIdx = 4;
  OSAL_pvMemoryCopy(&u32MsgLen, &m_pU8Data[dataIdx], sizeof(tU32));

  dataIdx = (tU8) (dataIdx + TUNING_DATA_OFFSET_IVI);

  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSetVolume Received %i bytes of volume data. ", u32MsgLen));


  if(u32MsgLen < 2 || u32MsgLen % 2 != 0)
  {
        ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI:vSetVolume Received invalid Volume Length."));
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        return;
  }

  tU32 u32LenCount = 0;
  while( u32LenCount < u32MsgLen)
  {
        if(TUNING_MSG_FOOTER_VALUE == m_pU8Data[dataIdx + u32LenCount])
          break;// end of msg reached

    // setup outgoing PO message
    VolumeData out;

    out.setVolumeType((midw_fi_tcl_e8_AudioVolumeType::tenType)m_pU8Data[dataIdx + u32LenCount]);
    u32LenCount = u32LenCount + 1;
    out.setVolume(m_pU8Data[dataIdx + u32LenCount]);
    u32LenCount = u32LenCount + 1;

    out.setResource(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
    out.m_Channel = midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_NOT_DEF;  // don't have this information here
    //tS16 m_VolumeGain;


    ID_CCA_Start_Volume outMsg(out);
    InternalCommunicationAdapter::POMessages->DeliverMsg(&outMsg);

      ETG_TRACE_USR2(("ArkamysTuningHandlerRNAIVI - vSetVolume: - forwarded Volume.Set for Type = %d, Resource = %d, Volume = %d"
           , out.getVolumeType(), out.getResource(), out.getVolume() ));
  }

  //send acknowledgement for the commands received
  vSendAckToClient(m_pU8Data[0]);
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vGetVolume()
 *
 * DESCRIPTION: This method sends the get volume message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vGetVolume()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vGetVolume entered "));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  m_bVolGetReq = true;
  tU8 Stream = (tU8)EN_AUDIO_SOURCE_STREAM_MAIN;
  vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_VOLUME, VD_ADR3_OPTYPE_GET
         , 1, &Stream, VOLUME_EXC_GET);//send data to ADR3

  //send acknowledgement for the commands received
  //vSendAckToClient(m_pU8Data[0]);
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSendMute()
 *
 * DESCRIPTION: This method sends the mute message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSendMute()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendMute Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);
  if(m_pU8Data[0] == TUNING_WriteCmd)
    vSetMuteDemute();
  else
    vGetMuteDemute();

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSetMuteDemute()
 *
 * DESCRIPTION: This method sends the set mute/demute message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSetMuteDemute()
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  tU32 u32MsgLen;
  tU8 dataIdx = 4;

  OSAL_pvMemoryCopy(&u32MsgLen, &m_pU8Data[dataIdx], sizeof(tU32));

  dataIdx = (tU8) (dataIdx + TUNING_DATA_OFFSET_IVI);

  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSetMuteDemute Received %i bytes of mute data. ", u32MsgLen));

  if(u32MsgLen != 1)
  {
        ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI:vSetMuteDemute Received invalid mute Length."));
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        return;
  }

  tU32 u32LenCount = 0;
  while( u32LenCount < u32MsgLen)
  {
        if(TUNING_MSG_FOOTER_VALUE == m_pU8Data[dataIdx + u32LenCount])
          break;// end of msg reached
        tU8 muteState = m_pU8Data[dataIdx + u32LenCount];
        u32LenCount++;

        ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSetMuteDemute Received mute state %i  . ", muteState));
        m_bMuteSetReq = true;
        if(muteState) //state 0 : mute, state 1: demute
        {
          AUD_POINTER_CHECK_CALL_NO_RET_VALUE(fc_audiomanager_tclApp::theServer()->poDBus()->poGetAudioProcProxy());
          tBool bRetVal = fc_audiomanager_tclApp::theServer()->poDBus()->poGetAudioProcProxy()->set_aplArkamysFlush(!muteState);//parameter mute-state is not used
          ETG_TRACE_USR4(("Audio Process aplArkamysFlush function call returned with (%d)", bRetVal));
          Demute(m_u8MuteRegId);
        }
        else
        {
          Mute(m_u8MuteRegId);
        }

}

  //send acknowledgement for the commands received
  //vSendAckToClient(m_pU8Data[0]);
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vGetMuteDemute()
 *
 * DESCRIPTION: This method sends the get mute/demute message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vGetMuteDemute()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vGetMuteDemute entered "));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  m_bMuteGetReq = true;
  tU8 Stream = (tU8)EN_AUDIO_SOURCE_STREAM_MAIN;
  vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_MUTE, VD_ADR3_OPTYPE_GET
         , 1, &Stream, MUTE_EXC);//send data to ADR3

  //send acknowledgement for the commands received
  //vSendAckToClient(m_pU8Data[0]);
}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSendSDEQ()
 *
 * DESCRIPTION: This method sends the SDEQ message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSendSDEQ()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendSDEQ Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);
  if(m_pU8Data[0] == TUNING_WriteCmd)
    vSetSDEQ();
  else
    vGetSDEQ();

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSetSDEQ()
 *
 * DESCRIPTION: This method sends the set SDEQ message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSetSDEQ()
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  tU8 dataIdx = 4;
  tU32 u32MsgLen;
  OSAL_pvMemoryCopy(&u32MsgLen, &m_pU8Data[dataIdx], sizeof(tU32));

  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSetSDEQ Received %i bytes of SDEQ HP data. ", u32MsgLen));

  if(u32MsgLen < 3 || u32MsgLen % 2 != 1)
  {
        ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI: vSetSDEQ Received invalid SDEQ Length."));
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        return;
  }

  tU8 u8PairsofSDEQParams;
  dataIdx = (tU8)(dataIdx * 2);//point to pair count in the message from tuning tool
  OSAL_pvMemoryCopy(&u8PairsofSDEQParams, &m_pU8Data[dataIdx], sizeof(tU8));
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSetSDEQ Received %i pairs of SDEQ HP data. ", u8PairsofSDEQParams));

  dataIdx = (tU8) (dataIdx + 1); //point to speed, frequency data

  //store values only if valid data length is received, deduct 1 byte to remove the pair count from tuning tool message
  vStoreAndSendSDEQParams((u32MsgLen-1),  &m_pU8Data[dataIdx]);

  //send acknowledgement for the commands received
  vSendAckToClient(m_pU8Data[0]);
}


/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vGetSDEQ()
 *
 * DESCRIPTION: This method sends the get SDEQ message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vGetSDEQ()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vGetSDEQ entered "));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  tU8 dataIdx = 4;
  tU32 u32MsgLen;
  OSAL_pvMemoryCopy(&u32MsgLen, &m_pU8Data[dataIdx], sizeof(tU32));

  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vGetSDEQ Request %i bytes of SDEQ HP data. ", u32MsgLen));

  if(!u32MsgLen)
  {
        ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI: vGetSDEQ Received invalid SDEQ Length."));
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        return;
  }

    //send the stored response back to tuning tool
  SDEQMessageRx(u32MsgLen, &m_u8SDEQData[1]);
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSendFMEnhancer1()
 *
 * DESCRIPTION: This method sends the FMEnhancer message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSendFMEnhancer1()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendFMEnhancer1 Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);
  if(m_pU8Data[0] == TUNING_WriteCmd)
    vSetFMEnhancer1();
  else
    vGetFMEnhancer1();
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSetFMEnhancer1()
 *
 * DESCRIPTION: This method sends the set FMEnhancer1 message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSetFMEnhancer1()
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  tU32 u32MsgLen;
  tU8 dataIdx = 4;
  OSAL_pvMemoryCopy(&u32MsgLen, &m_pU8Data[dataIdx], sizeof(tU32));

  dataIdx = (tU8) (dataIdx + TUNING_DATA_OFFSET_IVI);

  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSetFMEnhancer1 Received %i bytes of FM Enhancer data. ", u32MsgLen));

  if(u32MsgLen < 7 || u32MsgLen % 7 != 0)
  {
        ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI: vSetFMEnhancer1 Received invalid FMEnhancer Length."));
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        return;
  }

  tU32 u32LenCount; tU8 u8FilterCount;
  u32LenCount = 0 ; u8FilterCount = 0;

  while( u32LenCount < u32MsgLen )
  {
        if(TUNING_MSG_FOOTER_VALUE == m_pU8Data[dataIdx + u32LenCount])
          break;// end of msg reached

      if( u8FilterCount < NO_OF_FILTER_PARAMETERS )
      {
      // setup outgoing PO message
      SetFilterData outData;

      outData.u8FilterID = u8FilterCount;//0, 1, 2 are filter IDs for FM enhancer
      outData.u8FilterCharacteristic = m_pU8Data[dataIdx + u32LenCount];
      u32LenCount = u32LenCount + 1;

      outData.u8Order = m_pU8Data[dataIdx + u32LenCount];
      u32LenCount = u32LenCount + 1;

      OSAL_pvMemoryCopy(&outData.u16Frequency, &m_pU8Data[dataIdx + u32LenCount], sizeof(tU16));
      u32LenCount = u32LenCount + 2;

      OSAL_pvMemoryCopy(&outData.s16Gain, &m_pU8Data[dataIdx + u32LenCount], sizeof(tS16));
      u32LenCount = u32LenCount + 2;

      outData.u8QFactor = m_pU8Data[dataIdx + u32LenCount];
      u32LenCount = u32LenCount + 1;


      ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSetFMEnhancer1 u8FilterID %i, u8FilterCharacteristic %i, u8Order %i, u16Frequency %i, s16Gain %i, u8QFactor %i ",
          outData.u8FilterID, outData.u8FilterCharacteristic, outData.u8Order, outData.u16Frequency, outData.s16Gain, outData.u8QFactor));

      MSG_SetFilter oSetFilter(outData);
      InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSetFilter);

      u8FilterCount++;
      }

      ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI:vSetFMEnhancer1 u8FilterCount %i, u32LenCount = %i  ", u8FilterCount, u32LenCount));

  }

  //send acknowledgement for the commands received
  vSendAckToClient(m_pU8Data[0]);

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vGetFMEnhancer1()
 *
 * DESCRIPTION: This method sends the get FMEnhancer message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vGetFMEnhancer1()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vGetFMEnhancer1 entered "));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  m_bFMEnhancerGetReq = true;
    for(tU8 filterCount = 0; filterCount < NO_OF_FILTER_PARAMETERS; filterCount++)
    {
      tU8 u8filterId = filterCount;
      vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_SET_FILTER, VD_ADR3_OPTYPE_GET
             , 1, &u8filterId, (tU8)(GET_FILTER_1 +  u8filterId));//send data to ADR3
    }
  //send acknowledgement for the commands received
  //vSendAckToClient(m_pU8Data[0]);

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSendByPass()
 *
 * DESCRIPTION: This method sends the ByPass message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSendByPass()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSendBypass Received %i bytes of data. ", m_u32Size));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);
  if(m_pU8Data[0] == TUNING_WriteCmd)
    vSetByPass();
  else
    vGetByPass();

}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSetByPass()
 *
 * DESCRIPTION: This method sends the set Bypass message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSetByPass()
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  tU32 u32MsgLen;
  tU8 dataIdx = 4;
  OSAL_pvMemoryCopy(&u32MsgLen, &m_pU8Data[dataIdx], sizeof(tU32));

  dataIdx = (tU8) (dataIdx + TUNING_DATA_OFFSET_IVI);

  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSetByPass Received %i bytes of ByPass ADR3 data. ", u32MsgLen));

  if(u32MsgLen != 1)
  {
        ETG_TRACE_ERR(("ArkamysTuningHandlerRNAIVI:vSetByPass Received invalid Bypass Length."));
        vSendErrToClient((tS16)TUNING_BOSCH_LIB_ERROR_CODE);
        return;
  }

  tU32 u32LenCount = 0;
  while( u32LenCount < u32MsgLen )
  {
        if(TUNING_MSG_FOOTER_VALUE == m_pU8Data[dataIdx + u32LenCount])
          break;// end of msg reached

        tU8 u8Bypass;
        u8Bypass = m_pU8Data[dataIdx + u32LenCount];
        u32LenCount = u32LenCount + 1;

        ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vSetByPass Received %i  ByPass ADR3 data. ", ETG_CENUM(tenArkamysMode, u8Bypass)));
        if(u8Bypass == EN_ARKAMYS_OFF)
        {
          u8Bypass = ArkamysFeatureHandlerRNAIVI::getInstance()->enGetAudioMode();

          //if premium mode is set, change it to classic
          if(u8Bypass == EN_ARKAMYS_PREMIUM1 || u8Bypass == EN_ARKAMYS_PREMIUM2)
            u8Bypass = EN_ARKAMYS_CLASSIC;
        }

      vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_ARKAMYS_MODE, VD_ADR3_OPTYPE_SET
           , 1, &u8Bypass, ARKAMYS_MODE);//send data to ADR3

  }

  //send acknowledgement for the commands received
  vSendAckToClient(m_pU8Data[0]);
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vGetByPass()
 *
 * DESCRIPTION: This method sends the get Bypass message received from tuning tool to adr.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vGetByPass()
{
  ETG_TRACE_USR4(("ArkamysTuningHandlerRNAIVI: vGetByPass entered "));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pU8Data);

  m_bByPassGetReq = true;

  vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_ARKAMYS_MODE, VD_ADR3_OPTYPE_GET
         , 0, OSAL_NULL, ARKAMYS_MODE);//send data to ADR3

  //send acknowledgement for the commands received
  //vSendAckToClient(m_pU8Data[0]);
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSendAckToClient()
 *
 * DESCRIPTION: This method sends the acknowledgement to the received commands via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: tU8 u8CommandID.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSendAckToClient(tU8 u8CommandID)
{
   ETG_TRACE_USR4(("Send Acknowledge to '%i' command", ETG_CENUM(tenTuningCommandID, u8CommandID)));
   tPU8 pu8Message = OSAL_NEW tU8 [TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI];
     if (pu8Message == OSAL_NULL) // Lint wants a check
     {
        ETG_TRACE_FATAL(("Could not allocate memory."));
        return;
     }
    pu8Message[0] = TUNING_AckCmd;
    OSAL_pvMemorySet(&pu8Message[1], 0, TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI - 1); // error status and data length

    TuningDataService::getInstance() -> vDataTx(TuningDataService::ID_ARKAMYS_RNAIVI, m_txHandle, pu8Message, TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI);
      OSAL_DELETE []pu8Message;
}

/*******************************************************************************
 *
 * FUNCTION: ArkamysTuningHandlerRNAIVI::vSendErrToClient()
 *
 * DESCRIPTION: This method sends the error codes to the received commands via TCP/IP to the connected Tuning Tool.
 *
 * PARAMETER: tU16 u16ErrorCode.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid ArkamysTuningHandlerRNAIVI::vSendErrToClient(tS16 s16ErrorCode)
{
   ETG_TRACE_USR4(("Send Error status '%i' to tuning tool", s16ErrorCode));
   tPU8 pu8Message = OSAL_NEW tU8 [TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI];
     if (pu8Message == OSAL_NULL) // Lint wants a check
     {
        ETG_TRACE_FATAL(("Could not allocate memory."));
        return;
     }
    pu8Message[0] = TUNING_AckCmd;
    OSAL_pvMemoryCopy(&pu8Message[1], &s16ErrorCode, sizeof(tS16));
    OSAL_pvMemorySet(&pu8Message[3], 0, TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI - 3); // data length

    TuningDataService::getInstance() -> vDataTx(TuningDataService::ID_ARKAMYS_RNAIVI, m_txHandle, pu8Message, TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE_IVI);
      OSAL_DELETE []pu8Message;
}

