/**
 * @file EngineSoundModel.cpp
 * @brief Implements the data models of ESE feature
 * This class is resposible for maintaing the persistent data of ESE feature
 * @author pau4kor
 * @copyright (c) 2016 RBEI
 * @addtogroup fc_audiomanager
 * @{
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>
#include "fc_audiomanager_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_ENGINESOUND
#include "trcGenProj/Header/EngineSoundModel.cpp.trc.h"
#endif

#define FC_AUDIOMANAGER_S_IMPORT_INTERFACE_DP_FC_AUDIOMANAGER
#include "fc_audiomanager_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_diagdebug_if.h" // For EngineSound EOL Datapool
#include "dp_generic_if.h"
#include "dp_audio_if.h"

#include "EngineSoundModel.h"

#define DPID_SOUNDTYPE_ECO          0
#define DPID_SOUNDTYPE_COMFORT      1
#define DPID_SOUNDTYPE_NORMAL       2
#define DPID_SOUNDTYPE_DYNAMIC      3
#define DPID_SOUNDTYPE_PERSO        4
#define DPID_VOLUME_ECO_NEUTRAL     5
#define DPID_VOLUME_ECO_SOUND1      6
#define DPID_VOLUME_ECO_SOUND2      7
#define DPID_VOLUME_COMFORT_NEUTRAL 8
#define DPID_VOLUME_COMFORT_SOUND1  9
#define DPID_VOLUME_COMFORT_SOUND2  10
#define DPID_VOLUME_NORMAL_NEUTRAL  11
#define DPID_VOLUME_NORMAL_SOUND1   12
#define DPID_VOLUME_NORMAL_SOUND2   13
#define DPID_VOLUME_DYNAMIC_NEUTRAL 14
#define DPID_VOLUME_DYNAMIC_SOUND1  15
#define DPID_VOLUME_DYNAMIC_SOUND2  16
#define DPID_VOLUME_PERSO_NEUTRAL   17
#define DPID_VOLUME_PERSO_SOUND1    18
#define DPID_VOLUME_PERSO_SOUND2    19

#define DP_SET                      (tBool)TRUE
#define DP_GET                      (tBool)FALSE

#define DP_VAL_NO_CHANGE            0

/**
 *@brief Structure to hold the DP-IDs of possible settings in a MEX ESE
 *In each drive mode, user can select 1 sound type (out of 3) and for each sound type user can select a volume (out of 3)
 *The structure onlu holds the Dp IDS where the data is stored for that particular setting.
 */
typedef struct _eseMexSettings
{
   tU8 u8SoundType;
   tU8 au8Volume[3];
}tEseMexSettings;

/**
 *@brief A static array, that hold the Dp-Ids -> where the user settings of each drive mode is stored
 */
static tEseMexSettings ese_mex_settings[] =
{
   {
      DPID_SOUNDTYPE_ECO,
      {DPID_VOLUME_ECO_NEUTRAL,DPID_VOLUME_ECO_SOUND1,DPID_VOLUME_ECO_SOUND2}
   },
   {
      DPID_SOUNDTYPE_COMFORT,
      {DPID_VOLUME_COMFORT_NEUTRAL,DPID_VOLUME_COMFORT_SOUND1,DPID_VOLUME_COMFORT_SOUND2}
   },
   {
      DPID_SOUNDTYPE_NORMAL,
      {DPID_VOLUME_NORMAL_NEUTRAL,DPID_VOLUME_NORMAL_SOUND1,DPID_VOLUME_NORMAL_SOUND2}
   },
   {
      DPID_SOUNDTYPE_DYNAMIC,
      {DPID_VOLUME_DYNAMIC_NEUTRAL,DPID_VOLUME_DYNAMIC_SOUND1,DPID_VOLUME_DYNAMIC_SOUND2}
   },
   {
      DPID_SOUNDTYPE_PERSO,
      {DPID_VOLUME_PERSO_NEUTRAL,DPID_VOLUME_PERSO_SOUND1,DPID_VOLUME_PERSO_SOUND2}
   }
};

EngineSoundModel::EngineSoundModel():
                  m_u8ESEDriveMode(ENGINESOUND_DRIVEMODE_ECO),
                  m_u8ESESoundType(ENGINESOUND_SOUNDTYPE_NEUTRAL),
                  m_u8ESEVolumeLevel(ENGINESOUND_VOLUME_OFF),
                  m_u8RSoundStatus(RSOUND_ONOFF_OFF),
                  m_u8RSoundType(ENGINESOUND_SOUNDTYPE_RSOUND1),
                  m_u8RSoundVolumeLevel(7),
          m_u8RSoundActivationStatus(RSOUND_ACTIVATION_OFF)
{
  vReload();
}

EngineSoundModel::~EngineSoundModel()
{
}

tVoid EngineSoundModel::vReload()
{
   m_u8ESEDriveMode = u8GetESEDriveMode();
   m_u8ESESoundType = u8GetESESoundType(m_u8ESEDriveMode);
   m_u8ESEVolumeLevel = u8GetESEVolumeLevel(m_u8ESEDriveMode,m_u8ESESoundType);

   m_u8RSoundStatus = u8GetRSoundStatus();
   m_u8RSoundType = u8GetRSoundType();
   m_u8RSoundVolumeLevel = u8GetRSoundVolumeLevel();

   m_u8RSoundActivationStatus = u8GetRSoundActivationStatus();

   ETG_TRACE_USR4(("EngineSoundModel, Drive Mode %d, Sound Type %d, Volume %d, RSound Status %d, RSound Type %d, RSound Volume %d, RSoundActivation Status %d", m_u8ESEDriveMode,m_u8ESESoundType,m_u8ESEVolumeLevel,m_u8RSoundStatus,m_u8RSoundType,m_u8RSoundVolumeLevel, m_u8RSoundActivationStatus));
}

/***************************Functions to set/Get data***************************/

tVoid EngineSoundModel::vSetActiveESEDriveMode(tU8 u8DriveMode)
{
   m_u8ESEDriveMode = u8DriveMode;
   vStoreESEDriveMode(u8DriveMode);

   m_u8ESESoundType = u8GetESESoundType(m_u8ESEDriveMode);
   m_u8ESEVolumeLevel = u8GetESEVolumeLevel(m_u8ESEDriveMode,m_u8ESESoundType);
}

tU8 EngineSoundModel::u8GetActiveESEDriveMode() const
{
   return m_u8ESEDriveMode;
}

tVoid EngineSoundModel::vSetActiveESESoundType(tU8 u8SoundType)
{
   m_u8ESESoundType = u8SoundType;
   vStoreESESoundType(m_u8ESEDriveMode,u8SoundType);

   m_u8ESEVolumeLevel = u8GetESEVolumeLevel(m_u8ESEDriveMode,m_u8ESESoundType);
}

tU8 EngineSoundModel::u8GetActiveESESoundType() const
{
   return m_u8ESESoundType;
}

tVoid EngineSoundModel::vSetActiveESEVolumeLevel(tU8 u8VolumeLevel)
{
   m_u8ESEVolumeLevel = u8VolumeLevel;
   vStoreESEVolumeLevel(m_u8ESEDriveMode,m_u8ESESoundType,u8VolumeLevel);
}

tU8 EngineSoundModel::u8GetActiveESEVolumeLevel() const
{
   return m_u8ESEVolumeLevel;
}

tVoid EngineSoundModel::vSetActiveRSoundStatus(tU8 u8Status)
{
   m_u8RSoundStatus = u8Status;
   vStoreRSoundStatus(u8Status);
}

tU8 EngineSoundModel::u8GetActiveRSoundStatus() const
{
   return m_u8RSoundStatus;
}

tVoid EngineSoundModel::vSetActiveRSoundType(tU8 u8SoundType)
{
   m_u8RSoundType = u8SoundType;
   vStoreRSoundType(u8SoundType);
}

tU8 EngineSoundModel::u8GetActiveRSoundType() const
{
   return m_u8RSoundType;
}

tVoid EngineSoundModel::vSetActiveRSoundVolumeLevel(tU8 u8VolumeLevel)
{
   m_u8RSoundVolumeLevel = u8VolumeLevel;
   vStoreRSoundVolumeLevel(u8VolumeLevel);
}

tU8 EngineSoundModel::u8GetActiveRSoundVolumeLevel() const
{
   return m_u8RSoundVolumeLevel;
}

tVoid EngineSoundModel::vSetActiveRSoundActivationStatus(tU8 u8RSoundActivationStatus)
{
  m_u8RSoundActivationStatus = u8RSoundActivationStatus;
  vStoreRSoundActivationStatus(m_u8RSoundActivationStatus);
}

tU8 EngineSoundModel::u8GetActiveRSoundActivationStatus() const
{
   return m_u8RSoundActivationStatus;
}

/***********************************************************************************/

/**************************Functions for Dp Access**********************/

tVoid EngineSoundModel::vStoreESEDriveMode(tU8 u8DriveMode)
{
   dp_tclEngineSoundSettingsDPESEDriveMode oDriveModeDp;

   if(DP_S32_NO_ERR != oDriveModeDp.s32SetData(u8DriveMode))
      ETG_TRACE_ERR(("EngineSoundModel : Failed to store ESE Drive Mode"));

}
tVoid EngineSoundModel::vStoreESESoundType(tU8 u8DriveMode, tU8 u8SoundType)
{
   if(DP_S32_NO_ERR != s32SetGetDpItem(ese_mex_settings[u8DriveMode].u8SoundType, u8SoundType, DP_SET))
      ETG_TRACE_ERR(("EngineSoundModel : Failed to store ESE sound type %d for Mode %d", u8SoundType, u8DriveMode));

}

tVoid EngineSoundModel::vStoreESEVolumeLevel(tU8 u8DriveMode, tU8 u8SoundType, tU8 u8VolumeLevel)
{
  if(DP_S32_NO_ERR != s32SetGetDpItem(ese_mex_settings[u8DriveMode].au8Volume[u8SoundType-1], u8VolumeLevel, DP_SET))
      ETG_TRACE_ERR(("EngineSoundModel : Failed to store ESE volume %d for Mode %d and sound type %d", u8VolumeLevel, u8DriveMode, u8SoundType));
}

tVoid EngineSoundModel::vStoreRSoundStatus(tU8 u8Status)
{
   dp_tclEngineSoundSettingsDPRSoundOnOff oRSoundDp;

   if(DP_S32_NO_ERR != oRSoundDp.s32SetData(u8Status))
      ETG_TRACE_ERR(("EngineSoundModel : Failed to store RSound OnOff status"));
}

tVoid EngineSoundModel::vStoreRSoundType(tU8 u8SoundType)
{
   dp_tclEngineSoundSettingsDPRSoundType oRSoundDp;

   if(DP_S32_NO_ERR != oRSoundDp.s32SetData(u8SoundType))
      ETG_TRACE_ERR(("EngineSoundModel : Failed to store RSound sound type"));
}
tVoid EngineSoundModel::vStoreRSoundVolumeLevel(tU8 u8VolumeLevel)
{
   dp_tclEngineSoundSettingsDPRSoundVolume oRSoundDp;

   if(DP_S32_NO_ERR != oRSoundDp.s32SetData(u8VolumeLevel))
      ETG_TRACE_ERR(("EngineSoundModel : Failed to store RSound volume level"));
}

tVoid EngineSoundModel::vStoreRSoundActivationStatus(tU8 u8RSoundActivationStatus)
{
   dp_tclEngineSoundSettingsDPRSoundActivation oRSoundActivationDp;

   if(DP_S32_NO_ERR != oRSoundActivationDp.s32SetData(u8RSoundActivationStatus))
      ETG_TRACE_ERR(("EngineSoundModel : Failed to store RSoundActivation Status"));

}

tU8 EngineSoundModel::u8GetESEDriveMode()
{
   tU8 u8Val;
   dp_tclEngineSoundSettingsDPESEDriveMode oDriveModeDp;

   if(DP_S32_NO_ERR != oDriveModeDp.s32GetData(u8Val))
   {
      ETG_TRACE_ERR(("EngineSoundModel : Failed to read ESE Drive Mode"));
      u8Val = ENGINESOUND_DRIVEMODE_ECO;
   }

   if(u8Val > ENGINESOUND_DRIVEMODE_PERSO)
      u8Val = ENGINESOUND_DRIVEMODE_ECO;

   return u8Val;
}

tU8 EngineSoundModel::u8GetESESoundType(tU8 u8DriveMode)
{
   tU8 u8Val;

  if(DP_S32_NO_ERR != s32SetGetDpItem(ese_mex_settings[u8DriveMode].u8SoundType, u8Val, DP_GET))
   {
      ETG_TRACE_ERR(("EngineSoundModel : Failed to read ESE sound type for Mode %d", u8DriveMode));
      u8Val = ENGINESOUND_SOUNDTYPE_NEUTRAL;
   }

   /*ToDo : Possible default values for SoundType also contain value 0x00 'NoChange'. Currently it is not known how to handle this*/
   /*Hence if this is the value during init or defset, here we are hardcoding to another valid value. This should be changed*/
   if(DP_VAL_NO_CHANGE == u8Val)
      u8Val = ENGINESOUND_SOUNDTYPE_NEUTRAL;

   return u8Val;
}

tU8 EngineSoundModel::u8GetESEVolumeLevel(tU8 u8DriveMode,tU8 u8SoundType)
{
   tU8 u8Val;

  if(DP_S32_NO_ERR != s32SetGetDpItem(ese_mex_settings[u8DriveMode].au8Volume[u8SoundType-1], u8Val, DP_GET))
   {
      ETG_TRACE_ERR(("EngineSoundModel : Failed to read ESE volume for Mode %d and sound type %d", u8DriveMode, u8SoundType));
      u8Val = ENGINESOUND_VOLUME_OFF;
   }

   /*ToDo : Possible default values for Volume also contain value 0x00 'NoChange'. Currently it is not known how to handle this*/
   /*Hence if this is the value during init or defset, here we are hardcoding to another valid value. This should be changed*/
   if(DP_VAL_NO_CHANGE == u8Val)
      u8Val = ENGINESOUND_VOLUME_OFF;

   return u8Val;
}

tU8 EngineSoundModel::u8GetRSoundStatus()
{
   tU8 u8Val;
   dp_tclEngineSoundSettingsDPRSoundOnOff oRSoundDp;

   if(DP_S32_NO_ERR != oRSoundDp.s32GetData(u8Val))
   {
      ETG_TRACE_ERR(("EngineSoundModel : Failed to read RSound On/Off"));
      u8Val = RSOUND_ONOFF_OFF;
   }

   return u8Val;
}

tU8 EngineSoundModel::u8GetRSoundType()
{
   tU8 u8Val;
   dp_tclEngineSoundSettingsDPRSoundType oRSoundDp;

   if(DP_S32_NO_ERR != oRSoundDp.s32GetData(u8Val))
      ETG_TRACE_ERR(("EngineSoundModel : Failed to read RSound sound type"));

   return u8Val;
}

tU8 EngineSoundModel::u8GetRSoundVolumeLevel()
{
   tU8 u8Val;
   dp_tclEngineSoundSettingsDPRSoundVolume oRSoundDp;

   if(DP_S32_NO_ERR != oRSoundDp.s32GetData(u8Val))
   {
      ETG_TRACE_ERR(("EngineSoundModel : Failed to read RSound Volume level"));
      u8Val = 7;
   }

   return u8Val;
}

tU8 EngineSoundModel::u8GetRSoundActivationStatus()
{
   tU8 u8Val;
   dp_tclEngineSoundSettingsDPRSoundActivation oRSoundActivationDp;

   if(DP_S32_NO_ERR != oRSoundActivationDp.s32GetData(u8Val))
   {
      ETG_TRACE_ERR(("EngineSoundModel : Failed to read RSoundActivation status"));
      u8Val = RSOUND_ACTIVATION_OFF;
   }

   return u8Val;
}

#define SET_GET_DP(obj,val,set) \
   (1 == set)?obj.s32SetData(val):obj.s32GetData(val);

/**
 * @brief Set/Get the Dp item
 *
 * @param[in]     u8DpId   To identify which datapool item
 * @param[in/out] u8Value  Incase of SET, this contains the value to be set
 *                         Incase of GET, the value from Dp is copied to this variable
 * @param[in]     blSet    DP_SET -> To store the value in to Dp
 *                         DP_GET -> To read the value from Dp
 * @return        tS32     Error code returned by Dp Set/Get
 */
tS32 EngineSoundModel::s32SetGetDpItem(tU8 u8DpId, tU8& u8Value, tBool blSet)
{
   tS32 s32Res = DP_S32_ERR_UNKNOWN;

   switch(u8DpId)
   {
      case DPID_SOUNDTYPE_ECO:
         {
            dp_tclEngineSoundSettingsDPESESoundTypeEco oSoundTypeDp;
            s32Res = SET_GET_DP(oSoundTypeDp,u8Value,blSet);
         }
         break;

      case DPID_SOUNDTYPE_COMFORT:
         {
            dp_tclEngineSoundSettingsDPESESoundTypeComfort oSoundTypeDp;
            s32Res = SET_GET_DP(oSoundTypeDp,u8Value,blSet);
         }
         break;

      case DPID_SOUNDTYPE_NORMAL:
         {
            dp_tclEngineSoundSettingsDPESESoundTypeNormal oSoundTypeDp;
            s32Res = SET_GET_DP(oSoundTypeDp,u8Value,blSet);
         }
         break;

      case DPID_SOUNDTYPE_DYNAMIC:
         {
            dp_tclEngineSoundSettingsDPESESoundTypeDynamic oSoundTypeDp;
            s32Res = SET_GET_DP(oSoundTypeDp,u8Value,blSet);
         }
         break;

      case DPID_SOUNDTYPE_PERSO:
         {
            dp_tclEngineSoundSettingsDPESESoundTypePerso oSoundTypeDp;
            s32Res = SET_GET_DP(oSoundTypeDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_ECO_NEUTRAL:
         {
            dp_tclEngineSoundSettingsDPESEVolumeEcoNeutral oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_ECO_SOUND1:
         {
            dp_tclEngineSoundSettingsDPESEVolumeEcoSound1 oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_ECO_SOUND2:
         {
            dp_tclEngineSoundSettingsDPESEVolumeEcoSound2 oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_COMFORT_NEUTRAL:
         {
            dp_tclEngineSoundSettingsDPESEVolumeComfortNeutral oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_COMFORT_SOUND1:
         {
            dp_tclEngineSoundSettingsDPESEVolumeComfortSound1 oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_COMFORT_SOUND2:
         {
            dp_tclEngineSoundSettingsDPESEVolumeComfortSound2 oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_NORMAL_NEUTRAL:
         {
            dp_tclEngineSoundSettingsDPESEVolumeNormalNeutral oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_NORMAL_SOUND1:
         {
            dp_tclEngineSoundSettingsDPESEVolumeNormalSound1 oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_NORMAL_SOUND2:
         {
            dp_tclEngineSoundSettingsDPESEVolumeNormalSound2 oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_DYNAMIC_NEUTRAL:
         {
            dp_tclEngineSoundSettingsDPESEVolumeDynamicNeutral oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_DYNAMIC_SOUND1:
         {
            dp_tclEngineSoundSettingsDPESEVolumeDynamicSound1 oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_DYNAMIC_SOUND2:
         {
            dp_tclEngineSoundSettingsDPESEVolumeDynamicSound2 oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_PERSO_NEUTRAL:
         {
            dp_tclEngineSoundSettingsDPESEVolumePersoNeutral oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_PERSO_SOUND1:
         {
            dp_tclEngineSoundSettingsDPESEVolumePersoSound1 oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      case DPID_VOLUME_PERSO_SOUND2:
         {
            dp_tclEngineSoundSettingsDPESEVolumePersoSound2 oVolDp;
            s32Res = SET_GET_DP(oVolDp,u8Value,blSet);
         }
         break;

      default:
         break;
   }

   return s32Res;
}
/***********************************************************************************/
