/*******************************************************************************
* FILE:          fc_audiomanager_DiaglogClientHandler_Speaker.cpp
* PROJECT:
* SW-COMPONENT:  fc_audiomanager application
*-------------------------------------------------------------------------------
*
* DESCRIPTION:   ClientHandler for the Diaglog service of the Diagnosis
*
*-------------------------------------------------------------------------------
*
* COPYRIGHT:    (c) 2012 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author                  | Modification
* 3.3.14  | Puneeth  | Inital version
*******************************************************************************/

/*******************************************************************************
                        Includes
*******************************************************************************/
/* Include message framework (AMT, msgfw) */
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE

#include "../../PostOffice/PostOffice.hpp"
#include "../InternalComponentCommunication/DataTypes/TypeDefines/ADR3DiagType.h"
#include "../InternalComponentCommunication/Messages/Diag/IDNotifyDiagStatus.h"
#include "../InternalComponentCommunication/Messages/Diag/ID_DiagActiveSink.h"
#include "../InternalComponentCommunication/Messages/Diag/IDDiagRemoteControl.h"
#include "fc_audiomanager_DiaglogClientHandler_Speaker.h"
#include "fc_audiomanager_trace.h"
#include "../fc_audiomanager_service_Audio_Function.h"
#include "../aud_sinkmanager/vd_adr3/vd_adr3Msg_If.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_AUDIO_DIAGNOSIS
#include "trcGenProj/Header/fc_audiomanager_DiaglogClientHandler_Speaker.cpp.trc.h"
#endif

#if defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SMART
#define S2S 0x00
#define S4S 0x01
#define S2S_CONVERTIBLE 0x02
#endif

// Includes all header files for FI of DP-component
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_audio_if.h"
/*******************************************************************************
                  Specific defines for this component
*******************************************************************************/

// Static self reference to be used in function callbacks.
fc_audiomanager_tclApp*   fc_audiomanager_tclDiaglogClientHandler_Speaker::m_poCcaMainApp  = NULL;

/*******************************************************************************
                  Message map for FunctionIDs to be handled
*******************************************************************************/



/*******************************************************************************
                  Methods of class
*******************************************************************************/


/*******************************************************************************
**  FUNCTION:
*   fc_audiomanager_tclDiaglogClientHandler_Speaker();
*   Constructor
*******************************************************************************/
fc_audiomanager_tclDiaglogClientHandler_Speaker::fc_audiomanager_tclDiaglogClientHandler_Speaker(
  fc_audiomanager_tclApp* poMainAppl) : IF_MessageObserver<PO_MessageConfig::enID>("fc_audiomanager_tclDiaglogClientHandler_Speaker")
{
   ETG_TRACE_USR3(("fc_audiomanager_tclDiaglogClientHandler_Speaker()"));
   m_u16TroubleCode = 0;
   m_sResult = EN_PASSED;
   m_pPO = InternalCommunicationAdapter::getInstance();
   m_poResultHandler = NULL;
   m_u8DiagRemoteControlPhase = 0;
   m_bActiveSinkSet = FALSE;
   fc_audiomanager_tclDiaglogClientHandler_Speaker::m_poCcaMainApp = poMainAppl;

   vAddObserver();
}


/*******************************************************************************
** FUNCTION:   virtual ~fc_audiomanager_tclClientHandler_diaglogTemp();
*  Destructor
*******************************************************************************/
fc_audiomanager_tclDiaglogClientHandler_Speaker
      ::~fc_audiomanager_tclDiaglogClientHandler_Speaker(tVoid)
{
  InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);

  fc_audiomanager_tclDiaglogClientHandler_Speaker::m_poCcaMainApp = NULL;
  m_pPO = NULL;
  m_poResultHandler = NULL;
}


/*******************************************************************************
** FUNCTION:   tVoid vSaveTestResult();
* Save the Test Results to the CCA Msg for later sending to Diaglog
*******************************************************************************/

tVoid fc_audiomanager_tclDiaglogClientHandler_Speaker::vSaveTestResult(tU16 u16troublecode, tS16 sresult)
{
   tTestResult oTestData;

   oTestData.u16ID = u16troublecode;

   if (EN_PASSED == sresult)
   {
      oTestData.enResult = EN_PASSED;
      ETG_TRACE_USR3(("vSaveTestResult():=> Diagnosis Test Result = PASSED"));

   }
   else if (EN_FAILED == sresult)
   {
      oTestData.enResult = EN_FAILED;
      ETG_TRACE_USR3(("vSaveTestResult():=> Diagnosis Test Result = FAILED"));
   }
   else if (EN_NORESULT == sresult)
   {
      oTestData.enResult = EN_NORESULT;
      ETG_TRACE_USR3(("vSaveTestResult():=> Diagnosis Test Result = NO RESULT"));
   }

   if(m_oFiTestResultObject.empty())
   {
    //add the testData to the TestResultList if empty
    m_oFiTestResultObject.push_back(oTestData);
   }
   else
   {
    tTestResultList::iterator it = m_oFiTestResultObject.begin();

    for(; it !=  m_oFiTestResultObject.end(); ++it)
    {
      if(it->u16ID == u16troublecode)
      {
         it->enResult = (tenTestResult)sresult;
         break;
      }
    }

    if(it == m_oFiTestResultObject.end())
    {
      //add the testData to the TestResultList if entry not found
      m_oFiTestResultObject.push_back(oTestData);
    }
   }
}

/*******************************************************************************
** FUNCTION:   tVoid vSendTestResultToDiaglog();
* Send all the Test Results to Diaglog (1 CCA Msg for all results together)  (Method of this application)
*******************************************************************************/

tVoid fc_audiomanager_tclDiaglogClientHandler_Speaker::vSendTestResultToDiaglog(tVoid)
{
   ETG_TRACE_USR4(("vSendTestResultToDiaglog()"));
   if (NULL == m_poResultHandler)
   {
      ETG_TRACE_ERR(("DiagLog result handler not set"));
      return;
   }

   if (m_oFiTestResultObject.size())
   {
      ETG_TRACE_USR3(("vSendTestResultToDiaglog - sending %u results to result handler", m_oFiTestResultObject.size()));
      m_poResultHandler->vSaveDiagResult(m_oFiTestResultObject);

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
      tTestResultList::iterator it = m_oFiTestResultObject.begin();

      for(; it !=  m_oFiTestResultObject.end();)
      {
         switch(it->u16ID)
         {
         case ITC_AUDIO_SPK_LF_CIRCUIT_SHORT_GND:
         case ITC_AUDIO_SPK_LF_CIRCUIT_SHORT_VSS:
         case ITC_AUDIO_SPK_LR_CIRCUIT_SHORT_GND:
         case ITC_AUDIO_SPK_LR_CIRCUIT_SHORT_VSS:
         case ITC_AUDIO_SPK_RF_CIRCUIT_SHORT_GND:
         case ITC_AUDIO_SPK_RF_CIRCUIT_SHORT_VSS:
         case ITC_AUDIO_SPK_RR_CIRCUIT_SHORT_GND:
         case ITC_AUDIO_SPK_RR_CIRCUIT_SHORT_VSS:
         {
           if(it->enResult != EN_FAILED)
           {
             it = m_oFiTestResultObject.erase(it);
           }
           else
           {
              ++it;
           }
           break;
         }
         default:
         {
           it = m_oFiTestResultObject.erase(it);
           break;
         }

         }
      }
#else
      m_oFiTestResultObject.clear();
#endif
   }
   else
   {
     // For PSA request results again from ADR3. PSARCC21-1929
     #ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI

      ETG_TRACE_USR3(("vSendTestResultToDiaglog - No results available. Requesting from ADR"));

      fc_audiomanager_tclService_Audio_Function* poAudioFunction = m_poCcaMainApp -> pGetAudioFunctionptr();
      NORMAL_M_ASSERT(poAudioFunction != OSAL_NULL);
      if (poAudioFunction != OSAL_NULL)
      {
         poAudioFunction -> vRequestDiagResult(EN_AUDIO_DIAG_SPEAKER_LF);
         poAudioFunction -> vRequestDiagResult(EN_AUDIO_DIAG_SPEAKER_RF);
         poAudioFunction -> vRequestDiagResult(EN_AUDIO_DIAG_SPEAKER_LR);
         poAudioFunction -> vRequestDiagResult(EN_AUDIO_DIAG_SPEAKER_RR);
      }

     #else
      ETG_TRACE_USR3(("vSendTestResultToDiaglog - No Speaker Test triggered & No results available. "));
    #endif
  }
}

/*******************************************************************************
* FUNCTION:   vCheckErrorCodeSpeakerLf(tenErrorCode enErrorCode)
 *
 * DESCRIPTION:  This function is called to update the trouble code(ITC)
 *
 * PARAMETER:    tenErrorCode enErrorCode
 *
 * RETURNVALUE:  None
 *************************************************************************/
tVoid fc_audiomanager_tclDiaglogClientHandler_Speaker::vCheckErrorCodeSpeakerLf(tenErrorCode enErrorCode)
{
  ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLf received enErrorCode: 0x%4x.", enErrorCode));

  switch(enErrorCode)
  {
    case EN_AUDIO_DIAG_ERR_OPEN_WOOFER:
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLf::EN_AUDIO_DIAG_ERR_OPEN_WOOFER"));
      m_u16TroubleCode = ITC_AUDIO_SPK_LF_CIRCUIT_INTERRUPTED;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_GND :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLf::EN_AUDIO_DIAG_ERR_SHORT_GND"));
      m_u16TroubleCode = ITC_AUDIO_SPK_LF_CIRCUIT_SHORT_GND;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_VCC :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLf::EN_AUDIO_DIAG_ERR_SHORT_VCC"));
      m_u16TroubleCode = ITC_AUDIO_SPK_LF_CIRCUIT_SHORT_VSS;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_WIRE :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLf::EN_AUDIO_DIAG_ERR_SHORT_WIRE"));
      m_u16TroubleCode = ITC_AUDIO_SPK_LF_CIRCUIT_SHORT_ACROSS;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_OPEN_TWEETER :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLf::EN_AUDIO_DIAG_ERR_OPEN_TWEETER"));
      m_u16TroubleCode = ITC_AUDIO_TWEETER_LF_CIRCUIT_INTERRUPTED;
      m_sResult = EN_FAILED;
      break;

    default:
    ETG_TRACE_ERR(("vCheckErrorCodeSpeakerLf received wrong enErrorCode: 0x%4x.", enErrorCode));
        m_sResult = EN_NORESULT; //No Result for this Test
    break;
  }
}

/*******************************************************************************
* FUNCTION:   vCheckErrorCodeSpeakerLr(tenErrorCode enErrorCode)
 *
 * DESCRIPTION:  This function is called to update the trouble code(ITC)
 *
 * PARAMETER:    tenErrorCode enErrorCode
 *
 * RETURNVALUE:  None
 *************************************************************************/
tVoid fc_audiomanager_tclDiaglogClientHandler_Speaker::vCheckErrorCodeSpeakerLr(tenErrorCode enErrorCode)
{
  ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLr received enErrorCode: 0x%4x.", enErrorCode));

  switch(enErrorCode)
  {
    case EN_AUDIO_DIAG_ERR_OPEN_WOOFER:
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLr::EN_AUDIO_DIAG_ERR_OPEN_WOOFER"));
      m_u16TroubleCode = ITC_AUDIO_SPK_LR_CIRCUIT_INTERRUPTED;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_GND :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLr::EN_AUDIO_DIAG_ERR_SHORT_GND"));
      m_u16TroubleCode = ITC_AUDIO_SPK_LR_CIRCUIT_SHORT_GND;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_VCC :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLr::EN_AUDIO_DIAG_ERR_SHORT_VCC"));
      m_u16TroubleCode = ITC_AUDIO_SPK_LR_CIRCUIT_SHORT_VSS;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_WIRE :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLr::EN_AUDIO_DIAG_ERR_SHORT_WIRE"));
      m_u16TroubleCode = ITC_AUDIO_SPK_LR_CIRCUIT_SHORT_ACROSS;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_OPEN_TWEETER :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerLr::EN_AUDIO_DIAG_ERR_OPEN_TWEETER"));
      m_u16TroubleCode = ITC_AUDIO_TWEETER_LR_CIRCUIT_INTERRUPTED;
      m_sResult = EN_FAILED;
      break;

    default:
    ETG_TRACE_ERR(("vCheckErrorCodeSpeakerLr received wrong enErrorCode: 0x%4x.", enErrorCode));
        m_sResult = EN_NORESULT; //No Result for this Test
    break;
  }
}


/*******************************************************************************
* FUNCTION:   vCheckErrorCodeSpeakerRf(tenErrorCode enErrorCode)
 *
 * DESCRIPTION:  This function is called to update the trouble code(ITC)
 *
 * PARAMETER:    tenErrorCode enErrorCode
 *
 * RETURNVALUE:  None
 *************************************************************************/
tVoid fc_audiomanager_tclDiaglogClientHandler_Speaker::vCheckErrorCodeSpeakerRf(tenErrorCode enErrorCode)
{
  ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRf received enErrorCode: 0x%4x.", enErrorCode));

  switch(enErrorCode)
  {
    case EN_AUDIO_DIAG_ERR_OPEN_WOOFER:
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRf::EN_AUDIO_DIAG_ERR_OPEN_WOOFER"));
      m_u16TroubleCode = ITC_AUDIO_SPK_RF_CIRCUIT_INTERRUPTED;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_GND :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRf::EN_AUDIO_DIAG_ERR_SHORT_GND"));
      m_u16TroubleCode = ITC_AUDIO_SPK_RF_CIRCUIT_SHORT_GND;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_VCC :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRf::EN_AUDIO_DIAG_ERR_SHORT_VCC"));
      m_u16TroubleCode = ITC_AUDIO_SPK_RF_CIRCUIT_SHORT_VSS;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_WIRE :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRf::EN_AUDIO_DIAG_ERR_SHORT_WIRE"));
      m_u16TroubleCode = ITC_AUDIO_SPK_RF_CIRCUIT_SHORT_ACROSS;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_OPEN_TWEETER :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRf::EN_AUDIO_DIAG_ERR_OPEN_TWEETER"));
      m_u16TroubleCode = ITC_AUDIO_TWEETER_RF_CIRCUIT_INTERRUPTED;
      m_sResult = EN_FAILED;
      break;

    default:
    ETG_TRACE_ERR(("vCheckErrorCodeSpeakerRf received wrong enErrorCode: 0x%4x.", enErrorCode));
        m_sResult = EN_NORESULT; //No Result for this Test
    break;
  }
}

/*******************************************************************************
* FUNCTION:   vCheckErrorCodeSpeakerRr(tenErrorCode enErrorCode)
 *
 * DESCRIPTION:  This function is called to update the trouble code(ITC)
 *
 * PARAMETER:    tenErrorCode enErrorCode
 *
 * RETURNVALUE:  None
 *************************************************************************/
tVoid fc_audiomanager_tclDiaglogClientHandler_Speaker::vCheckErrorCodeSpeakerRr(tenErrorCode enErrorCode)
{
  ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRr received enErrorCode: 0x%4x.", enErrorCode));

  switch(enErrorCode)
  {
    case EN_AUDIO_DIAG_ERR_OPEN_WOOFER:
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRr::EN_AUDIO_DIAG_ERR_OPEN_WOOFER"));
      m_u16TroubleCode = ITC_AUDIO_SPK_RR_CIRCUIT_INTERRUPTED;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_GND :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRr::EN_AUDIO_DIAG_ERR_SHORT_GND"));
      m_u16TroubleCode = ITC_AUDIO_SPK_RR_CIRCUIT_SHORT_GND;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_VCC :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRr::EN_AUDIO_DIAG_ERR_SHORT_VCC"));
      m_u16TroubleCode = ITC_AUDIO_SPK_RR_CIRCUIT_SHORT_VSS;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_SHORT_WIRE :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRr::EN_AUDIO_DIAG_ERR_SHORT_WIRE"));
      m_u16TroubleCode = ITC_AUDIO_SPK_RR_CIRCUIT_SHORT_ACROSS;
      m_sResult = EN_FAILED;
      break;

    case EN_AUDIO_DIAG_ERR_OPEN_TWEETER :
      ETG_TRACE_USR4(("vCheckErrorCodeSpeakerRr::EN_AUDIO_DIAG_ERR_OPEN_TWEETER"));
      m_u16TroubleCode = ITC_AUDIO_TWEETER_RR_CIRCUIT_INTERRUPTED;
      m_sResult = EN_FAILED;
      break;

    default:
    ETG_TRACE_ERR(("vCheckErrorCodeSpeakerRr received wrong enErrorCode: 0x%4x.", enErrorCode));
        m_sResult = EN_NORESULT; //No Result for this Test
    break;
  }
}


/*******************************************************************************
* FUNCTION:   vCheckErrorSite(tenErrorSite enErrorSite, tenErrorCode enErrorCode)
 *
 * DESCRIPTION:  This function is called on receiving the diagnosis result
 *
 * PARAMETER:    tenErrorSite enErrorSite, tenErrorCode enErrorCode
 *
 * RETURNVALUE:  None
 *************************************************************************/
tVoid fc_audiomanager_tclDiaglogClientHandler_Speaker::vCheckErrorSite(tenErrorSite enErrorSite, tenErrorCode enErrorCode)
{
  ETG_TRACE_USR3(("vCheckErrorSite received enErrorSite: 0x%4x.", enErrorSite));
    switch(enErrorSite)
    {
      case EN_AUDIO_DIAG_SPEAKER_LF:
        ETG_TRACE_USR4(("EN_AUDIO_DIAG_SPEAKER_LF"));
        vCheckErrorCodeSpeakerLf(enErrorCode);
        break;
      case EN_AUDIO_DIAG_SPEAKER_RF:
        ETG_TRACE_USR4(("EN_AUDIO_DIAG_SPEAKER_RF"));
        vCheckErrorCodeSpeakerRf(enErrorCode);
        break;
      case EN_AUDIO_DIAG_SPEAKER_LR:
        ETG_TRACE_USR4(("EN_AUDIO_DIAG_SPEAKER_LR"));
        vCheckErrorCodeSpeakerLr(enErrorCode);
        break;
      case EN_AUDIO_DIAG_SPEAKER_RR:
        ETG_TRACE_USR4(("EN_AUDIO_DIAG_SPEAKER_RR"));
        vCheckErrorCodeSpeakerRr(enErrorCode);
        break;
      default:
      ETG_TRACE_ERR(("vCheckErrorSite received wrong enErrorSite: 0x%4x.", enErrorSite));
      m_sResult = EN_NORESULT; //No Result for this Test
      break;
    }

}
/*******************************************************************************
* FUNCTION:   vSend_SpeakerResultToDiaglog(tenErrorSite enErrorSite, tenErrorCode enErrorCode, tenErrorValue enErrorValue)
 *
 * DESCRIPTION:  This function is called to send the result to diaglog
 *
 * PARAMETER:    tenErrorSite enErrorSite, tenErrorCode enErrorCode, tenErrorValue enErrorValue
 *
 * RETURNVALUE:  None
 *************************************************************************/

tVoid fc_audiomanager_tclDiaglogClientHandler_Speaker::vSendSpeakerResultToDiaglog(tenErrorSite enErrorSite, tenErrorCode enErrorCode, tenErrorValue enErrorValue)
{

  ETG_TRACE_USR3(("send_diagResultToDiaglog received enErrorSite: 0x%4x.", enErrorSite));
    ETG_TRACE_USR3(("send_diagResultToDiaglog received enErrorCode: 0x%4x.", enErrorCode));
  ETG_TRACE_USR3(("send_diagResultToDiaglog received enErrorValue: 0x%4x.", enErrorValue));

    vCheckErrorSite(enErrorSite, enErrorCode);

  if(enErrorValue == EN_AUDIO_DIAG_ERRVAL_ERROR)
  {
    ETG_TRACE_USR4(("send_diagResultToDiaglog received with error"));
  }
  else if(enErrorValue == EN_AUDIO_DIAG_ERRVAL_NO_ERROR)
  {
    ETG_TRACE_USR4(("send_diagResultToDiaglog received without error" ));
    m_sResult = EN_PASSED;//test passed
  }

  else
  {
    ETG_TRACE_USR4(("send_diagResultToDiaglog received wrong error value" ));
    m_sResult = EN_NORESULT;//no result for this test
  }
  if(enErrorCode == EN_AUDIO_DIAG_ERR_OPEN_TWEETER)
  {
    fc_audiomanager_tclService_Audio_Function* poAudioFunction = m_poCcaMainApp -> pGetAudioFunctionptr();
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(poAudioFunction);
    if(poAudioFunction->bIsTweeterConfigured(enErrorSite) == FALSE)
    {
      ETG_TRACE_USR3(( "No Tweeter Configured for error Site, sending No Result to Diaglog"));
      m_sResult = EN_NORESULT;
    }

    if(poAudioFunction->bIsSpeakerModeACoff())  //donot send Tweeter Trouble code to diaglog when ACMode is off
      return;
  }
  ETG_TRACE_USR3(( "sResult : %d", m_sResult));
  ETG_TRACE_USR3(( "u16TroubleCode : %d", m_u16TroubleCode));
//  ETG_TRACE_USR3(( "bStartDiagTest : %d", bStartDiagTest));

  vSaveTestResult(m_u16TroubleCode, m_sResult);
  vSendTestResultToDiaglog();
}

/********************************************************************************
 * vAddObserver()
 *******************************************************************************/

void fc_audiomanager_tclDiaglogClientHandler_Speaker::vAddObserver()
{

  ETG_TRACE_USR3(("fc_audiomanager_tclDiaglogClientHandler_Speaker::vAddObserver"));

  if(m_pPO != NULL)
  {
    m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_CCA_GetDiagResult);
    m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifyDiagResult);
    m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifyDiagStatus);
    m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_DiagActiveSink);
    m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_DiagRemoteControl);
  }
  else
  {
        ETG_TRACE_FATAL(( "vAddObserver: InternalCommunicationAdapter Instance pointer is NULL" ));
    FATAL_M_ASSERT_ALWAYS();
  }

}

/********************************************************************************
 * MessageNotification(PO_MessageConfig::enID MsgId)
 *******************************************************************************/

void fc_audiomanager_tclDiaglogClientHandler_Speaker::MessageNotification(PO_MessageConfig::enID MsgId)
{
  ETG_TRACE_USR4(("fc_audiomanager_tclDiaglogClientHandler_Speaker::MessageNotification(%i) entered. ", ETG_CENUM(PO_MessageConfig::enID, MsgId)));

  fc_audiomanager_tclService_Audio_Function* poAudioFunction = m_poCcaMainApp -> pGetAudioFunctionptr();
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(poAudioFunction);

  switch(MsgId)
  {
  case(PO_MessageConfig::ID_NotifyDiagResult):
    {
        ETG_TRACE_USR4(("fc_audiomanager_tclDiaglogClientHandler_Speaker::MessageNotification: ID_NotifyDiagResult"));
      // Obtaining the Diag Message from PostOffice handler
      const ID_NotifyDiagResult*     pMsg = m_pPO->POMessages->QueryMessage<ID_NotifyDiagResult>(MsgId);
      AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
      //Receive the Diag Result Message from VD_ADR3.
      ETG_TRACE_USR3(("MessageNotification: Diagnose Result Observer received ErrorSite: 0x%4x ErrorCode: 0x%4x ErrorValue: 0x%4x", pMsg->enErrorSite, pMsg->enErrorCode, pMsg->enErrorValue));
      if(pMsg->enErrorSite != EN_AUDIO_DIAG_AMPLIFIER)
      {
        #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
        if((poAudioFunction->bCheckFaderAvailability() != EXTERNALAMP_BOSE2CH_MCAN) && (poAudioFunction->bCheckFaderAvailability() != HEAD_UNIT_ALONE))
        {
            if(pMsg->enErrorSite == EN_AUDIO_DIAG_SPEAKER_LR || pMsg->enErrorSite == EN_AUDIO_DIAG_SPEAKER_RR)
            {
              if((m_bActiveSinkSet == TRUE) ||(poAudioFunction->bCheckFaderAvailability() == INTERNALAMP_4SPEAKER)||(poAudioFunction->bCheckFaderAvailability() == EXTERNALAMP_BOSEWOMCAN))
              {
                vSendSpeakerResultToDiaglog(pMsg->enErrorSite, pMsg->enErrorCode, pMsg->enErrorValue);
              }
            }
            else
              vSendSpeakerResultToDiaglog(pMsg->enErrorSite, pMsg->enErrorCode, pMsg->enErrorValue);
        }

        #elif defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_SMART
          tU8 u8CarModelType[2];

           if((DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation","CarModel", &u8CarModelType[0], 2)))
           {
               ETG_TRACE_USR4(("fc_audiomanager_tclDiaglogClientHandler_Speaker:: carmodel %d",u8CarModelType[1]));
             if (u8CarModelType[1] == S2S || u8CarModelType[1]==S2S_CONVERTIBLE )
                {
                 if(pMsg->enErrorSite != EN_AUDIO_DIAG_SPEAKER_LR && pMsg->enErrorSite != EN_AUDIO_DIAG_SPEAKER_RR)
                      vSendSpeakerResultToDiaglog(pMsg->enErrorSite, pMsg->enErrorCode, pMsg->enErrorValue);
                }
            else if (u8CarModelType[1] == S4S)
               {
                     vSendSpeakerResultToDiaglog(pMsg->enErrorSite, pMsg->enErrorCode, pMsg->enErrorValue);
               }
           }
        
        
        #else
          vSendSpeakerResultToDiaglog(pMsg->enErrorSite, pMsg->enErrorCode, pMsg->enErrorValue);
        #endif
      }
      else
        ETG_TRACE_USR4(("Diagnose Result Observer received for ErrorSite other than speaker"));
      break;
    }
  case PO_MessageConfig::ID_CCA_GetDiagResult:
      {
         ETG_TRACE_USR4(("fc_audiomanager_tclDiaglogClientHandler_Speaker::MessageNotification: ID_CCA_GetDiagResult"));
         // request
         const ID_CCA_GetDiagResult* poMsg
            = InternalCommunicationAdapter::POMessages->QueryMessage<ID_CCA_GetDiagResult>(MsgId);
         if (NULL == poMsg)
         {
            ETG_TRACE_FATAL(("MessageNotification - could not retrieve message pointer for ID_CCA_GetDiagResult"))
            return;
         }
         m_poResultHandler = &(poMsg->roResultHandler);
         vSendTestResultToDiaglog(); // Send result if already available
     break;
    }
     case PO_MessageConfig::ID_DiagRemoteControl:
      {
        const ID_DiagRemoteControl* pMsgRemoteControl =  InternalCommunicationAdapter::POMessages->QueryMessage<ID_DiagRemoteControl>(MsgId);
        AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsgRemoteControl);
        ETG_TRACE_USR4(("RemoteControl phase received = %x",pMsgRemoteControl->enDiagRemoteControlPhase));
        m_u8DiagRemoteControlPhase = pMsgRemoteControl->enDiagRemoteControlPhase;
        if(m_u8DiagRemoteControlPhase == EN_REMOTECONTROL_UNFREEZE)
          m_bActiveSinkSet = FALSE;
        break;
      }
    case PO_MessageConfig::ID_DiagActiveSink:
      {
         const ID_DiagActiveSink* pMsgActiveSink =  InternalCommunicationAdapter::POMessages->QueryMessage<ID_DiagActiveSink>(MsgId);
         AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsgActiveSink);
         ETG_TRACE_USR4(("ID_DiagActiveSink received with Active Sink = %x",pMsgActiveSink->enDiagActiveSinkPhase));
         if((m_u8DiagRemoteControlPhase == EN_REMOTECONTROL_SET) && (pMsgActiveSink->enDiagActiveSinkPhase == EN_ACTIVESINK_SET))
          m_bActiveSinkSet = TRUE;
         break;
      }
  default:
    break;
  }
}
