/*
 * Booster.cpp
 *
 *  Created on: Aug 29, 2017
 *      Author: rjk2kor
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>   // implicitly links generic osal_if.h

#include <unistd.h>

#include "fc_audiomanager_main.h"
#include "aud_sinkmgr_main.h"

#include "fc_audiomanager_trace.h"
#include "fc_audiomanager_trace_macros.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AUD_SINKMGR_APPLICATION
#include "trcGenProj/Header/Booster.cpp.trc.h"
#endif

#include "../../PostOffice/PostOffice.hpp"
#include "InternalComponentCommunication/InternalCommunicationAdapter.h"
#include "InternalComponentCommunication/Messages/booster/IDSetBoosterState.h"
#include "InternalComponentCommunication/Messages/booster/IDNotifyBoosterState.h"
#include "InternalComponentCommunication/Messages/booster/IDNotifyBoosterTestResult.h"
#include "InternalComponentCommunication/Messages/Diag/IDDiagControlLine.h"
#include "fc_audiomanager_main.h"

#include "Booster.h"

/******************************************************************************/
/* constructor                                                                */
/******************************************************************************/
Booster::Booster(fc_audiomanager_tclApp* pMain)
   : IF_MessageObserver<PO_MessageConfig::enID>("Booster")
   ,m_pPO(NULL)
   ,m_Req_BoosterState(AUD_BOOSTER_OFF)
   ,m_AmpOnPinState(FALSE)
   ,m_DiagFreezeActive(FALSE)
   ,m_poMain(pMain)
   ,m_async(NULL)
   ,m_hBoosterSupervisionTimer(OSAL_C_INVALID_HANDLE)
   ,m_hBoosterRecoveryTimer(OSAL_C_INVALID_HANDLE)
{
  ETG_TRACE_USR1(("::Booster:: constructor"));

  NORMAL_M_ASSERT(m_poMain != NULL);

  //Turn off Booster Pin will be handled in vOnInit call

  //Update postoffice reference
  m_pPO = InternalCommunicationAdapter::getInstance();


  //Create timer for booster line diagnosis
  tS32 s32OsalError = OSAL_ERROR;

  /* create Booster line supervision timer */
  s32OsalError = OSAL_s32TimerCreate((OSAL_tpfCallback)vBoosterSupervisionTimerCallback, this, &m_hBoosterSupervisionTimer);

  if (s32OsalError == OSAL_OK){
    ETG_TRACE_USR4(("::Booster:: Supervision Timer created successfully."));
  }
  else
  {
    ETG_TRACE_FATAL(("::Booster:: Supervision Timer creation Failed !!"));
  }

  /* create Booster line supervision timer */
  s32OsalError = OSAL_s32TimerCreate((OSAL_tpfCallback)vBoosterRecoveryTimerCallback, this, &m_hBoosterRecoveryTimer);

  if (s32OsalError == OSAL_OK){
    ETG_TRACE_USR4(("::Booster:: Recovery Timer created successfully."));
  }
  else
  {
    ETG_TRACE_FATAL(("::Booster:: Recovery Timer creation Failed !!"));
  }

}
/**
 * Destructor
 */
Booster::~Booster()
{
  //ToDo: Do the deinitialization
  //1. Remove observers
  InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);

  //2. Stop timer
  if(m_hBoosterSupervisionTimer != OSAL_C_INVALID_HANDLE)
  {
    //Stop booster line supervision
    vStopBoosterLineSupervision();
    //Delete the timer
    OSAL_s32TimerDelete(m_hBoosterSupervisionTimer);
    //Set the handle to Invalid
    m_hBoosterSupervisionTimer = OSAL_C_INVALID_HANDLE;
  }

  //3. Unreference pointers
  m_poMain = NULL;
  m_async = NULL;
  m_pPO = NULL;
}

/**
 * Initialize Booster Module
 */
tVoid Booster::vInit()
{
  ETG_TRACE_USR2(("::Booster:: Initializing Module "));

  //1. Update pointer references
  if((m_async == NULL) && (m_poMain != NULL))
  {
    m_async = m_poMain->poAsyncCall();
  }
  //2. Update Booster State to OFF
  m_Req_BoosterState = AUD_BOOSTER_OFF;

  //3. Turn off Booster Pin (explicitly)
  bSetBoosterControlLine(FALSE);

  //4. Perform Short to VCC Test and Start Timer to perform further tests
  vPerformAmpOnDiagnosis();

  //5. Start Supervision of booster line
  bStartBoosterLineSupervision();

  //6. Setup observers
  if (m_pPO)
  {
    m_pPO->POMessages->AddObserver(this, PO_MessageConfig::ID_DiagControlLine);
    m_pPO->POMessages->AddObserver(this, PO_MessageConfig::IDSetBoosterState);
  }
  else
    FATAL_M_ASSERT_ALWAYS();
}
/******************************************************************************/
//  MessageNotification()
/******************************************************************************/
void Booster::MessageNotification(PO_MessageConfig::enID MsgId)
{
  ETG_TRACE_USR4(("::Booster:: MessageNotification entered. "));

  switch(MsgId)
  {
  case PO_MessageConfig::ID_DiagControlLine:
  {
    // Obtaining the Message from PostOffice handler
    const ID_DiagControlLine*     pMsg = m_pPO->POMessages->QueryMessage<ID_DiagControlLine>(MsgId);
    ETG_TRACE_USR4(("::Booster:: ID_DiagControlLine status: 0x%4x.", pMsg->enDiagControlLinePhase));
    vHandleDiagControlLineFreeze(pMsg->enDiagControlLinePhase);
    break;
  }
  case PO_MessageConfig::IDSetBoosterState:
  {
    const IDSetBoosterState*     pMsg = m_pPO->POMessages->QueryMessage<IDSetBoosterState>(MsgId);
    ETG_TRACE_USR4(("::Booster:: IDSetBoosterState Requested with State: %d.", pMsg->m_bActivateBooster));

    //Update requested state to internal member
    m_Req_BoosterState = pMsg->m_bActivateBooster;

    if(m_DiagFreezeActive)
    {
      ETG_TRACE_USR4(("::Booster:: IDSetBoosterState Requested when Diagnosis Control Line Freeze is active. Request Deferred"));
    }
    //Diagnosis freeze is not active. Set the Pin state accordingly.
    else if(m_AmpOnPinState != pMsg->m_bActivateBooster)
    {
      //Stop Booster line supervision, as we are going to change the pin state
      vStopBoosterLineSupervision();

      tBool bRes = FALSE;
      ETG_TRACE_USR4(("bRes : %d",bRes));

      if(AUD_BOOSTER_ON == m_Req_BoosterState)
      {
        bRes = bSetBoosterControlLine(TRUE);
      }
      else
      {
        bRes = bSetBoosterControlLine(FALSE);
      }

      //Enable booster line supervision
      bStartBoosterLineSupervision();
    }
    else
    {
      //There is no difference between the actual pin state and the requested state.
      //Send the current state of booster pin
      IDNotifyBoosterState oState(m_Req_BoosterState);

      //Update clients
      if(m_pPO)
        m_pPO->POMessages->DeliverMsg(&oState);
    }

    break;
  }
  default:
  {
    ETG_TRACE_ERR(("::Booster:: !!! Unhandled message !!!"));
  }
  break;
  }
}

/**
 * Function to handle diag freeze and unfreeze on AMPON Pin
 */
tVoid Booster::vHandleDiagControlLineFreeze(tenDiagControlLinePhase ePhase)
{
  ETG_TRACE_USR4(("::Booster:: vHandleDiagControlLineFreeze invoked with phase : 0x%4x.", ePhase));

  /** Act based on the current phase **/
  switch (ePhase)
  {
  case EN_CONTROLLINE_FREEZE:
    {
      ETG_TRACE_USR4(("::Booster:: vHandleDiagControlLineFreeze EN_CONTROLLINE_FREEZE"));
      m_DiagFreezeActive = TRUE;
    }
    break;
  case EN_CONTROLLINE_SET_ON:
    {
      ETG_TRACE_USR4(("::Booster:: vHandleDiagControlLineFreeze EN_CONTROLLINE_SET_ON"));
      //Process requests only when freeze is active
      if(m_DiagFreezeActive)
      {
        bSetBoosterControlLine(TRUE);
      }
    }
    break;
  case EN_CONTROLLINE_SET_OFF:
    {
      ETG_TRACE_USR4(("::Booster:: vHandleDiagControlLineFreeze EN_CONTROLLINE_SET_OFF"));
      if(m_DiagFreezeActive)
      {
        bSetBoosterControlLine(FALSE);
      }
    }
    break;
  case EN_CONTROLLINE_UNFREEZE:
    {
      ETG_TRACE_USR4(("::Booster:: vHandleDiagControlLineFreeze EN_CONTROLLINE_UNFREEZE"));
      //If freeze was active before, then unfreeze and restore the last set state of AMP ON Pin
      if(m_DiagFreezeActive)
      {
        m_DiagFreezeActive = FALSE;
        //Restore the last requested booster state.
        bSetBoosterControlLine(m_Req_BoosterState);
      }
      break;
    }
  default:
    ETG_TRACE_USR4(("::Booster:: vHandleDiagControlLineFreeze invalid parameter: 0x%4x.", ePhase));
    break;
  }
}
/**
 * Helper function to Set State of Booster Control Line (AMP ON GPIO Pin)
 */
tBool Booster::bSetBoosterControlLine(tBool bRequestedState)
{
  ETG_TRACE_USR2(("::Booster:: bSetBoosterControlLine() Requested State = %u ",bRequestedState));

  //disable for LSIM
#ifdef VARIANT_S_FTR_DISABLE_BOOSTER
  ETG_TRACE_USR2(("::Booster:: bSetBoosterControlLine() disabled for LSIM"));
  IDNotifyBoosterState oMsgLSIM(bRequestedState);
  if (m_pPO)
  {
    m_pPO->POMessages->DeliverMsg(&oMsgLSIM);
  }
  else
  {
    ETG_TRACE_ERR(("::Booster:: bSetBoosterControlLine() disabled for LSIM. Can not send PostOffice IDNotifyBoosterState"));
  }
  return true;
#endif
  //end

  tS32 _s32OutPutErrorCode =  OSAL_OK;

  //1. Open OSAL GPIO Device
  OSAL_tIODescriptor hDev;
  //Open device
  hDev = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
  if (hDev ==OSAL_ERROR)
  {
    ETG_TRACE_ERR(("::Booster:: ***********blTurnOnGPIOPin() Error in Opening Ext Amp DEVICE GPIO*********** hDev=%d",hDev));
    OSAL_s32IOClose(hDev);
    return FALSE;
  }

  //2. Update PIN details for GPIO set request
  OSAL_tGPIODevID devID;
  devID  = (OSAL_tGPIODevID)OSAL_EN_U140_SW_ENABLE;

  //3. Set the requested state on GPIO Pin.
  if(bRequestedState)
  {
    OSAL_trGPIOData Data = {devID, TRUE};
    ETG_TRACE_USR2(("::Booster:: ***********blTurnOnGPIOPin() turning on *********** "));
    _s32OutPutErrorCode=OSAL_s32IOControl(hDev, OSAL_C_32_IOCTRL_GPIO_SET_ACTIVE_STATE, (intptr_t) &Data);
  }
  else
  {
      OSAL_trGPIOData Data = {devID, FALSE};
      ETG_TRACE_USR2(("::Booster:: ***********blTurnOnGPIOPin() turning off *********** "));
     _s32OutPutErrorCode=OSAL_s32IOControl(hDev, OSAL_C_32_IOCTRL_GPIO_SET_INACTIVE_STATE , (intptr_t) &Data);
  }

  //4. Close the OSAL Gpio Device
  OSAL_s32IOClose(hDev);

  IDNotifyBoosterState oMsg(bRequestedState);

  //Sleep for 2 milliseconds, as this  is the requirement
  usleep(2000);

  //5. Evaluate the result of previous operation
  if(OSAL_OK  ==_s32OutPutErrorCode)
  {

    //Update the PIN State, as setting GPIO is successful
    m_AmpOnPinState = bRequestedState;

    ETG_TRACE_USR2(("::Booster:: GPIO pin opened/closed "));
  }
  else
  {
    ETG_TRACE_USR2(("::Booster:: ***********ERRROR in setting as Output *********** "));
    oMsg.m_BoosterActive = AUD_BOOSTER_OFF;//Due to error, set the state to FALSE
  }

  //6. Inform Clients about result of GPIO set operation with Current GPIO State
  if (m_pPO)
    m_pPO->POMessages->DeliverMsg(&oMsg);

  //7. Inform the caller about the result of this operation
  return (OSAL_OK == _s32OutPutErrorCode ? TRUE: FALSE);
}
/**
 * Helper function to retrieve the state of DiagPin
 */
tS8 Booster::s8GetDiagPinState() const
{
  ETG_TRACE_USR4(("::Booster:: s8GetDiagPinState Read state of DiagPin "));

  //disable for LSIM
#ifdef VARIANT_S_FTR_DISABLE_BOOSTER
  ETG_TRACE_USR2(("::Booster:: s8GetDiagPinState() disabled for LSIM, returning always 1"));
  return (tS8)1; //not sure if this is a correct return code here
#endif
  //end

  tS8 s8Result = 0;

  //1. Open OSAL GPIO Device
  OSAL_tIODescriptor hDev;
  //Open device
  hDev = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);

  OSAL_tGPIODevID devID;
  devID  = (OSAL_tGPIODevID)OSAL_EN_U140_SW_DIAG;
  ETG_TRACE_USR4(("devID : %d",devID));

    OSAL_tGPIODevID DevID = (OSAL_tGPIODevID)OSAL_EN_U140_SW_DIAG;
    OSAL_trGPIOData Data = {DevID, TRUE};
    Data.tId = (OSAL_tGPIODevID) OSAL_EN_U140_SW_DIAG;

  if( OSAL_ERROR != OSAL_s32IOControl ( hDev, OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE,(intptr_t)&Data) )
  {
    if(Data.unData.bState)
    {
      s8Result = 1;
      ETG_TRACE_USR2(("::Booster:: u8GPIO_Get_SW_DIAG : Pin Value High"));
    }
    else
    {
      s8Result = 0;
      ETG_TRACE_USR2(("::Booster:: u8GPIO_Get_SW_DIAG: Pin Value Low"));
    }
  }
  else
  {
    ETG_TRACE_FATAL(("::Booster:: u8GPIO_Get_SW_DIAG  :: DIAG GPIODevice could not be read!"));
  }

  //4. Close the OSAL Gpio Device
  OSAL_s32IOClose(hDev);


  return s8Result;
}
/**
 * Helper functoin that will be invoked to perform AmpON Diagnosis
 * This function has to be triggered on a regular basis.
 * Timer to be used to invoke this function regularly
 * If the request is to switch the Booster ON and there is a failure
 */
tVoid Booster::vPerformAmpOnDiagnosis()
{
  //If the test fails
  if(!bPerformTest())
  {
    //If Diagnosis freeze is active, then take no recovery action.
    if(m_DiagFreezeActive == FALSE)
    {
      //Diagnosis freeze is not active. So start recovery of booster line
      if(m_AmpOnPinState == AUD_BOOSTER_ON)
      {
        ETG_TRACE_ERR(("::Booster:: vPerformAmpOnDiagnosis: Booster Line Diagnosis failed. Turning OFF Booster Line."));
        bSetBoosterControlLine(AUD_BOOSTER_OFF);
        //Start recovery timer to switch it ON again
        ETG_TRACE_ERR(("::Booster:: vPerformAmpOnDiagnosis: Booster Line recovery activated."));
        bStartBoosterLineRecovery();
      }
      //In Short to VCC Case booster line is already OFF. Nothing to do.
    }
    else
    {
      ETG_TRACE_USR2(("Diagnosis freeze is active. Recovery handling deferred"));
    }

  }
}

/**
 * Function to be invoked regulary, which performs diagnosis
 */
tBool Booster::bPerformTest()
{
  //Initial result vars
    tU16 u16TroubleCode = ITC_AMPLIFIER_ON_SHORT_GND;
  tS16 s16Result = EN_NORESULT;


  tS8 s8DiagPinState = s8GetDiagPinState();

  //If the AMPON Pin State is high, short to GND Test is being performed
  if(m_AmpOnPinState)
  {
    u16TroubleCode = ITC_AMPLIFIER_ON_SHORT_GND;

    ETG_TRACE_USR4(("::Booster:: vPerformAmpOnDiagnosis Short to GND Started"));
    if(s8DiagPinState == 1)
    {
      s16Result = EN_FAILED;
      ETG_TRACE_ERR(("::Booster:: vPerformAmpOnDiagnosis Result : Short to GND Detected "));
    }
    else
    {
      s16Result = EN_PASSED;
      ETG_TRACE_USR4(("::Booster:: vPerformAmpOnDiagnosis Result : Test Passed "));
    }
  }
  else //AMPON Pin State is low, short to VCC Test is being performed
  {
    u16TroubleCode = ITC_AMPLIFIER_ON_SHORT_BAT;

    ETG_TRACE_USR4(("::Booster:: vPerformAmpOnDiagnosis Short to VCC Started"));

    if(s8DiagPinState == 0)
    {
      s16Result = EN_FAILED;
      ETG_TRACE_ERR(("::Booster:: vPerformAmpOnDiagnosis Result : Short to VCC Detected "));
    }
    else
    {
      s16Result = EN_PASSED;
      ETG_TRACE_USR4(("::Booster:: vPerformAmpOnDiagnosis Result : Test Passed "));
    }
  }

  //Post Test result to clients
  IDNotifyBoosterTestResult oMsg(u16TroubleCode,s16Result);

  if (m_pPO)
    m_pPO->POMessages->DeliverMsg(&oMsg);

  return !(s16Result == EN_FAILED);
}

/**
* Helper function to start timer
*/
tBool Booster::bStartBoosterLineSupervision()
{
  tBool ret = FALSE;

  tS32 result = OSAL_s32TimerSetTime(m_hBoosterSupervisionTimer, BOOSTER_SUPERVISION_TIMER_START_TIC, BOOSTER_SUPERVISION_TIMER_REPEAT_TIC);

  if(result == OSAL_OK)
  {
    ETG_TRACE_USR4(("::Booster:: Booster Line Supervision Started with Params : Start Tic = %d ms, repeat interval = %d ms",\
    BOOSTER_SUPERVISION_TIMER_START_TIC,\
    BOOSTER_SUPERVISION_TIMER_REPEAT_TIC));

    ret = TRUE;
  }
  else
  {
    ETG_TRACE_FATAL(("::Booster:: Failed to start Booster Line Supervision"));
  }
  return ret;
}

/**
* Helper function to stop timer
*/
tVoid Booster::vStopBoosterLineSupervision()
{
  tS32 result = OSAL_s32TimerSetTime(m_hBoosterSupervisionTimer, 0, 0);

  if(result == OSAL_OK)
  {
    ETG_TRACE_USR4(("::Booster:: Booster Line Supervision Stopped"));
  }
  else
  {
    ETG_TRACE_FATAL(("::Booster:: Failed to stop Booster Line Supervision"));
  }
}

/**
* Timer callback function
*/
tVoid Booster::vBoosterSupervisionTimerCallback(tVoid* pArg)
{
  if(pArg == NULL)
  {
    ETG_TRACE_FATAL(("::Booster:: vBoosterSupervisionTimerCallBack : pARG is NULL !!!"));
    return;
  }

  Booster* ptrSelf = static_cast<Booster*>(pArg);    /* unused parameter */
  if(ptrSelf->m_async)
  {
    //Make an async call to trigger AMP ON Pin Diagnosis
    ptrSelf->m_async->vCall<Booster>(ptrSelf, &Booster::vPerformAmpOnDiagnosis);
  }
  else
  {
    ETG_TRACE_FATAL(("::Booster:: vBoosterSupervisionTimerCallBack : m_async is NULL !!!"));
  }
}

/**
* Booster recovery timer callback function
*/
tVoid Booster::vBoosterRecoveryTimerCallback(tVoid* pArg)
{
  if(pArg == NULL)
  {
    ETG_TRACE_FATAL(("::Booster:: vBoosterRecoveryTimerCallback : pARG is NULL !!!"));
    return;
  }
  Booster* ptrSelf = static_cast<Booster*>(pArg);    /* unused parameter */
  if(ptrSelf->m_async)
  {
    //Make an async call to trigger Booster recovery
    ptrSelf->m_async->vCall<Booster>(ptrSelf, &Booster::vRecoverBooster);
  }
  else
  {
    ETG_TRACE_FATAL(("::Booster:: vBoosterSupervisionTimerCallBack : m_async is NULL !!!"));
  }
}
/**
* Booster recovery function
*/
tVoid Booster::vRecoverBooster()
{
  if(m_Req_BoosterState == m_AmpOnPinState)
  {
    ETG_TRACE_USR4(("::Booster:: Recovery not required as AMP ON pin is already in the requested state"));
    return;
  }
  if(m_DiagFreezeActive)
  {
    ETG_TRACE_USR4(("::Booster:: Recovery deferred. Diag Freeze is already active"));
    return;
  }

  //Set the booster in the requested state.
  bSetBoosterControlLine(m_Req_BoosterState);
}

/**
* Helper function to start timer
*/
tBool Booster::bStartBoosterLineRecovery()
{
  tBool ret = FALSE;

  tS32 result = OSAL_s32TimerSetTime(m_hBoosterRecoveryTimer, BOOSTER_RECOVERY_TIMER_START_TIC, BOOSTER_RECOVERY_TIMER_REPEAT_TIC);

  if(result == OSAL_OK)
  {
    ETG_TRACE_USR4(("::Booster:: Booster Line Recovery Started with Params : Start Tic = %d ms, repeat interval = %d ms",\
    BOOSTER_RECOVERY_TIMER_START_TIC,\
    BOOSTER_RECOVERY_TIMER_REPEAT_TIC));

    ret = TRUE;
  }
  else
  {
    ETG_TRACE_FATAL(("::Booster:: Failed to start Booster Line Recovery"));
  }
  return ret;
}

/**
* Helper function to stop timer
*/
tVoid Booster::vStopBoosterLineRecovery()
{
  tS32 result = OSAL_s32TimerSetTime(m_hBoosterRecoveryTimer, 0, 0);

  if(result == OSAL_OK)
  {
    ETG_TRACE_USR4(("::Booster:: Booster Line Recovery Stopped"));
  }
  else
  {
    ETG_TRACE_FATAL(("::Booster:: Failed to stop Booster Line Recovery"));
  }
}
