/*******************************************************************************
*
* FILE:          sndfile_backend_player.cpp
*
* SW-COMPONENT:  Beep file player application
*
* DESCRIPTION:   Beep file player implementation using libsndfile backend
*
* AUTHOR:        pmh7kor
*
* COPYRIGHT:    (c) 2015 RBEI, Bangalore
*
*******************************************************************************/


/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "../../fc_audiomanager_trace.h"
#include "../../fc_audiomanager_trace_macros.h"
#include "fc_audiomanager_main.h"

#include "sndfile_backend_player.h"

#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_BEEP
#include "trcGenProj/Header/sndfile_backend_player.cpp.trc.h"
#endif

using namespace std;

/******************************************************************************/
/*                                                                            */
/* MACROS                                                                    */
/*                                                                            */
/******************************************************************************/
//#define TRACE_MSG_LEN1 10


/******************************************************************************/
/*                                                                            */
/* GLOBAL VARIABLES                                                           */
/*                                                                            */
/******************************************************************************/

/* sndfile_backend_player()
*
* Brief  : This is the constructor for class sndfile_backend_player.Here
* class variables are initialized
*
* Params : Beep_playback_status_IF* m_ptrController
* Return : None
*/
sndfile_backend_player::sndfile_backend_player(Beep_playback_status_IF* m_ptrController)
:m_poController(m_ptrController)
,m_buffer(NULL)
,m_worker(0)
,bStopWorker(false)
,m_bPlayerStatus(TRC::enFileStopped)
{
  //Contructor of beep file player
}

/* ~sndfile_backend_player()
*
* Brief            : This is the destructor for class sndfile_backend_player.
* Here objects used in the class are appropriately freed.
*
* Input Parameters : None
* Return Value     : None
*/

sndfile_backend_player::~sndfile_backend_player()
{
  ETG_TRACE_USR3(("~sndfile_backend_player()"));
  //vTearDown();//Satisfy lint
  m_poController = NULL;
  if(m_buffer)
  {
    delete[] m_buffer;
    m_buffer = NULL;
  }
}
/*
* tBool sndfile_backend_player::bInitializePlayer()
*
* Brief            : Initializes the Player, by registering for TTFis loopback
*
* Input Parameters : None
* Return Value     : TRUE
*/
bool sndfile_backend_player::bInitializePlayer()
{
  ETG_TRACE_USR3(("sndfile_backend_player::bInitializePlayer"));
  //Multiple calls to this function is ok
  //fc_audiomanager_tclApp::theServer()->vRegisterTraceInputs(TRC::enBeepFilePlayer, this);
  return TRUE;
}
/*
* tBool sndfile_backend_player::bDeinitPlayer()
*
* Brief            : DeInitializes the Player
*
* Input Parameters : None
* Return Value     : returns TRUE if Deinit is done
*/

bool sndfile_backend_player::bDeinitPlayer()
{
  ETG_TRACE_USR3(("sndfile_backend_player::bDeinitPlayer"));
  if(bStop(STOP_AFTER_PLAYING_BUFFER) == TRUE)
    ETG_TRACE_USR3(("Beep play is stopped successfully"));
  vTearDown();
  return TRUE;
}

/*
* tVoid sndfile_backend_player::vTearDown()
*
* Brief            : unlinks the gstreamer elements and free them.
*
* Input Parameters : None
* Return Value     : None
*/

void sndfile_backend_player::vTearDown()
{
  ETG_TRACE_USR3(("sndfile_backend_player::vTearDown"));
  //Remove all file buffers
  vCleanupFileCollection();

  if(m_buffer)
  {
    delete[] m_buffer;
    m_buffer = NULL;
  }
  //sf_close(infile);
  ETG_TRACE_USR3(("Closing PCM device"));
  oDev.bDeInit();
}

/*
* tVoid* sndfile_backend_player::vWorkerTask()
*
* Brief            : Thread Function where player's mainloop runs
*
* Input Parameters : None
* Return Value     : None
*/
void* sndfile_backend_player::vWorkerTask(void * arg)
{
  ETG_TRACE_USR3(("Worker thread started"));
  sndfile_backend_player* self = static_cast<sndfile_backend_player*>(arg);

  unsigned int alsa_fdcount;

  //Get the count of poll descriptors
  alsa_fdcount = snd_pcm_poll_descriptors_count(self->oDev.poGetDevHandle());
  ETG_TRACE_USR3(("Number of fds to poll: %d",alsa_fdcount));
  pollfd* parray = new pollfd[alsa_fdcount];
  //Create a fd array for polling
  if(snd_pcm_poll_descriptors(self->oDev.poGetDevHandle(),parray,alsa_fdcount)< 0)
  {
    delete[] parray;
    ETG_TRACE_FATAL(("Failed to get poll descriptors"));
    return NULL;
  }

  snd_pcm_t* handle = self->oDev.poGetDevHandle();
  snd_pcm_state_t oldstate = SND_PCM_STATE_OPEN;

  //All Set.. Start Playing
  snd_pcm_start(self->oDev.poGetDevHandle());

  while(self->oDev.poGetDevHandle() && !self->bStopWorker)
  {
    poll(parray,alsa_fdcount,-1);
    unsigned short int received_events=0;

    //Demangle the received events according to alsa implementation
    snd_pcm_poll_descriptors_revents(self->oDev.poGetDevHandle(),parray,alsa_fdcount,&received_events);
    if(received_events & POLLOUT)
    {
      snd_pcm_state_t state = snd_pcm_state(handle);
      //If check only to reduce prints
      if(oldstate != state)
      {
        oldstate = state;
        self->vPrintPCMState(state);//Print the current state
      }

      switch(state)
      {
        case SND_PCM_STATE_SUSPENDED:
        {
          int err;
          while((err = snd_pcm_resume(handle)) == -EAGAIN)
          {
            sleep(1);//Wait until suspended flag is released
          }
          snd_pcm_prepare(handle);
        }
        break;//case SND_PCM_STATE_SUSPENDED:
        case SND_PCM_STATE_XRUN:
        {
          snd_pcm_prepare(handle);
        }
        //break;//case SND_PCM_STATE_XRUN:
        case SND_PCM_STATE_PREPARED:
        case SND_PCM_STATE_RUNNING:
        {
          snd_pcm_sframes_t  avail = snd_pcm_avail_update(handle);
          beep_player_error err = self->enWriteSamplestoPCM(handle,(sf_count_t)self->oDev.uGetPeriodSize(),(unsigned int)(avail/(snd_pcm_sframes_t)self->oDev.uGetPeriodSize()));
          if(err != BP_ERR_NO_ERROR)
          {
            //Check if we have reached end of file..
            self->bStopWorker = true;
          }
        }
        break;//case SND_PCM_STATE_RUNNING:
        default:
        {
          ETG_TRACE_USR3(("Unhandled State"));
          self->bStopWorker = true;
        }
        break;//default:
      }//switch(state)
    }
    if(received_events & POLLERR)
    {
      ETG_TRACE_USR3(("Received POLLERR"));
      self->bStopWorker = true;
    }
  }//while(self->oDev.poGetDevHandle() && !self->bStopWorker)
  delete[] parray;//Remove memory allocated for poll array
  ETG_TRACE_USR3(("PLAYER STATUS IS STOPPED"));
  self->vTriggerPlayStsUpdate(TRC::enFileStopped);
  ETG_TRACE_USR3(("Worker thread exited"));
  return NULL;
}

/*
* tBool sndfile_backend_player::bPrepare(const char *p_filePath, const char *p_deviceName,tBool repeatMode)
*
* Brief            :  Creates the buffer and a reader
*
* Input Parameters :
* p_filePath     the file to play
* p_deviceName      Alsa Device on which file to be played
* Return Value     : TRUE if success, else FALSE
*/
bool sndfile_backend_player::bPrepare(const char *p_filePath, const char *p_deviceName, bool bRepeat, unsigned int recurrence)
{
  if(p_filePath == NULL)
    return false;

  if(NULL == p_deviceName)
  {
    ETG_TRACE_FATAL(("Alsa device name cant be NULL"));
    return false;
  }
  //Create a file reader object
  beep_file_reader* tmpbuff = new beep_file_reader(p_filePath,bRepeat,recurrence);
  if(tmpbuff == NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
    return false;
  }

  if(tmpbuff->bIsValid())
  {
    //Initialize hardware
    if(oDev.bInitAlsaDev(p_deviceName,(unsigned int)tmpbuff->m_info.samplerate,(unsigned int)tmpbuff->m_info.channels))
    {
      //Allocate buffer for use
      //lint -e 826
      m_buffer = (short int*) new char[oDev.uGetPeriodSize()*oDev.getChannels()*snd_pcm_format_physical_width(SND_PCM_FORMAT_S16)/8];

      //Update the negotiated channels
      tmpbuff->vSetOutputChannels(oDev.getChannels());

      //Create a file buffer object
      m_filecollection[p_filePath]= tmpbuff;

      m_activefile = p_filePath;

      return true;
    }
    else
    {
      ETG_TRACE_FATAL(("Failed to Initialize Alsa Device !! "));
    }
  }
  else
  {
    ETG_TRACE_FATAL(("Format is not supported or File Doesnot exist !! "));
  }
  delete tmpbuff;
  return false;
}


/*
* tBool sndfile_backend_player::pGetActiveReader()
*
* Brief            :  Helper function to get an active reader
*
* Input Parameters :
* m_pipeline  : gstelement on which bus is created
* Return Value     : TRUE if success, else FALSE
*/
beep_file_reader* sndfile_backend_player::pGetActiveReader()
{
  std::map<std::string,beep_file_reader*>::iterator it = m_filecollection.find(m_activefile);
  if(it != m_filecollection.end())
  {
    return it->second;
  }
  return NULL;
}
/*
* tBool sndfile_backend_player::enWriteSamplestoPCM(snd_pcm_t *pcm, sf_count_t period_size, unsigned int periodcount)
*
* Brief            :  Helper function to write samples to Alsa Device
*
* Input Parameters :
* pcm          Pointer to PCM device handler
* period_size      Period size of alsa device
* periodcount      Number of periods to Write
*
* Return      : BP_ERR_NO_ERROR : If all the requested data is written to PCM data
            BP_ERR_EOF_REACHED : If end of file has reached
            BP_ERR_UNKNOWN : A Unknown error has occured
*/
beep_player_error sndfile_backend_player::enWriteSamplestoPCM(snd_pcm_t *pcm, sf_count_t period_size, unsigned int periodcount)
{
  beep_player_error err = BP_ERR_UNKNOWN;

  ETG_TRACE_USR4(("enWriteSamplestoPCM, PeriodsToWrite : %d, Period Size : %d",periodcount,(int)period_size));
    while(periodcount && pcm )
    {
      //sf_count_t readcount = sf_readf_short(infile, m_buffer, period_size);

      sf_count_t readcount = pGetActiveReader()->uFillBuffer(m_buffer,period_size);

      if(readcount)
      {
        snd_pcm_sframes_t written = snd_pcm_writei(pcm, m_buffer,(snd_pcm_uframes_t) readcount);
          if(written < 0)
          {
        ETG_TRACE_USR2(("Error writing to PCM device: %s",snd_strerror((int)written) ));
        written = snd_pcm_recover(pcm,(int)written,0);
        if(written < 0)
        {
          ETG_TRACE_FATAL(("Recover from Previous Error failed !! : %s",snd_strerror((int)written) ));
        }
        else //Recovery successful
        {
          //Retry writing the buffer
          written = snd_pcm_writei(pcm, m_buffer,(snd_pcm_uframes_t) readcount);
          if(written < 0)
          {
            ETG_TRACE_FATAL(("Giving Up after Retry to write after Recovery failed !! : %s",snd_strerror((int)written) ));
            break;
          }
          else if(readcount != written)
          {
            ETG_TRACE_USR2(("Error !! Frames Read from File : %d, Written to PCM : %d",(int)readcount,(int)written));
            break;
          }
          else
          {
            snd_pcm_start(pcm);//set the device to playing state
            err = BP_ERR_NO_ERROR;
          }
        }
          }
          else if(readcount != written)
          {
            ETG_TRACE_USR2(("Error !! Frames Read from File : %d, Written to PCM : %d",(int)readcount,(int)written));
            break;
          }
          else
          {
            err = BP_ERR_NO_ERROR;
          }
      }
      if(readcount<period_size)
      {
        err = BP_ERR_EOF_REACHED;
        ETG_TRACE_USR3(("Encountered EOF !"));
        return err;
      }
      periodcount--;
    }
     return err;
}

/*
* void sndfile_backend_player::vPrintPCMState(snd_pcm_state_t state)
*
* Brief            : Print the PCM State
*
* Input Parameters :
* state         State of PCM Device
*/
void sndfile_backend_player::vPrintPCMState(snd_pcm_state_t state)
{
  std::string s;
  switch(state)
  {
  case SND_PCM_STATE_OPEN: s = "SND_PCM_STATE_OPEN" ;break;
  case SND_PCM_STATE_SETUP:  s = "SND_PCM_STATE_SETUP";break;
  case SND_PCM_STATE_PREPARED:  s = "SND_PCM_STATE_PREPARED";break;
  case SND_PCM_STATE_RUNNING:  s = "SND_PCM_STATE_RUNNING";break;
  case SND_PCM_STATE_XRUN:  s = "SND_PCM_STATE_XRUN";break;
  case SND_PCM_STATE_DRAINING:  s = "SND_PCM_STATE_DRAINING";break;
  case SND_PCM_STATE_PAUSED:  s = "SND_PCM_STATE_PAUSED";break;
  case SND_PCM_STATE_SUSPENDED:  s = "SND_PCM_STATE_SUSPENDED";break;
  case SND_PCM_STATE_DISCONNECTED:  s = "SND_PCM_STATE_DISCONNECTED";break;
  default:
    s = "State is Unknown";break;
  }
  ETG_TRACE_USR3(("State of PCM Device : %s",s.c_str()));
}

/*
* tBool sndfile_backend_player::bPlay()
*
* Brief            : Function to play the Beep File
*
* Input Parameters : None
* Return Value     : True if playing success, else FALSE if State cant be set.
*/

bool sndfile_backend_player::bPlay()
{
  //No active reader
  if(pGetActiveReader() == NULL)
  {
    ETG_TRACE_FATAL(("There is no active reader at present... cant play"));
    vTriggerPlayStsUpdate(TRC::enFileStopped);
    return false;
  }

  ETG_TRACE_USR3(("sndfile_backend_player::bPlay() Enter "));
  snd_pcm_state_t  state = snd_pcm_state(oDev.poGetDevHandle());
  if(state == SND_PCM_STATE_PREPARED)
  {
    bStopWorker = false;
    if(BP_ERR_NO_ERROR == enWriteSamplestoPCM(oDev.poGetDevHandle(),(sf_count_t)oDev.uGetPeriodSize(),2))
    {
      /**
      * Create a worker thread with RT Priority
      */
      pthread_attr_t attr;
      sched_param param;
      param.sched_priority = sched_get_priority_max(SCHED_RR)-1;
      pthread_attr_init(&attr);
      NORMAL_M_ASSERT(0 == pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED));
      NORMAL_M_ASSERT(0 == pthread_attr_setschedpolicy(&attr, SCHED_RR) );
      NORMAL_M_ASSERT(0 == pthread_attr_setschedparam(&attr, &param) );
      NORMAL_M_ASSERT(0 == pthread_attr_setscope(&attr,PTHREAD_SCOPE_SYSTEM) );
      NORMAL_M_ASSERT(0 == pthread_attr_setdetachstate(&attr,PTHREAD_CREATE_DETACHED) );//Release resources allocated automatically on thread exit
      NORMAL_M_ASSERT(0 == pthread_create(&m_worker,&attr,&sndfile_backend_player::vWorkerTask,this));

      pthread_setname_np(m_worker,"BEEP_PLAYER_ALSA");
      pthread_attr_destroy(&attr);//Clear memory for pthread attributes
      //snd_pcm_start will be invoked in the thread entry function

      //Update the status as playing to Player
      ETG_TRACE_USR3(("Current Status is PLAYING"));
      m_bPlayerStatus = TRC::enFilePlaying;
      m_poController->vOnPlaybackStatus(TRC::enFilePlaying);
    }
    else
    {
      snd_pcm_start(oDev.poGetDevHandle());
      //Trigger status as playing and then stopped
      vTriggerPlayStsUpdate(TRC::enFilePlaying);//
      vTriggerPlayStsUpdate(TRC::enFileStopped);//
    }
    return true;
  }
  else
  {
    ETG_TRACE_FATAL(("Cannot Play as we are not in Prepared State"));
    vPrintPCMState(state);
    vTriggerPlayStsUpdate(TRC::enFileStopped);
  }
  return false;
}

/*
* tBool sndfile_backend_player::bStop(beep_player_stop_mode mode)
*
* Brief            : Function to stop playing the Beep File if playing
*
* Input Parameters :
*      mode   : Mode to use for stopping.
* Return Value     : True if STOP success, else FALSE if State cant be set.
*/

bool sndfile_backend_player::bStop(beep_player_stop_mode mode)
{
  ETG_TRACE_USR3(("sndfile_backend_player::bStop() Enter, Mode:%d",mode));
  if(mode == STOP_AFTER_EOF)
  {
    if(pGetActiveReader())
    {
      //Reset recurrence and looping parameters so that it will play to the end and stop
      pGetActiveReader()->vReset();
    }
  }
  else
  {
    bStopWorker = true;
  }
  return true;
}
/*
* tBool sndfile_backend_player::bUpdateRecurrence(int recurrence)
*
* Brief            : Function to update the recurrence of the file currently playing
*
* Input Parameters : recurrence : Recurrence period
* Return Value     : True if success, else FALSE if recurrence period cannot be set.
*/
bool sndfile_backend_player::bUpdateRecurrence(int recurrence)
{
  return (pGetActiveReader() == NULL? false : pGetActiveReader()->bSetRecurrencePeriod(recurrence));
}

/*
* tBool sndfile_backend_player::bUpdateActiveFile(const char* new_active_file)
*
* Brief            : Function
*
* Input Parameters : None
* Return Value     : True if playing success, else FALSE if State cant be set.
*/
bool sndfile_backend_player::bUpdateActiveFile(const char* new_active_file)
{
  bool ret = false;
  if(m_bPlayerStatus == TRC::enFileStopped)
  {
    ETG_TRACE_USR3(("Player is stopped, File cannot be updated"));
    return false;
  }
  if(new_active_file && (pGetActiveReader()!=NULL))
  {
    ETG_TRACE_USR3(("Updating the active file to : %s",new_active_file));
    //Check if there is a change
    if(m_activefile == std::string(new_active_file))
    {
      ETG_TRACE_USR3(("No Change in Active File"));
      return true;
    }

    beep_file_reader* tmpbuff = NULL;

    //If the file is a new file, add it to the collection
    std::map<std::string,beep_file_reader*>::iterator it = m_filecollection.find(new_active_file);
    if(it == m_filecollection.end())
    {
      ETG_TRACE_USR3(("This is a new file and doesnot exist in collection"));
      //This is a new file
      tmpbuff = new beep_file_reader(new_active_file,false);
      if(tmpbuff == NULL)
      {
        NORMAL_M_ASSERT_ALWAYS();
        return false;
      }

      if(tmpbuff->bIsValid())
      {
        if(tmpbuff->m_info.samplerate == pGetActiveReader()->m_info.samplerate)
        {
          m_filecollection[new_active_file] = tmpbuff;//Add it to collection
        }
        else
        {
          ETG_TRACE_FATAL(("Can update file as sampling rates %d != %d ",(int)tmpbuff->m_info.samplerate,(int)pGetActiveReader()->m_info.samplerate));
          delete tmpbuff;
          return false;
        }
      }
      else
      {
        delete tmpbuff;
        return false;
      }
    }
    else
    {
      tmpbuff = it->second;
    }
    //Copy the current settings to tmpbuff
    pGetActiveReader()->vCopySettingsTo(tmpbuff);

    //Todo: Pthread_lock here
    m_activefile = new_active_file;
    //Todo: pthread unlock here
    ETG_TRACE_USR3(("Active file updated to : %s",m_activefile.c_str() ));
  }
  return ret;
}
/*
* tBool sndfile_backend_player::vCleanupFileCollection()
*
* Brief            : Cleaunup memory allocated for file collection
*
* Input Parameters : None
*/
void sndfile_backend_player::vCleanupFileCollection()
{
  //Remove all file buffers
  for(map<std::string,beep_file_reader*>::iterator it=m_filecollection.begin(); it != m_filecollection.end();)
  {
    delete it->second;
    m_filecollection.erase(it);
    it = m_filecollection.begin();
  }
}
#if 0
/*
* tVoid sndfile_backend_player::vTraceRx(tU32 size, tPCUChar pcu8Data)
*
* Brief            : DeInitializes the gstreamer elements and loop
*
* Input Parameters :
* size          size of pcudata
* pcu8Data        data to be sent to beep controller
* Return Value     : None
*/
tVoid sndfile_backend_player::vTraceRx(tU32 size, tPCUChar pcu8Data)
{
  if(pcu8Data == NULL)
    return;

  if (size >= 2)
  {
    ETG_TRACE_USR4(("vTraceRx() pcu8Data[0] [1] =%u %u",(tU16)pcu8Data[0],(tU16)pcu8Data[1]));
    if(m_poController != NULL)
    {
      m_poController->vOnPlaybackStatus(pcu8Data[1]); // Playback status to Beep Controller
    }
  }
}
#endif

/*
* tVoid sndfile_backend_player::vTriggerPlayStsUpdate(tU8 u8Sts)
*
* Brief            : Triggers playback status update using loopback
*
* Input Parameters :
* u8Sts          Playback status
* Return Value     : None
*/
tVoid sndfile_backend_player::vTriggerPlayStsUpdate(tU8 u8Sts)
{
  m_bPlayerStatus = u8Sts;
  ETG_TRACE_USR1(("Triggering beep play status update via loopback : %d",u8Sts));

  if(fc_audiomanager_tclApp::theServer()->poAsyncCall())
  {
    fc_audiomanager_tclApp::theServer()->poAsyncCall()->vCall<Beep_playback_status_IF,tU8>(m_poController,&Beep_playback_status_IF::vOnPlaybackStatus,u8Sts);
  }
  else
  {
    NORMAL_M_ASSERT_ALWAYS();
  }

#if 0
  tU8 u8DataCpy[TRACE_MSG_LEN1]; //check size later
  u8DataCpy[0] = TRC::enBeepFilePlayer;
  u8DataCpy[1] = u8Sts; //Fill the play status
  //Post the loopback message
  fc_audiomanager_tclApp::vRxFromOtherThreads(TRACE_MSG_LEN1, u8DataCpy);
#endif

}
