/*******************************************************************************
*
* FILE:          gst_backend_player.cpp
*
* SW-COMPONENT:  Beep file player application
*
* DESCRIPTION:   Beep file player class implementation
*
* AUTHOR:        pmh7kor
*
* COPYRIGHT:    (c) 2015 RBEI, Bangalore
*
*******************************************************************************/


/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "../../fc_audiomanager_trace.h"
#include "../../fc_audiomanager_trace_macros.h"
#include "fc_audiomanager_main.h"

#include "gst_backend_player.h"

#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_BEEP
#include "trcGenProj/Header/gst_backend_player.cpp.trc.h"
#endif


/******************************************************************************/
/*                                                                            */
/* MACROS                                                                    */
/*                                                                            */
/******************************************************************************/
#define PLAY_STATE_NULL 1
#define PLAY_STATE_READY  2
#define PLAY_STATE_PAUSED 3
#define PLAY_STATE_PLAYING 4
//#define TRACE_MSG_LEN1 10
//#define PLAY_STATE_STOPPED 3
#define GSTPLAYER_SUCCESS 1
#define GSTPLAYER_ERROR 0

/******************************************************************************/
/*                                                                            */
/* GLOBAL VARIABLES                                                           */
/*                                                                            */
/******************************************************************************/

/* gst_backend_player()
*
* Brief  : This is the constructor for class gst_backend_player.Here
* class variables are initialized
*
* Params : Beep_playback_status_IF* m_ptrController
* Return : None
*/

gst_backend_player::gst_backend_player(Beep_playback_status_IF* m_ptrController)
{
  ETG_TRACE_USR4(("gst_backend_player(Beep_playback_status_IF)"));
  m_poController = m_ptrController;
  playStatus = PLAY_STATE_NULL;
  gst_state = GST_STATE_NULL;
  m_bLinkComplete = false;
  m_pipeline = NULL;
  m_source = NULL;
  m_decoder = NULL;
  m_converter = NULL;
  m_audiosink = NULL;
  loop = NULL;
  m_Bus = NULL;
  m_thread = 0;
  m_uiBusWatchId = 0;
}

/* ~gst_backend_player()
*
* Brief            : This is the destructor for class gst_backend_player.
* Here objects used in the class are appropriately freed.
*
* Input Parameters : None
* Return Value     : None
*/

gst_backend_player::~gst_backend_player()
{
  ETG_TRACE_USR4(("~gst_backend_player()"));
  //vTearDown();
  if(m_Bus)
  {
    gst_object_unref(m_Bus);
    m_Bus = NULL;
  }
  //gst_element_set_state(m_pipeline,GST_STATE_NULL);
  //gst_element_unlink_many((GstElement*)m_source,m_decoder,m_converter,m_audiosink, NULL);
  m_bLinkComplete = false;
  if(m_source)
  {
    gst_object_unref(m_source);
    m_source = NULL;
  }
  if(m_decoder)
  {
    gst_object_unref(m_decoder);
    m_decoder = NULL;
  }
  if(m_converter)
  {
    gst_object_unref(m_converter);
    m_converter = NULL;
  }
  if(m_audiosink)
  {
    gst_object_unref(m_audiosink);
    m_audiosink = NULL;
  }
  if(m_pipeline)
  {
    gst_object_unref(m_pipeline);
    m_pipeline = NULL;
  }
}

/*
* tBool gst_backend_player::bInitializePlayer()
*
* Brief            : Initializes the gstreamer and creates thread
*
* Input Parameters : None
* Return Value     : TRUE if Thread Creation is success, else False
*/

bool gst_backend_player::bInitializePlayer()
{
  ETG_TRACE_USR4(("gst_backend_player::bInitializePlayer"));

  if(gst_is_initialized() == FALSE)
  {
    ETG_TRACE_USR4(("gst_backend_player::bInitializePlayer, Initializing gstreamer.."));
    gst_init(NULL, NULL);
  }
  else
  {
    ETG_TRACE_USR4(("gst_backend_player::bInitializePlayer, gstreamer already initialized"));
  }

  if(m_thread == 0)//Create thread, if not created before
  {
    if(pthread_create(&m_thread, NULL, &gst_backend_player::vThreadEntry,this) != 0)
    {
      ETG_TRACE_USR4(("gst_backend_player::bInitializePlayer, failed to create gstreamer thread"));
      return false;
    }
  }
  //Multiple calls to this function is ok
  //fc_audiomanager_tclApp::theServer()->vRegisterTraceInputs(TRC::enBeepFilePlayer, this);

  return true;
}


/*
* tBool gst_backend_player::bDeinitPlayer()
*
* Brief            : DeInitializes the gstreamer elements and loop
*
* Input Parameters : None
* Return Value     : returns TRUE if Deinit is done
*/

bool gst_backend_player::bDeinitPlayer()
{
  ETG_TRACE_USR4(("gst_backend_player::bDeinitPlayer"));
  if(bStop(STOP_AFTER_PLAYING_BUFFER) == TRUE)
    ETG_TRACE_USR4(("Beep play is stopped successfully"));

  vTearDown();

  if(loop)
  {
    if (g_main_loop_is_running(loop))
    {
      g_main_loop_quit(loop);
      g_main_loop_unref(loop);
      loop = NULL;
    }
  }

  if(m_thread != 0)
  {
    pthread_join(m_thread,NULL);
    m_thread = 0;
  }

  return true;
}


/*
* tVoid gst_backend_player::vTearDown()
*
* Brief            : unlinks the gstreamer elements and free them.
*
* Input Parameters : None
* Return Value     : None
*/

void gst_backend_player::vTearDown()
{
  ETG_TRACE_USR4(("gst_backend_player::vTearDown"));
  if(m_uiBusWatchId!= 0)
  {
    g_source_remove(m_uiBusWatchId);
    m_uiBusWatchId= 0;
  }
  if(m_Bus)
  {
    gst_object_unref(m_Bus);
    m_Bus = NULL;
  }
  //gst_element_set_state(m_pipeline,GST_STATE_NULL);
  gst_element_unlink_many((GstElement*)m_source,m_decoder,m_converter,m_audiosink, NULL);
  m_bLinkComplete = false;
  if(m_source)
  {
    gst_object_unref(m_source);
    m_source = NULL;
  }
  if(m_decoder)
  {
    gst_object_unref(m_decoder);
    m_decoder = NULL;
  }
  if(m_converter)
  {
    gst_object_unref(m_converter);
    m_converter = NULL;
  }
  if(m_audiosink)
  {
    gst_object_unref(m_audiosink);
    m_audiosink = NULL;
  }
  if(m_pipeline)
  {
    gst_object_unref(m_pipeline);
    m_pipeline = NULL;
  }
}

/*
* tVoid* gst_backend_player::vThreadEntry()
*
* Brief            : Thread Function where loop is created
*
* Input Parameters : None
* Return Value     : None
*/
tVoid* gst_backend_player::vThreadEntry(tVoid* pVarg)
{
  ETG_TRACE_USR4(("gst_backend_player::creating gstreamer Thread"));
  gst_backend_player* m_thisPlayer;
  m_thisPlayer = reinterpret_cast<gst_backend_player*>(pVarg);

  if(m_thisPlayer == NULL)
  {
    FATAL_M_ASSERT_ALWAYS();
    return 0;
  }

  m_thisPlayer->loop = g_main_loop_new(NULL, FALSE);

  ETG_TRACE_USR4(("gst_backend_player::Mainloop Running"));
  g_main_loop_run(m_thisPlayer->loop);
  ETG_TRACE_USR4(("gst_backend_player::Mainloop Ended"));

  return 0;
}

/*
* tBool gst_backend_player::bPrepare(const char *p_filePath, const char *p_deviceName)
*
* Brief            :  Creates the gstreamer elements and links them to create pipeline
*
* Input Parameters :
* p_filePath     the file to play
* p_deviceName      Alsa Device on which file to be played
* Return Value     : TRUE if success, else FALSE
*/
bool gst_backend_player::bPrepare(const char *p_filePath, const char *p_deviceName,bool /*bRepeat*/ ,unsigned int /*recurrence*/)
{
  void* pvPipeline;
  if((p_deviceName == NULL) || (p_filePath == NULL))
  {
    ETG_TRACE_USR4(("gst_backend_player::bPrepare, Path Name or Alsa device are invalid"));
    return false;
  }
  ETG_TRACE_USR4(("gst_backend_player::bPrepare, for alsa device: %s ",p_deviceName));

  //Validate path and alsa device
  if(!bCheckFile_Device(p_filePath,p_deviceName))
  {
    ETG_TRACE_ERR(("gst_backend_player::bPrepare, File doesnot exist : %s",p_filePath));
    return false;
  }

  if((m_pipeline != NULL)&&(m_source != NULL) && (m_decoder != NULL)&&(m_converter != NULL)&&(m_audiosink != NULL))
  {
    ETG_TRACE_ERR(("gst_backend_player::bPrepare,! ! ! E R R O R ! ! ! Prepare invoked before teardown !!!"));
    return false;
  }

  // creating a gstreamer pipeline
  m_pipeline = gst_pipeline_new("Beep_Player");

  m_source = gst_element_factory_make("filesrc","source");
  m_decoder = gst_element_factory_make("decodebin2", "play");
  m_converter = gst_element_factory_make("audioconvert","convert");
  m_audiosink = gst_element_factory_make("alsasink", "sink");

  if ((m_pipeline == NULL) || (m_source == NULL) || (m_decoder == NULL) || (m_converter == NULL) || (m_audiosink == NULL))
  {
    ETG_TRACE_USR4(("Not able to create new gst element"));
    vTearDown();
    return false;
  }
  g_object_set((GObject*) m_source, "location", p_filePath, NULL);
  g_object_set((GObject*) m_audiosink, "device", p_deviceName, NULL);


  pvPipeline = (void *)m_pipeline;
  gst_bin_add_many((GstBin*)pvPipeline,m_source, m_decoder, m_converter, m_audiosink, NULL);

    m_bLinkComplete = false;
  //Linking source and decode bin
  if (!gst_element_link (m_source, m_decoder))
  {
    ETG_TRACE_USR4 (("source and decoder Elements could not be linked.\n"));
    gst_object_unref (m_pipeline);
    return false;
  }

  //Linking audioconverter and sink
  if (!gst_element_link (m_converter, m_audiosink))
  {
    ETG_TRACE_USR4 (("converter and sink Elements could not be linked.\n"));
    gst_object_unref (m_pipeline);
    return false;
  }

  //callback to link dynamically the decodebin2 to audioconverter
  g_signal_connect (m_decoder, "pad-added", G_CALLBACK(pad_added_handler), this);

  return createBus(m_pipeline);
}


/*
* tBool gst_backend_player::createBus(GstElement* m_pipeline)
*
* Brief            :  Creates the gstreamer bus and calls bus handler to handle the gstreamer msgs
*
* Input Parameters :
* m_pipeline  : gstelement on which bus is created
* Return Value     : TRUE if success, else FALSE
*/

tBool gst_backend_player::createBus(GstElement* pipeline)
{
  void* pvPipeline;
  pvPipeline = (void *)pipeline;
  m_Bus = gst_pipeline_get_bus((GstPipeline*)pvPipeline);

  if (!m_Bus)
  {
    ETG_TRACE_USR4(("Unable to create pipeline so Bus is null\n"));
    return GSTPLAYER_ERROR;
  }
  GSource* l_pSource = gst_bus_create_watch (m_Bus);
  if (l_pSource != NULL)
  {
    ETG_TRACE_USR4 (("CreatePipeline: Created GSource"));
    g_source_set_callback (l_pSource, (GSourceFunc) bus_callback, this, NULL);
    m_uiBusWatchId = g_source_attach (l_pSource, g_main_context_default());
    ETG_TRACE_USR4 (("CreatePipeline: Attached GSource. Id %d",m_uiBusWatchId));
    g_source_unref (l_pSource);
    return GSTPLAYER_SUCCESS;
  }
  return GSTPLAYER_ERROR;
}

/*
* tBool gst_backend_player::bCheckFile_Device(const char *p_filePath, const char *p_deviceName)
*
* Brief            :  Function to validate the beep file and alsa device
*
* Input Parameters :
* p_filePath     the file to play
* p_deviceName      Alsa Device on which file to be played
* Return Value     : GSTPLAYER_SUCCESS if success, else GSTPLAYER_ERROR
*/

tBool gst_backend_player::bCheckFile_Device(const char *p_filePath, const char *p_deviceName)
{
  tU16 fileSize = 0;
  struct stat sb;
  ETG_TRACE_USR4(("checkFile_device"));

  if ((p_filePath == NULL) || (strrchr(p_filePath,'.') == NULL))
  {
    ETG_TRACE_USR4(("Invalid File"));
    if (m_poController)
       m_poController->vOnPlaybackStatus(TRC::enFileStopped);
    return GSTPLAYER_ERROR;
  }

  ETG_TRACE_USR4(("File Ready to Play = %s ", p_filePath));
  if (stat(p_filePath, &sb) == -1)
  {
    ETG_TRACE_USR4(("Empty File or Not Regular"));
    if (m_poController)
       m_poController->vOnPlaybackStatus(TRC::enFileStopped);
    return GSTPLAYER_ERROR;
  }
  if(((S_ISREG( sb.st_mode) != 1) || (fileSize = (tU16)sb.st_size) != 0)) //is regular file and not empty
  {
    ETG_TRACE_USR4(("Regular file & File Size: %d",fileSize));
  }
  if(p_deviceName == NULL)
  {
    ETG_TRACE_USR4(("No Alsa Device Chosen, "));
    if (m_poController)
       m_poController->vOnPlaybackStatus(TRC::enFileStopped);
    return GSTPLAYER_ERROR;
  }
  return GSTPLAYER_SUCCESS;
}
/*
* void gst_backend_player::pad_added_handler (GstElement *src, GstPad *new_pad, gpointer data)
*
* Brief            : callback for dynamic linking of gst elements.Gets called on gst_set_element_state
*
* Input Parameters :
* src         source pad to be linked
* new_pad       New pad to form after dynamic link
* data         Data passed by caller of this function.
* Return Value     : None
*/

/* This function will be called by the pad-added signal */
void gst_backend_player::pad_added_handler (GstElement */*src*/, GstPad *new_pad, gpointer data)
{
  ETG_TRACE_USR4 (("pad_Added_handler"));

  gst_backend_player* m_thisPlayer;
  m_thisPlayer = reinterpret_cast<gst_backend_player*>(data);
  GstPad *sink_pad = gst_element_get_static_pad (m_thisPlayer->m_converter, "sink");
  GstPadLinkReturn ret;
  GstCaps *new_pad_caps = NULL;
  GstStructure *new_pad_struct = NULL;
  const gchar *new_pad_type = NULL;

  //ETG_TRACE_USR4 (("Received new pad '%s' from '%s':\n", GST_PAD_NAME (new_pad), GST_ELEMENT_NAME (src)));

  /* If our converter is already linked, we have nothing to do here */
  if (gst_pad_is_linked (sink_pad))
  {
    ETG_TRACE_USR4 (("  We are already linked. Ignoring.\n"));
    /* Unreference the new pad's caps, if we got them */

    /* Unreference the sink pad */
    gst_object_unref (sink_pad);
    return ;
  }

  /* Check the new pad's type */
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
  new_pad_caps = gst_pad_get_current_caps (new_pad);
#else
  new_pad_caps = gst_pad_get_caps(new_pad);
#endif

  new_pad_struct = gst_caps_get_structure (new_pad_caps, 0);
  new_pad_type = gst_structure_get_name (new_pad_struct);
  if (!g_str_has_prefix (new_pad_type, "audio/x-raw"))
  {
    ETG_TRACE_USR4 (("  It has type '%s' which is not raw audio. Ignoring.\n", new_pad_type));

    /* Unreference the new pad's caps, if we got them */
    if (new_pad_caps != NULL)
      gst_caps_unref (new_pad_caps);

    /* Unreference the sink pad */
    gst_object_unref (sink_pad);
    return ;
  }

  /* Attempt the link */
  ret = gst_pad_link (new_pad, sink_pad);
  if (GST_PAD_LINK_FAILED (ret))
  {
    ETG_TRACE_USR4(("  Type is '%s' but link failed.\n", new_pad_type));
  }
  else
  {
    m_thisPlayer->m_bLinkComplete = true;
    ETG_TRACE_USR4 (("  Link succeeded (type '%s').\n", new_pad_type));
  }
  ETG_TRACE_USR4(("Ready To Play"));
}


/*
* tBool gst_backend_player::bPlay()
*
* Brief            : Function to play the Beep File
*
* Input Parameters : None
* Return Value     : True if playing success, else FALSE if State cant be set.
*/

bool gst_backend_player::bPlay()
{
  if(NULL == m_pipeline)
    return false;

  //GstStateChangeReturn ret = GST_STATE_CHANGE_FAILURE;
  tBool ret = TRUE;
  ETG_TRACE_USR4(("gst_backend_player::bPlay, Current play sts: %d  ",playStatus));


  if(playStatus == PLAY_STATE_NULL)
  {
    ETG_TRACE_USR4 (("Setting pipeline status to playing"));
    //Check if linking is complete
    ret = set_element_state(m_pipeline, GST_STATE_PLAYING);
  }
  else if(playStatus == PLAY_STATE_PLAYING) //if already playing,
  {
    ETG_TRACE_USR4 (("Beep is already Playing"));
    if (m_poController)
       m_poController->vOnPlaybackStatus(TRC::enFilePlaying);
  }
  else //playStatus == PLAY_STATE_PAUSED
  {
    ETG_TRACE_USR4 (("Beep Play is in Pause State"));
    bResume();
  }
  return ret;
}

/*
* tBool gst_backend_player::bStop()
*
* Brief            : Function to stop playing the Beep File if playing
*
* Input Parameters : None
* Return Value     : True if STOP success, else FALSE if State cant be set.
*/

bool gst_backend_player::bStop(beep_player_stop_mode /*eMode*/)
{
  tBool ret = TRUE;
  ETG_TRACE_USR4(("gst_backend_player::bStop"));

  if(m_pipeline && (playStatus != PLAY_STATE_NULL))
  {
    /**
    * Setting pipeline state to NULL fails sometimes and the pipeline remains in ready state.
    * To overcome this, we are using a EOS event and posting it to the pipeline whenever a stop is requested.
    */
    /** Create a EOS event and send it to pipeline **/
    GstEvent *l_pEvent;
    l_pEvent = gst_event_new_eos();
    gst_element_send_event(m_pipeline, l_pEvent);
    ret = TRUE;

    //ret = set_element_state(m_pipeline, GST_STATE_NULL);
  }
  else
  {
     if (m_poController)
        m_poController->vOnPlaybackStatus(TRC::enFileStopped);
    ETG_TRACE_USR4(("Beep player is already Stopped"));
  }
  return ret;
}
/*
* tBool gst_backend_player::bPause()
*
* Brief            : Function to pause the Beep File if playing
*
* Input Parameters : None
* Return Value     : True if pause success, else FALSE if State cant be set.
*/

tBool gst_backend_player::bPause()
{
  tBool ret = TRUE;
  ETG_TRACE_USR4(("gst_backend_player::bPause"));

  if(m_pipeline && ((playStatus == PLAY_STATE_PLAYING)||(playStatus == PLAY_STATE_READY)))
  {
    ret = set_element_state(m_pipeline, GST_STATE_PAUSED);
    //playStatus = PLAY_STATE_PAUSED;
  }
  else if(playStatus == PLAY_STATE_PAUSED)
  {
    ETG_TRACE_USR4(("Already Paused"));
    if (m_poController)
       m_poController->vOnPlaybackStatus(TRC::enFilePaused);
  }
  else
  {
    ETG_TRACE_USR4(("No PlayBack"));
    return FALSE;
  }
  return ret;
}

/*
* tBool gst_backend_player::bResume()
*
* Brief            : Function to Resume playing the Beep File
*
* Input Parameters : None
* Return Value     : True if playing success, else FALSE if State cant be set.
*/

tBool gst_backend_player::bResume()
{
  tBool ret = TRUE;
  ETG_TRACE_USR4(("gst_backend_player::resume"));

  if(m_pipeline && ((playStatus == PLAY_STATE_PAUSED)||(playStatus == PLAY_STATE_READY)))
  {
    ret = set_element_state(m_pipeline, GST_STATE_PLAYING);
    //  playStatus = PLAY_STATE_PLAYING;
  }
  else if(playStatus == PLAY_STATE_PLAYING)
  {
    ETG_TRACE_USR4(("Already Playing"));
    if (m_poController)
       m_poController->vOnPlaybackStatus(TRC::enFilePlaying);
  }
  else
  {
    ETG_TRACE_USR4(("No Playback"));
    return FALSE;
  }
  return ret;
}


/*
* tBool gst_backend_player::set_element_state(GstElement* m_pipeline,GstState gst_state)
*
* Brief            : Function to Resume playing the Beep File
*
* Input Parameters :
* m_pipeline  gst element
* gst_state    gst state
* Return Value     : True if playing success, else FALSE if State cant be set.
*/

tBool gst_backend_player::set_element_state(GstElement* pipeline,GstState state)
{
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
  ret = gst_element_set_state(pipeline, state);
  if(ret == GST_STATE_CHANGE_FAILURE)
  {
    ETG_TRACE_USR4 (("Unable to set the pipeline to the state:%d",state));
    return FALSE;
  }
  return TRUE;
}

/*
* gboolean gst_backend_player::bus_callback(GstBus *bus_local, GstMessage *msg, gpointer data)
*
* Brief            : bus handler function to handle the gstreamer state changes
*
* Input Parameters :
* bus_local     Not Used currently
* msg        status message of play state
* data        data to be used in handler
* Return Value     : TRUE if success
*/
gboolean gst_backend_player::bus_callback(GstBus *bus_local, GstMessage *msg, gpointer data)
{
  //ETG_TRACE_USR1 (("gst_backend_player::bus_callback received"));//Disabled to reduce traces
  gst_backend_player* bPlayer  = (gst_backend_player *)data;
  (void) bus_local;
  if (msg == NULL)
    return TRUE;

  if(bPlayer != NULL)
  {
    //Process the message
    bPlayer->vProcessBusCallBack(msg);
  }
  else
  {
    NORMAL_M_ASSERT_ALWAYS();
  }

  return TRUE;
}

/*
* tVoid gst_backend_player::vProcessBusCallBack(GstMessage *msg)
*
* Brief            : Helper function to process the received message
*
* Input Parameters :
* msg      gst message
* Return Value     : None
*/

tVoid gst_backend_player::vProcessBusCallBack(GstMessage *msg)
{
  if(msg == NULL)
    return;

  switch (GST_MESSAGE_TYPE(msg))
  {
  case GST_MESSAGE_EOS:
    {
      // media file finished
      ETG_TRACE_USR4(("EOS STREAM RECEIVED STOPPING THE PLAYER"));
      //Set the pipeline state to NULL
      gst_element_set_state(m_pipeline,GST_STATE_NULL);
      playStatus = PLAY_STATE_NULL;
      //Invoke message informing that the playback has stopped
      vTriggerPlayStsUpdate(TRC::enFileStopped);
      break;
    }
  case GST_MESSAGE_STATE_CHANGED:
    {
      //Handle state change message
      vHandleGstStateChanged(msg);
      break;
    }
  case GST_MESSAGE_WARNING:
    {
      //Handle warning message
      vHandleWarningMsg(msg);
      break;
    }
  case GST_MESSAGE_ERROR:
    {
      //Handler error message
      vHandleErrorMsg(msg);
      playStatus = PLAY_STATE_NULL;
      break;
    }
  default:
    {
      //Unhandle messages, ignore
      break;
    }
  }
}


/*
* tVoid gst_backend_player::vHandleGstStateChanged(GstMessage *msg)
*
* Brief            : Function to handle the gst state changes
*
* Input Parameters :
* msg      gst message
* Return Value     :
* None
*/

tVoid gst_backend_player::vHandleGstStateChanged(GstMessage *msg)
{
  if(NULL == msg)
  {
    return;
  }
  // gst_play_pipeline changed state, let's forward to instance
  GstState oldState = GST_STATE_NULL;
  GstState newState = GST_STATE_NULL;
  GstState pending = GST_STATE_NULL;
  gst_message_parse_state_changed(msg, &oldState, &newState, &pending);

#if 0
  if(!m_bLinkComplete)//Linking not completed yet, ignore messages
    return;
#endif

  if (newState != gst_state)
  {
    gst_state = newState;

    switch (gst_state)
    {
    case GST_STATE_PLAYING:
      {
        playStatus = PLAY_STATE_PLAYING;
        ETG_TRACE_USR4(("Pipeline state : GST_STATE_PLAYING"));

        //Send playback status
        vTriggerPlayStsUpdate(TRC::enFilePlaying);
        break;
      }
    case GST_STATE_PAUSED:
      {
        playStatus = PLAY_STATE_PAUSED;
        ETG_TRACE_USR4(("Pipeline state : GST_STATE_PAUSED"));
        vTriggerPlayStsUpdate(TRC::enFilePaused);
        break;
      }
    case GST_STATE_READY:
      {
        playStatus = PLAY_STATE_READY;
        ETG_TRACE_USR4(("Pipeline state : GST_STATE_READY"));
        vTriggerPlayStsUpdate(TRC::enFileReady);
        break;
      }
    case GST_STATE_NULL:
      {
        playStatus = PLAY_STATE_NULL;
        ETG_TRACE_USR4(("Pipeline state : GST_STATE_NULL"));
        vTriggerPlayStsUpdate(TRC::enFileStopped);
        break;
      }
    default:
      {
        //Ignore other states
        break;
      }
    }
  }
}

/*
* tVoid gst_backend_player::vHandleErrorMsg(GstMessage *msg)
*
* Brief            : Handler when gst error is received.
*
* Input Parameters :
* msg     gst message
* Return Value     : None
*/

tVoid gst_backend_player::vHandleErrorMsg(GstMessage *msg)
{
  if(NULL == msg)
    return;

  ETG_TRACE_ERR(("G S T:  E R R O R   R E C E I V E D"));
  gchar *debug = NULL;
  GError *error = NULL;
  gst_message_parse_error(msg, &error, &debug);

  ETG_TRACE_USR4((" ERROR FROM BUS_CALL =%s", error->message));

  //ToDo: In this case, send the update as stopped
  vTriggerPlayStsUpdate(TRC::enFileStopped);

  g_free(debug);
  g_error_free(error);
}

/*
* tVoid gst_backend_player::vHandleWarningMsg(GstMessage *msg)
*
* Brief            : Handler when gst warning is received.
*
* Input Parameters :
* msg   gst message
* Return Value     : None
*/

tVoid gst_backend_player::vHandleWarningMsg(GstMessage *msg)
{
  if(NULL == msg)
    return;

  ETG_TRACE_ERR(("G S T:  W A R N I N G   R E C E I V E D"));
  gchar *debug = NULL;
  GError *error = NULL;
  gst_message_parse_warning(msg, &error, &debug);
  if((error != NULL)&&(error->message!= NULL))
  {
    ETG_TRACE_ERR(("Warning Received: %s",error->message));
  }
  g_free(debug);
  g_error_free(error);
}

#if 0
/*
* tVoid gst_backend_player::vTraceRx(tU32 size, tPCUChar pcu8Data)
*
* Brief            : DeInitializes the gstreamer elements and loop
*
* Input Parameters :
* size          size of pcudata
* pcu8Data        data to be sent to beep controller
* Return Value     : None
*/
tVoid gst_backend_player::vTraceRx(tU32 size, tPCUChar pcu8Data)
{
  if(pcu8Data == NULL)
    return;

  if (size >= 2)
  {
    ETG_TRACE_USR4(("vTraceRx() pcu8Data[0] [1] =%u %u",(tU16)pcu8Data[0],(tU16)pcu8Data[1]));
    if(m_poController != NULL)
    {
      m_poController->vOnPlaybackStatus(pcu8Data[1]); // Playback status to Beep Controller
    }
  }
}
#endif

/*
* tVoid gst_backend_player::vTriggerPlayStsUpdate(tU8 u8Sts)
*
* Brief            : Triggers playback status update using loopback
*
* Input Parameters :
* u8Sts          Playback status
* Return Value     : None
*/
tVoid gst_backend_player::vTriggerPlayStsUpdate(tU8 u8Sts)
{
  ETG_TRACE_USR1(("Triggering beep play status update via loopback : %d",u8Sts));

  if(fc_audiomanager_tclApp::theServer()->poAsyncCall())
  {
    fc_audiomanager_tclApp::theServer()->poAsyncCall()->vCall<Beep_playback_status_IF,tU8>(m_poController,&Beep_playback_status_IF::vOnPlaybackStatus,u8Sts);
  }
  else
  {
    NORMAL_M_ASSERT_ALWAYS();
  }

#if 0
  tU8 u8DataCpy[TRACE_MSG_LEN1]; //check size later
  u8DataCpy[0] = TRC::enBeepFilePlayer;
  u8DataCpy[1] = u8Sts; //Fill the play status
  //Post the loopback message
  fc_audiomanager_tclApp::vRxFromOtherThreads(TRACE_MSG_LEN1, u8DataCpy);
#endif

}


bool gst_backend_player::bUpdateActiveFile(const char* /*active_file*/)
{
  return false;
}

bool gst_backend_player::bUpdateRecurrence(int /*recurrence*/)
{
  return false;
}
