/*******************************************************************************
 *
 * FILE:          Beep_DataStore.cpp
 *
 * SW-COMPONENT:  Beep Data Store
 *
 * PROJECT:
 *
 * DESCRIPTION:   Central store where all the beep related data is stored
 *
 * AUTHOR:       Ranjit K
 *
 * COPYRIGHT:    (c) 2015 RBEI, Bangalore
 *
 *******************************************************************************/
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "../../fc_audiomanager_trace.h"
#include "../../fc_audiomanager_trace_macros.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_BEEP
#include "trcGenProj/Header/Beep_DataStore.cpp.trc.h"
#endif

#include "Beep_defines.h"
#include "Beep_DataStore.h"
#include "Priority/beepprioconf.h"
#include "vd_adr3Msg_If.h"


#define DP_S_IMPORT_INTERFACE_FI
#include "dp_audio_if.h"
/***************************
* Default beep data type
****************************/
//Constructor
beep_datatype::beep_datatype(tU8 BeepId, tU8 u8BeepExtId, tU8 uSource, tS16 volume, tS16 s16Gain, tU8 Next, tU16 Repeat,tU8 u8Generator)
:
m_u8BeepId(BeepId),
m_u8BeepExtId(u8BeepExtId),
m_u8Source(uSource),
m_s16Volume(volume),
m_s16Gain(s16Gain),
m_u8Next(Next),
m_u16Repeat(Repeat),
m_u8Generator(u8Generator)
{
  //Done
}
/***************************
* Beep data type for ADR
****************************/
//Constructor
beep_datatype_adr::beep_datatype_adr(tU8 u8BeepId, tU8 u8BeepExtId, tU8 u8Source,
    tU16 u16Freq,
    tS16 s16Volume,
    tS16 s16Gain,
    tU8 u8Next,
    tU16 u16Repeat,
    tU16 u16RiseTime,
    tU16 u16OnTime,
    tU16 u16FallTime,
    tU16 u16PeriodTime,
    tU8 u8SignalToneAction,
    tU8 u8Generator )
:
beep_datatype::beep_datatype(u8BeepId,u8BeepExtId,u8Source,s16Volume,s16Gain,u8Next,u16Repeat,u8Generator),
m_u16Freq(u16Freq),
m_u16RiseTime(u16RiseTime),
m_u16OnTime(u16OnTime),
m_u16FallTime(u16FallTime),
m_u16PeriodTime(u16PeriodTime),
m_u8SignalToneAction(u8SignalToneAction)
{
  //Done
}
/***************************
* Beep data type for IMX
****************************/
beep_datatype_imx::beep_datatype_imx(tU8 BeepId,tU8 u8BeepExtId,tU8 Source,string File_Path,tS16 s16Volume,tS16 Gain,tU8 Next,tU16 Repeat,tU8 Generator)
:
beep_datatype::beep_datatype(BeepId,u8BeepExtId,Source,s16Volume,Gain,Next,Repeat,Generator),
m_filepath(File_Path)
{
}
/**********************************************
* Data Store implementation
***********************************************/
/**
* Constructor
*
*/
beep_datastore::beep_datastore()
:m_bAdr3BeepConfigured(false)
,m_bImxBeepConfigured(false)
,m_sAlsaDev(BEEP_ALSA_DEV_GST_PLAYER_DEFAULT)
,m_beep_store()
,m_BeepPrioConfTable()
{
  ETG_TRACE_USR4(("Beep_DataStore, Constructor."));
  vInit();
}
/**
* Destructor
*
*/
beep_datastore::~beep_datastore()
{
  vDeInit();
}

/**
* Helper function to initialize the data store
*
*/
void beep_datastore::vInit()
{
  ETG_TRACE_USR4(("Beep_DataStore, Initializing"));
  //1. Load ADR beeps
  vLoadAdrBeepData();
  //2. Load IMX beeps
  vLoadImxBeepData();
  //3. Load beep priorities
  vLoadBeepPrio();//-> This is project specific and implementation is auto generated
  //4. Load project specific alsa device configuration, if any
  vLoadImxBeepConfig();
  ETG_TRACE_USR4(("Beep_DataStore, Initialized"));
}

/**
* Helper function that loads ADR Beep data from the configuration file
*
*/
void beep_datastore::vLoadAdrBeepData()
{
  ETG_TRACE_USR4(("Beep_DataStore, Loading ADR beep data"));
  size_t current_size = m_beep_store.size();
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
  tU8 u8AmpConfig;
  if((DP_S32_NO_ERR == DP_s32GetConfigItem("SystemConfiguration1","OutputInformation", &u8AmpConfig, 1)))
  {
    ETG_TRACE_USR4(("Amplifier configuration : %x", u8AmpConfig));
  }
  else
  {
    ETG_TRACE_ERR(("Error in diagnosis read "));
  }
#endif
  //Try loading ADR3 beeps from config
   #define BEEP_CONFIG
    #define AUD_BEEP_DATA(u8BeepId, u8BeepExtID, u8Source, u16Freq, s16Volume, s16Gain, u8Next, u16Repeat, u16RiseTime, u16OnTime, u16FallTime, u16PeriodTime, u8SignalToneAction, u8Generator ) \
             m_beep_store[u8BeepId]= new beep_datatype_adr(u8BeepId, u8BeepExtID, u8Source, u16Freq, s16Volume, s16Gain, u8Next, u16Repeat, u16RiseTime, u16OnTime, u16FallTime, u16PeriodTime, u8SignalToneAction, u8Generator);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
      if(u8AmpConfig == EXTERNALAMP_BOSE2CH_MCAN)
      {
        ETG_TRACE_FATAL(("Loading Buzzer Configuration"));
        #define BUZZER_CONFIG
        #include "Config/Beep.conf"
        #undef BUZZER_CONFIG
      }
      else
      {
        ETG_TRACE_FATAL(("Loading Normal Configuration"));
        #include "Config/Beep.conf"
      }
#else
      #include "Config/Beep.conf"
#endif
    #undef AUD_BEEP_DATA
   #undef BEEP_CONFIG

  //Check if any adr3 beeps got added
  if(m_beep_store.size() > current_size)
  {
    ETG_TRACE_USR4(("Beep_DataStore, ADR3 beeps are supported. Number of beeps: %d",(m_beep_store.size() - current_size)));
    m_bAdr3BeepConfigured = true;
  }
  else
  {
    ETG_TRACE_USR4(("Beep_DataStore, ADR3 beeps are NOT supported."));
  }
}
/**
* Helper function that loads IMX Beep data from the configuration file
*
*/
void beep_datastore::vLoadImxBeepData()
{
  ETG_TRACE_USR4(("Beep_DataStore, Loading IMX beep data"));
  size_t current_size = m_beep_store.size();
#ifdef VARIANT_S_IMX_BEEP_DATA
  //Try loading IMX beeps from config
  #define BEEP_CONFIG
      #define AUD_IMX_BEEP_DATA(BeepId,u8BeepExtID,Source,Filepath,volume,Gain,Next,Repeat,Generator) \
        m_beep_store[BeepId]=new beep_datatype_imx(BeepId,u8BeepExtID,Source,std::string(BEEP_FILE_BASE_DIR)+std::string(Filepath),volume,Gain,Next,Repeat,Generator);
      #include "Config/Beep.conf"
      #undef AUD_IMX_BEEP_DATA
  #undef BEEP_CONFIG
#endif
  //Check if any IMX beeps got added
  if(m_beep_store.size() > current_size)
  {
    ETG_TRACE_USR4(("Beep_DataStore, IMX beeps are Configured. Number of beeps: %d",(m_beep_store.size() - current_size)));
    m_bImxBeepConfigured = true;
  }
  else
  {
    ETG_TRACE_USR4(("Beep_DataStore, IMX beeps are NOT supported."));
  }
}

/**
 * Helper function to load the imx beep configuration
 *
 */
void beep_datastore::vLoadImxBeepConfig()
{
  ETG_TRACE_USR4(("Beep_DataStore, Loading Beep Priorities "));
#ifdef VARIANT_S_IMX_BEEP_DATA
  #define BEEP_CONFIG
    #define AUD_IMX_BEEP_CONFIG
      #include "Config/Beep.conf"

      #ifdef BEEP_ALSA_DEV_GST_PLAYER
        m_sAlsaDev = BEEP_ALSA_DEV_GST_PLAYER;
      #else
        m_sAlsaDev = BEEP_ALSA_DEV_GST_PLAYER_DEFAULT;
      #endif
    #undef AUD_IMX_BEEP_CONFIG
  #undef BEEP_CONFIG
#endif
  ETG_TRACE_USR4(("Beep_DataStore, Alsa device configured for project : %s",m_sAlsaDev.c_str()));
}

/**
* Helper function to deinitialize the data store
*
*/
void beep_datastore::vDeInit()
{
  //1. Clear beep priorites
  for (vector<BeepPrioConf*>::iterator it = m_BeepPrioConfTable.begin();it!= m_BeepPrioConfTable.end();it++)
  {
    delete (*it);
  }
  m_BeepPrioConfTable.clear();

  //2. Clear beep data
  for(map<tU8,beep_datatype*>::iterator it = m_beep_store.begin(); it != m_beep_store.end(); it++)
  {
    delete it->second;
  }
  m_beep_store.clear();
}

/**
* Helper function to get the beep config data by using its beep id
*
*/
beep_datatype* beep_datastore::pGetBeepData(tU8 u8BeepID)
{
  if(m_beep_store.end() == m_beep_store.find(u8BeepID))//Not found, return NULL
    return NULL;
  return m_beep_store[u8BeepID];
}
/**
* Helper function to check if the provided Beep ID is valid or not
*
*/
bool beep_datastore::bIsBeepValid(tU8 u8BeepId)
{
  if(m_beep_store.end() != m_beep_store.find(u8BeepId))
    return true;
  return false;
}
/**
* Helper function to get the alsa device to be used for a particular resource
* //TODO: Read the alsa device from configuration
*/
string beep_datastore::sGetAlsaDevice(tU8 /*u8Resource*/)
{
  return m_sAlsaDev;
};

bool beep_datastore::bIsAdrBeep(tU8 u8BeepId)
{
  //If generator
  if(m_beep_store.end() != m_beep_store.find(u8BeepId))
  {
    if(m_beep_store[u8BeepId] != NULL)
    {
      ETG_TRACE_USR4(("Beep_DataStore, Generator Type of Beep is :%d",m_beep_store[u8BeepId]->m_u8Generator));
      switch(m_beep_store[u8BeepId]->m_u8Generator)
      {
      case TRIANGULAR:
      case SINUSOIDAL:
        return true;
      default:
        break;
      }
    }
  }
  return false;
}
bool beep_datastore::bIsImxBeep(tU8 u8BeepId)
{
  //If generator
  if(m_beep_store.end() != m_beep_store.find(u8BeepId))
  {
    if(m_beep_store[u8BeepId] != NULL)
    {
      ETG_TRACE_USR4(("Beep_DataStore, Generator Type of Beep is :%d",m_beep_store[u8BeepId]->m_u8Generator));
      if(m_beep_store[u8BeepId]->m_u8Generator == BEEP_GST_PLAYER)
      {
        return true;
      }
    }
  }
  return false;
}

/**
* Helper function to convert internal id to external id.
*/
tU8  beep_datastore::u8GetExtID(tU8 u8BeepName)
{
  tU8 u8Retval = midw_fi_tcl_e8_AudioBeepType::FI_EN_AUDIO_BEEP_WARN_NOT_DEF;
  if(m_beep_store.end() != m_beep_store.find(u8BeepName))
  {
    if(m_beep_store[u8BeepName] != NULL)
    {
      u8Retval = m_beep_store[u8BeepName]->m_u8BeepExtId;
    }
  }
  return u8Retval;
}

/**
* Helper function to convert an external Id to an internal ID
*/
tU8 beep_datastore::u8GetIntID(tU8 u8ExternalBeepName)
{
  tU8 u8Retval = EN_AUDIO_BEEP_INVALID;
  for(map<tU8,beep_datatype*>::iterator it = m_beep_store.begin(); it != m_beep_store.end();it++)
  {
    if(it->second != NULL)
    {
      if(u8ExternalBeepName == it->second->m_u8BeepExtId)
      {
        u8Retval = it->first;
      }
    }
  }
  return u8Retval;
}
