/*******************************************************************************
 *
 * FILE:          Beep_Controller.cpp
 *
 * SW-COMPONENT:  Beep application
 *
 * DESCRIPTION:   Beep Controller Class implementation
 *
 * AUTHOR:        Ramesh Babu K
 *
 * COPYRIGHT:    (c) 2012 RBEI, Bangalore
 *
 *******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_BEEP
#include "trcGenProj/Header/Beep_Controller.cpp.trc.h"
#endif


/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include "Beep_file_Player.h"
#include "Priority/BeepPrioHandler_IF.h"
#include "Beep_Controller.h"
#include "fc_audiomanager_trace.h"
#include "../fc_audiomanager_service_Audio_Function.h"
#include "../InternalComponentCommunication/Messages/Beep/IDSignalToneParameters.h"
#include "../InternalComponentCommunication/Messages/Beep/IDChimeParameters.h"
#include "../InternalComponentCommunication/Messages/Beep/IDSignalTone.h"
#include "../InternalComponentCommunication/Messages/Volume/ID_Amp_SetVolume.h"
#include "../InternalComponentCommunication/Messages/Source/IDCCAStartAmpDeviceAudioRoute.h"
#include "../InternalComponentCommunication/Messages/Volume/VolumeManager/ID_CCA_Start_VolumeMode.h"
#include "../InternalComponentCommunication/Messages/Beep/IDNotifySignalTone.h"
#include "../InternalComponentCommunication/Messages/mute/IDNotifySystemMute.h"
#include "../InternalComponentCommunication/Messages/Beep/IDNotifyChimeParameters.h"
#include "../InternalComponentCommunication/DataTypes/TypeDefines/VolumeModeDefines.h"
//#include "../InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeOffsetData.h"
#include "../mute/stream/StreamMuteNotify_IF.h"
#include "../InternalComponentCommunication/Messages/Volume/IDSetVolumeOffset.h"

#include "../InternalComponentCommunication/DataTypes/TypeDefines/ADASBeepType.h"

#include "fc_audiomanager_main.h"
#include "Beep_playback_status_IF.h"

#include "Beep_VolumeOffsets.h"
#include "InternalComponentCommunication/Messages/Volume/VolumeManager/ID_PDC_Attenuation.h"
#include "../InternalComponentCommunication/Messages/Diag/IDNotifyDiagResult.h"

#include "config/dp/AudioUserDpIfSelect.h"
#include "../vd_amp_Msg/MCANBose/vd_amp_MCANBose_MasterAmplifier_Defines.h"
/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/
#define BEEP_RESOURCE_TYPE_LS    0x01
//#define BEEP_VOLUME_MODE         0x05
//#define NORMAL_VOLUME_MODE       0x00
//#define BEEP_MIX_CHANNEL         0x02
#define BEEP_MUTE_REASON_SRC_SEL     0x10

#define BEEP_STREAM_MUTE    0x01
#define BEEP_STREAM_DEMUTE    0x02
#define MAX_PAYLOAD 16
#define C_OFF  0
#define C_ON   1
#define BEEPLEVEL_LOW -6
#define BEEPLEVEL_MID 0
#define BEEPLEVEL_HIGH 6

#define KDS_BEEPLEVEL_0 -9
#define KDS_BEEPLEVEL_1 -6
#define KDS_BEEPLEVEL_2 -3
#define KDS_BEEPLEVEL_3  0
#define KDS_BEEPLEVEL_4  3
#define KDS_BEEPLEVEL_5  6
#define KDS_BEEPLEVEL_6  9
#define KDS_BEEPLEVEL_7  12
#define SET_WELCOMESOUND_UPADTE_STATUS 1

//#define VOLOFFSET_RESOLUTION 4;
#define LOW 1
#define MID 2
#define HIGH 3

/******************************************************************************/
/*                                                                            */
/* GLOBAL VARIABLES                                                           */
/*                                                                            */
/******************************************************************************/


/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/


/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::Beep_Controller()
 *
 * DESCRIPTION: Constructor.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/

Beep_Controller::Beep_Controller(void):IF_MessageObserver<PO_MessageConfig::enID>("BeepController Observer")
{
  u8BeepName = 0xFF; /*AUDIO_BEEP_WARN_NOT_DEF*/
  u8BeepControlState = BEEP_STATE_INIT;
  u8BeepSourceSelected = C_OFF;
  u8BeepStreamDeMuted = C_OFF;
  u8SignalToneStatus = C_OFF;
  u8BeepId = 0xFF;
  u8BeepSource = 0xFF;
  u8NewBeepSource = 0xFF;
  u8BeepStopType = 0xFF;
  m_poMainAppl = NULL;
  poBeep_PrioHander = NULL;
  pBeepParameter_IF = NULL;
  pBeepNotification_IF = NULL;
  pStreamMute = NULL;
  u8muteRegId = 0;
  m_playerstatus = TRC::enFileStopped;
  m_player = NULL;
  m_pBeepMain = NULL;
  bSystemMuteActive = false;
  m_beepVolOffsets = NULL;
  enActiveStreamId = EN_AUDIO_SOURCE_STREAM_DEFAULT;
  u8AmpConnectionState = AMP_NOT_CONNECTED;
  m_pdcStatus = false;

}

Beep_Controller::Beep_Controller(BeepPrioHandler_IF* pBeep_PrioHander, StreamMute_IF* m_pStreamMute, fc_audiomanager_tclApp* poMainAppl,Beep_Main* pBeepMain) :
  IF_MessageObserver<PO_MessageConfig::enID>("Beepcontroller Observer")
{
  ETG_TRACE_USR4(("Beep_Controller constructor"));
  poBeep_PrioHander = pBeep_PrioHander;
  m_poMainAppl = poMainAppl;
  pBeepNotification_IF = NULL;
  pBeepParameter_IF = NULL;
  u8muteRegId = 0;
  m_beepVolOffsets = new Beep_VolumeOffsets(poMainAppl);
    if(NULL == poBeep_PrioHander)
    {
        FATAL_M_ASSERT_ALWAYS();
    }

  m_pBeepMain = pBeepMain;
  if(NULL == pBeepMain)
    {
        FATAL_M_ASSERT_ALWAYS();
    }

    //m_VolumeModeHandler = new VolumeModeHandler;

    pStreamMute = m_pStreamMute;
    if(NULL == pStreamMute)
    {
        FATAL_M_ASSERT_ALWAYS();
    }
    else
    {
      u8muteRegId = pStreamMute->registerForStreamMute(StreamMute_IF::INTERNAL_BEH, this);
    }

  InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifySignalTone);
  InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifyChimeParameters);
  InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifySystemMute);
  InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifyDiagResult);
  InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this,PO_MessageConfig::ID_SetActiveStream);
  InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this,PO_MessageConfig::ID_BOSE_Amplifier_ConnectionStatus);
  u8BeepName = 0xFF; /*AUDIO_BEEP_WARN_NOT_DEF*/
  u8BeepControlState = BEEP_STATE_INIT;
  u8BeepSourceSelected = C_OFF;
  u8BeepStreamDeMuted = C_OFF;
  u8SignalToneStatus = C_OFF;
  u8BeepId = 0xFF;
  u8BeepSource = 0xFF;
  u8NewBeepSource = 0xFF;
  u8BeepStopType = 0xFF;
  m_playerstatus = TRC::enFileStopped;
  bSystemMuteActive = false;
  enActiveStreamId = EN_AUDIO_SOURCE_STREAM_DEFAULT;
  u8AmpConnectionState = AMP_NOT_CONNECTED;
  //create instance of player if it is configured in the beep conf files.
  if(m_pBeepMain->BeepData.bIsImxBeepConfigured())
  {
    m_player = new Beep_file_Player(this);
    if(m_player != NULL)
    {
      NORMAL_M_ASSERT(m_player->bInitializePlayer());//Assert, if player initialization fails
    }
  }
  else
  {
    m_player = NULL;
  }
  m_pdcStatus = false;
  MuteDemuteStream((tU8)BEEP_STREAM_MUTE, FC_AUDIO_SOURCE_STREAM_BEEP);
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::Beep_Controller()
 *
 * DESCRIPTION: Destructor.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
Beep_Controller::~Beep_Controller()
{
  ETG_TRACE_USR4(("Beep_Controller destructor"));
  InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);
  poBeep_PrioHander = NULL;
  pBeepNotification_IF = NULL;
  pBeepParameter_IF = NULL;
    pStreamMute = NULL;
  m_poMainAppl = NULL;
  m_pBeepMain = NULL;

  if (m_beepVolOffsets != NULL)
  {
    delete m_beepVolOffsets;
    m_beepVolOffsets = NULL;
  }

  if(NULL != m_player)
  {
    m_player->bDeinitPlayer();//Free resources
    delete m_player;
    m_player = NULL;
  }
}


/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::CheckAndMuteOtherStreams(tVoid)
 *
 * DESCRIPTION: Mute other stream if required
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid Beep_Controller::CheckAndMuteOtherStreams(tVoid)
{


#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
  ETG_TRACE_USR4(("Beep_Controller CheckAndMuteOtherStreams"));
  //Mute the Mix1 active streams if amplifier is connected and if UPA Beep is being played
  if((u8AmpConnectionState == AMP_CONNECTED) && (u8BeepName == EN_AUDIO_BEEP_UPA_OBSTACLE_ZONE))
  {
    ETG_TRACE_USR4(("muting Active Mix Stream : %d",enActiveStreamId));
    if(enActiveStreamId == EN_AUDIO_SOURCE_STREAM_MIX1)
      MuteDemuteStream((tU8)BEEP_STREAM_MUTE,enActiveStreamId);
  }
#endif
}
/**
 * Muting Other Streams. Supposed to be used by actuator
 */
tVoid Beep_Controller::MuteOtherStreams(tVoid)
{
  ETG_TRACE_USR4(("Beep_Controller MuteOtherStreams"));

    MuteDemuteStream((tU8)BEEP_STREAM_MUTE, EN_AUDIO_SOURCE_STREAM_MAIN); //Mute the EXC stream
    MuteDemuteStream((tU8)BEEP_STREAM_MUTE, EN_AUDIO_SOURCE_STREAM_MIX1); //Mute the MIX1 stream
    MuteDemuteStream((tU8)BEEP_STREAM_MUTE, EN_AUDIO_SOURCE_STREAM_MIX2); //Mute the MIX2 stream
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::CheckAndDemuteOtherStreams(tVoid)
 *
 * DESCRIPTION: Demute other stream if required
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid Beep_Controller::CheckAndDemuteOtherStreams(tVoid)
{
  ETG_TRACE_USR4(("Beep_Controller CheckAndDemuteOtherStreams"));

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
  ETG_TRACE_USR4(("Beep_Controller CheckAndDemuteOtherStreams"));
  //Demute the Mix1,Mix2 active streams if amplifier is connected and if UPA Beep is being played
  if((u8AmpConnectionState == AMP_CONNECTED) && (u8BeepName == EN_AUDIO_BEEP_UPA_OBSTACLE_ZONE))
  {
    ETG_TRACE_USR4(("Demuting Active Mix Stream : %d",enActiveStreamId));
    if(enActiveStreamId == EN_AUDIO_SOURCE_STREAM_MIX1)
      MuteDemuteStream((tU8)BEEP_STREAM_DEMUTE,enActiveStreamId);
  }
#endif
}
/*
 * Fucntion is supposed to beused by actuator tests
 */
tVoid Beep_Controller::DemuteOtherStreams()
{
  ETG_TRACE_USR4(("Beep_Controller DemuteOtherStreams"));
    MuteDemuteStream((tU8)BEEP_STREAM_DEMUTE, EN_AUDIO_SOURCE_STREAM_MAIN); //Mute the EXC stream
    MuteDemuteStream((tU8)BEEP_STREAM_DEMUTE, EN_AUDIO_SOURCE_STREAM_MIX1); //Mute the MIX1 stream
    MuteDemuteStream((tU8)BEEP_STREAM_DEMUTE, EN_AUDIO_SOURCE_STREAM_MIX2); //Mute the MIX2 stream
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::IsMuteBeepStreamNeeded(tVoid)
 *
 * DESCRIPTION: Returns whether beep stream to be muted or not
 *
 * PARAMETER: None
 *
 * RETURNVALUE: Bool
 *
 *******************************************************************************/
tBool Beep_Controller::IsMuteBeepStreamNeeded(tVoid) const
{
  tBool tMute = FALSE;

  if((u8NewBeepSource != u8BeepSource)&&(u8BeepStreamDeMuted == C_ON))  /*Source change required, Beep stream demuted*/
  {
    tMute = TRUE;
  }
  else if((TRUE == isActuatorTestActivated())&&(u8BeepStreamDeMuted == C_ON))
  {
    tMute = TRUE;
  }
  else
  {
    //nothing
  }

  ETG_TRACE_USR4(("Beep_Controller IsMuteBeepStreamRequired: %d", tMute));
  return tMute;
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::SetSourceAndVolume(tVoid)
 *
 * DESCRIPTION: Set the source and volume
 *
 * PARAMETER: None
 *
 * RETURNVALUE: Bool
 *
 *******************************************************************************/
tVoid Beep_Controller::SetSourceAndVolume(tVoid)
{
  ETG_TRACE_USR4(("Beep_Controller SetSourceAndVolume"));
  u8BeepSource = u8NewBeepSource;
  SetBeepSource();  //Select new beep source
  tenVolumeMode tVolMode = GetBeepVolumeMode();
  if(FALSE == isActuatorTestActivated())
     SetBeepVolumeMode(tVolMode, (midw_fi_tcl_e8_AudioChannel::tenType)FC_AUDIO_SOURCE_STREAM_BEEP);//Dont forget to extend midw_fi_tcl_e8_AudioChannel for the corresponding value of beep stream if applicable.
  else
     SetBeepVolumeMode(tVolMode, midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_NOT_DEF);

    SetBeepVolume();
     SetChannelOffsets();
}

tBool Beep_Controller::isActuatorTestActivated () const
{
  switch(u8BeepName)
  {
  case EN_AUDIO_BEEP_ACTUATOR_HMI:
  case EN_AUDIO_BEEP_ACTUATOR_DIAGNOSIS:
    return TRUE;
  default: //Satisfy lint
    break;
  }
  return (FALSE);
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::ExecuteStateMachine(BeepControllerEvent tEvent)
 *
 * DESCRIPTION: Beep controller State Machine.
 *
 * PARAMETER: BeepControllerEvent tEvent.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::ExecuteStateMachine(tU8 tEvent)
{
  ETG_TRACE_USR4(("Beep_Controller_ExecuteStateMachine Entry ControlState: %d ControlEvent: %d",
    ETG_CENUM(BeepControlState, u8BeepControlState),
    ETG_CENUM(BeepControlEvent, tEvent)));

  switch(u8BeepControlState)
  {
  case BEEP_STATE_INIT:                // New state added to check whether ADR can handle Beep requests.
    vHandleBeepStateInit(tEvent);
    break;
  case BEEP_STATE_OFF:
    vHandleBeepStateOff(tEvent);
    break;
  case BEEP_STATE_MUTE_IP:
    vHandleBeepStateMuteIP(tEvent);
    break;
  case BEEP_STATE_DEMUTE_IP:
    vHandleBeepStateDeMuteIP(tEvent);
    break;
  case BEEP_STATE_VOL_READY:
    vHandleBeepStateVolReady(tEvent);
    break;
  case BEEP_STATE_ON:
    vHandleBeepStateOn(tEvent);
    break;
  case BEEP_STATE_STOP_IP:
    vHandleBeepStateStopIP(tEvent);
    break;
  default:
    break;
  }
  ETG_TRACE_USR4(("Beep_Controller_ExecuteStateMachine Exit ControlState: %d ControlEvent: %d",
    ETG_CENUM(BeepControlState, u8BeepControlState),
    ETG_CENUM(BeepControlEvent, tEvent)));
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::vHandleBeepStateInit(tU8 tEvent)
 *
 * DESCRIPTION: Beep state Init Handler.
 *
 * PARAMETER: BeepControllerEvent tEvent.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::vHandleBeepStateInit(tU8 tEvent)
{
  // To check whether ADR can handle the beep request from Fc_AudioManager.
  // ADR returns Error for SignalChimeParameters command if V850 beeps are running already.

  if(tEvent == EN_BEEP_CHECK_ADRSTATUS)
  {
    if(m_pdcStatus)
    {
      vResetStateMachineAndClearRequest();
      return;
    }

  ChimeData SignalChimeData;

  SignalChimeData.Pulses = 0;
  SignalChimeData.AttackTime = 0;
  SignalChimeData.AttackTimeLog = 0;
  SignalChimeData.AttackTimeLin = 0;
  SignalChimeData.ReleaseTime = 0;
  SignalChimeData.ReleaseTimeLog = 0;
  SignalChimeData.ReleaseTimeLin = 0;
  SignalChimeData.SignalToneAction = 0;

  ETG_TRACE_USR4(("Beep_Controller_SignalChimeData deliverMessage: %d, %d, %d, %d, %d, %d, %d, %d",
      SignalChimeData.Pulses,
      SignalChimeData.AttackTime,
      SignalChimeData.AttackTimeLog,
      SignalChimeData.AttackTimeLin,
      SignalChimeData.ReleaseTime,
      SignalChimeData.ReleaseTimeLog,
      SignalChimeData.ReleaseTimeLin,
      SignalChimeData.SignalToneAction));
  MSG_ChimeParameters oChimeParam(SignalChimeData);
  InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oChimeParam);

  return ;
  }


  if(  tEvent == EN_STOP_BEEP  )
    {
      if (TRUE == isActuatorTestActivated() || (EN_AUDIO_BEEP_IPA_9 == u8BeepName))
      {
        u8BeepControlState = BEEP_STATE_OFF;
        u8SignalToneStatus = C_OFF;
        NotifyBeepStatus();
      }
      else
      {
        u8BeepControlState = BEEP_STATE_STOP_IP;
        StopRunningBeep();

      }



    }

}

/*******************************************************************************
*
* FUNCTION: Beep_Controller::vHandleBeepStateOff(tU8 tEvent)
*
* DESCRIPTION: Beep State Off Handler
*
* PARAMETER: BeepControllerEvent tEvent.
*
* RETURNVALUE: None.
*
*******************************************************************************/

tVoid Beep_Controller::vHandleBeepStateOff(tU8 tEvent)
{
  if(tEvent == EN_NEW_BEEP_REQUEST)
  {
    if(IsMuteBeepStreamNeeded() == TRUE)
    {
      u8BeepControlState = BEEP_STATE_MUTE_IP;
      MuteDemuteStream((tU8)BEEP_STREAM_MUTE, FC_AUDIO_SOURCE_STREAM_BEEP); //Mute the Beep stream
    }
    else
    {
      SetSourceAndVolume();
      if(u8BeepStreamDeMuted == C_ON)
      {
        if ((TRUE == isActuatorTestActivated()) || (EN_AUDIO_BEEP_IPA_9 == u8BeepName))
        {
          SetAndTriggerBeep();
          u8BeepControlState = BEEP_STATE_ON;
          u8SignalToneStatus = C_ON;
          NotifyBeepStatus();
        }
        else
        {
          u8BeepControlState = BEEP_STATE_VOL_READY;
          SetAndTriggerBeep();
        }
      }
      else
      {
        if ((TRUE == isActuatorTestActivated()) || (EN_AUDIO_BEEP_IPA_9 == u8BeepName))
        {
          SetAndTriggerBeep();
          u8BeepControlState = BEEP_STATE_DEMUTE_IP;
          u8SignalToneStatus = C_ON;
          MuteDemuteStream((tU8)BEEP_STREAM_DEMUTE, FC_AUDIO_SOURCE_STREAM_BEEP);
        }
        else
        {
          u8BeepControlState = BEEP_STATE_DEMUTE_IP;
          MuteDemuteStream((tU8)BEEP_STREAM_DEMUTE, FC_AUDIO_SOURCE_STREAM_BEEP);
        }
      }
    }
  }
  if( tEvent == EN_STOP_BEEP)
  {
            u8BeepControlState = BEEP_STATE_OFF;
        u8SignalToneStatus = C_OFF;
        NotifyBeepStatus();
  }
}


/*******************************************************************************
*
* FUNCTION: Beep_Controller::vHandleBeepStateMuteIP(tU8 tEvent)
*
* DESCRIPTION: Beep State Mute IP Handler
*
* PARAMETER: BeepControllerEvent tEvent.
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid Beep_Controller::vHandleBeepStateMuteIP(tU8 tEvent)
{
  if(tEvent == EN_NOTIFY_MUTE)
  {
    SetSourceAndVolume();
    if ((TRUE == isActuatorTestActivated()) || (EN_AUDIO_BEEP_IPA_9 == u8BeepName))
    {
      SetAndTriggerBeep();
      u8BeepControlState = BEEP_STATE_DEMUTE_IP;
      u8SignalToneStatus = C_ON;
      MuteDemuteStream((tU8)BEEP_STREAM_DEMUTE, FC_AUDIO_SOURCE_STREAM_BEEP);
    }
    else
    {
      u8BeepControlState = BEEP_STATE_DEMUTE_IP;
      MuteDemuteStream((tU8)BEEP_STREAM_DEMUTE, FC_AUDIO_SOURCE_STREAM_BEEP);
    }
  }



  if( tEvent == EN_STOP_BEEP)
    {
      if (TRUE == isActuatorTestActivated() || (EN_AUDIO_BEEP_IPA_9 == u8BeepName))
      {
        u8BeepControlState = BEEP_STATE_OFF;
        u8SignalToneStatus = C_OFF;
        MuteDemuteStream((tU8)BEEP_STREAM_MUTE, FC_AUDIO_SOURCE_STREAM_BEEP); //Mute the Beep stream
        u8BeepStreamDeMuted = C_OFF;
        NotifyBeepStatus();
      }
      else
      {
        u8BeepControlState = BEEP_STATE_STOP_IP;
        StopRunningBeep();

      }

    }

}

/*******************************************************************************
*
* FUNCTION: Beep_Controller::vHandleBeepStateDeMuteIP(tU8 tEvent)
*
* DESCRIPTION: Beep State Demute IP Handler
*
* PARAMETER: BeepControllerEvent tEvent.
*
* RETURNVALUE: None.
*
*******************************************************************************/

tVoid Beep_Controller::vHandleBeepStateDeMuteIP(tU8 tEvent)
{
  if(tEvent == EN_NOTIFY_DEMUTE)
  {
    if ((TRUE == isActuatorTestActivated()) || (EN_AUDIO_BEEP_IPA_9 == u8BeepName))
    {
      if(u8GetBeepStatus() == C_ON)
      {
        u8BeepControlState = BEEP_STATE_ON;
        u8SignalToneStatus = C_ON;
        NotifyBeepStatus();
      }
      else
      {
        u8BeepControlState = BEEP_STATE_OFF;
        SetBeepVolumeMode(EN_INT_VOLMODE_NORMAL, (midw_fi_tcl_e8_AudioChannel::tenType)FC_AUDIO_SOURCE_STREAM_BEEP);
        CheckAndDemuteOtherStreams();
        NotifyBeepStatus();
      }
    }
    else
    {
      u8BeepControlState = BEEP_STATE_VOL_READY;
      SetAndTriggerBeep();
    }
  }


  if(tEvent == EN_STOP_BEEP)
    {
      if (TRUE == isActuatorTestActivated() || (EN_AUDIO_BEEP_IPA_9 == u8BeepName))
      {
        u8BeepControlState = BEEP_STATE_OFF;
        u8SignalToneStatus = C_OFF;
        SetBeepVolumeMode(EN_INT_VOLMODE_NORMAL, midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_NOT_DEF);
        CheckAndDemuteOtherStreams();
        MuteDemuteStream((tU8)BEEP_STREAM_MUTE, FC_AUDIO_SOURCE_STREAM_BEEP); //Mute the Beep stream
        u8BeepStreamDeMuted = C_OFF;
        NotifyBeepStatus();
      }
      else
      {
        u8BeepControlState = BEEP_STATE_STOP_IP;

      }
      StopRunningBeep();

    }


}

/*******************************************************************************
*
* FUNCTION: Beep_Controller::vHandleBeepStateVolReady(tU8 tEvent)
*
* DESCRIPTION: Beep State Vol Ready Handler
*
* PARAMETER: BeepControllerEvent tEvent.
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid Beep_Controller::vHandleBeepStateVolReady(tU8 tEvent)
{
  if(tEvent == EN_NOTIFY_BEEP_STATE)
  {
    if(u8GetBeepStatus() == C_ON)
    {
      u8BeepControlState = BEEP_STATE_ON;
      NotifyBeepStatus();
      vTriggerNextEnvelopeforIPABeep();
    }
    else
    {
      u8BeepControlState = BEEP_STATE_OFF;
      SetBeepVolumeMode(EN_INT_VOLMODE_NORMAL, (midw_fi_tcl_e8_AudioChannel::tenType)FC_AUDIO_SOURCE_STREAM_BEEP);
      CheckAndDemuteOtherStreams();
      NotifyBeepStatus();
    }
  }
  else if(tEvent == EN_STOP_BEEP)
  {
    if ((FALSE == isActuatorTestActivated()) && (EN_AUDIO_BEEP_IPA_9 != u8BeepName)) //Applicable only for IPA beeps
    {
      u8BeepControlState = BEEP_STATE_STOP_IP;
      StopRunningBeep();
    }
  }
  else
  {
    //Nothing to do
  }

}

/*******************************************************************************
*
* FUNCTION: Beep_Controller::vHandleBeepStateOn(tU8 tEvent)
*
* DESCRIPTION: Beep State ON Handler.
*
* PARAMETER: BeepControllerEvent tEvent.
*
* RETURNVALUE: None.
*
*******************************************************************************/

tVoid Beep_Controller::vHandleBeepStateOn(tU8 tEvent)
{
  if(tEvent == EN_NOTIFY_BEEP_STATE)
  {
    if(u8GetBeepStatus() == C_OFF)
    {
      u8BeepControlState = BEEP_STATE_OFF;
      SetBeepVolumeMode(EN_INT_VOLMODE_NORMAL, (midw_fi_tcl_e8_AudioChannel::tenType)FC_AUDIO_SOURCE_STREAM_BEEP);
      CheckAndDemuteOtherStreams();
      NotifyBeepStatus();
    }
    else
    {
      vTriggerNextEnvelopeforIPABeep();
    }
  }
  else if(tEvent == EN_STOP_BEEP)
  {
    if (TRUE == isActuatorTestActivated() || (EN_AUDIO_BEEP_IPA_9 == u8BeepName))
    {
      u8BeepControlState = BEEP_STATE_OFF;
      u8SignalToneStatus = C_OFF;
      SetBeepVolumeMode(EN_INT_VOLMODE_NORMAL, midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_NOT_DEF);
      CheckAndDemuteOtherStreams();
      MuteDemuteStream((tU8)BEEP_STREAM_MUTE, FC_AUDIO_SOURCE_STREAM_BEEP); //Mute the Beep stream
      u8BeepStreamDeMuted = C_OFF;
      NotifyBeepStatus();
    }
    else
    {
      u8BeepControlState = BEEP_STATE_STOP_IP;

    }
    StopRunningBeep();
  }
  else
  {
    /*Nothing*/
  }

}


/*******************************************************************************
*
* FUNCTION: Beep_Controller::vHandleBeepStateStopIP(tU8 tEvent)
*
* DESCRIPTION: Beep State Stop IP Ready Handler
*
* PARAMETER: BeepControllerEvent tEvent.
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid Beep_Controller::vHandleBeepStateStopIP(tU8 tEvent)
{
  if(tEvent == EN_NOTIFY_BEEP_STATE)
  {
    if(u8GetBeepStatus() == C_OFF)
    {
      u8BeepControlState = BEEP_STATE_OFF;
      SetBeepVolumeMode(EN_INT_VOLMODE_NORMAL, (midw_fi_tcl_e8_AudioChannel::tenType)FC_AUDIO_SOURCE_STREAM_BEEP);
      CheckAndDemuteOtherStreams();
      NotifyBeepStatus();
    }
  }
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::StartBeep(tU8 tUBeepName)
 *
 * DESCRIPTION: start beep sequence
 *
 * PARAMETER: tU8 tUBeepName.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::StartBeep(BeepData oBeepData)
{
  //This check should always fail..., else it is a bug somewhere
  if(!oBeepData.m_bIsValid)
  {
    NORMAL_M_ASSERT_ALWAYS();
    NORMAL_M_ASSERT(oBeepData.bGetExtendedData());
  }

  //For A-IVI project no need to check if V850 is using the mix3 channel. So directly proceed with OFF State
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CSM_RNAIVI
  u8BeepControlState = BEEP_STATE_INIT;
  //Copy the recieved request locally
  m_beepdata = oBeepData;

  u8BeepName = oBeepData.ubBeepName;
  u8BeepId = oBeepData.ubBeepID;
  pBeepNotification_IF = oBeepData.poBeepNotification_IF;
  pBeepParameter_IF = oBeepData.poBeepParameter_IF;
  ETG_TRACE_USR4(("Beep_Controller StartBeep BeepName :%d, Beep Id : %d", u8BeepName,u8BeepId));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);
  m_poMainAppl->bStopTimer(FC_AUDIOMANAGER_TIMER_ID_BEEP);
  pBeepNotification_IF->SetBeepIdnStatus(u8BeepName,u8BeepId);
  u8NewBeepSource =  m_beepdata.m_u8Source;
  ExecuteStateMachine(EN_BEEP_CHECK_ADRSTATUS);
#else

  u8BeepControlState = BEEP_STATE_OFF;
  //Copy the recieved request locally
  m_beepdata = oBeepData;

  u8BeepName = oBeepData.ubBeepName;
  u8BeepId = oBeepData.ubBeepID;
  pBeepNotification_IF = oBeepData.poBeepNotification_IF;
  pBeepParameter_IF = oBeepData.poBeepParameter_IF;
  ETG_TRACE_USR4(("Beep_Controller StartBeep BeepName :%d, Beep Id : %d", u8BeepName,u8BeepId));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);
  m_poMainAppl->bStopTimer(FC_AUDIOMANAGER_TIMER_ID_BEEP);
  pBeepNotification_IF->SetBeepIdnStatus(u8BeepName,u8BeepId);

  if(m_pBeepMain->BeepData.bIsAdrBeep(u8BeepName) && (u8AmpConnectionState == AMP_CONNECTED))
  {
    m_pBeepMain->vSendBeepRequestToBuzzer(u8BeepName);
    return;
  }

  u8NewBeepSource =  m_beepdata.m_u8Source;

  CheckAndMuteOtherStreams();
  vMuteSignalToneGenerator(TRIANGULAR);
  vMuteSignalToneGenerator(SINUSOIDAL);


  ExecuteStateMachine(EN_NEW_BEEP_REQUEST);

#endif
}


 /*******************************************************************************
  *
  * FUNCTION: Beep_Controller::IdleBeep(tU8 tUBeepName)
  *
  * DESCRIPTION: stop the current running beep
  *
  * PARAMETER: tU8 tUBeepName.
  *
  * RETURNVALUE: None.
  *
  *******************************************************************************/
tVoid Beep_Controller::IdleBeep(/*tU8 tUBeepName,*/ tU8 tStopType)
{
  ETG_TRACE_USR4(("Beep_Controller IdleBeep BeepName:%d  StopType:%d", u8BeepName, tStopType));
  u8BeepStopType = tStopType;
  ExecuteStateMachine(EN_STOP_BEEP);
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::MuteDemuteStream(tU8 tmuteaction)
 *
 * DESCRIPTION: Mute/Demute Beep stream
 *
 * PARAMETER: tU8 tmuteaction.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::MuteDemuteStream(tU8 tmuteaction, tU8 tStream)
{
   StreamMute_IF::tenStreamMuteaction action = (StreamMute_IF::tenStreamMuteaction)tmuteaction;

   ETG_TRACE_USR4(("Beep_Controller MuteDemuteStream Action:%d Stream:%d", tmuteaction, tStream));

   if(NULL != pStreamMute)
    {
          pStreamMute->muteStream(u8muteRegId, action, tenFadingReason(BEEP_MUTE_REASON_SRC_SEL), tenStream(tStream));
    }
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::StreamMuteNotify(tenMuteNotification muteAck, tU8 stream)
 *
 * DESCRIPTION: Mute Notification handler
 *
 * PARAMETER: tenMuteNotification muteAck, tU8 stream.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
void Beep_Controller::StreamMuteNotify(tenMuteNotification muteAck, tU8 stream, tU16 sink)
{
  ETG_TRACE_USR4(("Beep_Controller_StreamMuteNotify called for %d and sink %d", muteAck, sink));

  if(stream == FC_AUDIO_SOURCE_STREAM_BEEP)
  {
    switch(muteAck)
    {
      case MUTE_ACK_DEMUTE:
        u8BeepStreamDeMuted = C_ON;
        ExecuteStateMachine(EN_NOTIFY_DEMUTE);
        break;
      case MUTE_ACK_MUTE:
        u8BeepStreamDeMuted = C_OFF;
        ExecuteStateMachine(EN_NOTIFY_MUTE);
        break;
      default:
      break;
    }
  }
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::SetBeepSource(tVoid)
 *
 * DESCRIPTION: Select the source for the requested Beeptype
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::SetBeepSource(tVoid)
{
  ETG_TRACE_USR4(("Beep_Controller_SetBeepSource called for %d", u8NewBeepSource));

  //1. Set the Source in ADR so that settings can be loaded. Source (0xF13).
  MSG_SetSource oSource(FC_AUDIO_SOURCE_STREAM_BEEP,(tenSourceType)u8NewBeepSource,midw_fi_tcl_e8_AudSource::FI_EN_CUE, 1);//ToDo > Check if the 3rd parameter is valid
  InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSource);

  //2. Set the route in ADR, so that audio routed is heard.
  ID_CCAStart_AmpDeviceAudioRoute oAmpDeviceAudioRoute(BEEP_RESOURCE_TYPE_LS, u8NewBeepSource, FC_AUDIO_SOURCE_STREAM_BEEP);

   InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oAmpDeviceAudioRoute);
   u8BeepSourceSelected = C_ON;
//   ExecuteStateMachine(EN_BEEP_SOURCE_SELECTED);
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::GetBeepVolumeMode(tVoid)
 *
 * DESCRIPTION: Returns the volume mode to be set for beep type
 *
 * PARAMETER: None
 *
 * RETURNVALUE: tenVolumeMode.
 *
 *******************************************************************************/
tenVolumeMode Beep_Controller::GetBeepVolumeMode(tVoid) const
{
  tenVolumeMode tVolumeMode;

  if(EN_AUDIO_BEEP_ACTUATOR_HMI == u8BeepName)
  {
    tVolumeMode = EN_INT_VOLMODE_ACTUATOR_TEST_HMI;
  }
  else if(EN_AUDIO_BEEP_ACTUATOR_DIAGNOSIS == u8BeepName)
  {
    tVolumeMode = EN_INT_VOLMODE_ACTUATOR_TEST_DIAGNOSIS;
  }
  else
  {
    tVolumeMode = EN_INT_VOLMODE_BEEP;
  }
  ETG_TRACE_USR4(("Beep_Controller_GetBeepVolumeMode called for %d", tVolumeMode));

  return tVolumeMode;
}

/*******************************************************************************
   *
   * FUNCTION: Beep_Controller::SetBeepVolumeMode(tVoid)
   *
   * DESCRIPTION: Set volume mode to Beep
   *
   * PARAMETER: tenVolumeMode.
   *
   * RETURNVALUE: None.
   *
 *******************************************************************************/
tVoid Beep_Controller::SetBeepVolumeMode(tenVolumeMode tVolMode, midw_fi_tcl_e8_AudioChannel::tenType Channel) const
{
  VolumeModeData Data;
  Data.m_Resource = (midw_fi_tcl_e8_ResourceNo::tenType)BEEP_RESOURCE_TYPE_LS;
  Data.m_VolumeModeType = tVolMode;
  Data.m_Channel = Channel;
  ETG_TRACE_USR4(("SetVolumeMode VolumeModeType: %d", tVolMode));

  ID_CCA_Start_VolumeMode oSetVolumeMode(Data);
  InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSetVolumeMode);
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::SetChannelOffsets(tVoid)
 *
 * DESCRIPTION: Set Channeloffsets
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::SetChannelOffsets(tVoid)
{
  tU8 u8Edtype = 0;
  VolumeOffsetData VolumeOffsetDataset;

  //Set Channeloffsets for Beep. Its done only for actuator tests now.
  if (EN_AUDIO_BEEP_CLICK == u8BeepName)
    u8Edtype = (tU8) EN_ED_INPUTGAINOFFSET_TONE_TOUCHSCREEN;
  else
    u8Edtype = (tU8) EN_ED_INPUTGAINOFFSET_TONE_BEEP;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pBeepNotification_IF);
  if(pBeepNotification_IF->bIsAdasSoundActive(u8BeepName))
  {
    vSetADASIputGainOffset(VolumeOffsetDataset,m_beepdata.m_channeloffsets);
  }
  else
#endif
  {
#if defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
  vSetIputGainOffset_PSA(VolumeOffsetDataset, m_beepdata.m_channeloffsets, u8Edtype);
#else
  vSetIputGainOffset_Normal(VolumeOffsetDataset, m_beepdata.m_channeloffsets, u8Edtype);
#endif
  }
  MSG_SetVolumeOffset oSetVolumeOffset(VolumeOffsetDataset);
  InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSetVolumeOffset);
}

/*******************************************************************************
 *
* FUNCTION: Beep_Controller::vSetIputGainOffset_Normal
   (VolumeOffsetData &VolumeOffsetDataset,BeepControlData::ChVolOff &ChannelOffsets,tU8 Edtype) const
 *
 * DESCRIPTION: Set vSetIputGainOffset
 *
 * PARAMETER: VolumeOffsetData &VolumeOffsetDataset,BeepControlData::ChVolOff &ChannelOffsets,tU8 Edtype
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::vSetIputGainOffset_Normal(VolumeOffsetData &VolumeOffsetDataset,BeepControlData::ChVolOff &ChannelOffsets,tU8 Edtype) const
{
  tS32 *s32EngineeringData = OSAL_NEW tS32[MAX_PAYLOAD];
  dp_tclAudioManagerDPEngineeringData oDPEngineeringData;
  if(s32EngineeringData != OSAL_NULL)
  {

    oDPEngineeringData.s32GetData(s32EngineeringData, MAX_PAYLOAD);
    tS16 EngineeringDataOffsetVal = (tS16)(s32EngineeringData[Edtype] *VOLOFFSET_RESOLUTION);

    tU8 u8Beeplevel = 0;

      if(AudioUserDpIfSelect::pGetAudUsrDpIf())
         AudioUserDpIfSelect::pGetAudUsrDpIf()->s32GetBeepLevel(u8Beeplevel);

    tS16 s16BeepLevelOffset = s16GetBeepOffset(u8Beeplevel);

    ETG_TRACE_USR4(("vSetIputGainOffset_Normal called, s16BeepLevelOffset : %d, EngineeringDataOffsetVal : %d", s16BeepLevelOffset, EngineeringDataOffsetVal));
    VolumeOffsetDataset.m_enStream = FC_AUDIO_SOURCE_STREAM_BEEP;

    VolumeOffsetDataset.s16Offset_LF = (tS16)(ChannelOffsets.LF + EngineeringDataOffsetVal + s16BeepLevelOffset);
    VolumeOffsetDataset.s16Offset_RF = (tS16)(ChannelOffsets.RF + EngineeringDataOffsetVal + s16BeepLevelOffset); //gen4 [-wconversion] compiler warning fix
    VolumeOffsetDataset.s16Offset_CE = ChannelOffsets.CE;
    VolumeOffsetDataset.s16Offset_SUB = ChannelOffsets.SUB;

    VolumeOffsetDataset.s16Offset_LR = ChannelOffsets.LR;
    VolumeOffsetDataset.s16Offset_RR = ChannelOffsets.RR;
    VolumeOffsetDataset.u16RampLin = ChannelOffsets.RampLin;
    VolumeOffsetDataset.u16RampdB = ChannelOffsets.RampDB;
  }
  OSAL_DELETE[] s32EngineeringData;
}
/*******************************************************************************
*
* FUNCTION: Beep_Controller::vSetIputGainOffset_PSA
(VolumeOffsetData &VolumeOffsetDataset,BeepControlData::ChVolOff &ChannelOffsets,tU8 Edtype) const
*
* DESCRIPTION: Set vSetIputGainOffset
*
* PARAMETER: VolumeOffsetData &VolumeOffsetDataset,BeepControlData::ChVolOff &ChannelOffsets,tU8 Edtype
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid Beep_Controller::vSetIputGainOffset_PSA(VolumeOffsetData &VolumeOffsetDataset,BeepControlData::ChVolOff &ChannelOffsets,tU8 Edtype) const
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pBeepMain);
  tS32 *s32EngineeringData = OSAL_NEW tS32[MAX_PAYLOAD];
  dp_tclAudioManagerDPEngineeringData oDPEngineeringData;
  if(s32EngineeringData != OSAL_NULL)
  {
    oDPEngineeringData.s32GetData(s32EngineeringData, MAX_PAYLOAD);
    tS16 EngineeringDataOffsetVal = (tS16)(s32EngineeringData[Edtype] * VOLOFFSET_RESOLUTION);

    tU8 u8Beeplevel = 0;

      if(AudioUserDpIfSelect::pGetAudUsrDpIf())
         AudioUserDpIfSelect::pGetAudUsrDpIf()->s32GetBeepLevel(u8Beeplevel);

    tS16 s16BeepLevelOffset = 0;
    if(m_pBeepMain->BeepData.bIsImxBeep(u8BeepName))
    {
      s16BeepLevelOffset = s16GetBeepOffset(u8Beeplevel);
    }
    ETG_TRACE_USR4(("vSetIputGainOffset_PSA called, s16BeepLevelOffset : %d, EngineeringDataOffsetVal : %d", s16BeepLevelOffset, EngineeringDataOffsetVal));
    VolumeOffsetDataset.m_enStream = FC_AUDIO_SOURCE_STREAM_BEEP;

    tS16 s16BeepLevelOffset_Kds = 0;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_KDS_HIFI_PSA_LINUX
    if(u8BeepName == EN_AUDIO_BEEP_GST_14) // for Roger Beep
    {
      tU8 KdsOffsetLevel = 0;
      dp_tclKdsCAL_Fct_AAS oDPKdsCAL_Fct_AAS;
      oDPKdsCAL_Fct_AAS.u8GetAASVolumeLevelFrontAASBeeps(KdsOffsetLevel);

      ETG_TRACE_USR4(("KdsOffsetLevel:  %d", KdsOffsetLevel));

      s16BeepLevelOffset_Kds = s16GetBeepOffset_Kds(KdsOffsetLevel);
      s16BeepLevelOffset = 0; //beeplevel offset should not affect radio memorization beeps for PSA.
    }
#endif
    if(pBeepParameter_IF != NULL)
    {
      if (pBeepParameter_IF->GetSoundSystemConfig() == SOUND_CONFIG_ITEM_VALUE_SOUNDSYSTEM4) //For PSA, Beep to be heard on SUBWOOFER & CENTRE Speakers for Gen3 Amplifier
      {
        VolumeOffsetDataset.s16Offset_CE = (tS16)(ChannelOffsets.CE + EngineeringDataOffsetVal + s16BeepLevelOffset + s16BeepLevelOffset_Kds);
        VolumeOffsetDataset.s16Offset_SUB = (tS16)(ChannelOffsets.SUB + EngineeringDataOffsetVal + s16BeepLevelOffset + s16BeepLevelOffset_Kds);
        VolumeOffsetDataset.s16Offset_LF = ChannelOffsets.LF;
        VolumeOffsetDataset.s16Offset_RF = ChannelOffsets.RF;
      }
      else // For Gen1,Gen2,HiFi Surround(Analog) Amplifiers, KDS,level offsets should be set to Front,CE
      {
        VolumeOffsetDataset.s16Offset_LF = (tS16)(ChannelOffsets.LF + EngineeringDataOffsetVal + s16BeepLevelOffset + s16BeepLevelOffset_Kds);
        VolumeOffsetDataset.s16Offset_RF = (tS16)(ChannelOffsets.RF + EngineeringDataOffsetVal + s16BeepLevelOffset+ s16BeepLevelOffset_Kds);
        VolumeOffsetDataset.s16Offset_CE = (tS16)(ChannelOffsets.CE + EngineeringDataOffsetVal + s16BeepLevelOffset+ s16BeepLevelOffset_Kds);
        VolumeOffsetDataset.s16Offset_SUB = ChannelOffsets.SUB;
      }
    }
    VolumeOffsetDataset.s16Offset_LR = ChannelOffsets.LR;
    VolumeOffsetDataset.s16Offset_RR = ChannelOffsets.RR;
    VolumeOffsetDataset.u16RampLin = ChannelOffsets.RampLin;
    VolumeOffsetDataset.u16RampdB = ChannelOffsets.RampDB;
  }
  OSAL_DELETE[] s32EngineeringData;
}

/*******************************************************************************
*
* FUNCTION: Beep_Controller::vSetADASIputGainOffset
(VolumeOffsetData &VolumeOffsetDataset,BeepControlData::ChVolOff &ChannelOffsets,tU8 Edtype) const
*
* DESCRIPTION: Set vSetIputGainOffset
*
* PARAMETER: VolumeOffsetData &VolumeOffsetDataset,BeepControlData::ChVolOff &ChannelOffsets,tU8 Edtype
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid Beep_Controller::vSetADASIputGainOffset(VolumeOffsetData &VolumeOffsetDataset,BeepControlData::ChVolOff &ChannelOffsets) const
{
  ETG_TRACE_USR4(("Beep_Controller::vSetADASIputGainOffset() called"));
  VolumeOffsetDataset.m_enStream = FC_AUDIO_SOURCE_STREAM_BEEP;
  VolumeOffsetDataset.s16Offset_LF = ChannelOffsets.LF;
  VolumeOffsetDataset.s16Offset_RF = ChannelOffsets.RF;

  VolumeOffsetDataset.s16Offset_LR = ChannelOffsets.LR;
  VolumeOffsetDataset.s16Offset_RR = ChannelOffsets.RR;
  VolumeOffsetDataset.s16Offset_CE = ChannelOffsets.CE;
  VolumeOffsetDataset.s16Offset_SUB = ChannelOffsets.SUB;
  VolumeOffsetDataset.u16RampLin = ChannelOffsets.RampLin;
  VolumeOffsetDataset.u16RampdB = ChannelOffsets.RampDB;
  m_beepVolOffsets->vRemapVolumeOffsets(VolumeOffsetDataset);
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::s16GetBeepOffset(tU8 beeplevel) const
 *
 * DESCRIPTION: Returns corresponding Offset based on beep level.
 *
 * PARAMETER: tU8 beeplevel
 *
 * RETURNVALUE: tS16 beepOffset.
 *
 *******************************************************************************/
tS16 Beep_Controller::s16GetBeepOffset(tU8 beeplevel) const
{
  ETG_TRACE_USR4(("s16GetBeepOffset BeepLevel : %d", beeplevel));

  tS16 beepOffset = 0;
  switch(beeplevel)
  {
  case LOW:
    beepOffset = BEEPLEVEL_LOW * VOLOFFSET_RESOLUTION;
    break;
  case MID:
    beepOffset = BEEPLEVEL_MID * VOLOFFSET_RESOLUTION;
    break;
  case HIGH:
    beepOffset = BEEPLEVEL_HIGH * VOLOFFSET_RESOLUTION;
    break;
  default:
    break;
  }
  ETG_TRACE_USR4(("s16GetBeepOffset calculated : %d", beepOffset));
  return beepOffset;
}


/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::s16GetBeepOffset(tU8 beeplevel) const
 *
 * DESCRIPTION: Returns corresponding Offset based on beep level.
 *
 * PARAMETER: tU8 beeplevel
 *
 * RETURNVALUE: tS16 beepOffset.
 *
 *******************************************************************************/
tS16 Beep_Controller::s16GetBeepOffset_Kds(tU8 beeplevel) const
{
  ETG_TRACE_USR4(("s16GetBeepOffset_Kds BeepLevel : %d", beeplevel));

  tS16 beepOffset = 0;
  switch(beeplevel)
  {
  case 1:
    beepOffset = KDS_BEEPLEVEL_0 * VOLOFFSET_RESOLUTION;
    break;
  case 2:
    beepOffset = KDS_BEEPLEVEL_1 * VOLOFFSET_RESOLUTION;
    break;
  case 3:
    beepOffset = KDS_BEEPLEVEL_2 * VOLOFFSET_RESOLUTION;
    break;
  case 4:
    beepOffset = KDS_BEEPLEVEL_3 * VOLOFFSET_RESOLUTION;
    break;
  case 5:
    beepOffset = KDS_BEEPLEVEL_4 * VOLOFFSET_RESOLUTION;
    break;
  case 6:
    beepOffset = KDS_BEEPLEVEL_5 * VOLOFFSET_RESOLUTION;
    break;
  case 7:
    beepOffset = KDS_BEEPLEVEL_6 * VOLOFFSET_RESOLUTION;
    break;
  case 8:
    beepOffset = KDS_BEEPLEVEL_7 * VOLOFFSET_RESOLUTION;
    break;
  default:
    break;
  }
  ETG_TRACE_USR4(("s16GetBeepOffset_Kds calculated : %d", beepOffset));
  return beepOffset;
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::SetBeepVolume(tVoid)
 *
 * DESCRIPTION: Set volume for requested beeptype
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::SetBeepVolume(tVoid)
{
  ETG_TRACE_USR4(("SetBeepVolume() : u8AmpConnectionState : %d", u8AmpConnectionState));
  if(u8AmpConnectionState == AMP_CONNECTED)
  {
    ETG_TRACE_USR4((" MCAN Bose Connected, Not setting any volume"));
    return;
  }
  AmpVolumeData AmplifierDataSet;

  /* Set Volume for Beep */   /*Hardcoded values. To be changed*/
  AmplifierDataSet.m_enStream = FC_AUDIO_SOURCE_STREAM_BEEP; //ToDO: use real data
  //AmplifierDataSet.m_VolStep = 0x00; //ToDO: use real data
  AmplifierDataSet.m_VoldB = m_beepdata.m_s16Gain;
  AmplifierDataSet.m_VolStep = (tU8)(m_beepdata.m_s16VolumeStep);   //updating with step value in case of incremental volume type //gen4 [-wconversion] compiler warning fix
  AmplifierDataSet.m_Sink = 1; //Hardcoded to LoudSpeaker.
  AmplifierDataSet.m_RampLin = 10; //ToDO: use real data
  AmplifierDataSet.m_RampdB = 300; //ToDO: use real data
  ID_Amp_SetVolume oSetVolume(AmplifierDataSet);
  InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSetVolume);
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::SetSignalTone(tVoid)
 *
 * DESCRIPTION: Set SignalTone. Called for Click and TestTones.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::SetSignalTone(tVoid) const
{
  tU8 tSignalToneType, tSignalToneAction;
  tSignalToneType = 0x01;
  if (EN_AUDIO_BEEP_CLICK == u8BeepName)
    tSignalToneType = 0x11;
  tSignalToneAction = 0x04;
  ETG_TRACE_USR4(("Beep_Controller_SignalTone deliverMessage: %d, %d",
      tSignalToneType,
      tSignalToneAction));
  MSG_SignalTone oSignalTone(tSignalToneType, tSignalToneAction);
  InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSignalTone);
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::SetSignalToneParam(tVoid)
 *
 * DESCRIPTION: Set SignalTone parameters. Called for all beeptypes except Click.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::SetSignalToneParam(tVoid)
{
  SignalToneParameterData SignalToneData;

  SignalToneData.SignalToneID = m_beepdata.m_beepparam.BeepId;
  SignalToneData.Frequency = m_beepdata.m_beepparam.Freq;
  SignalToneData.Gain = (tS16)(m_beepdata.m_beepparam.Gain * 128);


  ETG_TRACE_USR4(("Beep_Controller_SignalToneData deliverMessage: %d, %d, %d",
      SignalToneData.SignalToneID,
      SignalToneData.Frequency,
      SignalToneData.Gain));

  MSG_SignalToneParameters oSignalToneParam(SignalToneData);
  InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSignalToneParam);
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::StopRunningBeep(tVoid)
 *
 * DESCRIPTION: Function to stop the currently running Beeps. Especially needed for IPA beeps.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::StopRunningBeep(tVoid) const
{
  ChimeData SignalChimeData;

  SignalChimeData.Pulses = 0;
  SignalChimeData.AttackTime = 0;
  SignalChimeData.AttackTimeLog = 0;
  SignalChimeData.AttackTimeLin = 0;
  SignalChimeData.ReleaseTime = 0;
  SignalChimeData.ReleaseTimeLog = 0;
  SignalChimeData.ReleaseTimeLin = 0;
  SignalChimeData.SignalToneAction = u8BeepStopType;

  ETG_TRACE_USR4(("Beep_Controller_SignalChimeData deliverMessage: %d, %d, %d, %d, %d, %d, %d, %d",
      SignalChimeData.Pulses,
      SignalChimeData.AttackTime,
      SignalChimeData.AttackTimeLog,
      SignalChimeData.AttackTimeLin,
      SignalChimeData.ReleaseTime,
      SignalChimeData.ReleaseTimeLog,
      SignalChimeData.ReleaseTimeLin,
      SignalChimeData.SignalToneAction));
  MSG_ChimeParameters oChimeParam(SignalChimeData);
  InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oChimeParam);

  //Stop beep player, if playing any beep
  if(m_player != NULL)
  {
    m_player->bStop();
  }

}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::SetSignalChimeParam(tVoid)
 *
 * DESCRIPTION: Function to trigger SignalChimeParameter to ADR.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::SetSignalChimeParam(tVoid)
{
  BeepControlData::ChimePara& ChimeParameter= m_beepdata.m_chimeparam;

  ChimeData SignalChimeData;

  SignalChimeData.Pulses = ChimeParameter.Pulses;
  SignalChimeData.AttackTime = ChimeParameter.AttackTime;
  SignalChimeData.AttackTimeLog = ChimeParameter.AttackTimeLog;
  SignalChimeData.AttackTimeLin = ChimeParameter.AttackTimeLin;
  SignalChimeData.ReleaseTime = ChimeParameter.ReleaseTime;
  SignalChimeData.ReleaseTimeLog = ChimeParameter.ReleaseTimeLog;
  SignalChimeData.ReleaseTimeLin = ChimeParameter.ReleaseTimeLin;
  SignalChimeData.SignalToneAction = ChimeParameter.SignalToneAction;

  ETG_TRACE_USR4(("Beep_Controller_SignalChimeData deliverMessage: %d, %d, %d, %d, %d, %d, %d, %d",
      SignalChimeData.Pulses,
      SignalChimeData.AttackTime,
      SignalChimeData.AttackTimeLog,
      SignalChimeData.AttackTimeLin,
      SignalChimeData.ReleaseTime,
      SignalChimeData.ReleaseTimeLog,
      SignalChimeData.ReleaseTimeLin,
      SignalChimeData.SignalToneAction));
  MSG_ChimeParameters oChimeParam(SignalChimeData);
  InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oChimeParam);
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::vTriggerNextEnvelopeforIPABeep(tVoid)
 *
 * DESCRIPTION: Function to trigger SignalChimeParameter to generate next pulse for infinite duration beeps
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::vTriggerNextEnvelopeforIPABeep(tVoid)
{
  if ((EN_AUDIO_BEEP_IPA_4 == u8BeepName) || (EN_AUDIO_BEEP_IPA_5 == u8BeepName) ||
    (EN_AUDIO_BEEP_IPA_6 == u8BeepName))
  {
    SetSignalChimeParam();
  }
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::SetAndTriggerBeep(void)
 *
 * DESCRIPTION: Set SignalTone and ChimeParameters
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::SetAndTriggerBeep(tVoid)
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pBeepMain);
  if(m_pBeepMain->BeepData.bIsAdrBeep(u8BeepName))
  {
    ETG_TRACE_USR4(("SetAndTriggerBeep: ADR beep : %d",  u8BeepName));

  switch(u8BeepName)
  {
    case EN_AUDIO_BEEP_CLICK:
    {
      SetSignalTone();
    }
    break;
    case EN_AUDIO_BEEP_ACTUATOR_HMI:
    case EN_AUDIO_BEEP_ACTUATOR_DIAGNOSIS:
    case EN_AUDIO_BEEP_IPA_9:
    {
      SetSignalToneParam();
    }
        break;
    default:  //Beep types with chime
    {
      SetSignalToneParam();
      SetSignalChimeParam();
    }
        break;
  }
}
  else if(m_pBeepMain->BeepData.bIsImxBeep(u8BeepName))
  {
    if(u8GetBeepStatus() == C_ON)
      return;
    ETG_TRACE_USR4(("SetAndTriggerBeep: IMX beep : %d",  u8BeepName));
    if((m_player != NULL)&&(pBeepNotification_IF != NULL))
    {
      std::string beep_file_path = m_beepdata.m_filepath;
      ETG_TRACE_USR4(("SetAndTriggerBeep: IMX beep : %s",  beep_file_path.c_str()));
      //To use pBeepController_IF & get the beep data.. extend beep controller interface
      if(!beep_file_path.empty())
      {
        //Retrieve alsa device info and file info
        //Prepare the player
        bool bRepeat = false;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
        if((u8BeepName == EN_AUDIO_BEEP_UPA_OBSTACLE_ZONE) ||(u8BeepName == EN_AUDIO_BEEP_UPA_SETTING_SND)||(u8BeepName == EN_AUDIO_BEEP_LDW_SETTING_SND)|| (u8BeepName ==EN_AUDIO_BEEP_RCTA_SND) || (u8BeepName == EN_AUDIO_BEEP_LDW_SOUND_ALERT) )

        {
          bRepeat = true;
        }

#endif
        if(m_player->bPrepare(beep_file_path.c_str(),m_pBeepMain->BeepData.sGetAlsaDevice(BEEP_RESOURCE_TYPE_LS).c_str(),bRepeat,pBeepNotification_IF->GetRecurrencePeriod(u8BeepName)))
        {
          //Start playing
          if(!m_player->bPlay())
          {
            ETG_TRACE_ERR(("SetAndTriggerBeep: Failed to set pipeline to playing ! ! !, file: %s",beep_file_path.c_str()));
          }
        }
        else
        {
          ETG_TRACE_ERR(("SetAndTriggerBeep: Prepare failed ! ! !"));
        }
      }
      else
      {
        ETG_TRACE_ERR(("SetAndTriggerBeep: File path is empty ! ! !"));
      }
    }
  }
}

/*************************************************************************
 * FUNCTION:     Beep_Controller::NotifyBeepStatus(tVoid)
 *
 * DESCRIPTION:  Notification for Beep status.
 *
 * PARAMETER:    -
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid Beep_Controller::NotifyBeepStatus(tVoid)
{
  if(u8GetBeepStatus() == C_OFF)
    {
      if(NULL != poBeep_PrioHander)
      {
      poBeep_PrioHander->ClearRequest(u8BeepName);
        CheckMuteBeepStream();
      }
    }
  else
  {
    if(NULL != pBeepNotification_IF)
    {
      pBeepNotification_IF->NotifyBeepState(u8BeepName, u8BeepId, u8GetBeepStatus(),m_beepdata.u32BeepToken);
    }
  }
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::CheckMuteBeepStream(tVoid)
 *
 * DESCRIPTION: Start the timer before mute the Beep stream
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::CheckMuteBeepStream(tVoid) const
{
  if(NULL != poBeep_PrioHander)
  {
    if((poBeep_PrioHander->getSizeOfQueue() == 0x00)&&(u8BeepStreamDeMuted == C_ON))
    {
      AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poMainAppl);
        if(TRUE == m_poMainAppl->bStartTimer(FC_AUDIOMANAGER_TIMER_ID_BEEP, FC_AUDIOMANAGER_TIMER_BEEP_FIRST_TICK, FC_AUDIOMANAGER_TIMER_BEEP_TICK_INTERVAL))
      {
        ETG_TRACE_USR4(("Timer in Beep controller returned true"));
      }
      else
      {
        ETG_TRACE_USR4(("Timer in Beep controller returned false"));
      }
    }
  }
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::DoMuteBeepStream(tVoid)
 *
 * DESCRIPTION: Mute the Beep stream
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::DoMuteBeepStream(tVoid)
{

  if(NULL != poBeep_PrioHander)
  {
    if(poBeep_PrioHander->getSizeOfQueue() == 0x00)
    {
      MuteDemuteStream((tU8)BEEP_STREAM_MUTE, FC_AUDIO_SOURCE_STREAM_BEEP); //Mute the Beep stream
      u8BeepStreamDeMuted = C_OFF;
    }
  }
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::SetBeepLocation(tVoid)
 *
 * DESCRIPTION: Set new parameters for testtones when it is running.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::SetBeepLocation(tVoid)
{
  NORMAL_M_ASSERT(m_beepdata.bGetExtendedData());
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pBeepMain);
  if(EN_AUDIO_BEEP_ACTUATOR_DIAGNOSIS == u8BeepName)
  {
    SetBeepVolume();
  }
  SetChannelOffsets();
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pBeepNotification_IF);
  //AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_player);
  if((u8GetBeepStatus() == C_ON) && pBeepNotification_IF->bIsAdasSoundActive(u8BeepName))
  {
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_player);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
    if(u8BeepName == EN_AUDIO_BEEP_UPA_OBSTACLE_ZONE)
      m_player->bUpdateRecurrence(pBeepNotification_IF->GetRecurrencePeriod(u8BeepName));
#endif
    std::string sfile = pBeepNotification_IF->GetBeepFilePath(u8BeepName);
    m_player->bUpdateActiveFile(sfile.c_str());
  }
  else
  {
    if((EN_AUDIO_BEEP_ACTUATOR_DIAGNOSIS == u8BeepName)||(EN_AUDIO_BEEP_ACTUATOR_HMI == u8BeepName))
      SetAndTriggerBeep();
  }

  if(m_pBeepMain->BeepData.bIsAdrBeep(u8BeepName))
  {
  u8SignalToneStatus = C_ON;
  NotifyBeepStatus();
  }

}



tVoid Beep_Controller::UpdateBeepVolume(tVoid)
{
  NORMAL_M_ASSERT(m_beepdata.bGetExtendedData());
  SetBeepVolume();
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::vMuteSignalToneGenerator(U8 u8Generator)
 *
 * DESCRIPTION: Set gain to -100 dB to mute the other tone generators before start of new beep.
 *
 * PARAMETER: tU8.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Controller::vMuteSignalToneGenerator(tU8 u8Generator) const
{
  SignalToneParameterData SignalToneData;

   ETG_TRACE_USR4(("Beep_Controller_MuteSignalToneGenerator entered Generator:%d", u8Generator));

  SignalToneData.SignalToneID = u8Generator;
  SignalToneData.Frequency = 100;
  SignalToneData.Gain = (-100 * 128);   // Mutes unused generators

  ETG_TRACE_USR4(("Beep_Controller_SignalToneData deliverMessage: %d, %d, %d",
      SignalToneData.SignalToneID,
      SignalToneData.Frequency,
      SignalToneData.Gain));

  MSG_SignalToneParameters oSignalToneParam(SignalToneData);
  InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSignalToneParam);
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::MessageNotification(PO_MessageConfig::enID MsgId)
 *
 * DESCRIPTION: This function is called when any of the post office messsages addressed to Beep_Controller is posted
 *
 * PARAMETER:PO_MessageConfig::enID MsgId
 *
 * RETURN TYPE : void
 *
 *******************************************************************************/
void Beep_Controller::MessageNotification(PO_MessageConfig::enID MsgId)
{
   ETG_TRACE_USR4(("Beep_Controller_Message Notification function entered"));
   AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pBeepMain);

   switch(MsgId)
   {
   case(PO_MessageConfig::ID_NotifyChimeParameters):
  {
     const ID_NotifyChimeParameters* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_NotifyChimeParameters>(MsgId);
     u8SignalToneStatus = pMsg->enBeepState;
     ETG_TRACE_USR4(("IDNotifyChimeParametersObserver: MessageNotification: %d",
         u8SignalToneStatus));

      vHandleChimeParameterSignalStatus();
    break;
  }
   case(PO_MessageConfig::ID_NotifySignalTone):
  {
     const ID_NotifySignalTone* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_NotifySignalTone>(MsgId);
     ETG_TRACE_USR4(("IDNotifySignalToneObserver: MessageNotification: %d, %d",
          pMsg->ubSignalToneType, pMsg->ubSignalToneStatus));
     u8SignalToneStatus = pMsg->ubSignalToneStatus;
     if(m_pBeepMain->BeepData.bIsAdrBeep(u8BeepName))
     {
    ExecuteStateMachine(EN_NOTIFY_BEEP_STATE);
     }
     else
     {
       ETG_TRACE_USR4(("Beep_Controller_Message Current played beep : %d,is not ADR beep, State Machine Trigger not sent ",u8BeepName));
     }
    break;
  }
   case (PO_MessageConfig::ID_NotifySystemMute):
    {
     const ID_NotifySystemMute* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_NotifySystemMute>(MsgId);

     ETG_TRACE_USR4(("ID_NotifySystemMute: MessageNotification: Requester: %x, Reason: %x , Resource:%x , State : %x",
        pMsg->Requester,pMsg->Reason,pMsg->Resource,pMsg->State));
     AUD_POINTER_CHECK_CALL_NO_RET_VALUE(poBeep_PrioHander);  //lint fix
     if(pMsg->State == EN_AUDIO_MUTE_MUTEACTION_On){
      ETG_TRACE_USR4(("ID_NotifySystemMute, System Mute is activated !!"));

      bSystemMuteActive = true;
      //Stop immediateley beep if any playing
      ETG_TRACE_USR4(("ID_NotifySystemMute, System Mute is activated, Stop beep immediately if any !!"));
      m_pBeepMain->m_poBeepController->IdleBeep(BEEP_STOP_IMMEDIATE);

      ETG_TRACE_USR4(("ID_NotifySystemMute,SystemMute is Active, print beeps in the queue "));
      poBeep_PrioHander->PrintBeepsInTheQueue();

      //clear Beep Queue
      ETG_TRACE_USR4(("ID_NotifySystemMute, System Mute is activated, Clearing Beep queue !!"));
      poBeep_PrioHander->vClearQueue();
      CheckMuteBeepStream();
     }else{
       ETG_TRACE_USR4(("ID_NotifySystemMute, System Mute is removed!"));
       bSystemMuteActive = false;
     }
    break; //lintfix
  }

   case(PO_MessageConfig::ID_NotifyDiagResult) :
    {
        ETG_TRACE_USR4(("Beep_Controllers::MessageNotification: ID_NotifyDiagResult"));
        // Obtaining the Diag Message from PostOffice handler
        const ID_NotifyDiagResult*  pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_NotifyDiagResult>(MsgId);

        //Receive the Diag Result Message from VD_ADR3.
        ETG_TRACE_USR3(("MessageNotification: Diagnose Result Observer received ErrorSite: 0x%4x ErrorCode: 0x%4x ErrorValue: 0x%4x", pMsg->enErrorSite, pMsg->enErrorCode, pMsg->enErrorValue));

    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_beepVolOffsets);
        //Updating ChannelErrorStatusMap with Error Value for the received Error site
        m_beepVolOffsets->vUpdateErrorSiteCodeMap(pMsg->enErrorSite, pMsg->enErrorCode, pMsg->enErrorValue);

    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pBeepNotification_IF);
    if(pBeepNotification_IF->bIsAdasSoundActive(u8BeepName))
      SetChannelOffsets();
  }
  break;

   case (PO_MessageConfig::ID_SetActiveStream):
     {
        ETG_TRACE_USR4(("Beep_Controllers::MessageNotification:ID_SetActiveStream"));

      const ID_SetActiveStream* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_SetActiveStream>(MsgId);
      AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
      enActiveStreamId = pMsg->activeStreamId;
       ETG_TRACE_USR4(("Current active stream is %u", ETG_CENUM(tenStream, enActiveStreamId)));
     }
     break;
   case (PO_MessageConfig::ID_BOSE_Amplifier_ConnectionStatus):
     {
        const ID_BOSE_Amplifier_ConnectionStatus* pBoMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_BOSE_Amplifier_ConnectionStatus>(MsgId);
      AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pBoMsg);
        ETG_TRACE_USR4(("Beep_Controllers::MessageNotification:ID_BOSE_Amplifier_ConnectionStatus : %d",pBoMsg->connectionstate));
      u8AmpConnectionState = pBoMsg->connectionstate;
     }
     break;
     case PO_MessageConfig::ID_PDC_Attenuation:
    {
      const ID_PDC_Attenuation* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage <ID_PDC_Attenuation>(MsgId);
      AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);

      if(pMsg->m_u8PDCAttenuationCommand == midw_fi_tcl_e8_PDCAttenuation::FI_EN_AUDIO_PDC_COMMAND_START)
      {
        m_pdcStatus = true;
        if((u8BeepControlState == BEEP_STATE_ON)||(u8BeepControlState == BEEP_STATE_VOL_READY))
          IdleBeep(BEEP_STOP_IMMEDIATE);

        poBeep_PrioHander->vClearQueue();
        u8BeepControlState = BEEP_STATE_OFF;
        u8SignalToneStatus = C_OFF;
        u8BeepStreamDeMuted = C_OFF;
      }
      else
        m_pdcStatus = false;

      ETG_TRACE_USR4(("Beep_Controllers::MessageNotification:ID_PDC_Attenuation, m_pdcStatus: %d",m_pdcStatus));
    }
      break;

   default:
     break;
   }
 }

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::vOnPlaybackStatus(tU8 sts)
 *
 * DESCRIPTION: This function is called when a play back status update is received from beep player
 *
 * PARAMETER:PlaybackStatus sts: Playback status
 *
 * RETURN TYPE : void
 *
 *******************************************************************************/
void Beep_Controller::vOnPlaybackStatus(tU8 sts)
{
  tU8 u8Status = SET_WELCOMESOUND_UPADTE_STATUS;
  ETG_TRACE_USR4(("Beep_Controller_PlaybackStatus Notification. Status : %d",sts));


  //Only consider play and stopped updates, will extend if required later
  switch(sts)
  {
  case TRC::enFileStopped:
  case TRC::enFilePlaying:
    break;
  default:
    return;
  }

  m_playerstatus = sts;//Update locally


  //Tear down
  if((m_player != NULL)&& (m_playerstatus == TRC::enFileStopped ))
  {
    m_player->vTearDown();
    if(m_poMainAppl != NULL && m_poMainAppl->pGetAudioFunctionptr() != NULL )
    {
       if(m_poMainAppl->pGetAudioFunctionptr()->IsOEMRenault())
       {
         m_poMainAppl->pGetAudioFunctionptr()->vSetRemoveWelcomeSound(u8Status);
         if (m_poMainAppl->pGetAudioFunctionptr()->updateClients(MIDW_MASCFFI_C_U16_REMOVEWELCOMESOUND) != AIL_EN_N_NO_ERROR)
         {
           ETG_TRACE_USR3(("Beep_ExtAmpHandlerMain::Updating clients for removeWelcomeSound status"));
         }
       }
   }
  }
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pBeepMain);
  if(m_pBeepMain->BeepData.bIsImxBeep(u8BeepName))
  {
    ExecuteStateMachine(EN_NOTIFY_BEEP_STATE);//Trigger update to statemachine
  }
  else
  {
    ETG_TRACE_USR4(("Beep_Controller_PlaybackStatus Current played beep : %d,is not IMX beep, State Machine Trigger disabled",u8BeepName));
  }
}
/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::u8GetBeepStatus()
 *
 * DESCRIPTION: Returns the status of the active Beep.
 *
 * PARAMETER: None.
 *
 * RETURN TYPE : tU8.
 *
 *******************************************************************************/
tU8 Beep_Controller::u8GetBeepStatus()
{
  tU8 u8Ret = C_OFF;
  AUD_POINTER_CHECK_CALL(m_pBeepMain);
  if(m_pBeepMain->BeepData.bIsAdrBeep(u8BeepName))
  {
    u8Ret = u8SignalToneStatus;
  }
  else if(m_pBeepMain->BeepData.bIsImxBeep(u8BeepName))
  {
     ETG_TRACE_USR4(("u8GetBeepStatus() : %d",m_playerstatus));
    //This is a IMX beep
    if(m_playerstatus == TRC::enFilePlaying)
    {
      u8Ret = C_ON;
    }
  }
  return u8Ret;
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::vResetStateMachineAndClearRequest()
 *
 * DESCRIPTION: Resets Beep Statemachine and clears Beep requests in Queue.
 *
 * PARAMETER: None.
 *
 * RETURN TYPE : void
 *
 *******************************************************************************/
tVoid Beep_Controller::vResetStateMachineAndClearRequest()
{
  u8BeepControlState = BEEP_STATE_OFF;
  u8SignalToneStatus = C_OFF;
  NotifyBeepStatus();
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Controller::vHandleChimeParameterSignalStatus()
 *
 * DESCRIPTION: Handles vHandleChimeParameterSignalStatus.
 *
 * PARAMETER: None.
 *
 * RETURN TYPE : void
 *
 *******************************************************************************/
tVoid Beep_Controller::vHandleChimeParameterSignalStatus()
{
       ETG_TRACE_USR4(("bSystemMuteActive : %d", bSystemMuteActive));
     AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pBeepMain);
     if(u8SignalToneStatus == (tU8)EN_AUDIO_BEEP_FAILURE)
     {
        //Reset the StateMachine and Clear Request when Error response is received from ADR.
      vResetStateMachineAndClearRequest();
     }
     else if(u8BeepControlState == BEEP_STATE_INIT && bSystemMuteActive == false && m_pdcStatus == false)
     {
       u8BeepControlState = BEEP_STATE_OFF;
       CheckAndMuteOtherStreams();
       vMuteSignalToneGenerator(TRIANGULAR);
       vMuteSignalToneGenerator(SINUSOIDAL);

      //start the beep sequence...
      ExecuteStateMachine(EN_NEW_BEEP_REQUEST);
     }
     else
     {
      // if(m_pBeepMain->BeepData.bIsAdrBeep(u8BeepName))
       {
         ExecuteStateMachine(EN_NOTIFY_BEEP_STATE);
       }
       //else
       {
       //  ETG_TRACE_USR4(("Beep_Controller_Message Current played beep : %d,is not ADR beep, State Machine Trigger not sent ",u8BeepName));
       }
     }
}

