/*******************************************************************************
 *
 * FILE:          Beep_Adas.cpp
 *
 * SW-COMPONENT:  Beep application
 *
 * DESCRIPTION:   Beep Adas Class implementation
 *
 * AUTHOR:        pmh7kor
 *
 * COPYRIGHT:    (c) 2016 RBEI, Bangalore
 *
 *******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_BEEP
#include "trcGenProj/Header/Beep_Adas.cpp.trc.h"
#endif


/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include "Beep_Adas.h"
#include "../fc_audiomanager_trace.h"
#include "../mute/stream/StreamMute.h"

#include "InternalComponentCommunication/Messages/Beep/IDNotifyBeepState.h"
#include "InternalComponentCommunication/Messages/SoundConfig/IDSetSoundSystemConfig.h"
#include "InternalComponentCommunication/Messages/Beep/IDNotifyBeepFileStatus.h"

#include "InternalComponentCommunication/Messages/Beep/ID_NotifyParkAssistVolume.h"
#include "InternalComponentCommunication/Messages/Beep/ID_ADASSoundCollection.h"
#include "InternalComponentCommunication/Messages/Beep/ID_ADASBeepActive.h"
#include "InternalComponentCommunication/Messages/mute/IDNotifySystemMute.h"
#include "InternalComponentCommunication/Messages/settings/IDProfileDataChanged.h"
#include "InternalComponentCommunication/Messages/Volume/VolumeManager/ID_CANAttenuation.h"
#include "Beep_Actuator.h"
#include "Beep_Controller.h"
#include "Priority/Beep_PrioHandler.h"
#include "../fc_audiomanager_service_Audio_Function.h"
#include "Beep_defines.h"
#include "Beep_CSMSignalInterface.h"

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::Beep_Adas()
 *
 * DESCRIPTION: Constructor.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/

Beep_Adas::Beep_Adas(void):IF_MessageObserver<PO_MessageConfig::enID>("Beep_Adas Constructor")
{
  ETG_TRACE_USR4(("Beep_Adas() entered."));
  m_poBeepPrioHandler = NULL;
  m_poBeepController = NULL;
  m_poBeepMainAppl = NULL;
  m_pStreamMute = NULL;
  m_poAudio_Function = NULL;
  m_poBeep_CSMSignalInterface = NULL;
    u8BeepName = 0xFF;
  u8RunningBeepName = 0xFF;
    u8BeepId = 0xFF;
  u8BeepState = 0xFF;
  m_u8SoundSystemconfig = SOUND_CONFIG_ITEM_VALUE_SOUNDSYSTEM1;
  m_rsp_beep_token = 0;
  m_poAdasNotificationIF = NULL;
  vResetInternalData();
  m_poVolumeMgr = NULL;
  pAudUsrDpIf   = NULL;
  m_AdasNotificationMap.clear();
}

Beep_Adas::Beep_Adas(Beep_Main* poMainAppl, BeepPrioHandler_IF* poBeepPrioHandler,VolumeManager::VolumeManager* poVolumeMgr)
   : IF_MessageObserver<PO_MessageConfig::enID>("Beep_Adas")
{
  m_pStreamMute = StreamMute::getInstance();
  m_AdasNotificationMap.clear();
  m_poBeepMainAppl = poMainAppl;
  m_poAudio_Function = poMainAppl->m_poAudio_Function;
  m_poBeepController = poMainAppl->m_poBeepController;
  m_poBeepPrioHandler = poBeepPrioHandler;
  m_poAdasNotificationIF = NULL;
  m_poVolumeMgr = poVolumeMgr;
  u8BeepName = 0xFF;
  u8RunningBeepName = 0xFF;
    u8BeepId = 0xFF;
  u8BeepState = 0xFF;
  m_u8SoundSystemconfig = SOUND_CONFIG_ITEM_VALUE_SOUNDSYSTEM1;
  m_rsp_beep_token = 0;
  pAudUsrDpIf   = AudioUserDpIfSelect::pGetAudUsrDpIf();
  m_poBeep_CSMSignalInterface = new Beep_CSMSignalInterface(poMainAppl,this);

   vResetInternalData();
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::IDSetSoundSystemConfig);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifyParkAssistVolume);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_ADASSoundCollection);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_NotifySystemMute);
   InternalCommunicationAdapter::getInstance()->POMessages->AddObserver(this, PO_MessageConfig::ID_ProfileDataChanged);
   ETG_TRACE_USR2(("Beep_Adas: Observer added for ID_CCAStartADAS_Beep & IDSetSoundSystemConfig"));
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::~Beep_Adas()
 *
 * DESCRIPTION: Destructor.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
Beep_Adas::~Beep_Adas()
{
  ETG_TRACE_USR4(("~Beep_Adas() entered."));
  InternalCommunicationAdapter::getInstance()->POMessages->DeRegisterObserver(this);
  m_poBeepPrioHandler = NULL;
  m_poBeepController = NULL;
  m_poBeepMainAppl = NULL;
  m_poVolumeMgr = NULL;
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_pStreamMute);
  m_pStreamMute->vDeleteMuteResponseTimer();
  m_pStreamMute = NULL;
    m_poAudio_Function = NULL;
  u8BeepName = 0xFF;
  u8RunningBeepName = 0xFF;
  u8BeepId = 0xFF;
  u8BeepState = 0xFF;
  m_u8SoundSystemconfig = 0x0;
  m_poAdasNotificationIF = NULL;
  m_AdasNotificationMap.clear();
  if(m_poBeep_CSMSignalInterface != NULL)
  {
    delete (m_poBeep_CSMSignalInterface);
    m_poBeep_CSMSignalInterface = NULL;
  }
}



/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::MessageNotification(PO_MessageConfig::enID MsgId)
 *******************************************************************************/

// This function is called by post office on new incoming messages.
void Beep_Adas::MessageNotification(PO_MessageConfig::enID MsgId)
{
  switch(MsgId)
  {
  case (PO_MessageConfig::IDSetSoundSystemConfig):
    {
      const IDSetSoundSystemConfig* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage < IDSetSoundSystemConfig > (MsgId);
      AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
      m_u8SoundSystemconfig = pMsg->u8SoundConfig;
      ETG_TRACE_USR2(("Beep_Adas: MessageNotification(IDSetSoundSystemConfig): %d", m_u8SoundSystemconfig));
      break;
    }
  case (PO_MessageConfig::ID_NotifyParkAssistVolume) :
  {
    const ID_NotifyParkAssistVolume* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_NotifyParkAssistVolume>(MsgId);
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
    ETG_TRACE_USR2(("ID_NotifyParkAssistVolume received : %d", pMsg->m_volume));
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poAdasNotificationIF);
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poBeepController);
    if (m_poAdasNotificationIF->bIsAdasSoundActive()) //updating the volume if any UPA beep is active
    { m_poBeepController->UpdateBeepVolume();

        if(pMsg->m_volume != VOLUME_LEVEL_0)
        {
          tU8 BeepName=0;  //fix compilerwarning

          ID_ADASBeepActive oAdasBeepActive(TRUE,EN_ADAS_BEEPTYPE_UPA,(m_poAdasNotificationIF->GetBeepVolume_withoutCANAttenuation())/4); // Posting PO message to enable attenuation of main stream
           InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oAdasBeepActive);
           (void)BeepName;
        }
    }
    break;
  }
  case (PO_MessageConfig::ID_ADASSoundCollection) :
  {
    const ID_ADASSoundCollection* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_ADASSoundCollection>(MsgId);
    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
    ETG_TRACE_USR2(("ID_ADASSoundCollection received : %d", pMsg->m_SoundType));
    tS32 s32Res = 0;
    tU8 AdasUPASndtype=0;;
    s32Res =  pAudUsrDpIf->s32GetUPASoundtype(AdasUPASndtype);
    if(pMsg->m_SoundType !=AdasUPASndtype)
    {    s32Res =  pAudUsrDpIf->s32SetUPASoundtype(pMsg->m_SoundType);
        (tVoid)s32Res;
         ETG_TRACE_USR4(("Persistent Memory storage Result for pMsg->m_SoundType = %x", s32Res));
         AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poAdasNotificationIF);
         AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poBeepController);

     if (m_poAdasNotificationIF->bIsAdasSoundActive())
             m_poBeepController->SetBeepLocation();

  }
    break;
  }
  case (PO_MessageConfig::ID_NotifySystemMute):
    {
     const ID_NotifySystemMute* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<ID_NotifySystemMute>(MsgId);
     AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
     ETG_TRACE_USR4(("ID_NotifySystemMute: MessageNotification: Requester: %x, Reason: %x , Resource:%x , State : %x",
        pMsg->Requester,pMsg->Reason,pMsg->Resource,pMsg->State));
     AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poBeep_CSMSignalInterface);
     if(pMsg->State == EN_AUDIO_MUTE_MUTEACTION_On){
       m_poBeep_CSMSignalInterface->vResetAdasData();
     }
     else
      m_poBeep_CSMSignalInterface->vForceSignalRead();// triggering force read of CAN signals only on system mute off if obstacle exists
        break;
    }

    case (PO_MessageConfig::ID_ProfileDataChanged):
    {
        const MSG_ProfileDataChanged* pMsg = InternalCommunicationAdapter::getInstance()->POMessages->QueryMessage<MSG_ProfileDataChanged>(MsgId);
        AUD_POINTER_CHECK_CALL_NO_RET_VALUE(pMsg);
        ETG_TRACE_USR4(("ID_ProfileDataChanged: MessageNotification"));

         if(MSG_ProfileDataChanged::DATA_CHANGE_PROFILE_CHANGED == pMsg->m_enDataChangedType)
            {
                 AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poAdasNotificationIF);
              if (m_poAdasNotificationIF->bIsAdasSoundActive()) //updating the volume if any UPA beep is active
              {  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poBeepController);
                 m_poBeepController->SetBeepLocation();
              }
            }
        break;
    }

    default:
        break;
    }

}

/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::setAdasBeep(tU8 tBeepType)
 *
 * DESCRIPTION: Function called on new Beep PO message.
 *
 * PARAMETER: tU8 tBeepType
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Adas::setAdasBeep(tU8 tBeepType,AdasNotification_IF* poAdasNotification)
{
  tU8 tU8State = BEEP_STOPPED;
  tU8 tU8BeepId = 0xFF;
  u8BeepName = tBeepType;
  ETG_TRACE_USR4(("Beep_Adas::setADASBeep() BeepName: %d", ETG_ENUM(Adas_Beep, u8BeepName)));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poBeepMainAppl);

    if(m_poBeepMainAppl->BeepData.bIsBeepValid(u8BeepName))
    {
    map<tU8,AdasNotification_IF*>::iterator it;

    it = m_AdasNotificationMap.find(u8BeepName);

    if(it == m_AdasNotificationMap.end())
    {
      ETG_TRACE_USR4(("Beep not found in map, hence inserting beep into map"));
      m_AdasNotificationMap.insert(it,pair<tU8,AdasNotification_IF*>(u8BeepName,poAdasNotification));
    }

      if(NULL != m_poBeepPrioHandler)
      {
        bUpdateInternalData();
        tU8State = m_poBeepPrioHandler->u8SetRequest(u8BeepName, this,this, &tU8BeepId);
      }
      switch(tU8State)
      {
        case BEEP_REJECTED:
        NotifyBeepState( u8BeepName, tU8BeepId, BEEP_REJECTED,0);
        m_AdasNotificationMap.erase(u8BeepName);
          break;
        case BEEP_UNIQUE_REJECTED:
          break;
        default:
        {
        while (m_poBeepMainAppl->BeepData.m_beep_store[u8BeepName]->m_u8Next != 0xFF)
          {
          tBeepType = m_poBeepMainAppl->BeepData.m_beep_store[u8BeepName]->m_u8Next;
          ETG_TRACE_USR4((" Updating Data for Next Beep in sequence : %d", ETG_ENUM(Adas_Beep, tBeepType)));
            if(NULL != m_poBeepPrioHandler)
            {
            it = m_AdasNotificationMap.find(tBeepType);

            if (it == m_AdasNotificationMap.end())
            {
              ETG_TRACE_USR4(("Beep not found in map, hence inserting beep into map"));
              m_AdasNotificationMap.insert(it, pair<tU8, AdasNotification_IF*>(tBeepType, poAdasNotification));
            }
               u8UpdateBeepMainData(tBeepType);
               m_poBeepPrioHandler->u8SetRequest(tBeepType, this, this,&tU8BeepId);
            }
          }
        }
        break;
        }
    }
  else
  {
    ETG_TRACE_ERR(("!!! E R R O R !!! Beep_Adas_setBeep Invalid BeepName:%d", u8BeepName));
  }
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::GetBeepSource(tU8 tUBeepName)
 *******************************************************************************/

tU8 Beep_Adas::GetBeepSource(tU8 tUBeepName)
{
  ETG_TRACE_USR4(("Beep_Adas::GetBeepSource() : %d",  tUBeepName));
  AUD_POINTER_CHECK_CALL(m_poBeepMainAppl);
  return m_poBeepMainAppl->BeepData.m_beep_store[tUBeepName]->m_u8Source;
}
/*******************************************************************************
 * FUNCTION: Beep_Adas::u8UpdateBeepMainData(tU8 tBeepName)
 *
 * DESCRIPTION: Function to update Beep info for the beep in queue.
 *
 * PARAMETER: tU8 tBeepName
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Adas::u8UpdateBeepMainData(tU8 tBeepName)
{
  ETG_TRACE_USR4(("Beep_Adas::u8UpdateBeepMainData() :  %d",  tBeepName));
  u8BeepName = tBeepName;
  bUpdateInternalData();
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::GetSignalTonePara()
 *
 * DESCRIPTION: Interface function to get Signal Tone parameters.
 *
 * PARAMETER: tU8 tUBeepName, BeepControlData::BeepPara *BeepParameter
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Adas::GetSignalTonePara(tU8 tUBeepName, BeepControlData::BeepPara *BeepParameter)
{
  ETG_TRACE_USR4(("Beep_Adas::GetSignalTonePara() %d",  tUBeepName));
  memcpy(BeepParameter,&m_req_beep_BeepPara,sizeof(BeepControlData::BeepPara));
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::GetSignalChimePara()
 *
 * DESCRIPTION: Interface function to get Signal Chime parameters.
 *
 * PARAMETER: tU8 tUBeepName, BeepControlData::ChimePara *ChimeParameter
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Adas::GetSignalChimePara(tU8 tUBeepName, BeepControlData::ChimePara *ChimeParameter)
{
  ETG_TRACE_USR4(("Beep_Adas::GetSignalChimePara() %d",  tUBeepName));
  memcpy(ChimeParameter,&m_req_beep_ChimePara,sizeof(BeepControlData::ChimePara));
}

/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::s16GetParkAssistVolumeOffset()
 *
 * DESCRIPTION: Interface function to get Signal Chime parametersPark assist volume offset.
 *
 * PARAMETER: parkAssistVolLevel
 *
 * RETURNVALUE: tS16.
 *
 *******************************************************************************/
tS16 Beep_Adas::s16GetAdasVolumeOffset(tU8 tAdasVolLevel)
{
  ETG_TRACE_USR4(("Beep_Adas::s16GetAdasVolumeOffset(),tAdasVolLevel :%d", tAdasVolLevel));
  tS16 ret = 0;
  switch (tAdasVolLevel)
  {
    case VOLUME_LEVEL_0:
      ret = VOLOFFSET_LVL0 * VOLOFFSET_RESOLUTION;
      break;
    case VOLUME_LEVEL_1:
      ret = VOLOFFSET_LVL1 * VOLOFFSET_RESOLUTION;
      break;
    case VOLUME_LEVEL_2:
      ret = VOLOFFSET_LVL2 * VOLOFFSET_RESOLUTION;
      break;

    case VOLUME_LEVEL_3:
      ret = VOLOFFSET_LVL3 * VOLOFFSET_RESOLUTION;
      break;

    case VOLUME_LEVEL_4:
      ret = VOLOFFSET_LVL4 * VOLOFFSET_RESOLUTION;
      break;

    case VOLUME_LEVEL_5:
      ret = VOLOFFSET_LVL5 * VOLOFFSET_RESOLUTION;
      break;

    default:
      break;
  }
  return ret;
}



/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::GetBeepVolumeOffsets(tU8 tUBeepName, BeepControlData::ChVolOff *VolOffsets)
 *
 * DESCRIPTION: Interface function to get volume offsets for Beep
 *
 * PARAMETER: tU8 tUBeepName
 *
 * RETURNVALUE: tVoid
 *
 *******************************************************************************/
tVoid Beep_Adas::GetBeepVolumeOffsets(tU8 tUBeepName, BeepControlData::ChVolOff *VolOffsets)
{
  ETG_TRACE_USR4(("Beep_Adas::GetBeepVolumeOffsets()  BeepName:%d", tUBeepName));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poAdasNotificationIF);
  m_poAdasNotificationIF->GetBeepVolumeOffsets(tUBeepName,VolOffsets);
  ETG_TRACE_USR4(("GetBeepVolumeOffsets() : VolOffsets->LF : %d, VolOffsets->RF : %d, VolOffsets->LR :  %d, VolOffsets->RR : %d",VolOffsets->LF, VolOffsets->RF,VolOffsets->LR,VolOffsets->RR));
}


/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::GetSoundSystemConfig()
 *
 * DESCRIPTION: Interface function to sound system configuration
 *
 * PARAMETER: None
 *
 * RETURNVALUE: tU8
 *
 *******************************************************************************/
tU8 Beep_Adas::GetSoundSystemConfig()
{
  return m_u8SoundSystemconfig;
}



/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::NotifyBeepState(tU8 tBeepID, tU8 tBeepState)
 *
 * DESCRIPTION: Beep state Notification
 *
 * PARAMETER: tU8 tBeepID, tU8 tBeepState
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid Beep_Adas::NotifyBeepState(tU8 tUBeepName,tU8 tBeepID, tU8 tBeepState, tU32 u32ResponseToken)
{
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poAdasNotificationIF);
  ETG_TRACE_USR4(("Entered NotifyBeepState BeepName : %d BeepID:%d, State:%d, token:%d",tUBeepName,tBeepID, tBeepState,u32ResponseToken));
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_CSM_RNAIVI
  if( (((tUBeepName >= EN_AUDIO_BEEP_HFP_LVL1_PHASE1) && (tUBeepName <= EN_AUDIO_BEEP_PSM_PHASE2))||((tUBeepName >= EN_AUDIO_BEEP_UPA_ERROR_SND) && (tUBeepName <= EN_AUDIO_BEEP_LDW_SETTING_SND)) ) || (tUBeepName==EN_AUDIO_BEEP_RCTA_SND))
  {
    if(tBeepState == BEEP_START_IP)
    {
      ID_ADASBeepActive oAdasBeepActive(TRUE,EN_ADAS_BEEPTYPE_UPA,(m_poAdasNotificationIF->GetBeepVolume_withoutCANAttenuation(tUBeepName))/4); // Posting PO message to enable attenuation of main stream
      InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oAdasBeepActive);
    }
    else if(tBeepState == BEEP_STOPPED)
    {
      vClearAdasSoundStatus(tUBeepName);
      ID_ADASBeepActive oAdasBeepActive(FALSE,EN_ADAS_BEEPTYPE_UPA,0);
      InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oAdasBeepActive);// Posting PO message to enable attenuation of main stream
    }// added incase of beep reject  when ADAS beep in playing , we should not sent ADAS inactive to volume, Hence beep reject handled seperately here.
   else if(tBeepState ==BEEP_REJECTED)
    {
     vClearAdasSoundStatus(tUBeepName);
     ETG_TRACE_USR4(("Entered NotifyBeepState BeepActive not sent"));
    }
    else
    {}
  }
  if ((tUBeepName == EN_AUDIO_BEEP_UPA_SETTING_SND) || (tUBeepName == EN_AUDIO_BEEP_LDW_SETTING_SND))
  {
    if (tBeepState == 0x01)
    {
      tBeepState = midw_fi_tcl_e8_AudioBeepStatus::FI_EN_AUDIO_BEEP_STATUS_ON;
    }
    else if(tBeepState==BEEP_STOPPED||tBeepState==BEEP_REJECTED) //reject reponse to client for setting sound(response as stopped)
    {
      tBeepState = midw_fi_tcl_e8_AudioBeepStatus::FI_EN_AUDIO_BEEP_STATUS_OFF;
    }

    u8BeepState = tBeepState;
    m_rsp_beep_token = u32ResponseToken;

    AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poAudio_Function);

    if (m_poAudio_Function->updateClients(MIDW_MASCFFI_C_U16_BEEPSTATUS) != AIL_EN_N_NO_ERROR)
    {
      ETG_TRACE_ERR(("vSendBeepSatus : eUpdateClients():-> Updating the clients failed"));
    }
    else
    {
      ETG_TRACE_USR4(("vSendBeepSatus: eUpdateClients():-> Updating the clients passed"));
    }
  }
#endif
}


/*******************************************************************************
 *
 * FUNCTION:  Beep_Adas::bSendCCAStatusMessage(amt_tclServiceData& roOutMsg, amt_tclServiceData* poInMsg)

 *******************************************************************************/
tBool Beep_Adas::bSendCCAStatusMessage(amt_tclServiceData& roOutMsg, amt_tclServiceData* poInMsg ) const
 {
   (tVoid)poInMsg;
   (tVoid)roOutMsg;

   midw_mascffi_tclMsgBeepStatusStatus oBeepStatus;
   tBool bRetVal = false;

  oBeepStatus.BeepStatus.enType = (midw_fi_tcl_e8_AudioBeepStatus::tenType)u8BeepState;
  oBeepStatus.Resource.enType = (midw_fi_tcl_e8_ResourceNo::tenType)0x01;
  oBeepStatus.u32Token = m_rsp_beep_token;
  fi_tclVisitorMessage oCCaMsg(oBeepStatus);

   oBeepStatus.vDestroy();
   bRetVal = oCCaMsg.bHandOver(&roOutMsg);
   return bRetVal;
 }

/*******************************************************************************
 *
 * FUNCTION:  Beep_Adas::bIsBeepIdValid(tU8 tUBeepName)

 *******************************************************************************/
tBool Beep_Adas::bIsBeepIdValid(tU8 tUBeepName)
{
  AUD_POINTER_CHECK_CALL(m_poBeepMainAppl);
  return m_poBeepMainAppl->BeepData.bIsBeepValid(tUBeepName);
}
/*******************************************************************************
 *
 * FUNCTION:  Beep_Adas::GetImxBeepData(tU8 tUBeepName)

 *******************************************************************************/
std::string Beep_Adas::GetBeepFilePath(tU8 tUBeepName)
{
  AUD_POINTER_CHECK_CALL(m_poAdasNotificationIF);
  m_req_beep_path = std::string(BEEP_FILE_BASE_DIR) + m_poAdasNotificationIF->GetBeepFilePath(tUBeepName);
  return m_req_beep_path;
}

/*******************************************************************************
 *
 * FUNCTION:  Beep_Adas::GetBeepVolume(tU8 tUBeepName)

 *******************************************************************************/

tS16 Beep_Adas::GetBeepVolume(tU8 tUBeepName)
{
  AUD_POINTER_CHECK_CALL(m_poAdasNotificationIF);
  m_req_beep_volume = m_poAdasNotificationIF->GetBeepVolume(tUBeepName);
  //AdasNotification_IF* AdasNotification = GetAdasNotificationPtr(tUBeepName);
  //AUD_POINTER_CHECK_CALL(AdasNotification);
  //m_req_beep_volume = m_poAdasNotificationIF->GetBeepVolume(tUBeepName);
  ETG_TRACE_USR4(("Beep_Adas::GetBeepVolume, beep volume is :%d ",m_req_beep_volume));
  return m_req_beep_volume;
};

/*******************************************************************************
 *
 * FUNCTION:  Beep_Adas::GetBeepVolume(tU8 tUBeepName)

 *******************************************************************************/
tU32 Beep_Adas::GetRequestToken()
{
  ETG_TRACE_USR4(("Beep_Adas::GetRequestToken, beep token is :%d ",m_req_beep_token));
  return m_req_beep_token;
}

/*******************************************************************************
 *
 * FUNCTION:  Beep_Adas::vResetInternalData()
 * Description : Resets the internal data variables : m_req_beep_ChimePara, m_req_beep_BeepPara, m_req_beep_volume to initial state
 *******************************************************************************/
tVoid Beep_Adas::vResetInternalData()
{
  memset(&m_req_beep_ChimePara,0,sizeof(BeepControlData::ChimePara));
  memset(&m_req_beep_BeepPara,0,sizeof(BeepControlData::BeepPara));
  m_req_beep_volume = (-120 * ADR_DB_RESOLUTION);//Mute
  m_req_beep_token = 0;
}

/*******************************************************************************
 *
 * FUNCTION:  Beep_Adas::bUpdateInternalData()
 * Description : Updates the internal data variables : m_req_beep_ChimePara, m_req_beep_BeepPara, m_req_beep_path, m_req_beep_volume to initial state
 *******************************************************************************/
tBool Beep_Adas::bUpdateInternalData(const BeepFile_Config* ptrConfig, tU32 u32Token)
{
  ETG_TRACE_USR4(("Beep_Adas_bUpdateInternalData %d",  u8BeepName));
  AUD_POINTER_CHECK_CALL(m_poBeepMainAppl);
  if(!m_poBeepMainAppl->BeepData.bIsBeepValid(u8BeepName))
  {
    ETG_TRACE_USR4(("Invalid Beep Id, Cannot process this request "));
    return false;
  }

  tBool bretval = false;

  vResetInternalData();//Clear the data first
  AUD_POINTER_CHECK_CALL(m_poBeepMainAppl);
  beep_datatype* ptrBeepData = m_poBeepMainAppl->BeepData.m_beep_store[u8BeepName];
  if(ptrBeepData == NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
    (void)bretval;
    return false;

  }

  /*************************
   * UPDATE VOLUME FOR BEEP
   *************************/
  //Common parameter : Get the volume of the beep
  m_req_beep_volume  = ptrBeepData->m_s16Gain;
  m_req_beep_token   = u32Token;

  //Check and update, if user specific volume has to be applied for this beep.
  if(ptrConfig != NULL)
  {
    //Beep file path is already updated in Beep_Adas::setBeepFile. So just update the volume
    if(BEEP_VOL_TYPE_USER_SPECIFIC == ptrConfig->VolumeType)//User specific gain has to be applied
    {
      m_req_beep_volume = tS16 ( ptrConfig->Volume * ADR_DB_RESOLUTION);
      ETG_TRACE_USR4(("Applying User Specific Volume: %d *1/4 dB",m_req_beep_volume));
    }
  }

  /*************************************
   * UPDATE DATA REQUIRED TO PLAY BEEP
   *************************************/
  //Check if the requested beep is a ADR beep. For ADR beep we donot support any user configuration.
  //All the required data has to be fetched from Data store in case of ADR beep.
  if(m_poBeepMainAppl->BeepData.bIsAdrBeep(u8BeepName))
  {
    ETG_TRACE_USR4(("bUpdateInternalData,bIsAdrBeep"));
    beep_datatype_adr* tmp = dynamic_cast<beep_datatype_adr*>(ptrBeepData);
    if(tmp != NULL)
    {
      //1. Fetch Chime parameters
      m_req_beep_ChimePara.Pulses       = tmp->m_u16Repeat;
      m_req_beep_ChimePara.AttackTime     =tU16(tmp->m_u16RiseTime + tmp->m_u16OnTime) ;
      m_req_beep_ChimePara.AttackTimeLog     = 0x7FFF;
      m_req_beep_ChimePara.AttackTimeLin     = tmp->m_u16RiseTime;
      m_req_beep_ChimePara.ReleaseTime     =  tU16( tmp->m_u16PeriodTime - (tmp->m_u16RiseTime + tmp->m_u16OnTime));
      m_req_beep_ChimePara.ReleaseTimeLog   = 0x7FFF;
      m_req_beep_ChimePara.ReleaseTimeLin   = tmp->m_u16FallTime;
      m_req_beep_ChimePara.SignalToneAction   = tmp->m_u8SignalToneAction;
      //2. Fetch Beep Parameters
      m_req_beep_BeepPara.BeepId         = tmp->m_u8Generator;
      m_req_beep_BeepPara.Freq         = tmp->m_u16Freq;
      m_req_beep_BeepPara.Gain         = -6;  //-6; //currently configured to -6db

      bretval = true;
    }
    else
    {
      NORMAL_M_ASSERT_ALWAYS();
    }
  }
  //This beep is not an ADR beep. So this should be and IMX beep.
  //For IMX beep, we support user configuration of path.
  //For user configured imx beeps, m_req_beep_path variable is updated in Beep_Adas::setBeepFile() function.
  //So if the user has not provided any configuration, then fetch the file path information from data store.
   else if((ptrConfig == NULL)&&(m_poBeepMainAppl->BeepData.bIsImxBeep(u8BeepName)))
  {
    ETG_TRACE_USR4(("Beep_Adas_bUpdateInternalData, Beep data is preconfigured. Imx_Beep. %d",  u8BeepName));
    beep_datatype_imx* tmp = dynamic_cast<beep_datatype_imx*>(ptrBeepData);
    if(tmp != NULL)
    {
      //1. Fetch Path
      bretval = true;
    }
  }
  return false;
}


/*******************************************************************************
 *
 * FUNCTION: Beep_Adas::StopAdasBeep(tU8 tBeepType)
 *
 * DESCRIPTION: Function called to stop the active ADAS Beep.
 *
 * PARAMETER: tU8 tBeepType
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid Beep_Adas::StopAdasBeep(tU8 tBeepName)
{
    ETG_TRACE_USR4(("StopAdasBeep() called for tBeepName  : %d, u8BeepId : %d",tBeepName,u8BeepId));
    if ((NULL != m_poBeepPrioHandler) && ((tBeepName == u8RunningBeepName)||(tBeepName == 0)))
      {
        m_poBeepPrioHandler->vIdleBeep(BEEP_STOP_IMMEDIATE, u8BeepId);
      }
}

/*******************************************************************************
 *
 * FUNCTION:  Beep_Adas::vClearAdasSoundStatus()

 *******************************************************************************/
tVoid Beep_Adas::vClearAdasSoundStatus(tU8 tBeepName)
{
  ETG_TRACE_USR4(("Beep_Adas::vClearAdasSoundStatus() called : %d", tBeepName));
  AdasNotification_IF* AdasNotification = GetAdasNotificationPtr(tBeepName);
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(AdasNotification);
    AdasNotification->vClearAdasSoundStatus(tBeepName);
}

/*******************************************************************************

 *
 * FUNCTION:  Beep_Adas::bIsAdasSoundActive()

 *******************************************************************************/
bool Beep_Adas::bIsAdasSoundActive(tU8 tBeepName)
{
  // AUD_POINTER_CHECK_CALL(m_poAdasNotificationIF);
   AdasNotification_IF* AdasNotification = GetAdasNotificationPtr(tBeepName);
   AUD_POINTER_CHECK_CALL(AdasNotification);
   return AdasNotification->bIsAdasSoundActive(tBeepName);
}

/*******************************************************************************

 *
 * FUNCTION:  Beep_Adas::SetAdasNotificationPointer()

 *******************************************************************************/
tVoid Beep_Adas::SetAdasNotificationPointer(tU8 tBeepName)
{
  ETG_TRACE_USR4(("Beep_Adas::SetAdasNotificationPointer for Beep :%d", tBeepName));

  map<tU8,AdasNotification_IF*>::iterator it;

  it = m_AdasNotificationMap.find(tBeepName);
  if(it != m_AdasNotificationMap.end())
    m_poAdasNotificationIF = it->second;
  else
    ETG_TRACE_FATAL((" ERROR in retrieving AdasNotification pointer"));
}
/*******************************************************************************

 *
 * FUNCTION:  Beep_Adas::GetAdasNotificationPtr()

 *******************************************************************************/
AdasNotification_IF* Beep_Adas::GetAdasNotificationPtr(tU8 tBeepName)
{
  ETG_TRACE_USR4(("Beep_Adas::GetAdasNotificationPtr : %d", tBeepName));

  map<tU8,AdasNotification_IF*>::iterator it;

  it = m_AdasNotificationMap.find(tBeepName);
  if(it != m_AdasNotificationMap.end())
    return it->second;
  else
  {
    ETG_TRACE_FATAL((" ERROR in retrieving AdasNotification pointer"));
    return NULL;
  }
}
/*******************************************************************************

 *
 * FUNCTION:  Beep_Adas::s16GetDBValue()

 *******************************************************************************/
tS16 Beep_Adas::s16GetDBValue(tU8 tU8VolumeStep,tenStream enStream)
{
  AUD_POINTER_CHECK_CALL(m_poVolumeMgr);
  return (m_poVolumeMgr->s16getDBValue(tU8VolumeStep,enStream)); //enStream = EN_AUDIO_SOURCE_STREAM_DEFAULT
}
/*******************************************************************************

 *
 * FUNCTION:  Beep_Adas::SetAdasSoundStatus()

 *******************************************************************************/
tVoid Beep_Adas::SetBeepIdnStatus(tU8 tBeepName, tU8 tBeepId)
{
  //AdasNotification_IF* AdasNotification = GetAdasNotificationPtr(tBeepName);
  //AUD_POINTER_CHECK_CALL_NO_RET_VALUE(AdasNotification);
  u8BeepId = tBeepId;
  u8RunningBeepName = tBeepName;
  ETG_TRACE_USR4((" Current Running Beep :%d", u8RunningBeepName));
  AUD_POINTER_CHECK_CALL_NO_RET_VALUE(m_poAdasNotificationIF);
  m_poAdasNotificationIF->SetAdasSoundStatus(tBeepName);
}
/*******************************************************************************

 *
 * FUNCTION:  Beep_Adas::GetRecurrencePeriod()

 *******************************************************************************/
tU16 Beep_Adas::GetRecurrencePeriod(tU8 tUBeepName)
{
  //ETG_TRACE_USR4(("Beep_Adas::GetRecurrencePeriod() called"));
  //AdasNotification_IF* AdasNotification = GetAdasNotificationPtr(tUBeepName);
  //AUD_POINTER_CHECK_CALL(AdasNotification);
  AUD_POINTER_CHECK_CALL(m_poAdasNotificationIF);
  return (m_poAdasNotificationIF->GetRecurrencePeriod(tUBeepName));
}

/*******************************************************************************
*
* FUNCTION: Beep_Adas::GetClusterSoundLvlForAttenuation()
*******************************************************************************/
tU8 Beep_Adas::GetClusterSoundLvlForAttenuation()
{
  tU8 u8ClusterStepValue = 0;

  //read item from the KDS key's bytes- stream
  if ((DP_S32_NO_ERR == DP_s32GetConfigItem("SoundVolumeConfig", "ClusterSoundLevel", &u8ClusterStepValue, 1)))
  {
    ETG_TRACE_USR4((" Cluster Sound configured in EOL: %d", u8ClusterStepValue));
  }
  else
  {
    ETG_TRACE_ERR(("Error in reading EOL Parameter for cluster sound level "));
  }
  return u8ClusterStepValue;
}
