/**
 * @file ArkamysEolInterfaceRNAIVI.cpp
 * @author pau4kor
 * @copyright (c) 2015 RBEI
 * @addtogroup fc_audiomanager
 * @{
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include <etrace_if.h>
#include "fc_audiomanager_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_ARKAMYS
#include "trcGenProj/Header/ArkamysEolInterfaceRNAIVI.cpp.trc.h"
#endif

#include "util/Macro.h"
#include "ArkamysEolInterfaceRNAIVI.h"
#include "ArkamysFeatureHandlerRNAIVI.h"
#include "Arkamys/ArkamysEOLConverter.h"
#include <netinet/in.h>

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_audio_if.h"
#include "dp_diagdebug_if.h" // For Arkamys EOL Datapool

//#define SYSTEM_S_IMPORT_INTERFACE_QUEUE
//#include <stl_pif.h>



#include <queue>
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"


#include "vd_adr3Msg_If.h"
#define FM_ENHANCER_PARAMETERS_LEN (24)
#define ARKAMYS_USED_EOL_DATA_SIZE_CLASSIC (3028)
#define ARKAMYS_USED_EOL_DATA_SIZE_AUDITORIUM (12288)

#define STATIC_ASSERT(_t, _msg) {typedef char (_msg)[(_t) ? 1 : -1]; _msg a; (void)a;} // Some lint maneuvers

// Number of elements
#define NELT(_array) (sizeof(_array) / sizeof(_array[0]))

static const tU8  cArrayArkamysEolDefaultData[]            = { Arkamys_EolDefaultData };
static const tU8  cArrayArkamysAuditoriumEolDefaultData[]  = { Arkamys_AuditoriumEOLDefaultData };//auditorium EOL default data

ArkamysEolInterfaceRNAIVI::ArkamysEolInterfaceRNAIVI()
{
   m_enArkamysMode   = EN_ARKAMYS_BYPASS;

   m_pEolData = m_u8EolData;
   memset(m_pEolData, 0, sizeof(m_u8EolData));
   STATIC_ASSERT(sizeof(m_u8EolData) == ARKAMYS_EOL_MAX_DP_SIZE, Check_EolSizes);
   STATIC_ASSERT(sizeof(m_u8EolData) >= sizeof(ArkamysConfigRNAIVI::tArkamysEolHeader), CheckHeaderAndEolSize);

   m_bDataLoadedFromDatapool = false;

   ASSERT_STATIC(sizeof(cArrayArkamysEolDefaultData) == ARKAMYS_EOL_DATA_SIZE_BYTE);
   /*
    * Auditorium features
    */
   m_enCurAmbiance = ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_FLAT;
   ASSERT_STATIC(sizeof(cArrayArkamysAuditoriumEolDefaultData) <= ARKAMYS_USED_EOL_DATA_SIZE_AUDITORIUM);
}

ArkamysEolInterfaceRNAIVI::~ArkamysEolInterfaceRNAIVI()
{
   m_pEolData = OSAL_NULL;
}

ArkamysEolInterfaceRNAIVI *ArkamysEolInterfaceRNAIVI::getInstance()
{
   static ArkamysEolInterfaceRNAIVI theInstance;
   return &theInstance;
}

tBool ArkamysEolInterfaceRNAIVI::bLoadEOLDataFromFile(tS32 &s32Size)
{
   ETG_TRACE_USR4(("bLoadEOLDataFromFile() called."));

   tBool bRetVal = false;
   const char* D_EOL_FILE = "/var/opt/bosch/dynamic/audiomanager/AIVI_EOL_DATA.bin";

   FILE* eolfile = NULL;

   eolfile = fopen(D_EOL_FILE, "rb");

   if(NULL == eolfile)
   {
     ETG_TRACE_ERR(("bLoadEOLDataFromFile::fail to open the arkamys eol file:  %s", D_EOL_FILE));
   }
   else
   {
     /* find out the length of the file
      */
     fseek(eolfile, 0L, SEEK_END);
     s32Size = (tS32)ftell(eolfile);
     fseek(eolfile, 0L, SEEK_SET); /* go back to the beginning */
     ETG_TRACE_USR2(("bLoadEOLDataFromFile:: size of arkamys EOL file %d", s32Size));
     if((s32Size > 0) && (s32Size <= ARKAMYS_EOL_MAX_DP_SIZE))
     {
       if(s32Size != (tS32)fread((void*)m_pEolData, 1, s32Size, eolfile))
       {
         ETG_TRACE_USR2(("bLoadEOLDataFromFile:: fail to read arkamys eol file"));
       }
       else
       {
         ETG_TRACE_USR2(("bLoadEOLDataFromFile:: reading of arkamys EOL file success"));
         bRetVal = true;
       }
     }

     fclose(eolfile);
   }

   return bRetVal;
}
tBool ArkamysEolInterfaceRNAIVI::loadEolData()
{
   ETG_TRACE_USR4(("loadEolData() called."));
   tS32 dataSize = 0;

   // Fill m_pEolData with data from file
   tBool bIsLoadedFromFile = false;
   bIsLoadedFromFile = bLoadEOLDataFromFile(dataSize);
   ETG_TRACE_USR4(("EOL data loaded from file(%s)", bIsLoadedFromFile?"yes":"no"));

   //if file not exists then load from datapool
   if(!bIsLoadedFromFile)
   {
     // Fill m_pEolData with data from datapool
     dp_tclArkamysDPArkamys_Eol oDpEol;
     dataSize = oDpEol.s32GetData(m_pEolData, ARKAMYS_EOL_MAX_DP_SIZE);
     ETG_TRACE_USR4(("EOL data loaded from datapool"));
   }

   ETG_TRACE_USR4(("EOL data size() %d", dataSize));
   ETG_TRACE_USR4(("EOL data: %04x", ETG_LIST_LEN(ARKAMYS_EOL_DATA_SIZE_BYTE), ETG_LIST_PTR_T8(m_pEolData)));

   // Check EOL header and header size read from datapool
   if ((!checkEOLFileSize(dataSize)) || (!checkEOLHeader()))
   {
      ETG_TRACE_SYS_MIN(("Check of EOL file failed. Falling back to default data: %x", ETG_LIST_LEN(sizeof(cArrayArkamysEolDefaultData)), ETG_LIST_PTR_T8(cArrayArkamysEolDefaultData)));
      STATIC_ASSERT(sizeof(cArrayArkamysEolDefaultData) == ARKAMYS_EOL_DATA_SIZE_BYTE, Check_EolSizes);
      OSAL_pvMemoryCopy(m_pEolData, &cArrayArkamysEolDefaultData, ARKAMYS_EOL_DATA_SIZE_BYTE);
    m_bDataLoadedFromDatapool = false;
   }
   else
   {
      ETG_TRACE_USR4(("EOL data from datapool ok. Loading %i bytes.", getEOLFileSize()));
      NORMAL_M_ASSERT(m_pEolData == reinterpret_cast<tU8*>(m_u8EolData));
      m_bDataLoadedFromDatapool = true;
   }

   ETG_TRACE_USR4(("EOL data loaded. Version: %i ", ntohl(getEOLADR3LibVersion())));

   return true;
}

tBool ArkamysEolInterfaceRNAIVI::checkEOLFileSize(tS32 s32FileSize)
{
  if(s32FileSize <= (tS32)ARKAMYS_EOL_HEADER_SIZE_BYTE)
  {
    ETG_TRACE_ERR(("ArkamysEolInterfaceRNAIVI::checkEOLFileSize(): Invalid file size (%i), no data available.", s32FileSize));
    return false;
  }

  if(
     ((ntohl(this->getEOLLibType()) != ARKAMYS_LIBTYPE_CLASSIC) || (s32FileSize < ARKAMYS_USED_EOL_DATA_SIZE_CLASSIC)) &&
     ((ntohl(this->getEOLLibType()) != ARKAMYS_LIBTYPE_AUDITORIUM) || (s32FileSize < ARKAMYS_USED_EOL_DATA_SIZE_AUDITORIUM))
    )
  {
    ETG_TRACE_ERR(("ArkamysEolInterfaceRNAIVI::checkEOLFileSize(): Invalid file size (%i) for library type (%i)", s32FileSize, ntohl(this->getEOLLibType())));
    return false;
  }
  return true;
}

tBool ArkamysEolInterfaceRNAIVI::checkEOLHeader()
{
  //todo : check the endianness from datapool
   if ( (ntohl(this->getEOLADR3LibVersion()) < ARKAMYS_ADR3_LIB_MIN_VERSION) ||
       (ntohl(this->getEOLADR3LibVersion()) > ARKAMYS_ADR3_LIB_MAX_VERSION) )
   {
      ETG_TRACE_ERR(("ArkamysEolInterfaceRNAIVI::checkEOLVersion(): ADR Version %i not supported.", ntohl(this->getEOLADR3LibVersion())));
      return false;
   }

   //0 - classic; 1 - Auditorium
   if( (ntohl(this->getEOLLibType()) != ARKAMYS_LIBTYPE_CLASSIC) &&
    (ntohl(this->getEOLLibType()) != ARKAMYS_LIBTYPE_AUDITORIUM) )
   {
     ETG_TRACE_ERR(("ArkamysEolInterfaceRNAIVI::checkEOLLibType(): Type %i not supported.", ntohl(this->getEOLLibType())));
     return false;
   }

   if( (ntohl(this->getEOLLibType()) == ARKAMYS_LIBTYPE_AUDITORIUM) )
   {
     if ( (ntohl(this->getEOLIMXLibVersion()) < ARKAMYS_IMX_LIB_MIN_VERSION) ||
         (ntohl(this->getEOLIMXLibVersion()) > ARKAMYS_IMX_LIB_MAX_VERSION) )
     {
       ETG_TRACE_ERR(("ArkamysEolInterfaceRNAIVI::checkEOLVersion(): IMX Version %i not supported.", ntohl(this->getEOLIMXLibVersion())));
       return false;
     }
   }

   return true;
}

void ArkamysEolInterfaceRNAIVI::setArkamysAudioMode(tenArkamysMode enArkamysMode)
{
   m_enArkamysMode = enArkamysMode;
   ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI::Arkamys audio mode set to %i.", ETG_CENUM(tenArkamysMode, m_enArkamysMode)));
   //update SDEQ-HP parameters from EOL only for auditorium mode
   if(m_enArkamysMode == EN_ARKAMYS_AUDITORIUM)
     vUpdateSDHP();
}

void ArkamysEolInterfaceRNAIVI::vUpdateSDHP()
{
  tU8 dataIdx = 4;

  tU32 u32SDHPLen;
  OSAL_pvMemoryCopy(&u32SDHPLen, &m_pEolData[ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SDEQHPOffsets + dataIdx], sizeof(tU32));

  ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI: vUpdateSDHP Received %i bytes of SDEQ HP data. ", u32SDHPLen));

  if(u32SDHPLen != (MAX_LEN_OF_SDEQ_PARAMETERS - 1))
  {
      ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI: Invalid length = %i",u32SDHPLen));
      return;
  }
    //store and send data to adr3
  ArkamysTuningHandlerRNAIVI::getInstance()->vStoreAndSendSDEQParams(u32SDHPLen, &m_pEolData[ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SDEQHPOffsets + (2 * dataIdx)]);

}

tU32 ArkamysEolInterfaceRNAIVI::getEOLADR3LibVersion()
{
   return reinterpret_cast<ArkamysConfigRNAIVI::tArkamysEolHeader*>(m_pEolData) -> ADR3LibVersion;
}

tU32 ArkamysEolInterfaceRNAIVI::getEOLIMXLibVersion()
{
   return reinterpret_cast<ArkamysConfigRNAIVI::tArkamysEolHeader*>(m_pEolData) -> IMXLibVersion;
}

tBool ArkamysEolInterfaceRNAIVI::isDataFromDatapool()
{
   return m_bDataLoadedFromDatapool;
}


tU32 ArkamysEolInterfaceRNAIVI::getEolFileGenerationDate()
{
   return reinterpret_cast<ArkamysConfigRNAIVI::tArkamysEolHeader*>(m_pEolData) -> EolFileGenerationDate;
}

tU32 ArkamysEolInterfaceRNAIVI::getEOLLibType()
{
   return reinterpret_cast<ArkamysConfigRNAIVI::tArkamysEolHeader*>(m_pEolData) -> LibType;
}

tU32 ArkamysEolInterfaceRNAIVI::getEOLSteeringConfigurationType()
{
   return reinterpret_cast<ArkamysConfigRNAIVI::tArkamysEolHeader*>(m_pEolData) -> ConfigurationType;
}


tU32 ArkamysEolInterfaceRNAIVI::getEOLFileSize()
{
   return reinterpret_cast<ArkamysConfigRNAIVI::tArkamysEolHeader*>(m_pEolData) -> DataSize;
}


/*
 * @description  Creates an instance of ArkamysStaticMessage and sends it to the ADR for writing.
 *               The function takes "EOL Style" data as an input and converts it to "ADR Style". Both representations
 *               are Address-Length-Value-Style, but in EOL, there are two addresses to distinguish left and right steering.
 *               Examples for the data styles are as follows.
 *               Address and Length fields are 32-bit values, the unit of the length fields is either 8-bit (reduced) or 32-bit
 *               Use Host Byte Order!
 *               EOL Style: LAddress, RAddress, Length, Data,  LAddress, RAddress, Length, Data, ...
 *               ADR Style: Address, Length, Values,  Address, Length, Values, ...
 * @param pData  Pointer to the first Left Address field of the EOL-Style Input Data
 * @param length Length of the complete data buffer in 8-bit (reduced) or 32-bit words
 */
tBool ArkamysEolInterfaceRNAIVI::WriteDataToADR(const tU32* pData, tU32 length)
{
   ETG_TRACE_USR3(("ArkamysEolInterfaceRNAIVI::WriteDataToADR(%p, %i): %x", pData, length, ETG_LIST_LEN(length * 4), ETG_LIST_PTR_T8(pData)));
   if (length < 4)
   {
      ETG_TRACE_ERR(("ArkamysEolInterfaceRNAIVI: Got data pointer with too small length parameter (%i words, but need at least 4 words)", length));
      return false;
   }

//   ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI::WriteDataToADR(): LAdr:%08X Length:%08X Data:%08X", pData[0], pData[2], ETG_LIST_LEN(pData[2]), ETG_LIST_PTR_T32(pData[3])));

   {
      queue<ArkamysStaticMessage> msgQueue;
      tU16 len = FillMessageQueue(&msgQueue, pData, length);
    ETG_TRACE_USR1(("ArkamysEolInterfaceRNAIVI: FillMessageQueue size : %i ", len));
      while (msgQueue.size() > 0)
      {
         ArkamysStaticMessage oMsg = msgQueue.front();
         msgQueue.pop();
         ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI::Sending Message (%i left in queue): %X", msgQueue.size(), ETG_LIST_LEN(sizeof(oMsg)), ETG_LIST_PTR_T8(reinterpret_cast<tU8*>(&oMsg))));
         if (ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pADRInterface != OSAL_NULL)
         {
            ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pADRInterface->SendToADR(oMsg);
         }
         else
         {
            ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pArkamysADRInterface == OSAL_NULL"));
         }
      }

      return true;
   }
   return false;
}

tU16 ArkamysEolInterfaceRNAIVI::FillMessageQueue(std::queue<ArkamysStaticMessage> *msgQueue, const tU32* pData, tU32 length)
{
   // From EOL spec:
   tU32 address    = pData[0];
   const tU32 dataLength  = pData[1];
   const tU32 headerSize  = 2 * sizeof(tU32); // Address + Length (both 4 byte)
   const tU32 messageSize = (tU32)(dataLength * sizeof(tU32) + headerSize);
   ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI::FillMessageQueue(): Address:%p Length:%i headerSize:%i msgSize:%i"
         ,address, dataLength, headerSize, messageSize));
   if (address == 0)
   {
      ETG_TRACE_ERR(("ArkamysEolInterfaceRNAIVI: FillMessageQueue (%p, %i): Target address == 0. Will not send.", pData, length));
      return 0;
   }
   if (dataLength + 2 > length)
   {
      ETG_TRACE_ERR(("ArkamysEolInterfaceRNAIVI: FillMessageQueue(): Data Length (%u) would cause buffer read overflow (length: %u). Will not send.", dataLength, length));
      return 0;
   }
   tU32 remainingSize = messageSize;
   tU32 dataIdx = 2; // index in pData where the actual data starts
   while (remainingSize > headerSize)
   {
      const tU32 cropSize = (remainingSize > sizeof(ArkamysStaticMessage::data)) ?
            (sizeof(ArkamysStaticMessage::data) - headerSize) :
            (remainingSize - headerSize);
      remainingSize = remainingSize - cropSize;
      ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI: FillMessageQueue(): Message Loop: remaining:%i(from total:%i) max:%i crop to:%i adr:%08X"
            , remainingSize, messageSize, sizeof(ArkamysStaticMessage::data), cropSize + headerSize, address));
      const tU32 dataSize = cropSize + headerSize;
      NORMAL_M_ASSERT(dataSize <= sizeof(ArkamysStaticMessage::data));
      tU32 cropLength = (tU32)(cropSize / sizeof(tU32));
      tU32 addressNtw = htonl(address);
      tU32 cropLenNtw = htonl(cropLength);
      address = address + cropSize; // Advance address to next chunk
      ArkamysStaticMessage oASMsg;
      oASMsg.command = ArkamysStaticMessage::ArkamysOpTypeSet;
      oASMsg.size    = dataSize;
      int dataOffset = 0;
      OSAL_pvMemoryCopy(&oASMsg.data[dataOffset], &addressNtw, sizeof(tU32));   // Address
      dataOffset = (tU32)(dataOffset + sizeof(tU32));
      OSAL_pvMemoryCopy(&oASMsg.data[dataOffset], &cropLenNtw, sizeof(tU32));   // Length
      dataOffset = (tU32)(dataOffset + sizeof(tU32));
      ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI: FillMessageQueue(): Message Loop: Copy %i bytes (Len:%i) from %p(end:%p max:%p) to %p(end:%p max:%p)."
            , cropLength * sizeof(tU32), cropLength
            , &pData[dataIdx], &pData[dataIdx + cropLength], &pData[length] // from, end, max
            , &oASMsg.data[dataOffset], &oASMsg.data[dataOffset + (cropLength * sizeof(tU32))], &oASMsg.data[ARKAMYS_ADR_MAX_MESSAGE_SIZE_BYTES] // to, end, max
            ));
      FATAL_M_ASSERT(&pData[dataIdx + cropLength] <= &pData[length]); // Assert read boundaries
      FATAL_M_ASSERT(&oASMsg.data[dataOffset + (cropLength * sizeof(tU32))] <= &oASMsg.data[ARKAMYS_ADR_MAX_MESSAGE_SIZE_BYTES]);  // Assert write boundaries
      //Data (swap every word)
      for (uint curWord=0; curWord < cropLength; curWord++)
      {
         tU32 tmp = htonl(pData[dataIdx + curWord]);
         OSAL_pvMemoryCopy(&oASMsg.data[dataOffset], &tmp, sizeof(tU32));
         dataOffset = (tU32)(dataOffset + sizeof(tU32));
      }
      dataIdx += cropLength;
      ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI: FillMessageQueue(): Message Loop: Push message: size:%i data:%x", oASMsg.size, ETG_LIST_LEN(oASMsg.size), ETG_LIST_PTR_T8(oASMsg.data)));
      msgQueue -> push(oASMsg);
   }
   return ((tU16)(msgQueue -> size()));
}

/*
 * Send an ArkamysParameter "Get" request to ADR.
 */
void ArkamysEolInterfaceRNAIVI::RequestDataFromADR(tU32 address, tU32 length)
{
   ETG_TRACE_USR4(("RequestDataFromADR(0x%X, %i) entered.", address, length));
   ArkamysStaticMessage oArkamysStaticMessage;

   oArkamysStaticMessage.command = ArkamysStaticMessage::ArkamysOpTypeGet;
   oArkamysStaticMessage.size = 2 * sizeof(tU32);
   tU32 u32NtwAddr = htonl(address);
   tU32 u32NtwLen  = htonl(length);
   OSAL_pvMemoryCopy(&oArkamysStaticMessage.data[0], &u32NtwAddr, sizeof(tU32)); // Address
   OSAL_pvMemoryCopy(&oArkamysStaticMessage.data[sizeof(tU32)], &u32NtwLen, sizeof(tU32)); // Length

   if (ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pADRInterface != OSAL_NULL)
   {
      ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pADRInterface->SendToADR(oArkamysStaticMessage);
   }
   else
   {
      ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pArkamysADRInterface == OSAL_NULL"));
   }
}

tBool ArkamysEolInterfaceRNAIVI::bSetFMEnhancer(const tU8 *u8Data, tU32 u32Length)
{
   ETG_TRACE_USR4(("vSetFMEnhancer(%i) entered.",  u32Length));

   bool bretVal = false;
   tU8 dataIdx = 8;
   tU32 u32LenCount; tU8 u8FilterCount;
   u32LenCount = 0 ; u8FilterCount = 0;
   tU8 u8FMEnhancerData[FM_ENHANCER_PARAMETERS_LEN];
   tU32 DataOffset = 0;
   while( u32LenCount < u32Length )
    {
        if( u8FilterCount < FM_ENHANCER_FILTERS )
        {
        // setup outgoing PO message
        u8FMEnhancerData[DataOffset] = u8FilterCount;//0, 1, 2 are filter IDs for FM enhancer
        DataOffset = DataOffset + 1;

        OSAL_pvMemoryCopy(&u8FMEnhancerData[DataOffset], &u8Data[dataIdx + u32LenCount], sizeof(tU16));//filter characteristic & order
        u32LenCount = u32LenCount + 2;
        DataOffset = DataOffset + 2;

        tU16 u16Frequency;//to convert the endianness
        OSAL_pvMemoryCopy(&u16Frequency, &u8Data[dataIdx + u32LenCount], sizeof(tU16));//frequency
        u32LenCount = u32LenCount + 2;

        u8FMEnhancerData[DataOffset] = (tU8) (((u16Frequency) & 0xFF00)>>8);
        DataOffset = DataOffset + 1;

        u8FMEnhancerData[DataOffset] = (tU8) ((u16Frequency) & 0x00FF);
        DataOffset = DataOffset + 1;

        tS16 s16Gain;//to convert the endianness
        OSAL_pvMemoryCopy(&s16Gain, &u8Data[dataIdx + u32LenCount], sizeof(tS16));//gain
        u32LenCount = u32LenCount + 2;

        u8FMEnhancerData[DataOffset] = (tU8) (((s16Gain) & 0xFF00)>>8);
        DataOffset = DataOffset + 1;

        u8FMEnhancerData[DataOffset] = (tU8) ((s16Gain) & 0x00FF);
        DataOffset = DataOffset + 1;

        u8FMEnhancerData[DataOffset] = u8Data[dataIdx + u32LenCount];//Qfactor
        u32LenCount = u32LenCount + 1;
        DataOffset = DataOffset + 1;

        u8FilterCount++;
        }

        ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI:vSetFMEnhancer1 u8FilterCount %i, u32LenCount = %i  ", u8FilterCount, u32LenCount));
        bretVal = true;
    }

   if(bretVal)
   {
     ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI: vSetFMEnhancer(): Send multiple message: size:%i data:%x", FM_ENHANCER_PARAMETERS_LEN, ETG_LIST_LEN(FM_ENHANCER_PARAMETERS_LEN), ETG_LIST_PTR_T8(u8FMEnhancerData)));
     vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_SET_FILTER, VD_ADR3_OPTYPE_SETGET
         , FM_ENHANCER_PARAMETERS_LEN, u8FMEnhancerData, (tU8)(SET_FILTER_3));//send data to ADR3
   }
   return bretVal;
}

tBool ArkamysEolInterfaceRNAIVI::setFMEnhancer(tBool bOn)
{
  tBool bSuccess = false;

   if (bOn)
   {
       const tU8 *pDataSet;
       pDataSet = &m_pEolData[ArkamysConfigRNAIVI::oArkamys_sEolOffsets.FMEnhancerOnOffsets];
       bSuccess = bSetFMEnhancer(pDataSet, FM_ENHANCER_MSG_LENGTH);

   }

   else
   {
       const tU8 *pDataSet;
       pDataSet = &cArrayArkamysEolDefaultData[ArkamysConfigRNAIVI::oArkamys_sEolOffsets.FMEnhancerOffOffsets];
       bSuccess = bSetFMEnhancer(pDataSet, FM_ENHANCER_MSG_LENGTH);

   }
   return bSuccess;
}



tBool ArkamysEolInterfaceRNAIVI::setVirtualSubwoofer(tU8 u8State, tUInt uiDataSetIdx)
{
  ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI:setVirtualSubwoofer received with state %d", u8State));
  tBool bSuccess = false;

   if (u8State == (tU8)ArkamysConfigRNAIVI::STATE_ON)
   {
     if (uiDataSetIdx < NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.BassExciterOnOffsets))
     {
       const tU8 *pDataSet;
       pDataSet = &m_pEolData[ArkamysConfigRNAIVI::oArkamys_sEolOffsets.BassExciterOnOffsets[uiDataSetIdx]];
        vSendMultipleDataToADR(pDataSet, 2);
       bSuccess = true;
     }
     else
     {
      ETG_TRACE_ERR(("Index out of bounds: Index %i is greater than Number of elements in array ArkamysConfigRNAIVI::oArkamys_sEolOffsets.BassExciterOnOffsets (%i)", uiDataSetIdx, NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.BassExciterOnOffsets)));
     }
   }

   else
   {
     if (uiDataSetIdx < NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.BassExciterOffOffsets))
     {
       const tU8 *pDataSet;
       pDataSet = &m_pEolData[ArkamysConfigRNAIVI::oArkamys_sEolOffsets.BassExciterOffOffsets[uiDataSetIdx]];

       //for premium 1 & 2 amplifier and linear,bypass mode load default data
       if(u8State == (tU8)ArkamysConfigRNAIVI::STATE_LINEAR)
         pDataSet = &cArrayArkamysEolDefaultData[ArkamysConfigRNAIVI::oArkamys_sEolOffsets.BassExciterOffOffsets[uiDataSetIdx]];

       vSendMultipleDataToADR(pDataSet, 2);
       bSuccess = true;
     }
     else
     {
      ETG_TRACE_ERR(("Index out of bounds: Index %i is greater than Number of elements in array ArkamysConfigRNAIVI::oArkamys_sEolOffsets.BassExciterOffOffsets (%i)", uiDataSetIdx, NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.BassExciterOffOffsets)));
     }
   }
   return bSuccess;

}


tBool ArkamysEolInterfaceRNAIVI::SendResetPreProc()
{
   tBool bSuccess = true;
   ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI::SendResetPreProc()"));

  vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_ARKAMYS_RESET, VD_ADR3_OPTYPE_SET
        , 0, OSAL_NULL, ARKAMYS_RESET);//send data to ADR3

   return bSuccess;
}

tBool ArkamysEolInterfaceRNAIVI::setArkamysMode(tU8 u8ArkamysMode)
{
   tBool bSuccess = true;
   ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI::setArkamysMode to %i", ETG_CENUM(tenArkamysMode, u8ArkamysMode)));

   if(u8ArkamysMode == EN_ARKAMYS_PREMIUM1 || u8ArkamysMode == EN_ARKAMYS_PREMIUM2)
     u8ArkamysMode = EN_ARKAMYS_CLASSIC;

  vd_adr3Msg_If::vSendMsg(VD_ADR3_INST_ID_LS_1, VD_ADR3_FKT_ID_ARKAMYS_MODE, VD_ADR3_OPTYPE_SET
        , 1, &u8ArkamysMode, ARKAMYS_MODE);//send data to ADR3

   return bSuccess;
}

/**
 * @param iDataSetIdx Index of dataset to set (zero based). The total count of datasets depends on the array sizes
 *       (tArkamys_sEolOffsets4ch.Driver_Offsets and .AllPass_Offsets) in ArkamysConfigRNAIVI.h
 */
tBool ArkamysEolInterfaceRNAIVI::setSSS4(tBool bOn, tUInt uiDataSetIdx)
{
   ETG_TRACE_USR4(("setSSS4(%i, %i) entered.", bOn, uiDataSetIdx));

   tBool bSuccess = false;

   if(bOn)
   {
     const tU8 *pDataSet;
     pDataSet = &m_pEolData[ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStagingOffsets[uiDataSetIdx]];
     if (uiDataSetIdx < NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStagingOffsets))
     {
       if(uiDataSetIdx)
       {
           vSendMultipleDataToADR(pDataSet, 4);
           bSuccess = true;
       }
       else
       {
         tU32 u32DataSize = ArkamysEOLConverter::GetInflatedDatasetSizeRNAIVI(pDataSet);
         tU32 *pData = OSAL_NEW tU32[u32DataSize / sizeof(tU32)]; // inflated is always a multiple of 32-bit
         u32DataSize = ArkamysEOLConverter::InflateDatasetRNAIVI(pDataSet, (tU8 *)pData, u32DataSize);
         bSuccess = WriteDataToADR(pData, u32DataSize);

         OSAL_DELETE[] pData;
       }
     }
     else
     {
      ETG_TRACE_ERR(("Index out of bounds: Index %i is greater than Number of elements in array ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStagingOffsets (%i)", uiDataSetIdx, NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStagingOffsets)));
     }
   }
   else
   {
     if (uiDataSetIdx < NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStagingOffsets))
     {
       const tU8 *pDataSet;
       pDataSet = &cArrayArkamysEolDefaultData[ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStagingOffsets[uiDataSetIdx]];
       if(uiDataSetIdx)
       {
           vSendMultipleDataToADR(pDataSet, 4);
           bSuccess = true;
       }
       else
       {
         tU32 u32DataSize = ArkamysEOLConverter::GetInflatedDatasetSizeRNAIVI(pDataSet);
         tU32 *pData = OSAL_NEW tU32[u32DataSize / sizeof(tU32)]; // inflated is always a multiple of 32-bit
         ArkamysEOLConverter::InflateDatasetRNAIVI(pDataSet, (tU8*)pData, u32DataSize);
         bSuccess = WriteDataToADR(pData, u32DataSize);
         OSAL_DELETE[] pData;
       }
     }
     else
     {
      ETG_TRACE_ERR(("Index out of bounds: Index %i is greater than Number of elements in array ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStagingOffsets (%i)", uiDataSetIdx, NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStagingOffsets)));
     }

   }
   return bSuccess;
}

void ArkamysEolInterfaceRNAIVI::ResetSSS4()
{
   // Fill area with zeroes (to be defined by Arkamys if needed)
   ETG_TRACE_USR4(("ResetSSS4() entered."));
}


tBool ArkamysEolInterfaceRNAIVI::requestLibVersion()
{
   RequestDataFromADR(ARKAMYS_LIB_VERSION_ARM_ADR, 1);
   return true;
}

void ArkamysEolInterfaceRNAIVI::DebugWriteEol()
{
   FILE *f;
   f = fopen("/tmp/EOL.bin", "w");
   if (f == NULL)
   {
      ETG_TRACE_SYS_MIN(("ArkamysEolInterfaceRNAIVI::DebugWriteEol(): could not open file /tmp/EOL.bin for writing."));
   }
   else
   {
      size_t written = fwrite(m_u8EolData, 1, sizeof(m_u8EolData), f);
      ETG_TRACE_SYS_MIN(("ArkamysEolInterfaceRNAIVI::DebugWriteEol(): %i/%i bytes written to /tmp/EOL.bin", written, sizeof(m_u8EolData)));
      fclose(f);
   }
}

/*
 * Auditorium features
 */

/**
 * @param iDataSetIdx Index of dataset to set (zero based). The total count of datasets depends on the array sizes
 *        in ArkamysConfigRNAIVI.h
 */
tBool ArkamysEolInterfaceRNAIVI::setAmbiance(ArkamysConfigRNAIVI::tenAuditoriumAmbiance enAkAmbiance, tUInt uiDataSetIdx)
{
   ETG_TRACE_USR4(("setAmbiance(%i, %i) entered.", ETG_CENUM(ArkamysConfigRNAIVI::tenAuditoriumAmbiance, enAkAmbiance), uiDataSetIdx));

   tBool bSuccess = false;
   const tU8 *pEOLData;
   pEOLData = cArrayArkamysAuditoriumEolDefaultData;//assign flat data by default

   if(enAkAmbiance != ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_FLAT)
   {
     m_enCurAmbiance = enAkAmbiance;
     pEOLData = m_pEolData;
   }

   const tU32 *pOffsets = (m_enCurAmbiance == ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_NATURAL) ? ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStageNaturalOffsets :
                          (m_enCurAmbiance == ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_CLUB)    ? ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStageClubOffsets :
                          (m_enCurAmbiance == ArkamysConfigRNAIVI::ARKAMYS_AMBIANCE_LOUNGE)  ? ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStageLoungeOffsets :
                          ArkamysConfigRNAIVI::oArkamys_sEolOffsets.SoundStageLiveOffsets; // Live

   const tU8 *pDataSet = &pEOLData[pOffsets[uiDataSetIdx]];

   if(uiDataSetIdx == 1)
   {
     vSendMultipleDataToADR(pDataSet, 4);
     bSuccess = true;
   }
   else if(uiDataSetIdx == 5)
   {
     vSendMultipleDataToADR(pDataSet, 2);
     bSuccess = true;
   }
   else
   {
     tU32 u32DataSize = ArkamysEOLConverter::GetInflatedDatasetSizeRNAIVI(pDataSet);
     tU32 *pData = OSAL_NEW tU32[u32DataSize / sizeof(tU32)]; // inflated is always a multiple of 32-bit
     u32DataSize = ArkamysEOLConverter::InflateDatasetRNAIVI(pDataSet, (tU8 *)pData, u32DataSize);
     bSuccess = WriteDataToADR(pData, u32DataSize);
     OSAL_DELETE[] pData;
   }

   return bSuccess;
}


tBool ArkamysEolInterfaceRNAIVI::setFMEnhancer2(tBool bOn, tUInt uiDataSetIdx)
{

  tBool bSuccess = false;

   if (bOn)
   {
    if (uiDataSetIdx < NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.FMEnhancer2Offsets))
     {
       const tU8 *pDataSet;
       pDataSet = &m_pEolData[ArkamysConfigRNAIVI::oArkamys_sEolOffsets.FMEnhancer2Offsets[uiDataSetIdx]];

       vSendMultipleDataToADR(pDataSet, 2);
       bSuccess = true;
     }
     else
     {
      ETG_TRACE_ERR(("Index out of bounds: Index %i is greater than Number of elements in array ArkamysConfigRNAIVI::oArkamys_sEolOffsets.FMEnhancer2Offsets (%i)", uiDataSetIdx, NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.FMEnhancer2Offsets)));
     }
   }

   else
   {
     if (uiDataSetIdx < NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.FMEnhancer2Offsets))
     {
       const tU8 *pDataSet;
       pDataSet = &cArrayArkamysAuditoriumEolDefaultData[ArkamysConfigRNAIVI::oArkamys_sEolOffsets.FMEnhancer2Offsets[uiDataSetIdx]];

       vSendMultipleDataToADR(pDataSet, 2);
       bSuccess = true;
     }
     else
     {
      ETG_TRACE_ERR(("Index out of bounds: Index %i is greater than Number of elements in array ArkamysConfigRNAIVI::oArkamys_sEolOffsets.FMEnhancer2Offsets (%i)", uiDataSetIdx, NELT(ArkamysConfigRNAIVI::oArkamys_sEolOffsets.FMEnhancer2Offsets)));
     }
   }
   return bSuccess;
}

void ArkamysEolInterfaceRNAIVI::vSendMultipleDataToADR(const tU8* pu8Data, tUInt iterations)
{
  tU32 EOLdataOffset = 0;
  int dataIdx = 0;
  tU32* pData = OSAL_NULL;
  ArkamysStaticMessage oASMsg;
  oASMsg.command = ArkamysStaticMessage::ArkamysOpTypeSet;
  oASMsg.size = 0;

  tUInt noOfIter = 0;
  while(noOfIter < iterations)
  {
    tU32 EOLDataSize = 0;
    OSAL_pvMemoryCopy(&EOLDataSize, &pu8Data[EOLdataOffset + sizeof(tU32)], sizeof(tU32));   // Length
    ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI::vSendMultipleDataToADR() msg size:%i", EOLDataSize));

    tU32 u32DataSize = ArkamysEOLConverter::GetInflatedDatasetSizeRNAIVI(&pu8Data[EOLdataOffset]);
    pData = OSAL_NEW tU32[u32DataSize / sizeof(tU32)]; // inflated is always a multiple of 32-bit
    if(pData == OSAL_NULL)
      return;
    u32DataSize = ArkamysEOLConverter::InflateDatasetRNAIVI(&pu8Data[EOLdataOffset], (tU8 *)pData, u32DataSize);

    tU32 dataOffset = 0;
    // From EOL spec:
    const tU32 address    = pData[dataOffset]; //address
    dataOffset++;
    const tU32 dataLength  = pData[dataOffset]; //length
    dataOffset++;

    const tU32 headerSize  = 2 * sizeof(tU32); // Address + Length (both 4 byte)
    const tU32 messageSize = (tU32)(dataLength * sizeof(tU32) + headerSize);
    ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI::vSendMultipleDataToADR(): Adr:%p Length:%i headerSize:%i msgSize:%i"
        ,address, dataLength, headerSize, messageSize));

    if (address == 0)
    {
      ETG_TRACE_ERR(("ArkamysEolInterfaceRNAIVI: vSendMultipleDataToADR (%p, %i): Target address == 0. Will not send.", pData, dataLength));
      OSAL_DELETE [] pData;
      return;
    }

    oASMsg.size    += messageSize;
    tU32 cropSize = messageSize - headerSize;
    tU32 cropLength = (tU32)(cropSize / sizeof(tU32));
    tU32 addressNtw = htonl(address);
    tU32 cropLenNtw = htonl(cropLength);
    OSAL_pvMemoryCopy(&oASMsg.data[dataIdx], &addressNtw, sizeof(tU32));   // Address
    dataIdx = (tU32)(dataIdx + sizeof(tU32));
    OSAL_pvMemoryCopy(&oASMsg.data[dataIdx], &cropLenNtw, sizeof(tU32));   // Length
    dataIdx = (tU32)(dataIdx + sizeof(tU32));

    ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI: vSendMultipleDataToADR(): Message Loop: Copy %i bytes (Len:%i) from %p(end:%p max:%p) to %p(end:%p max:%p)."
        , cropLength * sizeof(tU32), cropLength
        , &pData[dataOffset], &pData[dataOffset + cropLength], &pData[dataLength] // from, end, max
                                        , &oASMsg.data[dataIdx], &oASMsg.data[dataIdx + (cropLength * sizeof(tU32))], &oASMsg.data[ARKAMYS_ADR_MAX_MESSAGE_SIZE_BYTES] // to, end, max
    ));
    //Data (swap every word)
    for (uint curWord=0; curWord < cropLength; curWord++)
    {
      tU32 tmp = htonl(pData[dataOffset + curWord]);
      //ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI: vSendMultipleDataToADR(): tmp word (%x)", tmp));
      OSAL_pvMemoryCopy(&oASMsg.data[dataIdx], &tmp, sizeof(tU32));
      dataIdx = (tU32)(dataIdx + sizeof(tU32));
    }

    EOLdataOffset = EOLdataOffset + EOLDataSize + headerSize;//as per eol spec - length + 8 bytes header size will give the next offset
    ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI::vSendMultipleDataToADR() actual dataoffset:%i", EOLdataOffset));
    noOfIter++;

    OSAL_DELETE [] pData;
  }

  //send to adr
  if ((ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pADRInterface != OSAL_NULL) &&
      (oASMsg.size))
  {
    ETG_TRACE_USR4(("ArkamysEolInterfaceRNAIVI: vSendMultipleDataToADR(): Send multiple message: size:%i data:%x", oASMsg.size, ETG_LIST_LEN(oASMsg.size), ETG_LIST_PTR_T8(oASMsg.data)));
    ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pADRInterface->SendToADR(oASMsg);
  }
  else
  {
    ETG_TRACE_ERR(("ArkamysFeatureHandlerRNAIVI::getInstance() -> m_pArkamysADRInterface == OSAL_NULL"));
  }

}

