/*
 * vdamp-plugin-send.h
 *
 *  Created on: Nov 16, 2013
 *      \author: rjk2kor
 */

#ifndef VDAMP_PLUGIN_SEND_H_
#define VDAMP_PLUGIN_SEND_H_

#include "vdamp-plugin-types.h"
#include "vdamp-plugin-sender-func.h"

//Description for Main page of Doxygen

/*!
 * \mainpage External Amplifier Plugin Interfaces description
 * \section intro_sec Introduction
 * Gen3 Audio management supports External Amplifiers using a plugin architecture.
 * The plugins will be handled by the VD AMP module of Gen3 Audio Management.
 * Developer can use the below guidelines to create a VD Amp Compatible Plugin.
 *
 * \subsection step1 Plugin Interface Model
 * VD Amp module manages the external amplifier related plugins.
 * VD Amp expects the plugins to support a standard interface using which VD Amp would talk to the plugins. <br>
 * This interface is called the <B>plugin sender interface</B>. <br>
 * The plugin sender interface is split into two parts
 *  - Plugin Management Interface
 *  - Plugin Functional Sender Interface(s)
 *
 * VD Amp provides an interface using which the plugins can talk back to it. This interface is called the <B>Plugin Receiver Interface</B>.
 *
 * \subsection step2 Plugin Management Interface
 * The Plugin Management interface is defined by the abstract class vdamp_plugin_sender.
 * Plugin Developer has to override the pure virtual functions of this class.
 *
 * \subsection step3 Plugin Functional Sender Interfaces(s)
 * Plugin functional sender interfaces implement the amplifier specific functionality.
 * For example SetVolume at the amplifier is a seperate functional sender interface as is setMute.
 * The plugin developer would implement only those sender interfaces that are supported by the external amplifier he/she is trying to control.
 * The plugin functional sender interfaces can be registered with VD Amp using a vdamp_plugin_sender_interfaces object.
 * VD Amp requests the plugin to provide interface data by invoking oGetInterfaceData() method of the plugin management interface.
 * Plugin shall set the pointers to the instances of the implemented sender interfaces and return interface data object.
 * \note The unimplemented senders, if left unset, are by default initialized to NULL and the same is expected from the plugin.
 * The pointers provided in the interface data should remain valid till vdamp_plugin_sender::vDeInitPlugin() of the plugin is invoked.
 *
 * \subsection step4 Plugin Receiver Interface
 * Plugin Receiver Interface is defined by the abstract class vdamp_plugin_receiver.
 * Direction : Plugin to the VD Amp
 * VD Amp will provide a pointer to an instance of derived class.
 * Plugin can use the provided pointer to send the data to VD Amp.
 * \note All Calls made to this class have to be done in the context of VD Amp's CCA Thread. See section concurrency for more information.
 *
 * \subsection step5 Plugin Startup Concept
 * VD Amp at startup will read the KDS Configuration to verifiy if the system is configured to support an external amplifier.
 * If the system is configured to support external amplifier, it will try to load plugins/shared object files from <B>/opt/bosch/vdamp/plugins</B> folder.
 * VD Amp expects every plugin to provide a symbol <B>getpluginpointer()</B> which would return a pointer to instance of vdamp_plugin_sender.
 * Once all the plugins containing the symbol <B>getpluginpointer</B> are loaded, vdamp_plugin_sender::bInitPlugin( ) is invoked on each plugin.
 * \note Make sure that your plugins are placed in the folder and provide the symbol as mentioned above so that VD Amp would load your plugin.
 *
 * \subsection step6 Plugin Shutdown Concept
 * When a plugin has to be unloaded, VD Amp will invoke vdamp_plugin_sender::vDeInitPlugin() function of the plugin management interface.
 * It is expected that all the resources allocated by the plugin are freed here. VD Amp will then delete the instance of the plugin and will unload it.
 * \todo Introduce an additional interface in the plugin receiver interface to enable the plugin to acknowledge vDeInitPlugin().
 *
 * \subsection step7 Plugin Connect
 * Plugin can use their own algorithms/methods to findout if the connected amplifier is compatible with the loaded plugin.
 * VD Amp will not define the methods to do compatibility check.The result of the compatibility check, irrespective of whether success or failure has to
 * be informed to VD Amp using the vdamp_plugin_receiver::vPluginConnect() interface of Plugin receiver interface. VD Amp will unload all incompatible plugins.
 * <B> VD Amp will not communicate with the plugins till vPluginConnect() function is invoked </B>
 * \note At any point of time there should be a single compatible plugin for the connected amplifier.
 *  If more than one plugin reports compatible, the first plugin to inform compatibility will be retained and others will be unloaded.
 *
 * \subsection step8 Function Sender Interface(s) registration
 * VD Amp will request the compatible plugin to provide the Functional Sender Interface(s) for further handling.
 * For this, VD Amp will invoke the function vdamp_plugin_sender::oGetInterfaceData() function of the plugin�s management interface.
 * Plugin will send the pointers to the sender interfaces using the interface data object.
 * VD Amp will take this information and will update the corresponding handlers with the specific sender interfaces of the amplifier.
 *
 * \subsection step9 Concurrency Handling
 * <B> Calls to the plugin receiver interface should always be in the context of the CCA Thread of VD Amp, with the exception of context switcher function.</B>
 * For this purpose, the plugins can make use of the context switcher of VD Amp. The plugins can invoke the vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback() function of the receiver
 * interface with an arbitary stream of data from any of its worker threads. The VD Amp context switcher would send the data back to the plugin by invoking
 * vdamp_plugin_sender::vAmp_Rx_FromCCAThread(). This call will be made from the context of CCA thread of VD Amp. So from vAmp_Rx_FromCCAThread function it is safe to invoke the functions of
 * plugin receiver interface.
 *
 * \subsection step10 Value Ranges
 * For all the supported value range for different parameters, please refer <a href="https://hi-dms.de.bosch.com/docushare/dsweb/Services/Document-574139">SWCD_AudioAmplifierInterface.doc</a>
 * \todo Define enumeration for possible error codes
 */


/*******************************************************
 * MACRO To Track versioning of the plugin APIs
 *******************************************************/
/*
 * The below version would be changed, when ever an existing enumeration has been modified,
 * and you want to enforce the plugin to do a recompile.
 */
/**
 * \def VD_AMP_PLUGIN_SENDER_API_VERSION Sender API Version.
 */
#define VD_AMP_PLUGIN_SENDER_API_VERSION  (int (4))

//Feed forward declaration
class vdamp_plugin_receiver;

/**
 * \class vdamp_plugin_sender
 * \brief This class will implement the send interface for the plugin.
 * Direction : Plugin manager to the plugin. Plugin developer has to override the pure virtual functions of this interface class.
 *
 * \note VD Amp would make calls to the interfaces of this class from the context of its CCA Thread.
 */
class vdamp_plugin_sender
{
public:
  /*!
   * Default constructor
   */
  vdamp_plugin_sender(){}
  /*!
   * Default destructor
   */
  virtual ~vdamp_plugin_sender(){}
  /*!
   * \fn bool  bInitPlugin(vdamp_pluginID_t pluginID, vdamp_plugin_receiver* ptrReceiver)
   * \brief Once the plugin has been loaded by VD Amp, this function triggered to initialize the plugin.
   * Plugin ID is provided by the plugin manager
   * \param pluginID Plugin ID assigned to the Plugin
   * \param ptrReceiver Pointer to plugin receiver object.
   * \return TRUE    : Initialization success
   * \return FALSE  : Initialization failed
   */
  virtual bool bInitPlugin(vdamp_pluginID_t pluginID, vdamp_plugin_receiver* ptrReceiver) = 0;
  /*!
   * \fn void vDeInitPlugin()
   * \brief VD Amp would invoke this function before the plugin is about to be unloaded.
   * It is expected that all the resources allocated by the plugin are deallocated after this function returns.
   * \return None
   */
  virtual void vDeInitPlugin() = 0;
  /*!
   * \fn vdamp_plugin_sender_interfaces   oGetInterfaceData()
   * \brief Once the plugin has been initialized and the plugin has informed that it is compatible with the
   * connected amplifier, VD Amp will invoke this function to get the list of supported sender interfaces.
   * It is expected from the plugin that all the pointers to the supported interfaces is updated in the sender
   * interface data object and the unsupported sender interfaces are initialized to NULL.
   * \return vdamp_plugin_sender_interfaces : Object of sender interface data.
   */
  virtual vdamp_plugin_sender_interfaces oGetInterfaceData() = 0;
  /*!
   * \fn vdamp_plugin_amplifierdata oGetAmplifierInfo()
   * \brief Once the plugin has been initialized and the plugin has informed that it is compatible with the
   * connected amplifier, VD Amp will invoke this function to get the information about the connected amplifier.
   * The plugin is expected to provide the details of the connected amplifier here.
   *
   * \return vdamp_plugin_amplifierdata : Object of amplifier data.
   */
  virtual vdamp_plugin_amplifierdata oGetAmplifierInfo() = 0;
  /*!
   * \fn void   vAmp_Rx_FromCCAThread(uint8_t* ptrStream )
   * \brief <B>This is a context switcher related function.</B>
   * When the plugin uses vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback() function to send a stream of data
   * to be called in CCA server's context, the CCA server would trigger this function with
   * the stream of data sent using vAmp_Tx_ToCCALoopback, but in the CCA server thread's context
   * \param ptrStream pointer to stream of data.
   * \warning Donot free the pointer. It will be taken care by VD Amp.
   * \return none
   */
  virtual void vAmp_Rx_FromCCAThread (uint8_t* ptrStream , size_t length) = 0;
  /*!
   * \fn int iGetPluginSenderAPIVersion()
   * \brief This function is triggered by the plugin manager to get version of sender API that the plugin implements.
   *
   * \return PLUGIN_SENDER_API_VERSION
   */
  virtual int iGetPluginSenderAPIVersion()const
  {
    return VD_AMP_PLUGIN_SENDER_API_VERSION;
  }
};


#endif /* VDAMP_PLUGIN_SEND_H_ */
