/*
 * vdamp-plugin-receive.h
 *
 *  Created on: Nov 16, 2013
 *      Author: rjk2kor
 * *****************************************************************************************
 * Version History
 * Date      Modified by    Change Description
 * *****************************************************************************************
 * 11/26/2013  Ranjit K    1. Added vAmp_SetFixAdrVolume as suggested by Klotz Gerlinde (CM-AI/PJ-CB33)
 *                2. Changed vPluginCompatibility to vPluginConnect as suggested by Klotz Gerlinde (CM-AI/PJ-CB33)
 * *****************************************************************************************
 */

#ifndef VDAMP_PLUGIN_RECEIVE_H_
#define VDAMP_PLUGIN_RECEIVE_H_

#include "vdamp-plugin-types.h"
#include <sys/types.h>
#include <system_types.h>
/**
 * \class vdamp_plugin_receiver
 * \brief This class will implement the receive interface for the plugin.
 * Direction : Plugin to VD Amp.
 * VD Amp will provide a pointer to this class/derived class. Plugin can use the provided pointer to send the data to VD Amp.
 * \note Apart from vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback all the functions of this class have to be invoked from the conetxt of VD Amp CCA Thread.
 * \note See function usage for more info.
 */
class vdamp_plugin_receiver
{
public:
  /**
  * Default constructor
  */
  vdamp_plugin_receiver(){}
  /**
  * Default destructor
  */
  virtual ~vdamp_plugin_receiver(){}
  /*!
   * \fn void vPluginConnect(vdamp_pluginID_t  pluginID, bool bCompatible)
   * \brief  Once the plugin is initialized the plugin should validate if the connected amplifier
   *  This function has to be called by the plugin to inform the plugin manager whether
   *          or not if the connected amplifier is compatible with the loaded plugin.
   *          This function is similar to 5.2.1  vRegisterAmp of SWCD_AmplifierInterface.doc
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   * \param pluginID Plugin ID received from Plugin Manager
   * \param bCompatible Compatibility with the connected amplifier
   * \return None.
   */
  virtual void vPluginConnect(vdamp_pluginID_t  pluginID, bool bCompatible)= 0;
  /*!
   * \fn void   vAmp_IndPowerState(vdamp_pluginID_t  pluginID, amp_tenPowerStatus enStatus, amp_tenPowerState enState)
   * \brief  Plugin would call this function to indicate the power state of connected amplifier.
   *          When power state other than ON is indicated, the internal amplifier is muted.
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   * \param pluginID Plugin ID received from Plugin Manager
   * \param enStatus Current Status of the connected amplifier
   * \param enState Requested State of the connected amplifier
   * \return None.
   */
  virtual void vAmp_IndPowerState(vdamp_pluginID_t  pluginID, amp_tenPowerStatus enStatus, amp_tenPowerState enState) = 0;
  /*!
   * \fn void   vAmp_ErrPowerState(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate an error in retreiving the power state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
  virtual void vAmp_ErrPowerState(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)=0;
  /*!
   * \fn void   void vAmp_IndSource(vdamp_pluginID_t  pluginID, amp_tenStream enStream,amp_tenSource enSource)
   * \brief  Plugin would call this function to indicate the source of the connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   * \param pluginID Plugin ID received from Plugin Manager
   * \param enStream Stream
   * \param enSource Source
   * \return None.
   */
    virtual void vAmp_IndSource(vdamp_pluginID_t  pluginID, amp_tenStream enStream,amp_tenSource enSource)= 0;

  /*!
   * \fn void   void vAmp_IndSource_UnMapped(vdamp_pluginID_t  pluginID, amp_tenStream enStream,amp_tenSource_unmapped enSource)
   * \brief  Plugin would call this function to indicate the source of the connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   * \param pluginID Plugin ID received from Plugin Manager
   * \param enStream Stream
   * \param enSource Source
   * \return None.
   */

    virtual void vAmp_IndSource_UnMapped(vdamp_pluginID_t  pluginID, amp_tenStream enStream,amp_tenSource_unmapped enSource)= 0;

  /*!
   * \fn void   vAmp_ErrSource(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in updating source of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
    virtual void vAmp_ErrSource(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)=0;
  /*!
   * \fn void vAmp_IndVolume(vdamp_pluginID_t  pluginID, amp_tenStream enStream, uint8_t ubStep, int16_t swdB, amp_tenVolumeState enState)
   * \brief Plugin would call this function to indicate volume state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   * \param pluginID Plugin ID received from Plugin Manager
   * \param enStream Stream
   * \param ubStep  Steps
   * \param swdB    dB value
   * \param enState State
   * \return None.
   */
  virtual void vAmp_IndVolume(vdamp_pluginID_t  pluginID, amp_tenStream enStream, uint8_t ubStep, int16_t swdB, amp_tenVolumeState enState) = 0;
  /*!
   * \fn void   vAmp_IndVolume2(vdamp_pluginID_t  pluginID, uint8_t VolumeStep, amp_tenExtAmpVolumeType enVolGroup)
   * \brief  Plugin would call this function to set relative volume to connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
  virtual void vAmp_IndVolumePrem2(vdamp_pluginID_t  pluginID, uint8_t VolumeStep, amp_tenExtAmpVolumeType enVolGroup) = 0;

  /*!
   * \fn void   vAmp_IndVolumeListPrem2(vdamp_pluginID_t  pluginID, uint8_t* List)
   * \brief  Plugin would call this function to indicate volume list update from connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
  virtual void vAmp_IndVolumeListPrem2(vdamp_pluginID_t  pluginID, uint8_t* List) = 0;

  /*!
   * \fn void   vAmp_ErrVolume(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in volume state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
    virtual void vAmp_ErrVolume(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode) = 0;
  /*!
   * \fn void   vAmp_SetFixAdrVolume (vdamp_pluginID_t  pluginID,amp_tenStream enStream, uint8_t ubStep,int16_t swdB,uint16_t uwRampLin,uint16_t uwRampdB)
   * \brief  Plugin would call this function to set a fixed adr volume.
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param enStream Stream for which the volume has to set
   * \param ubStep Steps
   * \param swdB DB Value
   * \param uwRampLin Ramp Lin
   * \param uwRampdB Ramp DB
   *
   * \return None.
   */
  virtual void vAmp_SetFixAdrVolume (vdamp_pluginID_t  pluginID,amp_tenStream enStream, uint8_t ubStep,int16_t swdB,uint16_t uwRampLin,uint16_t uwRampdB)= 0;
  /*!
   * \fn void vAmp_IndVolumeOffset(vdamp_pluginID_t  pluginID, int16_t swOffset, amp_tenVolumeOffsetStatus enStatus)
   * \brief  Plugin would call this function to indicate Volume offset status.
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param swOffset volume offset
   * \param enStatus status
   *
   * \return None.
   */
    virtual void vAmp_IndVolumeOffset(vdamp_pluginID_t  pluginID, int16_t swOffset, amp_tenVolumeOffsetStatus enStatus)=0;
  /*!
   * \fn void   vamp_ErrVolumeOffset(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in VolumeOffset state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
    virtual void vamp_ErrVolumeOffset(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)    =0;
  /*!
   * \fn void vAmp_IndInputVolumeOffset(vdamp_pluginID_t  pluginID, amp_tenStream enStream, uint16_t uwOffset, amp_tenVolumeOffsetStatus enStatus)
   * \brief  Plugin would call this function to indicate Input Volume offset status.
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param enStream Stream
   * \param uwOffset offset
   * \param enStatus status
   *
   * \return None.
   */
    virtual void vAmp_IndInputVolumeOffset(vdamp_pluginID_t  pluginID, amp_tenStream enStream, uint16_t uwOffset, amp_tenVolumeOffsetStatus enStatus) =0;
  /*!
   * \fn void   vAmp_ErrInputVolumeOffset(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in InputVolumeOffset state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
    virtual void vAmp_ErrInputVolumeOffset(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)    =0;
  /*!
   * \fn void vAmp_IndSettings(vdamp_pluginID_t  pluginID, amp_tenStream enStream, amp_tenSettingType enSettingType, int16_t swSteps)
   * \brief  Plugin would call this function to indicate status of a particular setting.
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param enStream Stream
   * \param enSettingType Setting type
   * \param swSteps Current value
   *
   * \return None.
   */
    virtual void vAmp_IndSettings(vdamp_pluginID_t  pluginID, amp_tenStream enStream, amp_tenSettingType enSettingType, int16_t swSteps)   =0;
  /*!
   * \fn void   vAmp_ErrSettings(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in Settings state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
    virtual void vAmp_ErrSettings(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)    =0;
  /*!
   * \fn void   vAmp_AvailSettings(vdamp_pluginID_t  pluginID, uint16_t swValue)
   * \brief  Plugin would call this function to indicate available settings of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param swValue Available settings
   *
   * \return None.
   */
    virtual void vAmp_AvailSettings(vdamp_pluginID_t  pluginID, uint16_t swValue)    =0;
  /*!
   * \fn void vAmp_IndSetParamFilter(vdamp_pluginID_t  pluginID, uint8_t ubFilterID, amp_tenParamFilterChar enChar,uint8_t ubOrder, uint16_t uwFreq, int16_t swGain,uint8_t ubQ)
   * \brief  Plugin would call this function to indicate set param filter status
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubFilterID Filter ID. TBD.
   * \param enChar Filter Char
   * \param ubOrder  Range 0x01 - 0x04. Step size 1.
   * \param uwFreq  Range 0x0014 - 0x4E20. Step size 1. Unit: Hz.
   * \param swGain  Range 0xFF88 - 0x0078. Step size 1. Unit: quarter_DB
   * \param ubQ  Range 0x00 - 0x0A. Step size 1.
   *
   * \return None.
   */
    virtual void vAmp_IndSetParamFilter(vdamp_pluginID_t  pluginID, uint8_t ubFilterID, amp_tenParamFilterChar enChar,uint8_t ubOrder, uint16_t uwFreq, int16_t swGain,uint8_t ubQ)    =0;
  /*!
   * \fn void   vAmp_ErrSetParamFilter(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in SetParamFilter state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
    virtual void vAmp_ErrSetParamFilter(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)    =0;
  /*!
   * \fn void vAmp_IndSetGraphFilter(vdamp_pluginID_t  pluginID, uint8_t ubEqID, uint8_t ubBand, int16_t swGain, amp_tenFrequencyIndex enFreqIndex)
   * \brief  Plugin would call this function to indicate set graph filter status
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubEqID If more than one equaliser is present in an instance, this parameter is used to select one of these.
   * \param ubBand Each Band of the Equaliser has one filter that can be controlled. In case of Graphic Equaliser the frequency is constant.
   *               In case of an Parametric-EQ all parameters, including the frequency, can be changed. Step size 1.
   * \param swGain  Range 0xFF88 - 0x0078. Step size 1. Unit: quarter_DB
   * \param enFreqIndex  Frequency index
   *
   * \return None.
   */
    virtual void vAmp_IndSetGraphFilter(vdamp_pluginID_t  pluginID, uint8_t ubEqID, uint8_t ubBand, int16_t swGain, amp_tenFrequencyIndex enFreqIndex)    =0;
  /*!
   * \fn void   vAmpErr_SetGraphFilter(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in SetGraphFilter state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
    virtual void vAmpErr_SetGraphFilter(vdamp_pluginID_t  pluginID,uint8_t ubErrorCode)    =0;
  /*!
   * \fn void vAmp_IndMute(vdamp_pluginID_t  pluginID, amp_tenStream enStream,amp_tenMuteStatus enMuteStatus)
   * \brief  Plugin would call this function to indicate the mute status
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param enStream stream
   * \param enMuteStatus  mute status
   *
   * \return None.
   */
    virtual void vAmp_IndMute(vdamp_pluginID_t  pluginID, amp_tenStream enStream,amp_tenMuteStatus enMuteStatus)    =0;
  /*!
   * \fn void   vAmp_ErrMute(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in mute state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
    virtual void vAmp_ErrMute(vdamp_pluginID_t  pluginID,uint8_t ubErrorCode)    =0;
  /*!
   * \fn void vAmp_IndBeepParam(vdamp_pluginID_t  pluginID, amp_tenBeepID enSignalBeepID, uint8_t ubSignalToneID,uint16_t uwFreq,int16_t swGain)
   * \brief  Plugin would call this function to indicate the Beep param status
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param enSignalBeepID Beep ID
   * \param ubSignalToneID Signal tone ID. Range 0x01 - 0x04. Step size 1.
   * \param uwFreq  Range 0x0014 - 0x4E20. Step size 1. Unit: Hz.
   * \param swGain  Range 0xFF88 - 0x0078. Step size 1. Unit: quarter_DB
   *
   * \return None.
   */
    virtual void vAmp_IndBeepParam(vdamp_pluginID_t  pluginID, amp_tenBeepID enSignalBeepID, uint8_t ubSignalToneID,uint16_t uwFreq,int16_t swGain)    =0;
  /*!
   * \fn void   vAmp_ErrBeepParam(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in setting beep parameter of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
    virtual void vAmp_ErrBeepParam(vdamp_pluginID_t  pluginID,uint8_t ubErrorCode)    =0;
  /*!
   * \fn void vAmp_IndBeep(vdamp_pluginID_t  pluginID, amp_tenBeepID enSignalBeepID, amp_tenBeepState  enBeepState)
   * \brief  Plugin would call this function to indicate the Beep status
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param enSignalBeepID Beep ID
   * \param enBeepState beep state
   *
   * \return None.
   */
    virtual void vAmp_IndBeep(vdamp_pluginID_t  pluginID, amp_tenBeepID enSignalBeepID, amp_tenBeepState  enBeepState)    =0;
  /*!
   * \fn void   vAmp_ErrBeep(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in setting beep state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
    virtual void vAmp_ErrBeep(vdamp_pluginID_t  pluginID,uint8_t ubErrorCode)    =0;

  //vAmp_IndCarSetting is removed

  /*!
   * \fn void   vAmp_ErrCarSetting(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in setting car setting state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   */
    virtual void vAmp_ErrCarSetting(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)= 0;

  /*!
   * \fn void vAmp_IndSurround(vdamp_pluginID_t  pluginID,uint8_t ubMode, int8_t sbLevel, int8_t sbCenterLevel)
   * \brief  Plugin would call this function to indicate the Surround status
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubMode Mode.
   * \param sbLevel Level.
   * \param sbCenterLevel Center Level.
   *
   * \return None.
   */
    virtual void vAmp_IndSurround(vdamp_pluginID_t  pluginID,uint8_t ubMode, int8_t sbLevel, int8_t sbCenterLevel)= 0;
  /*!
   * \fn void   vAmp_ErrSurround(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)
   * \brief  Plugin would call this function to indicate error in Setting surround state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ubErrorCode Error Code
   *
   * \return None.
   *
   */
    virtual void vAmp_ErrSurround(vdamp_pluginID_t  pluginID, uint8_t ubErrorCode)   =0;
  /*!
   * \fn void vAmp_Tx_ToCCALoopback (vdamp_pluginID_t  pluginID, uint8_t *    ptrStream  , size_t datalength)
   * \brief  <B>Context switcher function</B>.
   * Messages from plugin worker threads need to be put to CCA thread before further processing.
   * The plugin transmits the message from worker thread to vd_amp_IF using vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback() first.
   * VD Amp sends the message back on CCA thread by invoking vdamp_plugin_sender::vAmp_Rx_FromCCAThread().
   *
   * \note This function can be called from the threads created by the plugin.
   * \warning VD Amp will not consume the pointer sent to this function. Plugin has to take care of memory de allocation.
   * \param pluginID Plugin ID received from Plugin Manager
   * \param ptrStream pointer to stream of data
   * \param datalength length of the stream in bytes
   *
   * \return None.
   */
  /*!
   * \fn void   vAmp_ConnectionState(vdamp_pluginID_t  pluginID, uint8_t swValue)
   * \brief  Plugin would call this function to indicate state of connected amplifier
   * \note  This function has be to called from the context of CCA Thread of VD Amp. Context switcher has to be used. Refer vdamp_plugin_receiver::vAmp_Tx_ToCCALoopback & vdamp_plugin_sender::vAmp_Rx_FromCCAThread for more info.
   *
   * \param pluginID Plugin ID received from Plugin Manager
   * \param swValue Available settings
   *
   * \return None.
   */
    virtual void vAmp_ConnectionState(vdamp_pluginID_t  pluginID, uint8_t swValue)    =0;
  virtual void vAmp_IndPDCVolOffset(vdamp_pluginID_t  pluginID, int16_t swSteps) =0;
  virtual void vAmp_ErrPDCVolOffset(vdamp_pluginID_t  PluginID, uint8_t ErrorCode) =0;
    virtual void vAmp_Tx_ToCCALoopback (vdamp_pluginID_t  pluginID, uint8_t *    ptrStream, size_t datalength  )   =0;
    virtual void vAmp_SetADRSoundConfigItem(tU8 ubSoundSystem) = 0;
    virtual void vAmp_SetADRSpeakerSoundConfigItem(tU8 ubSpeakerSoundSystem) = 0;
    virtual void vAmp_MainAudioAttenuation(vdamp_pluginID_t  pluginID, int8_t dbValue)    =0;
  virtual void vChangeConfig(tU8 AmplifierConfig) = 0;

};


#endif /* VDAMP_PLUGIN_RECEIVE_H_ */

